// (c) Copyright Microsoft Corporation.
// This source is subject to the Microsoft Public License.
// See http://www.microsoft.com/opensource/licenses.mspx#Ms-PL.
// All other rights reserved.

using System;
using System.Collections.Generic;
using Microsoft.Web.Testing;
using Microsoft.Web.Testing.UI;
using AjaxControlToolkit.Testing.Client;
using AjaxControlToolkit.Testing.Client.Tabs;

/// <summary>
/// Ensure headers are not visible when setting TabPanel.Visible = false
/// </summary>
/// <TestSuite>
///   <Path>Tabs\HeaderVisibility.aspx</Path>
///   <ComponentType>AjaxControlToolkit.Testing.Client.TabContainerBehavior</ComponentType>
///   <WrapperType>AjaxControlToolkit.Testing.Client.Tabs.HeaderVisibilityPage</WrapperType>
/// </TestSuite>
[WebTestClass]
[WebTestTag("Tabs.HeaderVisibility")]
public class Tabs_HeaderVisibility : Tabs_Common
{
    /// <summary>
    /// Verify the expected visibility of the tab headers.
    /// </summary>
    /// <param name="page">Page with the tabs.</param>
    /// <param name="visible">Expected visibility of the second header.</param>
    private void AssertVisible(HeaderVisibilityPage page, bool visible)
    {
        Assert.IsTrue(Exists(page.RootElement, "Tabs_FirstTab_FirstTabHeader"), "Could not find first header!");
        Assert.AreEqual(visible, Exists(page.RootElement, "Tabs_SecondTab_SecondTabHeader"),
            visible ?
                "Could not find second header even though it should be there!" :
                "Found second header even though it should not be there!");
        Assert.IsTrue(Exists(page.RootElement, "Tabs_ThirdTab_ThirdTabHeader"), "Could not find third header!");
    }

    /// <summary>
    /// Verify that invisible headers are not rendered. 
    /// </summary>
    [WebTestMethod]
    public void InvisibleNotInitiallyRendered()
    {
        AssertVisible(new HeaderVisibilityPage(), false);
    }

    /// <summary>
    /// Verify that invisible headers can be restored.
    /// </summary>
    [WebTestMethod]
    public void VisibilityRestored()
    {
        HeaderVisibilityPage page = new HeaderVisibilityPage();
        AssertVisible(page, false);

        page.btnChange.Click(true, true);
        AssertVisible(page, true);
    }

    /// <summary>
    /// Check if a control has been rendered.
    /// </summary>
    /// <param name="page">Page containing the controls.</param>
    /// <param name="id">Id of the control to check for.</param>
    /// <returns>true if rendered, false if not.</returns>
    private bool Exists(HtmlElement element, string id)
    {
        Queue<HtmlElement> remaining = new Queue<HtmlElement>();
        remaining.Enqueue(element);

        while (remaining.Count > 0)
        {
            HtmlElement current = remaining.Dequeue();
            if (current.Id == id)
            {
                return true;
            }

            foreach (HtmlElement child in current.ChildElements)
            {
                remaining.Enqueue(child);
            }
        }

        return false;
    }
}