// (c) Copyright Microsoft Corporation.
// This source is subject to the Microsoft Public License.
// See http://www.microsoft.com/opensource/licenses.mspx#Ms-PL.
// All other rights reserved.

using System;
using System.Collections.Generic;
using System.Globalization;
using Microsoft.Web.Testing;
using Microsoft.Web.Testing.UI;

namespace AjaxControlToolkit.Testing.Client.Calendar
{
    /// <summary>
    /// Common Calendar tests
    /// </summary>
    /// <TestSuite>
    ///   <ComponentType>AjaxControlToolkit.Testing.Client.CalendarBehavior</ComponentType>
    /// </TestSuite>
    [WebTestTag("Calendar")]
    [WebTestTag("Popup")]
    public abstract class Calendar_Common
    {
        /// <summary>
        /// Calendar behaviors
        /// </summary>
        public abstract IEnumerable<CalendarBehavior> Calendars
        {
            get;
        }

        /// <summary>
        /// Focus the calendar
        /// </summary>
        [WebTestMethod]
        public void Common_Focus()
        {
            foreach (CalendarBehavior calendar in Calendars)
            {
                calendar.Show();
            }
        }

        /// <summary>
        /// Blur the calendar
        /// </summary>
        [WebTestMethod]
        public void Common_Blur()
        {
            foreach (CalendarBehavior calendar in Calendars)
            {
                calendar.Show();
                calendar.Hide();
            }
        }

        /// <summary>
        /// Manually set a date in the textbox
        /// </summary>
        [WebTestMethod]
        public void Common_ParseDate()
        {
            foreach (CalendarBehavior calendar in Calendars)
            {
                DateTime date = new DateTime(2000, 1, 15);

                // Set the textbox to a specific date
                calendar.Element.Focus();
                string format = string.IsNullOrEmpty(calendar.Format.Value) ? "d" : calendar.Format.Value;
                calendar.Element.SetText(date.ToString(format, calendar.Culture.DateTimeFormat), false);
                calendar.Element.DispatchEvent(HtmlEventName.Change);
                calendar.Show();

                // Make sure the calendar's selected date matches
                calendar.SelectedDate.Invalidate();
                Assert.AreEqual(date, calendar.SelectedDate.Reference);

                calendar.Hide();
            }
        }

        /// <summary>
        /// Pick a date
        /// </summary>
        [WebTestMethod]
        public virtual void Common_PickDate()
        {
            foreach (CalendarBehavior calendar in Calendars)
            {
                DateTime date = new DateTime(2000, 1, 1);
                int dayToSelect = 12;

                // Set the calendar to the first day in the month
                calendar.SelectedDate.Value = date.ToLongDateString();
                calendar.Show();

                // Click the selected day
                HtmlElement[,] days = calendar.GetDayElements();
                int index = calendar.GetFirstDayOfMonthOffset(date) + dayToSelect - 1;
                days[index / 7, index % 7].Click();

                // Make sure the textbox has the right day
                Assert.AreEqual(
                    date.AddDays(dayToSelect - 1),
                    DateTime.Parse(calendar.Element.GetAttributes().Value, calendar.Culture));
            }
        }
        
        /// <summary>
        /// Verify the dates are correct for the current month
        /// </summary>
        [WebTestMethod]
        public void Common_VerifyCurrentDates()
        {
            foreach (CalendarBehavior calendar in Calendars)
            {
                calendar.AssertCalendarDates(DateTime.Now);
            }
        }

        /// <summary>
        /// Verify a number of months have correct dates
        /// </summary>
        [WebTestMethod]
        [WebTestTag("Stress")]
        public void Common_VerifyDates()
        {
            foreach (CalendarBehavior calendar in Calendars)
            {
                // Leap years
                calendar.AssertCalendarDates(new DateTime(2000, 2, 14));
                calendar.AssertCalendarDates(new DateTime(2000, 1, 1));
                calendar.AssertCalendarDates(new DateTime(2000, 12, 31));
                calendar.AssertCalendarDates(new DateTime(2004, 2, 14));
                calendar.AssertCalendarDates(new DateTime(1900, 2, 14)); // Shouldn't be a leap year

                // DST issue with work item 8531
                calendar.AssertCalendarDates(new DateTime(2007, 10, 14));

                // Y2K
                calendar.AssertCalendarDates(new DateTime(1999, 12, 1));
                calendar.AssertCalendarDates(new DateTime(2000, 1, 1));

                // AMT and KG
                calendar.AssertCalendarDates(new DateTime(1912, 6, 23));
                calendar.AssertCalendarDates(new DateTime(1906, 4, 28));
            }
        }
    }
}