// (c) Copyright Microsoft Corporation.
// This source is subject to the Microsoft Public License.
// See http://www.microsoft.com/opensource/licenses.mspx#Ms-PL.
// All other rights reserved.

using System;
using System.Collections.Generic;
using Microsoft.Web.Testing;
using Microsoft.Web.Testing.UI;

namespace AjaxControlToolkit.Testing.Client.AutoComplete
{
    /// <summary>
    /// Common AutoComplete tests
    /// </summary>
    /// <TestSuite>
    ///   <ComponentType>AjaxControlToolkit.Testing.Client.AutoCompleteBehavior</ComponentType>
    /// </TestSuite>
    [WebTestTag("AutoComplete")]
    [WebTestTag("Popup")]
    public abstract class AutoComplete_Common
    {
        public abstract IEnumerable<AutoCompleteBehavior> AutoCompletes
        {
            get;
        }

        /// <summary>
        /// Make sure that autocomplete respects the min prefix length property.
        /// </summary>
        [WebTestMethod]
        [WebTestFailureTag(Browsers.Opera | Browsers.Safari, "TODO: Fix the way keyboard events are being simulated.")]
        public void Common_CheckFlyoutVisible()
        {
            int minPrefixLength;
            string lessThanMinPrefixLength;
            string equalToMinPrefixLength;
            string greaterThanPrefixLength;
            // reasonably long string; most autocompletes should have minimum prefix lengths in
            // the range of 0-5 before they invoke the webservice to get suggestions
            string randomText = "abW39843Wsfklj133sgsdgsfdg3jwljkWWIOWEHHQOWPOWNHJWOIWRWOI";
            // this test should be invoked only when the the autocomplete webservice is expected to
            // return results no matter what the prefix. If not it will fail.
            foreach (AutoCompleteBehavior autoComplete in AutoCompletes)
            {
                minPrefixLength = autoComplete.MinimumPrefixLength.Value;
                Assert.IsTrue(minPrefixLength < randomText.Length);
                try
                {
                    if (minPrefixLength > 0)
                    {
                        lessThanMinPrefixLength = randomText.Substring(0, minPrefixLength - 1);
                        // will throw because the flyout will not appear and hence the test
                        // should pass.
                        autoComplete.SetTextWaitForFlyout(lessThanMinPrefixLength);
                        Assert.Fail("The flyout should not have showed up");
                    }
                }
                catch (WebTestingException)
                {
                    Assert.AreEqual(0, autoComplete.ActualCompletionSetCount);
                }

                equalToMinPrefixLength = randomText.Substring(0, minPrefixLength);
                autoComplete.SetTextWaitForFlyout(equalToMinPrefixLength);

                greaterThanPrefixLength = randomText.Substring(0, minPrefixLength + 1);
                autoComplete.SetTextWaitForFlyout(greaterThanPrefixLength);
            }
        }

        /// <summary>
        /// Ensure that css styles are correctly applied. Will not work if webservice does
        /// not return at least than two values in response to the string
        /// entered in the textbox.
        /// </summary>
        [WebTestMethod]
        [WebTestTag("Css")]
        public void Common_AutoCompleteStyles()
        {
            int minPrefixLength;
            foreach (AutoCompleteBehavior autoComplete in AutoCompletes)
            {
                minPrefixLength = autoComplete.MinimumPrefixLength.Value;
                string randomText = "abcddddddddddddddddddddddddddd";
                Assert.IsTrue(minPrefixLength < randomText.Length);
                try
                {
                    autoComplete.SetTextWaitForFlyout(randomText);
                }
                catch(WebTestingException)
                {
                    continue;
                }
                if (autoComplete.CompletionListElement.Reference.ChildElements.Count <= 0)
                {
                    continue;
                }
                if (!String.IsNullOrEmpty(autoComplete.CompletionListCssClass.Value))
                {
                    Assert.StringContains(autoComplete.CompletionListElement.Reference.GetAttributes().Class,
                        autoComplete.CompletionListCssClass.Value);
                }
                if (!String.IsNullOrEmpty(autoComplete.CompletionListItemCssClass.Value) && !autoComplete.FirstRowSelected.Value)
                {
                    Assert.StringContains(autoComplete.CompletionListElement.Reference.ChildElements[0].GetAttributes().Class,
                        autoComplete.CompletionListItemCssClass.Value);
                }
                if (!String.IsNullOrEmpty(autoComplete.CompletionListHighlightedItemCssClass.Value) && autoComplete.FirstRowSelected.Value)
                {
                    Assert.StringContains(autoComplete.CompletionListElement.Reference.ChildElements[0].GetAttributes().Class,
                            autoComplete.CompletionListHighlightedItemCssClass.Value);
                }
                else if (!String.IsNullOrEmpty(autoComplete.CompletionListHighlightedItemCssClass.Value))
                {
                    autoComplete.CompletionListElement.Reference.ChildElements[0].DispatchEvent(new HtmlMouseEvent("mouseover")); 
                    Assert.StringContains(autoComplete.CompletionListElement.Reference.ChildElements[0].GetAttributes().Class,
                        autoComplete.CompletionListHighlightedItemCssClass.Value);
                }
                if (!String.IsNullOrEmpty(autoComplete.CompletionListItemCssClass.Value) && (autoComplete.CompletionListElement.Reference.ChildElements.Count > 1))
                {
                    autoComplete.CompletionListElement.Reference.ChildElements[1].DispatchEvent(new HtmlMouseEvent("mouseover"));
                    Assert.StringContains(autoComplete.CompletionListElement.Reference.ChildElements[0].GetAttributes().Class,
                        autoComplete.CompletionListItemCssClass.Value);
                }
            }
        }
    }
}