<?php

class Scanner {
    const WORD         = 1;
    const QUOTE        = 2;
    const APOS         = 3;
    const WHITESPACE   = 6;
    const EOL          = 8;
    const CHAR         = 99;
    const EOF          = 0;

    protected $in;
    protected $line_no = 1;
    protected $char_no = 0;
    protected $token;
    protected $token_type;
    protected $regexps;
    public $resultstack = array();

    function __construct( $in ) {
        $this->in = $in;
        $this->setRegexps();
        $this->nextToken();
        $this->eatWhiteSpace();
    }

    // odłożenie wyniku na stos
    function pushResult( $mixed ) {
        array_push( $this->resultstack, $mixed );
    }

    // zdjęcie wyniku ze stosu
    function popResult( ) {
        return array_pop( $this->resultstack );
    }

    // liczba elementów na stosie
    function resultCount() {
        return count( $this->resultstack );
    }

    // zwraca ostatni (szczytowy) element stosu (ale bez zdejmowania elementu)
    function peekResult( ) {
        if ( empty( $this->resultstack ) ) {
            throw new Exception( "pusty stos" );
        }
        return $this->resultstack[count( $this->resultstack ) -1 ];
    }

    // wyrażenia regularne dla wyrazów
    private function setRegexps() {
        $this->regexps = array( 
                      self::WHITESPACE => '[ \t]', 
                      self::EOL => '\n', 
                      self::WORD => '[a-zA-Z0-9_-]+\b',
                      self::QUOTE => '"',
                      self::APOS => "'",
        );

        $this->typestrings = array( 
                      self::WHITESPACE => 'WHITESPACE', 
                      self::EOL => 'EOL', 
                      self::WORD => 'WORD',
                      self::QUOTE => 'QUOTE',
                      self::APOS => "APOS",
                      self::CHAR  => 'CHAR',
                      self::EOF => 'EOF'
        );
    }

    // pomijanie znaków odstępu
    function eatWhiteSpace( ) {
        $ret = 0;
        if ( $this->token_type != self::WHITESPACE && 
             $this->token_type != self::EOL ) {
            return $ret;
        }
        while ( $this->nextToken() == self::WHITESPACE || 
                $this->token_type == self::EOL ) {
            $ret++;
        }
        return $ret;
    }

    // dla stałej zwraca symbol w postaci ciągu, np.
    // 1 => 'WORD',
    function getTypeString( $int=-1 ) {
        if ( $int<0 ) {
            $int=$this->token_type(); 
        }
        return $this->typestrings[$int];
    }

    // typ bieżącego wyrazu
    function token_type() {
        return $this->token_type;
    }

    // skanowany tekst jest skracany w miarę
    // wyodrębniania kolejnych wyrazów
    function input() {
        return $this->in;
    }

    // bieżący wyraz
    function token() {
        return $this->token;
    }

    // numer bieżącego wiersza (przydatny w komunikatach o błędach)
    function line_no() {
        return $this->line_no;
    }

    // numer bieżącego znaku
    function char_no() {
        return $this->char_no;
    }

    // próba wyodrębnienia kolejnego wyrazu z ciągu wejściowego;
    // w przypadku braku dopasowania wyrazu, wyodrębniany jest znak
    function nextToken() {
        if ( ! strlen( $this->in ) ) {
            return ( $this->token_type = self::EOF );
        }

        $ret = 0;
        foreach ( $this->regexps as $type=>$regex ) {        
            if ( $ret = $this->testToken( $regex, $type ) ) {
                if ( $ret == self::EOL ) {
                    $this->line_no++;
                    $this->char_no = 0;
                } else {
                    $this->char_no += strlen( $this->token() );
                }
                return $ret;
            }
        }
        $this->token = substr( $this->in, 0, 1 );
        $this->in    = substr( $this->in, 1 );
        $this->char_no += 1;
        return ( $this->token_type = self::CHAR );
    }

    // dopasowywanie wyrazów
    private function testToken( $regex, $type ) {
        $matches = array();
        if ( preg_match( "/^($regex)(.*)/s", $this->in, $matches ) ) {
            $this->token = $matches[1];
            $this->in    = $matches[2];
            return ( $this->token_type  = $type );
        }
        return 0;
    }

    // klonowanie obiektu skanera
    function updateToMatch( Scanner $other ) {
        $this->in = $other->in;
        $this->token = $other->token;
        $this->token_type = $other->token_type;
        $this->char_no = $other->char_no;
        $this->line_no = $other->line_no;
        $this->resultstack = $other->resultstack;
    }
}
?>
