<?php
// Klasa do rysowania wykresw umoliwiajca zdefiniowanie konfiguracji
// za pomoc jej publicznych waciwoci.
class MultiGraph {
    // Publiczne skadowe klasy (wszystkie trzeba ustawi przed wykreleniem wykresu).
    public $border_color; // Kolor ta (tekstu).
    public $grid_color;   // Kolor linii siatki.
    public $data_color;   // Kolor blokw danych.
    public $g;            // Obiekt graficzny.
    public $data;         // Dane, na podstawie ktrych bdzie utworzony wykres
    public $draw_border = true;           // Czy naley wykreli obramowanie?
    public $horizontal_grid_segments = 5; // Liczba poziomych linii siatki?
    public $vertical_grid_segments = 0;   // Liczba pionowych linii siatki?
    public $x = 0;    // Wsprzdna X lewego grnego rogu obszaru wykresu.
    public $y = 0;    // Wsprzdna Y.
    public $w = 200;  // Szeroko obszaru wykresu.
    public $h = 100;  // Wysoko obszaru wykresu.
    public $point_size = 5;  // Rozmiar punktw danych w przypadku wykresu punktowego.
    public $y_axis_autolabel = true;  // Czy generowa opisy dla osi Y?
    public $x_axis_autolabel = true;  // Czy generowa opisy dla osi X?
    public $font = 'arial';  // Czcionka do wykorzystania.
    public $fontsize = 8;    // Rozmiar czcionki.

    // Konstruktor. Wymaga przekazania obiektu graficznego.
    public function __construct($graphics) {
        // Zapisanie obiektu graficznego.
        $this->g = $graphics;

        // Wczenie wygadzania (anti-aliasing).
        imageantialias($this->g, true);
    }

    // Metoda, ktra uatwia definiowanie wymiarw.
    public function setDimensions($x, $y, $w, $h) {
        // Ustawienie wymiarw.
        $this->x = $x;
        $this->y = $y;
        $this->h = $h;
        $this->w = $w;
    }

    // Metoda rysujca wykres liniowy na ekranie.
    public function drawLineGraph() {
        // Zadano narysowania wykresu liniowego. Najpierw wykrelamy obramowanie.
        $this->_drawBorder();

        // Obliczenie wspczynnikw konwersji w celu wyskalowania danych w przestrzeni wykresu.
        $yscale = $this->_calcYScale();
        $xscale = $this->_calcXScale();

        // Wykrelenie siatki, jeli tego zadano.
        $this->_drawHorizontal();
        $this->_drawVertical();

        // Mona teraz wykrela dane.
        // Narysujemy wykres liniowy przy zaoeniu, e kluczem s posortowane wartoci X.
        // Ze wzgldu na sortowanie utworzymy kopi, aby nie pomiesza rzeczywistych danych.
        $tmpdata = $this->data;
        ksort($tmpdata);

        // Przetwarzanie danych w ptli i wykrelanie linii.
        list($ldx, $ldy) = each($tmpdata);
        while (list($dx, $dy) = each($tmpdata)) {
            // Wykrelenie linii, poczwszy od ostatniej pary x,y do biecej.
            imageline($this->g,
                $this->x + $ldx * $xscale,
                $this->y + $this->h - $ldy * $yscale,
                $this->x + $dx * $xscale,
                $this->y + $this->h - $dy * $yscale,
                $this->data_color);

            // Zresetowanie danych w celu wykonania nastpnej iteracji.
            $ldx = $dx;
            $ldy = $dy;
        }
    }

    // Metoda rysujca wykres punktowy na ekranie.
    public function drawPointGraph() {
        // Zadano narysowania wykresu punktowego. Najpierw wykrelimy obramowanie.
        $this->_drawBorder();

        // Obliczenie wspczynnikw konwersji w celu wyskalowania danych w przestrzeni wykresu.
        $yscale = $this->_calcYScale();
        $xscale = $this->_calcXScale();

        // Wykrelenie siatki, jeli tego zadano.
        $this->_drawHorizontal();
        $this->_drawVertical();

        // Mona teraz wykrela dane. Przetwarzanie ich w ptli.
        foreach ($this->data as $dx => $dy) {
            // Wykrelenie punktu dla biecych danych.
            imagefilledellipse($this->g, $this->x + $dx * $xscale,
                $this->y + $this->h - $dy * $yscale,
                $this->point_size, $this->point_size, $this->data_color);
        }
    }

    // Metoda rysowania wykresu supkowego.
    // UWAGA:  W przypadku wykresw supkowych zakada si, e indeksy tablicy
    // s etykietami, a nie wsprzdnymi 'x'.
    public function drawBarGraph() {
        // Zadano narysowania wykresu supkowego. Najpierw wykrelimy obramowanie.
        $this->_drawBorder();

        // Obliczenie wspczynnikw konwersji w celu wyskalowania danych w przestrzeni wykresu.
        $yscale = $this->_calcYScale();

        // Wykrelenie siatki, jeli tego zadano.
        $this->_drawHorizontal();

        // Obliczenie kilku wartoci w celu poprawienia wydajnoci.
        // Szeroko kadej 'sekcji'.
        $section = (float) $this->w / count($this->data);
        // Szeroko supka  przyjmujemy, e bdzie miaa warto 70% szerokoci sekcji.
        $bar = (int) ($section * 0.7);

        // Przetwarzanie w ptli punktw danych  zakadamy, e s uporzdkowane.
        $count = 0;
        foreach ($this->data as $label => $val) {
            $count++;

            // Wykrelenie supka do odpowiedniej wysokoci i szerokoci,
            // najpierw w postaci prostokta wypenionego kolorem.
            imagefilledrectangle($this->g,
                $this->x + ($section * $count) - $bar,
                $this->y + $this->h,
                $this->x + ($section * $count),
                $this->y + $this->h - $val * $yscale,
                $this->data_color);

            // Teraz w postaci obrysu.
            imagerectangle($this->g, $this->x + ($section * $count) - $bar,
                $this->y + $this->h, $this->x + ($section * $count),
                $this->y + $this->h - $val * $yscale, $this->border_color);

            // Poniewa w przypadku wykresw supkowych nie istnieje pionowa siatka,
            // musimy samodzielnie utworzy etykiety.
            if ($this->x_axis_autolabel) {
                // Obliczenie szerokoci ramki otaczajcej.
                $box = imagettfbbox($this->fontsize, 270,
                    $this->font, $label);
                $texwidth = abs($box[4] - $box[0]);

                // Wykrelenie tekstu etykiety w pionie.
                imagettftext($this->g,
                    $this->fontsize, 270,
                    ($this->x + ($section * $count)) -
                            ($bar / 2) - ($texwidth / 2),
                    $this->y + $this->h + 4,
                    $this->border_color,
                    $this->font,
                    $label);
            }
        }
    }

    // Funkcje pomocnicze do rysowania obramowania.
    private function _drawBorder() {
        if ($this->draw_border) {
            // Linia pionowa.
            imageline($this->g, $this->x, $this->y, $this->x,
                $this->h + $this->y, $this->border_color);
            // Linia pozioma.
            imageline($this->g, $this->x, $this->h+$this->y,
                $this->w + $this->x, $this->h + $this->y,
                $this->border_color);
        }
    }

    // Funkcja pomocnicza do obliczania skali wsprzdnej Y.
    private function _calcYScale() {
        // Dodanie marginesu ze wzgldw wizualnych, tak by wykres nigdy nie sign grnej krawdzi.
        return (float) $this->h / $this->_calcYMax();
    }

    // Funkcja pomocnicza do obliczania skali wsprzdnej X.
    private function _calcXScale() {
        return (float) $this->w / $this->_calcXMax();
    }

    // Funkcja pomocnicza do obliczania maksymalnej wartoci wsprzdnej Y w celu skalowania itp.
    private function _calcYMax() {
        // Dodanie marginesu ze wzgldw wizualnych, tak by wykres nigdy nie sign grnej krawdzi.
        // Wymuszone zaokrglenie do 2 znaczcych cyfr,
        // aby wykres by przejrzysty.
        $max = (float) max($this->data) * 1.05;
        // Obliczenie 'dugoci' cigu.
        $len = strlen((int)$max);
        // Znalezienie dwucyfrowego zaokrglenia; jeli liczba skada si z mniej ni dwch cyfr  zwrcenie jej.
        if ($len < 2) {
            return $max;
        } else {
            // Zachowanie dwch pierwszych cyfr i uzupenienie zerami.
            return intval(substr($max, 0, 2) . str_repeat('0', $len - 2));
        }
    }

    // Funkcja pomocnicza do obliczania maksymalnej wartoci wsprzdnej X do skalowania itp.
    private function _calcXMax() {
        return max(array_keys($this->data));
    }

    // Funkcja pomocnicza rysujca siatk poziom.
    private function _drawHorizontal() {
        if ($this->horizontal_grid_segments) {
            // Wyznaczenie lokalizacji.
            foreach(range(1, $this->horizontal_grid_segments) as $hg) {
                // Wysoko Y siatki.
                $yheight = (int) $this->y + $this->h -
                    ($hg * ($this->h / $this->horizontal_grid_segments));
                imageline($this->g, $this->x + 1, $yheight,
                    $this->w + $this->x, $yheight, $this->grid_color);

                // Jeli skonfigurowano automatyczne opisy:
                if ($this->y_axis_autolabel) {
                    // Obliczenie wartoci i jej wywietlenie.
                    $ax_step = (int)(($this->_calcYMax() /
                        $this->horizontal_grid_segments) * $hg);

                    // Obliczenie szerokoci ramki potrzebnej do wywietlenia opisu.
                    // Wykorzystanie wsprzdnej X lewego dolnego rogu (0) oraz wsprzdnej X 
                    //  prawego grnego rogu (4) do obliczenia szerokoci:
                    $box = imagettfbbox($this->fontsize, 0,
                        $this->font, $ax_step);
                    $texwidth = abs($box[4] - $box[0]);
                    $texheight = abs($box[5] - $box[1]);

                    // Wywietlenie opisw wyrwnanych do prawej.
                    imagettftext($this->g, $this->fontsize, 0,
                        $this->x - 3 - $texwidth, $yheight + $texheight / 2,
                        $this->border_color, $this->font, $ax_step);
                }
            }
        }
    }

    // Funkcja pomocnicza rysujca siatk pionow.
    private function _drawVertical() {
        if ($this->vertical_grid_segments) {
            // Wyznaczenie lokalizacji.
            foreach(range(1, $this->vertical_grid_segments) as $vg) {
                // Lokalizacja siatki X.
                $xloc = (int) ($this->x +
                    ($vg * ($this->w / $this->vertical_grid_segments)));
                imageline($this->g, $xloc, $this->y,
                    $xloc, $this->y + $this->h - 1, $this->grid_color);

                // Jeli skonfigurowano automatyczne opisy:
                if ($this->x_axis_autolabel) {
                    // Obliczenie wartoci i jej wywietlenie.
                    $ax_step = (int)(($this->_calcXMax() /
                        $this->vertical_grid_segments) * $vg);

                    // Obliczenie szerokoci ramki potrzebnej do wywietlenia opisu.  
                    // Wykorzystanie wsprzdnej X lewego dolnego rogu (0) oraz wsprzdnej X 
                    //  prawego grnego rogu (4) do obliczenia szerokoci.
                    $box = imagettfbbox($this->fontsize, 270,
                        $this->font, $ax_step);
                    $texwidth = abs($box[4] - $box[0]);

                    // Wykrelenie tekstu etykiety w pionie.
                    imagettftext($this->g, $this->fontsize, 270,
                        $xloc - $texwidth / 2, $this->y + $this->h + 3,
                        $this->border_color, $this->font, $ax_step);
                }
            }
        }
    }
}

// Wstpna konfiguracja funkcji przetwarzania dat.
date_default_timezone_set('Europe/Warsaw');

// Utworzenie pustego obrazu do testowania.
$gfx = imagecreatetruecolor(950, 650);

// Zadeklarowanie kilku kolorw.
$red = imagecolorallocate($gfx, 255, 0, 0);
$manilla = imagecolorallocate($gfx, 255, 255, 245);
$black = imagecolorallocate($gfx, 0, 0, 0);
$lgray = imagecolorallocate($gfx, 200, 200, 200);

// Wypenienie kolorem ta.
imagefill($gfx, 0, 0, $manilla);

// Zainicjowanie klasy Multigraph.
$graph = new MultiGraph($gfx);
$graph->border_color = $black;
$graph->grid_color = $lgray;
$graph->data_color = $red;
$graph->vertical_grid_segments = 10;

// Utworzenie przykadowych danych.
// Wygenerowanie do 50 losowych punktw danych.
$chartdata = array();
for ($i = 0; $i < 50; $i++) {
    $chartdata[rand(1,200)] = rand(1,1000);
}
$graph->data = $chartdata;

// Wykrelenie wykresu liniowego.
$graph->setDimensions(50, 50, 300, 200);
$graph->drawLineGraph();

// Wykrelenie wykresu punktowego z kilkoma zmodyfikowanymi parametrami siatki.
$graph->horizontal_grid_segments = 9;
$graph->vertical_grid_segments = 15;
$graph->setDimensions(50, 350, 300, 200);
$graph->drawPointGraph();

// Utworzenie wykresu supkowego. Do jego narysowania potrzebne s nowe dane i opisy.
// Opisy bd wygenerowane automatycznie z wykorzystaniem nazw miesicy.
$bardata = array();
foreach(range(1,12) as $mon) {
    $bardata[date('F', mktime(0, 0, 0, $mon))] = rand(1,5000);
}

// Modyfikacja niektrych opcji i wywietlenie duego wykresu supkowego.
$cyan = imagecolorallocate($gfx, 0, 255, 255);
$graph->data_color = $cyan;
$graph->fontsize = 11;
$graph->horizontal_grid_segments = 12;
$graph->setDimensions(450, 40, 400, 500);
$graph->data = $bardata;
$graph->drawBarGraph();

// Wywietlenie obrazu w formacie PNG
header('Content-type: image/png');
imagepng($gfx);
?>

