<?php
// Biblioteka udostpniajca funkcje do wykonywania oblicze z dugoci i szerokoci geograficzn.

// W niektrych funkcjach do oblicze jest potrzebny promie Ziemi.
// Naley ustawi ponisz sta na warto wyraon w jednostkach, w jakich
// maj by wykonane obliczenia. Warto ta wynosi 6371 km,
// co odpowiada 3956,09 mili.
define('EARTH_R', 6371);

// Funkcja: _deg2rad_multi.
// Opis: Funkcja pomocnicza.  W wielu obliczeniach trzeba dokona konwersji
//  wartoci ze stopni na radiany.
function _deg2rad_multi() {
    // Pobranie wszystkich argumentw w postaci tablicy i wykonanie dla kadego elementu funkcji deg2rad.
    $arguments = func_get_args();
    return array_map('deg2rad', $arguments);
}

// Funkcja: latlon_convert.
// Opis:  funkcja przeksztacajca standardow notacj
//  wsprzdnych na format wymagany przez funkcje w tej bibliotece. 
//   Umoliwia wprowadzanie wsprzdnych w postaci oddzielnie stopni, minut i sekund.
//   Mona rwnie poda wartoci kierunku:  'N', 'S', 'E' lub 'W'. Wszystkie  parametry (z wyjtkiem stopni) s
//   opcjonalne i s danymi typu float.
//   Argumenty mona wprowadzi na przykad jako 77 34 45.5 lub 75 54.45644.
function latlon_convert($degrees, $minutes = 0, $seconds = 0, $dir = '') {
    // Przygotowanie wartoci, w ktrej bdzie zapisany wynik, i dodawanie do niej kolejnych czci.
    $final = $degrees;

    // Dodanie minut i sekund po ich przeksztaceniu na wartoci dziesitne.
    // W obliczeniach wykorzystano fakt, e stopie zawiera 60 minut i 3600 sekund.
    $final += $minutes / 60.0;
    $final += $seconds / 3600.0;

    // Zamiana wartoci na ujemn dla kierunkw zachodniego lub poudniowego,
    // w przypadku gdy kto wprowadzi warto dodatni i kierunek zachodni lub poudniowy.
    if (($dir == 'W') || ($dir == 'S')) {
        $final = abs($final) * -1.0;
    }

    return $final;
}

// Funkcja: latlon_distance_great_circle.
// Opis:  obliczenie najkrtszej odlegoci midzy dwoma parami wsprzdnych.
//  Funkcja oblicza dugo uku po wielkim kole przy zaoeniu, e Ziemia jest
//   idealn kul. Obliczenia zawieraj pewien bd, poniewa Ziemia nie jest
//   idealn kul, ale s do dokadne.
function latlon_distance_great_circle($lat_a, $lon_a, $lat_b, $lon_b) {
    // Konwersja stopni na radiany.
    list($lat1, $lon1, $lat2, $lon2) =
        _deg2rad_multi($lat_a, $lon_a, $lat_b, $lon_b);

    // Podstawienie wartoci do wzoru i zwrcenie wyniku.
    return acos(
            ( sin($lat1) * sin($lat2) ) +
            ( cos($lat1) * cos($lat2) * cos($lon2 - $lon1) )
            ) * EARTH_R;
}

// Funkcja: latlon_bearing_great_circle.
// Opis:  funkcja oblicza pocztkowy azymut, wedug ktrego naley podrowa
//  od punktu A do punktu B wzdu uku wielkiego koa. Wielokrotne wywoania tej
//  funkcji umoliwiaj obliczenie azymutu na kadym etapie podry.
function latlon_bearing_great_circle($lat_a, $lon_a, $lat_b, $lon_b) {
    // Konwersja stopni na radiany.
    list($lat1, $lon1, $lat2, $lon2) =
        _deg2rad_multi($lat_a, $lon_a, $lat_b, $lon_b);

    // Podstawienie wartoci do wzoru i zapisanie wyniku (w radianach).
    $rads = atan2(
            sin($lon2 - $lon1) * cos($lat2),
            (cos($lat1) * sin($lat2)) -
                (sin($lat1) * cos($lat2) * cos($lon2 - $lon1)) );

    // Konwersja radianw na stopnie w celu porwnania z kompasem.
    $degrees = rad2deg($rads);

    // W przypadku wartoci ujemnej odjcie jej od 360 w celu uzyskania azymutu o wartoci, do jakiej jestemy      
    // przyzwyczajeni.
    $degrees = ($degrees < 0) ? 360 + $degrees : $degrees;

    return $degrees;
}

// Funkcja: latlon_distance_rhumb.
// Opis:  oblicza odlego midzy dwoma punktami wzdu loksodromy.
// Loksodroma to linia midzy dwoma punktami prowadzona pod staym
// ktem. Odlegoci wzdu loksodrom s nieco wiksze w porwnaniu z odlegociami po uku wielkiego koa,
// ale pozwalaj na atwiejsz nawigacj.
function latlon_distance_rhumb($lat_a, $lon_a, $lat_b, $lon_b) {
    // Konwersja stopni na radiany.
    list($lat1, $lon1, $lat2, $lon2) =
        _deg2rad_multi($lat_a, $lon_a, $lat_b, $lon_b);

    // Jeli oba punkty le na tym samym poudniku, mamy do czynienia ze specjalnym przypadkiem.
    if ($lat1 == $lat2) {
        $mid = cos($lat1);
    } else {
        $delta = log( tan(($lat2 / 2) + (M_PI / 4))
            / tan(($lat1 / 2) + (M_PI / 4)) );
        $mid = ($lat2 - $lat1) / $delta;
    }

    // Obliczenie rnicy w dugoci geograficznej, a w przypadku wartoci wikszej ni 180, zmiana
    //  kierunku na przeciwny, poniewa bdzie to mniejsza odlego.
    $dlon = abs($lon2 - $lon1);
    $dlon = ($dlon > M_PI) ? (2 * M_PI - $dlon) : $dlon;
    $distance = sqrt( pow($lat2 - $lat1,2) +
                      (pow($mid, 2) * pow($dlon, 2)) ) * EARTH_R;

    return $distance;
}

// Funkcja: latlon_bearing_rhumb.
// Opis: oblicza azymut dla loksodromy czcej dwa punkty.
function latlon_bearing_rhumb($lat_a, $lon_a, $lat_b, $lon_b) {
    // Konwersja stopni na radiany.
    list($lat1, $lon1, $lat2, $lon2) =
        _deg2rad_multi($lat_a, $lon_a, $lat_b, $lon_b);

    // Wykonanie oblicze i zapisanie wynikw w radianach.
    $delta = log( tan(($lat2 / 2) + (M_PI / 4))
                / tan(($lat1 / 2) + (M_PI / 4)) );
    $rads = atan2( ($lon2 - $lon1), $delta);

    // Konwersja radianw na stopnie w celu porwnania z kompasem.
    $degrees = rad2deg($rads);

    // W przypadku wartoci ujemnej odjcie jej od 360 w celu uzyskania azymutu o wartoci, do jakiej jestemy 
        // przyzwyczajeni.
    $degrees = ($degrees < 0) ? 360 + $degrees : $degrees;

    return $degrees;
}

// Przygotowanie do wywietlenia wyniku.
echo '<pre>';

// Wykorzystanie funkcji konwersji w celu przeksztacenia dwch wartoci na format stopni w postaci dziesitnej.
$home_lat = latlon_convert(30, 25.773, 0, 'N');
$home_lon = latlon_convert(77, 06.272, 0, 'W');

// Wywietlenie wartoci. Powinny one mie nastpujc posta: szeroko = 30.42955, 
// dugo = -77.104533333333
echo "Wsprzdne po konwersji:  Szeroko = {$home_lat}, Dugo = {$home_lon}\n\n";

// Przygotowanie kolejnego zbioru wsprzdnych.
$z_lat = 37.318776;
$z_lon = -122.008452;

// Obliczenie odlegoci midzy tymi punktami po uku wielkiego koa: 4176.2068288585 km.
$distance = latlon_distance_great_circle(
                $home_lat, $home_lon, $z_lat, $z_lon);
echo "Odlego po uku wielkiego koa pomidzy punktem 'dom' i 'z': {$distance} km\n\n";

// Obliczenie pocztkowego azymutu dla wielkiego koa: 292.92619009409 stopni.
$bearing = latlon_bearing_great_circle(
                $home_lat, $home_lon, $z_lat, $z_lon);
echo "Azymut po uku wielkiego koa pomidzy punktem 'dom' i 'z': {$bearing} stopni\n\n";

// Obliczenie odlegoci po loksodromie: 4211.1489700467 km.
$rdistance = latlon_distance_rhumb($home_lat, $home_lon, $z_lat, $z_lon);
echo "Odlego po loksodromie pomidzy punktem 'dom' i 'z': {$rdistance} km\n\n";

// Obliczenie azymutu dla tej loksodromy: 280.48113814226 stopni.
$rbearing = latlon_bearing_rhumb($home_lat, $home_lon, $z_lat, $z_lon);
echo "Azymut loksodromy pomidzy punktem 'dom' i 'z': {$rbearing} stopni\n\n";?>
