<?php

declare(strict_types=1);

/**
 * Tworzy i modyfikuje kalendarz wydarzeń
 *
 * PHP w wersji 7
 *
 * LICENCJA: Plik źródłowy udostępniany na licencji MIT opisanej pod adresem
 * http://www.opensource.org/licenses/mit-license.html
 *
 * @author Jason Lengstorf <jason.lengstorf@ennuidesign.com>
 * @copyright 2009 Ennui Design
 * @license http://www.opensource.org/licenses/mit-license.html
 */
class Calendar extends DB_Connect
{

    /**
     * Data dla której powstaje kalendarz
     *
     * Przechowywana w formacie RRRR-MM-DD GG:MM:SS
     *
     * @var string data używana w kalendarzu
     */
    private $_useDate;

    /**
     * Miesiąc, dla którego powstaje kalendarz
     *
     * @var int the używany miesiąc
     */
    private $_m;

    /**
     * Rok, dla którego powstaje kalendarz
     *
     * @var int używany rok
     */
    private $_y;

    /**
     * Używana liczba dni w miesiącu
     *
     * @var int liczba dni w miesiącu
     */
    private $_daysInMonth;

    /**
     * Indeks dnia tygodnia, którym zaczyna się miesiąc (0-6)
     *
     * @var int dzień tygodnia, którym zaczyna się miesiąc
     */
    private $_startDay;

    /**
     * Tworzy obiekt bazy danych i zapisuje odpowiednie dane
     *
     * Pobiera obiekt bazy danych przy tworzeniu instancji klasy
     * i zapisuje go w prywatnej właściwości $_db, o ile nie jest równy null.
     * Jeżeli obiekt ma wartość null, tworzy i zapisuje zamiast niego
     * nowy obiekt PDO.
     *
     * W metodzie są też gromadzone i zapisywane dodatkowe informacje,
     * takie jak miesiąc, dla którego powstaje kalendarz,
     * liczba jego dni, w jakim dniu tygodnia się zaczyna
     * oraz aktualny dzień.
     *
     * @param object $dbo obiekt bazy danych
     * @param string $useDate data, dla której powstaje kalendarz
     * @return void
     */
    public function __construct($db=NULL, $useDate=NULL)
    {
        /*
         * Wywołaj konstruktor klasy nadrzędnej, żeby sprawdzić, czy
         * obiekt bazy danych istnieje
         */
        parent::__construct($db);

        /*
         * Uzyskaj i zapisz dane dotyczące miesiąca
         */
        if ( isset($useDate) )
        {
             $this->_useDate = $useDate;
        }
        else
        {
             $this->_useDate = date('Y-m-d H:i:s');
        }

        /*
         * Konwersja na znacznik czasu oraz wyznaczenie miesiąca
         * i roku, dla którego powstaje kalendarz
         */
        $ts = strtotime($this->_useDate);
        $this->_m = (int)date('m', $ts);
        $this->_y = (int)date('Y', $ts);

        /*
         * Określ liczbę dni w miesiącu
         */
        $this->_daysInMonth = cal_days_in_month(
                CAL_GREGORIAN,
                $this->_m,
                $this->_y
            );

        /*
         * Określ dzień tygodnia, jakim zaczyna się miesiąc
         */
        $ts = mktime(0, 0, 0, $this->_m, 1, $this->_y);
        $this->_startDay = date('w', $ts);
    }

    /**
     * Potwierdza i wykonuje operację usunięcia wydarzenia
     * 
     * Po kliknięciu przycisku, aby usunąć wydarzenie,
     * generuje okno z potwierdzeniem.
     * Jeżeli użytkownik potwierdzi,
     * usuwa wydarzenie z bazy danych i przekierowuje
     * go z powrotem do kalendarza. Jeżeli użytkownik
     * postanowi nie usuwać wydarzenia, przekierowuje go z powrotem do
     * kalendarza, nie usuwając wydarzenia.
     *
     * @param int $id identyfikator wydarzenia
     * @return mixed formularz w przypadku potwierdzania, brak wartości lub komunikat o błędzie w przypadku usuwania
     */
    public function confirmDelete($id)
    {
        /*
         * Upewnij się, że został przekazany identyfikator
         */
        if ( empty($id) ) { return NULL; }

        /*
         * Upewnij się, że identyfikator jest liczbą całkowitą
         */
        $id = preg_replace('/[^0-9]/', '', $id);

        /*
         * Jeżeli wysłano formularz potwierdzenia zawierający
         * poprawny token, sprawdź formularz
         */
        if ( isset($_POST['confirm_delete'])
                && $_POST['token']==$_SESSION['token'] )
        {
            /*
             * Jeżeli potwierdzono usunięcie, usuń wydarzenie
             * z bazy danych
             */
            if ( $_POST['confirm_delete']=="Tak, usuń wydarzenie" )
            {
                $sql = "DELETE FROM `events`
                        WHERE `event_id`=:id
                        LIMIT 1";
                try
                {
                    $stmt = $this->db->prepare($sql);
                    $stmt->bindParam(
                          ":id",
                          $id,
                          PDO::PARAM_INT
                        );
                    $stmt->execute();
                    $stmt->closeCursor();
                    header("Location: ./");
                    return;
                }
                catch ( Exception $e )
                {
                    return $e->getMessage();
                }
            }

            /*
             * W przypadku braku potwierdzona, przekieruj użytkownika do widoku głównego
             */
            else
            {
                header("Location: ./");
                return;
            }
        }

        /*
         * Jeżeli formularz potwierdzenia nie został wysłany, wyświetl go
         */
        $event = $this->_loadEventById($id);

        /*
         * Jeżeli nie ma obiektu do zwrócenia, wróć do widoku głównego
         */
        if ( !is_object($event) ) { header("Location: ./"); }

        return <<<CONFIRM_DELETE

    <form action="confirmdelete.php" method="post">
        <h2>
            Czy na pewno chcesz usunąć "$event->title"?
        </h2>
        <p>Tej operacji <strong>nie można cofnąć</strong>.</p>
        <p>
            <input type="submit" name="confirm_delete"
                  value="Tak, usuń wydarzenie" />
            <input type="submit" name="confirm_delete"
                  value="Nie! Tylko żartowałem!" />
            <input type="hidden" name="event_id"
                  value="$event->id" />
            <input type="hidden" name="token"
                  value="$_SESSION[token]" />
        </p>
    </form>
CONFIRM_DELETE;
    }

    /**
     * Ładuje informacje o wydarzeniu (lub wydarzeniach) do tablicy
     *
     * @param int $id opcjonalny identyfikator wydarzenia do filtrowania wyników
     * @return array tablica wydarzeń z bazy danych
     */
    private function _loadEventData($id=NULL)
    {
        $sql = "SELECT
                    `event_id`, `event_title`, `event_desc`,
                    `event_start`, `event_end`
                FROM `events`";

        /*
         * Jeżeli podano identyfikator wydarzenia, dodaj klauzulę WHERE,
         * aby zwrócone zostało tylko to wydarzenie
         */
        if ( !empty($id) )
        {
            $sql .= "WHERE `event_id`=:id LIMIT 1";
        }

        /*
         * W przeciwnym razie załaduj wszystkie wydarzenia z danego miesiąca
         */
        else
        {
            /*
             * Wyznacz pierwszy i ostatni dzień miesiąca
             */
            $start_ts = mktime(0, 0, 0, $this->_m, 1, $this->_y);
            $end_ts = mktime(23, 59, 59, $this->_m+1, 0, $this->_y);
            $start_date = date('Y-m-d H:i:s', $start_ts);
            $end_date = date('Y-m-d H:i:s', $end_ts);

            /*
             * Odfiltruj wydarzenia odbywające się w
             * aktualnie wybranym miesiącu
             */
            $sql .= "WHERE `event_start`
                        BETWEEN '$start_date'
                        AND '$end_date'
                    ORDER BY `event_start`";
        }

        try
        {
            $stmt = $this->db->prepare($sql);

            /*
             * Powiąż parametr z identyfikatorem, o ile został przekazany
             */
            if ( !empty($id) )
            {
                $stmt->bindParam(":id", $id, PDO::PARAM_INT);
            }

            $stmt->execute();
            $results = $stmt->fetchAll(PDO::FETCH_ASSOC);
            $stmt->closeCursor();

            return $results;
        }
        catch ( Exception $e )
        {
            die ( $e->getMessage() );
        }
    }

    /**
     * Składuje wszystkie wydarzenia dla miesiąca w tablicy
     *
     * @return array Informacje o wydarzeniach
     */
    private function _createEventObj()
    {
        /*
         * Pobierz tablicę wydarzeń
         */
        $arr = $this->_loadEventData();

        /*
         * Utwórz nową tablicę, a następnie pogrupuj wydarzenia
         * według dnia miesiąca, w którym się odbywają
         */
        $events = array();
        foreach ( $arr as $event )
        {
            $day = date('j', strtotime($event['event_start']));

            try
            {
                $events[$day][] = new Event($event);
            }
            catch ( Exception $e )
            {
                die ( $e->getMessage() );
            }
        }
        return $events;
    }

    /**
     * Zwraca kod HTML kalendarza i wydarzeń
     * 
     * Na podstawie informacji zawartych we właściwościach,
     * pobiera wydarzenia dla danego miesiąca, generuje
     * kalendarz i zwraca całość w postaci poprawnego kodu HTML.
     * 
     * @return string kod HTML kalendarza
     */
    public function buildCalendar()
    {
        /*
         * Wyznacz miesiąc i utwórz tablicę
         * skróconych nazw dni tygodnia do oznaczenia kolumn kalendarza
         */
        $cal_month = date('F Y', strtotime($this->_useDate));
        define('WEEKDAYS', array('Nd', 'Pn', 'Wt', 'Śr', 'Cz', 'Pt', 'So'));

        /*
         * Dodaj nagłówek w kodzie kalendarza
         */
        $html = "\n\t<h2>$cal_month</h2>";
        for ( $d=0, $labels=NULL; $d<7; ++$d )
        {
            $labels .= "\n\t\t<li>" . WEEKDAYS[$d] . "</li>";
        }
        $html .= "\n\t<ul class=\"weekdays\">"
            . $labels . "\n\t</ul>";

        /*
         * Pobierz dane wydarzeń
         */
        $events = $this->_createEventObj();

        /*
         * Generuj kod kalendarza
         */
        $html .= "\n\t<ul>"; // Początek nowej listy nieuporządkowanej
        for ( $i=1, $c=1, $t=date('j'), $m=date('m'), $y=date('Y');
                $c<=$this->_daysInMonth; ++$i )
        {
            /*
             * Dodaj klasę "fill" do pól przed
             * pierwszym dniem miesiąca
             */
            $class = $i<=$this->_startDay ? "fill" : NULL;

            /*
             * Dodaj klasę "today", jeśli bieżąca data odpowiada
             * dzisiejszej
             */
            if ( $c==$t && $m==$this->_m && $y==$this->_y )
            {
                $class = "today";
            }

            /*
             * Generuj znaczniki otwierające i zamykające element listy
             */
            $ls = sprintf("\n\t\t<li class=\"%s\">", $class);
            $le = "\n\t\t</li>";
            $event_info = NULL;
            
            /*
             * Dodaj dzień miesiąca w polu kalendarza
             */
            if ( $this->_startDay<$i && $this->_daysInMonth>=$c)
            {
                /*
                 * Formatuj dane wydarzenia
                 */
                //$event_info = NULL; // wyczyść zmienną
                if ( isset($events[$c]) )
                {
                    foreach ( $events[$c] as $event )
                    {
                        $link = '<a href="view.php?event_id='
                                . $event->id . '">' . $event->title
                                . '</a>';
                        $event_info .= "\n\t\t\t$link";
                    }
                }

                $date = sprintf("\n\t\t\t<strong>%02d</strong>",$c++);
            }
            else { $date="&nbsp;"; }

            /*
             * Jeśli bieżący dzień to sobota, zacznij nowy wiersz
             */
            $wrap = $i!=0 && $i%7==0 ? "\n\t</ul>\n\t<ul>" : NULL;

            /*
             * Generuj gotowy element listy
             */
            $html .= $ls . $date . $event_info . $le . $wrap;
        }

        /*
         * Wypełnij ostatni tydzień do końca
         */
        while ( $i%7!=1 )
        {
            $html .= "\n\t\t<li class=\"fill\">&nbsp;</li>";
            ++$i;
        }

        /*
         * Zamknij ostatnią listę nieuporządkowaną
         */
        $html .= "\n\t</ul>\n\n";

        /*
         * Wyświetl opcje administratora, jeżeli jest zalogowany
         */
        $admin = $this->_adminGeneralOptions();

        /*
         * Zwróć wygenerowany kod
         */
        return $html . $admin;
    }

    /**
     * Wyświetla informacje o danym wydarzeniu
     * 
     * @param int $id identyfikator wydarzenia
     * @return string kod HTML do wyświetlenia informacji o wydarzeniu
     */
    public function displayEvent($id)
    {
        /*
         * Upewnij się, że został przekazany identyfikator
         */
        if ( empty($id) ) { return NULL; }

        /*
         * Upewnij się, że identyfikator jest liczbą całkowitą
         */
        $id = preg_replace('/[^0-9]/', '', $id);

        /*
         * Pobierz informacje o wydarzeniach z bazy danych
         */
        $event = $this->_loadEventById($id);

        /*
         * Generuj łańcuchy daty, czasu rozpoczęcia i czasu zakończenia
         */
        $ts = strtotime($event->start);
        $date = date('F d, Y', $ts);
        $start = date('g:ia', $ts);
        $end = date('g:ia', strtotime($event->end));

        /*
         * Wyświetl opcje administratora, jeżeli użytkownik jest zalogowany
         */
        $admin = $this->_adminEntryOptions($id);

        /*
         * Generuj i zwróć kod HTML
         */
        return "<h2>$event->title</h2>"
            . "\n\t<p class=\"dates\">$date, $start&mdash;$end</p>"
            . "\n\t<p>$event->description</p>$admin";
    }    

    /**
     * Generuje formularz do tworzenia i edycji wydarzeń
     * 
     * @return string kod HTML formularza edycji
     */
    public function displayForm()
    {
        /*
         * Sprawdź, czy został przekazany identyfikator
         */
        if ( isset($_POST['event_id']) )
        {
            $id = (int)$_POST['event_id']; // Rzutuj na typ całkowity, aby zapewnić poprawność danych
        }
        else
        {
            $id = NULL;
        }

        /*
         * Inicjalizuj nagłówek i tekst przycisku
         */
        $submit = "Utwórz nowe wydarzenie";

        /*
         * W przypadku braku identyfikatora, utwórz pusty obiekt wydarzenia.
         */
        $event = new Event();

        /*
         * W przeciwnym razie załaduj odpowiednie wydarzenie
         */
        if ( !empty($id))
        {
            $event = $this->_loadEventById($id);

            /*
             * Jeżeli nie ma obiektu, zwróć NULL
             */
            if ( !is_object($event) ) { return NULL; }

            $submit = "Edytuj to wydarzenie";
        }

        /*
         * Generuj kod
         */
        return <<<FORM_MARKUP

    <form action="assets/inc/process.inc.php" method="post">
        <fieldset>
            <legend>$submit</legend>
            <label for="event_title">Nazwa wydarzenia</label>
            <input type="text" name="event_title"
                  id="event_title" value="$event->title" />
            <label for="event_start">Czas rozpoczęcia</label>
            <input type="text" name="event_start"
                  id="event_start" value="$event->start" />
            <label for="event_end">Czas zakończenia</label>
            <input type="text" name="event_end"
                  id="event_end" value="$event->end" />
            <label for="event_description">Opis wydarzenia</label>
            <textarea name="event_description"
                  id="event_description">$event->description</textarea>
            <input type="hidden" name="event_id" value="$event->id" />
            <input type="hidden" name="token" value="$_SESSION[token]" />
            <input type="hidden" name="action" value="event_edit" />
            <input type="submit" name="event_submit" value="$submit" />
            lub <a href="./">anuluj</a>
        </fieldset>
    </form>
FORM_MARKUP;
    }

    /**
     * Sprawdza poprawność formularza i zapisuje nowe wydarzenie lub edytuje istniejące
     * 
     * @return mixed TRUE w przypadku powodzenia, komunikat o błędzie w przypadku niepowodzenia
     */
    public function processForm()
    {
        /*
         * Zakończ, jeżeli parametr action ma nieprawidłową wartość
         */
        if ( $_POST['action']!='event_edit' )
        {
            return "Nieprawidłowe użycie metody processForm";
        }

        /*
         * Przygotowanie danych do zapisu w bazie
         */
        $title = htmlentities($_POST['event_title'], ENT_QUOTES);
        $desc = htmlentities($_POST['event_description'], ENT_QUOTES);
        $start = htmlentities($_POST['event_start'], ENT_QUOTES);
        $end = htmlentities($_POST['event_end'], ENT_QUOTES);

        /*
         * W przypadku braku identyfikatora wydarzenia, utwórz nowe
         */
        if ( empty($_POST['event_id']) )
        {
            $sql = "INSERT INTO `events`
                        (`event_title`, `event_desc`, `event_start`,
                            `event_end`)
                    VALUES
                        (:title, :description, :start, :end)";
        }

        /*
         * Aktualizuj wydarzenie, jeżeli jest edytowane
         */
        else
        {
            /*
             * Dla bezpieczeństwa rzutuj identyfikator wydarzenia na typ całkowity
             */
            $id = (int)$_POST['event_id'];
            $sql = "UPDATE `events`
                    SET
                        `event_title`=:title,
                        `event_desc`=:description,
                        `event_start`=:start,
                        `event_end`=:end
                    WHERE `event_id`=$id";
        }

        /*
         * Wykonaj zapytanie po podstawieniu wartości w miejsce parametrów
         */
        try
        {
            $stmt = $this->db->prepare($sql);
            $stmt->bindParam(":title", $title, PDO::PARAM_STR);
            $stmt->bindParam(":description", $desc, PDO::PARAM_STR);
            $stmt->bindParam(":start", $start, PDO::PARAM_STR);
            $stmt->bindParam(":end", $end, PDO::PARAM_STR);
            $stmt->execute();
            $stmt->closeCursor();
            return TRUE;
        }
        catch ( Exception $e )
        {
            return $e->getMessage();
        }
    }

    /**
     * Zwraca pojedynczy obiekt wydarzenia
     * 
     * @param int $id identyfikator wydarzenia
     * @return object obiekt wydarzenia
     */
    private function _loadEventById($id)
    {
        /*
         * Jeśli nie przekazano identyfikatora, zwróć wartość NULL
         */
        if ( empty($id) )
        {
            return NULL;
        }

        /*
         * Pobierz wydarzenie i zapisz je w tablicy
         */
        $event = $this->_loadEventData($id);

        /*
         * Zwróć obiekt wydarzenia
         */
        if ( isset($event[0]) )
        {
            return new Event($event[0]);
        }
        else
        {
            return NULL;
        }
    }

    /**
     * Generuje kod HTML opcji administracyjnych
     * 
     * @return string kod HTML opcji administracyjnych
     */
    private function _adminGeneralOptions()
    {
        /*
         * Wyświetl opcje administratora
         */
        return <<<ADMIN_OPTIONS

    <a href="admin.php" class="admin">+ Dodaj nowe wydarzenie</a>
ADMIN_OPTIONS;
    }

    /**
     * Generuje opcje edycji i usuwania dla danego identyfikatora wydarzenia
     * 
     * @param int $id identyfikator wydarzenia, dla którego mają zostać wygenerowane opcje
     * @return string kod HTML opcji edycji i usuwania
     */
    private function _adminEntryOptions($id)
    {
        return <<<ADMIN_OPTIONS

    <div class="admin-options">
    <form action="admin.php" method="post">
        <p>
            <input type="submit" name="edit_event"
                  value="Edytuj to wydarzenie" />
            <input type="hidden" name="event_id"
                  value="$id" />
        </p>
    </form>
    <form action="confirmdelete.php" method="post">
        <p>
            <input type="submit" name="delete_event"
                  value="Usuń to wydarzenie" />
            <input type="hidden" name="event_id"
                  value="$id" />
        </p>
    </form>
    </div><!-- end .admin-options -->
ADMIN_OPTIONS;
    }

}

?>