#!/usr/bin/perl -w
# plik: chat_client.pl
# klient pogawedki wykorzystujcy protokol UDP
use strict;
use IO::Socket;
use IO::Select;
use ChatObjects::ChatCodes;
use ChatObjects::Comm;

$SIG{INT} = $SIG{TERM} = sub { exit 0 };
my ($nickname,$server);

 # tablica rozdzielcza dla polecen uzytkownika
 my %COMMANDS = (
  channels=> sub { $server->send_event(LIST_CHANNELS)},
  join=> sub { $server->send_event(JOIN_REQ,shift)},
  part=> sub { $server->send_event(PART_REQ,shift)},
  users=> sub { $server->send_event(LIST_USERS)},
  public => sub { $server->send_event(SEND_PUBLIC,shift)},
  private=> sub { $server->send_event(SEND_PRIVATE,shift) },
  login=> sub { $nickname = do_login()},
  quit=> sub { undef },
 );

 # tablica rozdzielcza dla komunikatow z serwera
 my %MESSAGES = (
  ERROR()=> \&error,
  LOGIN_ACK() => \&login_ack,
  JOIN_ACK()=> \&join_part,
  PART_ACK()=> \&join_part,
  PUBLIC_MSG()=> \&public_msg,
  PRIVATE_MSG()=> \&private_msg,
  USER_JOINS()=> \&user_join_part,
  USER_PARTS()=> \&user_join_part,
  CHANNEL_ITEM() => \&list_channel,
  USER_ITEM() => \&list_user,
  );

 # Utworz oraz inicjalizuj gniazdo UDP
 my $servaddr = shift || 'localhost';
 my $servport = shift || 2027;
 $server = ChatObjects::Comm->new(PeerAddr=> "$servaddr:$servport") or die $@;

 # Proba rejestracji do systemu
 $nickname = do_login();
 die "Nie mozna zarejestrowac.\n" unless $nickname;

 # Czytaj polecenia od uzytkownika i komunikaty z serwera
 my $select = IO::Select->new($server->socket,\*STDIN);
 LOOP:
 while (1) {
 my @ready = $select->can_read;
 foreach (@ready) {
  if ($_ eq \*STDIN) {
 do_user(\*STDIN) || last LOOP;
  } else {
 do_server($_);
  }
 }
 }

 # wywolanie obslugi polecenia otrzymanego od uzytkownika
 sub do_user {
 my $h = shift;
 my $data;
 returnunless sysread($h,$data,1024);# najdluzszy wiersz
 return 1 unless $data =~ /\S+/;
 chomp($data);
 my($command,$args) = $data =~ m!^/(\S+)\s*(.*)!;
 ($command,$args) = ('public',$data) unless $command;
 my $sub = $COMMANDS{lc $command};
 return warn $command: nieznana komenda\n unless $sub;
 return $sub->($args);
 }

 # wywolanie obslugi komunikatu otrzymanego z serwera
 sub do_server {
 die "zle gniazdo" unless my $s = ChatObjects::Comm->sock2server(shift);
 die "nie mozna odebrac: $!" unless 
  my ($mess,$args) = $s->recv_event;
 my $sub = $MESSAGES{$mess} || return warn "$mess: nieznany komunikat serwera\n";
 $sub->($mess,$args);
 return $mess;
 }

 # proba rejestracji do systemu (ponawiana)
 sub do_login {
 $server->send_event(LOGOFF,$nickname) if $nickname;
 my $nick = get_nickname();# pobierz od uzytkownika
 my $select = IO::Select->new($server->socket);
 for (my $count=1; $count <= 5; $count++) {
  warn "proba rejestracji ($count)...\n";
  $server->send_event(LOGIN_REQ,$nick);
  next unless $select->can_read(6);
  return $nick if do_server($server->socket) == LOGIN_ACK;
  $nick = get_nickname();
 }

 }

 # monit do uzytkownika o podanie pseudonimu
 sub get_nickname {
 while (1) {
  local $| = 1;
  print "Podaj pseudonim: ";
  last unless defined(my $nick = <STDIN>);
  chomp($nick);
  return $nick if $nick =~ /^\S+$/;
  warn "Niepoprawny pseudonim.Nie moze zawierac spacji.\n";
 }
 }

# obsluga komunikatu serwera o bledzie
sub error {
my ($code,$args) = @_;
print "\t** BLAD: $args **\n";
}

# obsluga potwierdzenia serwera rejestracji w systemie
sub login_ack {
my ($code,$nickname) = @_;
print "\tUdana rejestracja w systemie.Witaj $nickname.\n";
}

# obsluga komunikatow serwera potwierdzajacych przylaczenie do kanalu lub odlaczenie od kanalu
sub join_part {
my ($code,$msg) = @_;
my ($title,$users) = $msg =~ /^(\S+) (\d+)/;
print $code == JOIN_ACK 
 ? "\tWitaj w kanale $title($users uczestnikow)\n"
 : "\tOpusiciles kanal $title \n";
}

# obsluga wydruku wykazu kanalow z serwera
sub list_channel {
my ($code,$msg) = @_;
my ($title,$count,$description) = $msg =~ /^(\S+) (\d+) (.+)/;
printf "\t%-20s %-40s %3d uczestnikow\n","[$title]",$description,$count;
}

# obsluga komunikatu publicznego z serwera
sub public_msg {
my ($code,$msg) = @_;
my ($channel,$user,$text) = $msg =~ /^(\S+) (\S+) (.*)/;
print "\t$user [$channel]: $text\n";
}

# obsluga komunikatu prywatnego z serwera
sub private_msg {
my ($code,$msg) = @_;
my ($user,$text) = $msg =~ /^(\S+) (.*)/;
print "\t$user [**prywatnie**]: $text\n";
}

# obsluga komunikatow z serwera dla uzytkownika o przylaczeniu do kanalu lub odlaczeniu od kanalu
sub user_join_part {
my ($code,$msg) = @_;
my $verb = $code == USER_JOINS ? 'rozpoczyna pogawedke' : 'konczy pogawedke';
my ($channel,$user) = $msg =~ /^(\S+) (\S+)/;
print "\t<$user $verb $channel>\n";
}

# obsluga komunikatow z serwera zawierajacych informacje o uzytkownikach
sub list_user {
my ($code,$msg) = @_;
my ($user,$timeon,$channels) = $msg =~ /^(\S+) (\d+) (.+)/;
my ($hrs,$min,$sec) = format_time($timeon);
printf "\t%-15s (on %02d:%02d:%02d) Kanaly: %s\n",$user,$hrs,$min,$sec,$channels;
}
# czytelne formatowanie czasu(godz, min sek)
sub format_time {
my $sec = shift;
my $hours = int( $sec/(60*60) );
$sec-= ($hours*60*60);
my $min= int( $sec/60 );
$sec-= ($min*60);
return ($hours,$min,$sec);
}

END {
if (defined $server) {
 $server->send_event(LOGOFF,$nickname);
 $server->close;
}
}