 #include "minunit.h"
 #include <lcthw/radixmap.h>
 #include <time.h>

 static int make_random(RadixMap * map)
 {
     size_t i = 0;

     for (i = 0; i < map->max - 1; i++) {
         uint32_t key = (uint32_t) (rand() | (rand() << 16));
         check(RadixMap_add(map, key, i) == 0, "Nie udało się dodać klucza %u.",
             key);
     }

     return i;

 error:
     return 0;
 }

 static int check_order(RadixMap * map)
 {
     RMElement d1, d2;
     unsigned int i = 0;

     // Sygnalizuj błędy tylko wtedy, gdy wystąpią (nie powinno ich być).
     for (i = 0; map->end > 0 && i < map->end - 1; i++) {
         d1 = map->contents[i];
         d2 = map->contents[i + 1];

         if (d1.data.key > d2.data.key) {
             debug("NIEPOWODZENIE:i=%u, klucz: %u, wartość: %u, osiągnęły wartość maksymalną? %d\n", i,
                 d1.data.key, d1.data.value,
                 d2.data.key == UINT32_MAX);
             return 0;
         }
     }

     return 1;
 }

 static int test_search(RadixMap * map)
 {
     unsigned i = 0;
     RMElement *d = NULL;
     RMElement *found = NULL;

     for (i = map->end / 2; i < map->end; i++) {
         d = &map->contents[i];
         found = RadixMap_find(map, d->data.key);
         check(found != NULL, "Nie znaleziono %u w %u.", d->data.key, i);
         check(found->data.key == d->data.key,
             "Otrzymano nieprawidłowy wynik: %p:%u, wyszukując %u w %u", found,
             found->data.key, d->data.key, i);
     }

     return 1;
 error:
     return 0;
 }

 // Sprawdzenie działania w przypadku bardzo dużej liczby elementów.
 static char *test_operations()
 {
     size_t N = 200;

     RadixMap *map = RadixMap_create(N);
     mu_assert(map != NULL, "Nie udało się utworzyć mapy.");
     mu_assert(make_random(map), "Nie udało się utworzyć losowo wybranej, nieprawdziwej mapy radix.");

     RadixMap_sort(map);
     mu_assert(check_order(map),
         "Nie udało się prawidłowo posortować RadixMap.");

     mu_assert(test_search(map), "Test wyszukiwania zakończył się niepowodzeniem.");
     mu_assert(check_order(map),
         "Mapa RadixMap nie jest posortowana po operacji wyszukiwania.");

     while (map->end > 0) {
         RMElement *el = RadixMap_find(map,
             map->contents[map->end / 2].data.key);
         mu_assert(el != NULL, "Powinien być otrzymany wynik.");

         size_t old_end = map->end;

         mu_assert(RadixMap_delete(map, el) == 0, "Nie usunięto.");
         mu_assert(old_end - 1 == map->end, "Nieprawidłowa wielkość po usunięciu.");

         // Sprawdzenie, czy na końcu znajduje się teraz stara wartość,
         // natomiast wartość wynosi UINT32_MAX, więc element jest usuwany.
         mu_assert(check_order(map),
             "Mapa RadixMap nie jest posortowana po operacji usunięcia."P
     }

     RadixMap_destroy(map);

     return NULL;
 }

 char *all_testsP
 {
     mu_suite_start();
     srand(time(NULL));

     mu_run_test(test_operations);

     return NULL;
 }

 RUN_TESTS(all_tests);
