/**
*  @fileoverview This is a file to describe the draggable Draggable object.  If the HTML content of the 
*  block is larger than the block, scrollbars will appear to allow the viewing of all of the block contents.
*  @author Lee S. Barney
*  @version 0.1
*/


/**
 * Construct a new Draggable object.
 * @class This is the Draggable class.  
 * It is used to represent and draw draggable entities
 * @constructor
 * @param {int} top The pixel location of the top of the block
 * @param {int} left The pixel location of the left side of the block
 * @param {int} width The pixel width of the block
 * @param {int} height The pixel height of the block
 * @param {String} color (Optional)The background color of the display.  If no color is declared the div will be transparent.
 * @param {HTMLElement} parent The container element for this block
 * @return A new Draggable
 */


function Draggable(width, height, color, parent, zIndex){
   if(!parent || !width || !height){
     return;
   }
	/**
	 * stores the previous x location of the mouse during a drag of the block.
	 * @type int
	 */
  this.oldMouseX = -1;
  /**
	 * stores the previous y location of the mouse during a drag of the block.
	 * @type int
	 */
  this.oldMouseY = -1;/**
	 * indicates that the the block is to be drug when the mouse move event fires.
	 * @type boolean
	 */
  this.isDragging = false;
  	/**
	 * stores the top location in pixels of the block.
	 * @type int
	 */
  this.width = width;
  	/**
	 * stores the height in pixels of the block.
	 * @type int
	 */
  this.height = height;
  
  	/**
  	 * the div that allows the block object to be displayed
	 * @type HTMLDivElement
	 */
  this.display = document.createElement('div');
  	
  this.display.style.width = width+'px';
  	
  this.display.style.height = height+'px';
  	
  if(zIndex){
  	this.display.style.zIndex = zIndex;
  }
  else{
    this.display.style.zIndex = 0;
  }
  	
  if(parent.style && parent.style.zIndex){
  	this.parentZIndex = parent.style.zIndex;
  }
  else{
    this.parentZIndex = 0;
  }
  	
  if(color){
  	this.display.style.backgroundColor = color ;
  }
  	
  this.display.style.overflow = 'show' ;
  	
  this.display.style.cursor = 'pointer' ;
  	
  this.display.draggable = this;
  
  this.parent = parent;
  parent.appendChild(this.display);
  /*
  * Set the onmousemove for the window so that if the mouse leaves the draggable's display while dragging, the display will catch up to the mouse.
  */
  window.onmousemove = this.display.onmousemove;


  	//startDragging method
    this.display.onmousedown = function(event){
        var target = event.target || event.sourceElement;
        var aDraggable = target.draggable;
        if(aDraggable.whenStartDragging){
          aDraggable.whenStartDragging(aDraggable.whenStartDraggingParams);
        }
        target.style.position = "relative";
        target.draggable.left = target.offsetLeft;
        target.draggable.top = target.offsetTop;
        
        window.draggable = aDraggable;
        aDraggable.isDragging = true;
        target.bringToFront();
        aDraggable.oldMouseX = event.screenX;
        aDraggable.oldMouseY = event.screenY;
    };
    
  	//done draggin method
  this.display.onmouseup = function(event){
        var target = event.target || event.sourceElement;
        var aDraggable = target.draggable;
        if(aDraggable.whenDropped){
            aDraggable.whenDropped(aDraggable.whenDroppedParams);
        }
        target.sendBackDown();
        window.draggable = null;
        aDraggable.isDragging = false;
    };
    
  	//while dragging method
  this.display.onmousemove = function(event){//if the mouse leaves the block, this event will be fired by the window as well.  If the mouse leaves the browser no event will be fired.
        var target = event.target || event.sourceElement;
        var aDraggable = target.draggable;
        document.getElementById('messageDiv').innerHTML += 'isDragging<br/>'
        if( aDraggable != null && aDraggable.isDragging){
            document.getElementById('messageDiv').innerHTML += "left: "+aDraggable.left
                +" xdiff: "+(event.screenX - aDraggable.oldMouseX)
                +" top: "+aDraggable.top+" ydiff: "+(event.screenY - aDraggable.oldMouseY)+'<br/>';
            aDraggable.left += event.screenX - aDraggable.oldMouseX;
            aDraggable.top += event.screenY - aDraggable.oldMouseY;
                
            aDraggable.oldMouseX = event.screenX;
            aDraggable.oldMouseY = event.screenY;
            if(aDraggable.whenDragging){
                aDraggable.whenDragging(aDraggable.whenDraggingParams);
            }   
         
        }
    };
  
  this.setWhenStartDragging = function(funcRef, params){
      this.whenStartDragging = funcRef;
      this.whenStartDraggingParams = params;
    };
  this.setWhenDragging = function(funcRef, params){
      this.whenDragging = funcRef;
      this.whenDraggingParams = params;
    };
  this.setWhenDropped = function(funcRef, params){
      this.whenDropped = funcRef;
      this.whenDroppedParams = params;
    };
    
  	/**
  	 * the method used to float the draggable to the top during a drag
	 * @private
	 */
  this.display.bringToFront = function(){
      this.style.zIndex = this.draggable.parentZIndex + 50;
      
    };
  
  	/**
  	 * the method used to undo floating the block to the top during a drag
	 * @private
	 */
  this.display.sendBackDown = function(){
      this.style.zIndex = this.draggable.parentZIndex + 50;
    };

  	/**
  	 * the method used to define the inner html of the block
	 * @private
	 */
  this.setHTMLContents = function (html){
      this.display.innerHTML = html;
    };
}

