/*
 * create_compound_trigger.sql
 * Rozdzia 10., Oracle Database 11g. Programowanie w jzyku PL/SQL
 * Michael McLaughlin
 *
 * UWAGI:
 *
 * Ten skrypt deklaruje i testuje wyzwalacz zoony.
 */

SET ECHO OFF
SET FEEDBACK ON
SET NULL '<Null>'
SET PAGESIZE 999
SET SERVEROUTPUT ON

-- Warunkowe usuwanie obiektw.
BEGIN
  FOR i IN (SELECT null
            FROM   user_tables
            WHERE  table_name = 'PRICE_EVENT_LOG') LOOP
    EXECUTE IMMEDIATE 'DROP TABLE price_event_log CASCADE CONSTRAINTS';
  END LOOP;
  FOR i IN (SELECT null
            FROM   user_sequences
            WHERE  sequence_name = 'PRICE_EVENT_LOG_S1') LOOP
    EXECUTE IMMEDIATE 'DROP SEQUENCE price_event_log_s1';
  END LOOP;
END;
/

-- Tworzenie tabeli.
CREATE TABLE price_event_log
( price_log_id     NUMBER
, price_id         NUMBER
, created_by       NUMBER
, creation_date    DATE
, last_updated_by  NUMBER
, last_update_date DATE );

-- Tworzenie sekwencji.
CREATE SEQUENCE price_event_log_s1;

-- Ustawia warto V$SESSION CLIENT_INFO value zgodnie z wartoci z tabeli SYSTEM_USER.
EXEC dbms_application_info.set_client_info('3');

-- Tworzenie wyzwalacza zoonego dla tabeli price.
CREATE OR REPLACE TRIGGER compound_price_update_t1
  FOR UPDATE ON price
  COMPOUND TRIGGER
    -- Deklaracja globalnego typu rekordowego.
    TYPE price_record IS RECORD
    ( price_log_id     price_event_log.price_log_id%TYPE
    , price_id         price_event_log.price_id%TYPE
    , created_by       price_event_log.created_by%TYPE
    , creation_date    price_event_log.creation_date%TYPE
    , last_updated_by  price_event_log.last_updated_by%TYPE
    , last_update_date price_event_log.last_update_date%TYPE );
    -- Deklaracja globalnego typu kolekcji.
    TYPE price_list IS TABLE OF PRICE_RECORD;
    -- Deklarowanie i inicjowanie kolekcji globalnej.
    price_updates  PRICE_LIST := price_list();

  BEFORE EACH ROW IS
    -- Deklarowanie i definiowanie zmiennych lokalnych sekcji czasowej.
    c       NUMBER;
    user_id NUMBER := NVL(TO_NUMBER(SYS_CONTEXT('userenv','client_info')),-1);
  BEGIN
    -- Przydzia pamici i przypisanie dynamicznej wartoci indeksu.
    price_updates.EXTEND;
    c := price_updates.LAST;
    price_updates(c).price_log_id := price_event_log_s1.nextval;
    price_updates(c).price_id := :old.price_id;
    price_updates(c).created_by := user_id;
    price_updates(c).creation_date := SYSDATE;
    price_updates(c).last_updated_by := user_id;
    price_updates(c).last_update_date := SYSDATE;
  END BEFORE EACH ROW;
  
  AFTER STATEMENT IS
  BEGIN
    FORALL i IN price_updates.FIRST..price_updates.LAST
      INSERT INTO price_event_log
      VALUES
      ( price_updates(i).price_log_id
      , price_updates(i).price_id
      , price_updates(i).created_by
      , price_updates(i).creation_date
      , price_updates(i).last_updated_by
      , price_updates(i).last_update_date );
  END AFTER STATEMENT;
END;
/

-- Aktualizacja tabeli PRICE.
UPDATE price
SET    last_updated_by = NVL(TO_NUMBER(SYS_CONTEXT('userenv','client_info')),-1);

-- Zapytanie do tabeli PRICE_EVENT_LOG.
SELECT * FROM price_event_log;