/*
 * load_clob_from_file.sql
 * Rozdzia 8., Oracle Database 11g. Programowanie w jzyku PL/SQL
 * Michael McLaughlin
 *
 * UWAGI:
 *
 * W systemie Linux i Unix trzeba umieci plik LOTRFellowship.txt w katalogu /tmp;
 * w systemie Windows trzeba doda ten plik do katalogu C:\WINDOWS\TEMP.
 * Mona te zdefiniowa katalog wirtualny GENERIC wskazujcy
 * na wybrany katalog fizyczny, trzeba jednak upewni si, e Oracle
 * ma uprawnienia do odczytu z tego katalogu.
 *
 * Ten skrypt definiuje procedur, ktra wczytuje duy plik znakowy do kolumny
 * typu CLOB.
 *
 * TRYB DIAGNOSTYCZNY:
 *
 * Naley wywoa w sesji ponisze polcenie.
 *
 * ALTER SESSION SET PLSQL_CCFLAGS = 'debug:1';
 */

-- Uywane do diagnozowania skryptu.
SET ECHO ON
SET FEEDBACK ON
SET PAGESIZE 49999
SET SERVEROUTPUT ON SIZE 1000000

-- Procedura do wczytywania danych typu CLOB.
CREATE OR REPLACE PROCEDURE load_clob_from_file
( src_file_name     IN VARCHAR2
, table_name        IN VARCHAR2
, column_name       IN VARCHAR2
, primary_key_name  IN VARCHAR2
, primary_key_value IN VARCHAR2 ) IS

  -- Definicja zmiennych lokalnych na potrzeby procedury DBMS_LOB.LOADCLOBFROMFILE.
  des_clob   CLOB;
  src_clob   BFILE := BFILENAME('GENERIC',src_file_name);
  des_offset NUMBER := 1;
  src_offset NUMBER := 1;
  ctx_lang   NUMBER := dbms_lob.default_lang_ctx;
  warning    NUMBER;
  
  -- Okrelenie rozmiaru przed wczytywaniem.
  src_clob_size NUMBER;
  -- Definicja zmiennej lokalnej uywanej w kodzie NDS.
  stmt VARCHAR2(2000);

BEGIN

  -- Otwarcie pliku rdowego jest konieczne.
  IF dbms_lob.fileexists(src_clob) = 1 AND NOT dbms_lob.isopen(src_clob) = 1 THEN
    src_clob_size := dbms_lob.getlength(src_clob);
    dbms_lob.open(src_clob,DBMS_LOB.LOB_READONLY);
  END IF;
  
  -- Dynamiczne przypisanie acucha znakw do instrukcji.
  stmt := 'UPDATE '||table_name||' '
       || 'SET    '||column_name||' = empty_clob() '
       || 'WHERE  '||primary_key_name||' = '||''''||primary_key_value||''' '
       || 'RETURNING '||column_name||' INTO :locator';

  -- Uruchomienie dynamicznej instrukcji.
  EXECUTE IMMEDIATE stmt USING OUT des_clob;

  -- Odczyt pliku i jego zapis do obiektu typu CLOB, zamknicie pliku i zatwierdzenie zmian.
  dbms_lob.loadclobfromfile( dest_lob     => des_clob
                           , src_bfile    => src_clob
                           , amount       => dbms_lob.getlength(src_clob)
                           , dest_offset  => des_offset
                           , src_offset   => src_offset
                           , bfile_csid   => dbms_lob.default_csid
                           , lang_context => ctx_lang
                           , warning      => warning );

  -- Zamknicie otwartego pliku.
  dbms_lob.close(src_clob);

  -- Zatwierdzenie zapisu i warunkowe potwierdzenie tej operacji.
  IF src_clob_size = dbms_lob.getlength(des_clob) THEN
    $IF $$DEBUG = 1 $THEN
      dbms_output.put_line('Sukces!');
    $END
    COMMIT;
  ELSE
    $IF $$DEBUG = 1 $THEN
      dbms_output.put_line('Niepowodzenie.');
    $END
    RAISE dbms_lob.operation_failed;
  END IF;
  
END load_clob_from_file;
/

SHOW ERRORS

-- Formatowanie kolumn.
COL item_id    FORMAT 9999
COL item_title FORMAT A50
COL size       FORMAT 9,999,990

-- Zapytanie o rozmiar kolumny przed wczytaniem danych.
SELECT item_id
,      item_title
,      dbms_lob.getlength(item_desc) AS "SIZE"
FROM   item
WHERE  dbms_lob.getlength(item_desc) > 0;

-- Wstawianie opisu we wszystkich pasujcych wierszach.
BEGIN
  FOR i IN (SELECT item_id
            FROM   item
            WHERE  item_title = 'The Lord of the Rings - Fellowship of the Ring'
            AND    item_type IN (SELECT common_lookup_id
                                 FROM   common_lookup
                                 WHERE  common_lookup_table = 'ITEM'
                                 AND    common_lookup_column = 'ITEM_TYPE'
                                 AND    REGEXP_LIKE(common_lookup_type,'^(dvd|vhs)*','i'))) LOOP
    -- Wywoanie procedury dla odpowiednich wierszy.
    load_clob_from_file( src_file_name     => 'LOTRFellowship.txt'
                       , table_name        => 'ITEM'
                       , column_name       => 'ITEM_DESC'
                       , primary_key_name  => 'ITEM_ID'
                       , primary_key_value => TO_CHAR(i.item_id) );
  END LOOP;
END;
/

-- Zapytanie o rozmiar kolumny po wczytaniu danych.
SELECT item_id
,      item_title
,      dbms_lob.getlength(item_desc) AS "SIZE"
FROM   item
WHERE  dbms_lob.getlength(item_desc) > 0;