/*
 * load_blob_from_file.sql
 * Rozdzia 8., Oracle Database 11g. Programowanie w jzyku PL/SQL
 * Michael McLaughlin
 *
 * UWAGI:
 *
 * W systemie Linux i Unix trzeba umieci plik LOTRFellowship.txt w katalogu /tmp;
 * w systemie Windows trzeba doda ten plik do katalogu C:\WINDOWS\TEMP.
 * Mona te zdefiniowa katalog wirtualny GENERIC wskazujcy
 * na wybrany katalog fizyczny, trzeba jednak upewni si, e Oracle
 * ma uprawnienia do odczytu z tego katalogu.
 *
 * Ten skrypt definiuje procedur, ktra wczytuje duy plik znakowy
 * do kolumny typu BLOB.
 *
 * TRYB DIAGNOSTYCZNY:
 *
 * Naley wywoa w sesji ponisze polcenie.
 *
 * ALTER SESSION SET PLSQL_CCFLAGS = 'debug:1';
 */

-- Uywane do diagnozowania skryptu.
SET ECHO ON
SET FEEDBACK ON
SET PAGESIZE 49999
SET SERVEROUTPUT ON SIZE 1000000

-- Procedura do wczytywania danych typu BLOB.
CREATE OR REPLACE PROCEDURE load_blob_from_file
( src_file_name     IN VARCHAR2
, table_name        IN VARCHAR2
, column_name       IN VARCHAR2
, primary_key_name  IN VARCHAR2
, primary_key_value IN VARCHAR2 ) IS

  -- Definicja zmiennych lokalnych na potrzeby procedury DBMS_LOB.LOADBLOBFROMFILE.
  des_blob      BLOB;
  src_blob      BFILE := BFILENAME('GENERIC',src_file_name);
  des_offset    NUMBER := 1;
  src_offset    NUMBER := 1;
  
  -- Okrelenie rozmiaru przed wczytywaniem.
  src_blob_size NUMBER;            ]

  -- Definicja zmiennej lokalnej uywanej w kodzie NDS.
  stmt VARCHAR2(2000);

BEGIN

  -- Otwarcie pliku rdowego jest niezbdne.
  IF dbms_lob.fileexists(src_blob) = 1 AND NOT dbms_lob.isopen(src_blob) = 1 THEN
    src_blob_size := dbms_lob.getlength(src_blob);
    dbms_lob.open(src_blob,DBMS_LOB.LOB_READONLY);
  END IF;
  
  -- Dynamiczne przypisanie acucha znakw do instrukcji.
  stmt := 'UPDATE '||table_name||' '
       || 'SET    '||column_name||' = empty_blob() '
       || 'WHERE  '||primary_key_name||' = '||''''||primary_key_value||''' '
       || 'RETURNING '||column_name||' INTO :locator';

  -- Uruchomienie instrukcji dynamicznej.
  EXECUTE IMMEDIATE stmt USING OUT des_blob;

  -- Odczyt pliku i zapis danych w kolumnie typu BLOB.
  dbms_lob.loadblobfromfile( dest_lob     => des_blob
                           , src_bfile    => src_blob
                           , amount       => dbms_lob.getlength(src_blob)
                           , dest_offset  => des_offset
                           , src_offset   => src_offset );

  -- Zamykanie otwartego pliku rdowego.
  dbms_lob.close(src_blob);
  
  -- Zatwierdzenie zapisu.
  IF src_blob_size = dbms_lob.getlength(des_blob) THEN
    $IF $$DEBUG = 1 $THEN
      dbms_output.put_line('Sukces!');
    $END
    COMMIT;
  ELSE
    $IF $$DEBUG = 1 $THEN
      dbms_output.put_line('Poraka.');
    $END
    RAISE dbms_lob.operation_failed;
  END IF;
  
END load_blob_from_file;
/

SHOW ERRORS

-- Formatowanie kolumn.
COL item_id    FORMAT 9999
COL item_title FORMAT A50
COL size       FORMAT 9,999,990

-- Zapytanie o rozmiar kolumny przed wczytaniem danych.
SELECT item_id
,      item_title
,      dbms_lob.getlength(item_blob) AS "SIZE"
FROM   item
WHERE  dbms_lob.getlength(item_blob) > 0;

-- Wstawianie opisu we wszystkich pasujcych wierszach.
BEGIN
  FOR i IN (SELECT item_id
            FROM   item
            WHERE  item_title = 'Harry Potter and the Sorcerer''s Stone'
            AND    item_type IN
             (SELECT common_lookup_id
              FROM   common_lookup
              WHERE  common_lookup_table = 'ITEM'
              AND    common_lookup_column = 'ITEM_TYPE'
              AND    REGEXP_LIKE(common_lookup_type,'^(dvd|vhs)*','i'))) LOOP
    -- Wywoanie procedury dla odpowiednich wierszy.
    load_blob_from_file( src_file_name     => 'HarryPotter1.png'
                       , table_name        => 'ITEM'
                       , column_name       => 'ITEM_BLOB'
                       , primary_key_name  => 'ITEM_ID'
                       , primary_key_value => TO_CHAR(i.item_id) );
  END LOOP;
END;
/

-- Zapytanie o rozmiar kolumny po wczytaniu danych.
SELECT item_id
,      item_title
,      dbms_lob.getlength(item_blob) AS "SIZE"
FROM   item
WHERE  dbms_lob.getlength(item_blob) > 0;

-- Wstawia rekord tylko wtedy, jeli nie go znajdzie (skrypt mona uruchomi wielokrotnie).
DECLARE
  -- Deklaracja numeru wiersza.
  row NUMBER :=0;

  -- Deklaracja kursora sprawdzajcego kolumn typu BLOB.
  CURSOR c IS
    SELECT item_id
    FROM   item
    WHERE  item_title = 'The Blob - Criterion Collection'
    AND    item_type IN
            (SELECT common_lookup_id
             FROM   common_lookup
             WHERE  common_lookup_table = 'ITEM'
             AND    common_lookup_column = 'ITEM_TYPE'
             AND    REGEXP_LIKE(common_lookup_type,'^(dvd|vhs)*','i'));
BEGIN
  OPEN c;
  LOOP
    FETCH c INTO row;
    IF c%NOTFOUND AND c%ROWCOUNT = 1 THEN
      INSERT INTO item VALUES
      ( item_s1.nextval
      ,'ASIN: B00004W3HE'
      ,(SELECT   common_lookup_id
        FROM     common_lookup
        WHERE    common_lookup_type = 'XBOX')
      ,'The Blob - Criterion Collection'
      ,''
      , empty_clob()
      , empty_blob()
      , NULL
      ,'NR'
      ,'MPAA'
      ,'14-NOV-2000'
      , 3, SYSDATE, 3, SYSDATE);
      EXIT;
    ELSE
      EXIT;
    END IF;
  END LOOP;
END;
/

-- Wstawianie opisu we wszystkich pasujcych wierszach.
BEGIN
  FOR i IN (SELECT item_id
            FROM   item
            WHERE  item_title = 'The Blob - Criterion Collection'
            AND    item_type IN
             (SELECT common_lookup_id
              FROM   common_lookup
              WHERE  common_lookup_table = 'ITEM'
              AND    common_lookup_column = 'ITEM_TYPE'
              AND    REGEXP_LIKE(common_lookup_type,'^(dvd|vhs)*','i'))) LOOP
    -- Wywoanie procedury dla odpowiednich wierszy.
    load_blob_from_file( src_file_name     => 'TheBlob.pdf'
                       , table_name        => 'ITEM'
                       , column_name       => 'ITEM_BLOB'
                       , primary_key_name  => 'ITEM_ID'
                       , primary_key_value => TO_CHAR(i.item_id) );
  END LOOP;
END;
/

