/*
 * create_add_contact2.sql
 * Rozdzia 6., Oracle Database 11g. Programowanie w jzyku PL/SQL
 * Michael McLaughlin
 *
 * UWAGI:
 *
 * Ten skrypt tworzy procedur o parametrach przekazywanych przez referencj.
 */

-- Uywane do diagnozowania skryptu.
SET ECHO ON
SET FEEDBACK ON
SET PAGESIZE 49999
SET SERVEROUTPUT ON SIZE 1000000

-- Definicja funkcji autonomicznej do zapisywania wartoci klucza sztucznego.
CREATE OR REPLACE FUNCTION get_sequence_value
(sequence_name VARCHAR2) RETURN NUMBER IS
  PRAGMA AUTONOMOUS_TRANSACTION;
  id_value  NUMBER;
  statement VARCHAR2(2000);
BEGIN
  -- Tworzenie dynamicznej instrukcji SQL jako anonimowego bloku jzyka PL/SQL.
  statement := 'BEGIN'                                ||CHR(10)
            || '  SELECT  '||sequence_name||'.nextval'||CHR(10)
            || '  INTO     :id_value'                 ||CHR(10)
            || '  FROM     dual;'                     ||CHR(10)
            || 'END;';

  -- Wykonanie dynamicznej instrukcji SQL.
  EXECUTE IMMEDIATE statement USING OUT id_value; 
  RETURN id_value;
END get_sequence_value;
/

CREATE OR REPLACE procedure add_contact
( contact_id           OUT NUMBER  -- Klucz gwny po wstawieniu danych.
, member_id         IN     NUMBER  -- Klucz zewntrzny przed wstawieniem danych.
, contact_type      IN     NUMBER
, last_name         IN     VARCHAR2
, first_name        IN     VARCHAR2
, middle_initial    IN     VARCHAR2 := NULL
, created_by        IN     NUMBER
, creation_date     IN     DATE     := SYSDATE
, last_updated_by   IN     NUMBER
, last_update_date  IN     DATE     := SYSDATE) IS
BEGIN
  -- Ustawianie punktu zapisu, aby zagwarantowa, e program 
  -- ukoczy wszystkie operacje lub nie wykona adnej z nich.
  SAVEPOINT add_contact;

  -- Sugestia rozwinicia funkcji get_sequence_value w miejscu wywoania.
  PRAGMA INLINE(get_sequence_value,'YES');

  -- Przypisanie nastpnej wartoci z sekwencji i wstawienie rekordu.
  contact_id := get_sequence_value('CONTACT_S1');
  INSERT INTO contact VALUES
  ( contact_id
  , member_id
  , contact_type
  , last_name
  , first_name
  , middle_initial
  , created_by
  , creation_date
  , last_updated_by
  , last_update_date);
EXCEPTION
  WHEN others THEN
    ROLLBACK TO add_contact;
    RAISE_APPLICATION_ERROR(-20001,SQLERRM);
END add_contact;
/

CREATE OR REPLACE procedure add_address
( address_id           OUT NUMBER   -- Klucz gwny po wstawieniu danych.
, contact_id        IN     NUMBER   -- Klucz zewntrzny przed wstawieniem danych.
, address_type      IN     NUMBER   := NULL
, street_address    IN     VARCHAR2 := NULL
, city              IN     VARCHAR2 := NULL
, state_province    IN     VARCHAR2 := NULL
, postal_code       IN     VARCHAR2 := NULL
, created_by        IN     NUMBER
, creation_date     IN     DATE     := SYSDATE
, last_updated_by   IN     NUMBER
, last_update_date  IN     DATE     := SYSDATE) IS

  -- Deklaracja zmiennej przechowujcej klucz sztuczny.
  street_address_id NUMBER;
BEGIN
  -- Ustawianie punktu zapisu, aby zagwarantowa, e program ukoczy wszystkie operacje lub nie wykona adnej z nich.
  SAVEPOINT add_address;

  -- Sprawdzanie stanu danych i wstawianie ich do tabeli ADDRESS.
  IF address_type IS NOT NULL   AND
     city IS NOT NULL           AND
     state_province IS NOT NULL AND
     postal_code IS NOT NULL    THEN

    -- Przypisanie nastpnej wartoci z sekwencji i wstawienie rekordu.
    address_id := get_sequence_value('ADDRESS_S1');
    INSERT INTO address VALUES
    ( address_id
    , contact_id
    , address_type
    , city
    , state_province
    , postal_code
    , created_by
    , creation_date
    , last_updated_by
    , last_update_date);

  -- Sprawdzanie obecnoci danych i wstawianie ich do tabeli STREET_ADDRESS.
    IF street_address IS NOT NULL THEN
      -- Przypisanie nastpnej wartoci z sekwencji i wstawienie rekordu.
      street_address_id := get_sequence_value('STREET_ADDRESS_S1');
      INSERT INTO street_address VALUES
      ( street_address_id
      , address_id
      , street_address
      , created_by
      , creation_date
      , last_updated_by
      , last_update_date);
    END IF;
  END IF;
EXCEPTION
  WHEN others THEN
    ROLLBACK TO add_address;
    RAISE_APPLICATION_ERROR(-20001,SQLERRM);
END add_address;
/

-- Blok anonimowy testujcy procedury ADD_CONTACT i ADD_ADDRESS.
DECLARE
  -- Deklaracje zmiennych przechowujcych klucze sztuczne.
  member_id  NUMBER;
  contact_id NUMBER;
  address_id NUMBER;

  -- Deklaracja funkcji lokalnej pobierajcej typ.
  FUNCTION get_type
  ( table_name  VARCHAR2
  , column_name VARCHAR2
  , type_name   VARCHAR2) RETURN NUMBER IS
    retval NUMBER;
  BEGIN
    SELECT   common_lookup_id
    INTO     retval
    FROM     common_lookup
    WHERE    common_lookup_table = table_name
    AND      common_lookup_column = column_name
    AND      common_lookup_type = type_name;
    RETURN retval;
  END get_type;

  -- Definicja funkcji autonomicznej do obsugi wartoci kluczy sztucznych.
  FUNCTION get_member_id RETURN NUMBER IS
    PRAGMA AUTONOMOUS_TRANSACTION;
    id_value NUMBER;
  BEGIN
    SELECT member_s1.nextval INTO id_value FROM dual;
    RETURN id_value;
  END;
BEGIN
  -- Deklaracja zmiennej przechowujcej klucz sztuczny.
  member_id := get_member_id;
  INSERT INTO member VALUES
  ( member_id
  ,(SELECT common_lookup_id
    FROM   common_lookup
    WHERE  common_lookup_table = 'MEMBER'
    AND    common_lookup_column = 'MEMBER_TYPE'
    AND    common_lookup_type = 'GROUP')
  , '4563-98-71'
  , '5555-6363-1212-4343'
  ,(SELECT common_lookup_id
    FROM   common_lookup
    WHERE  common_lookup_table = 'MEMBER'
    AND    common_lookup_column = 'CREDIT_CARD_TYPE'
    AND    common_lookup_type = 'VISA_CARD')
  , 3
  , SYSDATE
  , 3
  , SYSDATE);

  -- Wywoanie procedury wstawiajcej rekordy do odpowiedniej tabeli.
  add_contact( member_id => member_id
             , contact_id => contact_id    -- To zmienna w trybie OUT.
             , contact_type => get_type('CONTACT','CONTACT_TYPE','CUSTOMER')
             , last_name => 'Prus'
             , first_name => 'Bolesaw'
             , created_by => 3
             , last_updated_by => 3);

  -- Wywoanie procedury wstawiajcej rekordy do odpowiednich tabel.
  add_address( address_id => address_id
             , contact_id => contact_id    -- To zmienna w trybie OUT.
             , address_type => get_type('ADDRESS','ADDRESS_TYPE','HOME')
             , street_address => 'Szeroka 25'
             , city => 'Warszawa'
             , state_province => 'WAW'
             , postal_code => '10451'
             , created_by => 3
             , last_updated_by => 3);
END;
/

