/*
 * nds_sqle.sql
 * Rozdzia 13, Oracle10g. Programowanie w jzyku PL/SQL
 * Ron Hardman, Mike McLaughlin i Scott Urman
 *
 * Cel:
 *   Dziaajcy samouczek dotyczcy jzyka NDS
 *   z bdem w postaci parametru w trybie OUT.
 *   Pakiet skompiluje si, ale zgosi wyjtek
 *   w czasie wykonywania
 */

-- Konfiguracja rodowiska SQL*PLUS na potrzeby skryptu
SET SERVEROUTPUT ON SIZE 1000000
SET ECHO OFF
SET FEEDBACK OFF
SET LINESIZE 90
SET PAGESIZE 0

@create_types.sql

--  Umieszczenie pakietu w dzienniku
SELECT 'CREATE OR REPLACE PACKAGE nds_tutorial' FROM dual;

-- Tworzenie specyfikacji pakietu
CREATE OR REPLACE PACKAGE nds_tutorial AS

  -- Definicje zmiennych formatujcych
  dline               VARCHAR2(80) := 
   '============================================================';
  sline               VARCHAR2(80) := 
   '------------------------------------------------------------';

  -- Procedura tworzca sekwencj przy uyciu czenia
  PROCEDURE create_sequence
    ( sequence_name           IN     VARCHAR2);

  -- Procedura tworzca tabel przy uyciu czenia
  PROCEDURE create_table
    ( table_name              IN     VARCHAR2
    , table_definition        IN     VARCHAR2);

  -- Procedura usuwajca sekwencj przy uyciu czenia
  PROCEDURE drop_sequence
    ( sequence_name           IN     VARCHAR2);

  -- Procedura usuwajca tabel przy uyciu czenia
  PROCEDURE drop_table
    ( table_name              IN     VARCHAR2);

  -- Procedura ukrywajca blok SELECT-INTO jzyka PL/SQL
  PROCEDURE increment_sequence
    ( sequence_name           IN     VARCHAR2
    , sequence_value          IN OUT NUMBER);

  -- Procedura demonstrujca instrukcj DML bez zmiennych powizanych
  PROCEDURE insert_into_table
    ( table_name              IN     VARCHAR2
    , table_column_value1     IN     NUMBER
    , table_column_value2     IN     VARCHAR2
    , table_column_value3     IN     VARCHAR2);

  -- Procedura demonstrujca instrukcj DML z uporzdkowanymi zmiennymi powizanymi
  PROCEDURE inserts_into_table
    ( table_name              IN     VARCHAR2
    , table_column_value1     IN     NUMBER
    , table_column_value2     IN     VARCHAR2
    , table_column_value3     IN     VARCHAR2);

  -- Procedura demonstrujca wielowierszowe polecenie DQL
  PROCEDURE multiple_row_return;

  -- Procedura demonstrujca wielowierszowe polecenie DQL
  PROCEDURE multiple_row_return
    ( table_name    VARCHAR2
    , column_name1  VARCHAR2
    , column_name2  VARCHAR2
    , column_name3  VARCHAR2 );

  -- Procedura demonstrujca jednowierszowe polecenie DQL
  PROCEDURE single_row_return;

  -- Procedura demonstrujca jednowierszowe polecenie DQL
  PROCEDURE single_row_return
    ( table_name    VARCHAR2
    , column_name1  VARCHAR2
    , column_name2  VARCHAR2
    , column_name3  VARCHAR2 );

END nds_tutorial;
/

-- ==========================================================================
--  Zastosowano technik zarzdzania diagnozowaniem i dziennikami w celu
--  zapisu kompilowanego kodu i komunikatw o bdach.
--  Naley oznaczy ten fragment komentarzem, kiedy kod jest gotowy do udostpnienia,
--  a przy diagnozowaniu zmian ponownie usun oznaczenie komentarzem
-- ==========================================================================

SPOOL nds_tutorial_spec.log

list

show errors

SPOOL OFF

--  Umieszczanie ciaa pakietu w dzienniku
SELECT 'CREATE OR REPLACE PACKAGE BODY nds_tutorial' FROM dual;

-- Tworzenie ciaa pakietu
CREATE OR REPLACE PACKAGE BODY nds_tutorial IS

  /*
  || =========================================================================
  ||  PODSUMOWANIE:
  ||  -------
  ||  Dziaajcy samouczek dotyczcy jzyka NDS
  ||  z prostymi przykadami zastosowania rnych technik.
  || =========================================================================
  */

  /*
  || ------------------------------------------------------------------
  */

  -- Procedura tworzca sekwencj przy uyciu czenia
  PROCEDURE create_sequence
    ( sequence_name           IN     VARCHAR2) IS

    -- Definicja zmiennej lokalnej
    statement           VARCHAR2(2000);

    -- Definicja funkcji lokalnej do sprawdzania, czy sekwencja nie istnieje
    FUNCTION verify_not_sequence
      ( sequence_name_in      IN     VARCHAR2)
    RETURN BOOLEAN IS

      -- Domylna zwracana warto
      retval                  BOOLEAN := TRUE;

      -- Kursor zwraca pojedynczy wiersz po znalezieniu sekwencji
      CURSOR find_sequence IS
        SELECT   null
        FROM     user_objects
        WHERE    object_name = sequence_name_in;
      
    BEGIN

      -- Ptla for ustawia warto logiczn po znalezieniu sekwencji
      FOR i IN find_sequence LOOP
        retval := FALSE;
      END LOOP;

      -- Zwraca warto logiczn
      RETURN retval;

    END verify_not_sequence;

  BEGIN

    -- Jeli sekwencja nie istnieje, trzeba j utworzy
    IF verify_not_sequence(sequence_name) = TRUE THEN

      -- Tworzy dynamiczn instrukcj SQL
      statement := 'CREATE SEQUENCE '||sequence_name||CHR(10)
                || '  INCREMENT BY   1'             ||CHR(10)
                || '  START WITH     1'             ||CHR(10)
                || '  CACHE          20'            ||CHR(10)
                || '  ORDER';

      -- Uycie NDS do uruchomienia instrukcji
      EXECUTE IMMEDIATE statement;

      -- Wywietlenie nazwy moduu
      dbms_output.put_line(
        '-> nds_tutorial.create_sequence');

      -- Wywietlenie wiersza rozdzielajcego
      dbms_output.put_line(sline);

      -- Wywietlenie danych wyjciowych
      dbms_output.put_line(
        'Utworzono sekwencje <'||sequence_name||'>');

    ELSE

      -- Wywietlenie nazwy moduu
      dbms_output.put_line(
        '-> nds_tutorial.create_sequence');

      -- Wywietlenie wiersza rozdzielajcego
      dbms_output.put_line(sline);

      -- Wywietlenie danych wyjciowych
      dbms_output.put_line(
        'Sekwencja <'||sequence_name||'> juz istnieje');

    END IF;

  END create_sequence;

  /*
  || ------------------------------------------------------------------
  */

  -- Procedura tworzca tabel przy uyciu czenia
  PROCEDURE create_table
    ( table_name              IN     VARCHAR2
    , table_definition        IN     VARCHAR2) IS

    -- Definicja lokalnej zmiennej NDS
    statement                 VARCHAR2(2000);

    -- Definicja funkcji lokalnej do sprawdzania, czy tabela nie istnieje
    FUNCTION verify_not_table
      ( object_name_in        IN     VARCHAR2)
    RETURN BOOLEAN IS

      -- Domylna zwracana warto
      retval                  BOOLEAN := TRUE;

      -- Kursor zwraca pojedynczy wiersz po znalezieniu tabeli
      CURSOR find_object IS
        SELECT   null
        FROM     user_objects
        WHERE    object_name = object_name_in;
      
    BEGIN

      -- Ptla for ustawia warto logiczn po znalezieniu tabeli
      FOR i IN find_object LOOP
        retval := FALSE;
      END LOOP;

      -- Zwraca warto logiczn
      RETURN retval;

    END verify_not_table;

  BEGIN

    -- Jeli tabelia nie istnieje, naley j utworzy
    IF verify_not_table(table_name) = TRUE THEN 

      -- Tworzy dynamiczn instrukcj SQL
      statement := 'CREATE TABLE '||table_name||CHR(10)
                || table_definition;

      -- Zastosowanie NDS do uruchomienia instrukcji
      EXECUTE IMMEDIATE statement;

      -- Wywietlenie nazwy moduu
      dbms_output.put_line(
        '-> nds_tutorial.create_table');

      -- Wywietlenie wiersza rozdzielajcego
      dbms_output.put_line(sline);

      -- Wywietlenie danych wyjciowych
      dbms_output.put_line('Utworzono tabele <'||table_name||'>');

    ELSE

      -- Wywietlenie nazwy moduu
      dbms_output.put_line(
        '-> nds_tutorial.create_table');

      -- Wywietlenie wiersza rozdzielajcego
      dbms_output.put_line(sline);

      -- Wywietlenie danych wyjciowych
      dbms_output.put_line(
        'Obiekt <'||table_name||'> juz istnieje');

    END IF;

  END create_table;

  /*
  || ------------------------------------------------------------------
  */

  -- Procedura usuwajca sekwencj przy uyciu czenia
  PROCEDURE drop_sequence
    ( sequence_name           IN     VARCHAR2) IS

    -- Definicja zmiennej lokalnej
    statement          VARCHAR2(2000);

    -- Definicja funkcji lokalnej do sprawdzania, czy sekwencja istnieje
    FUNCTION verify_sequence
      ( sequence_name_in      IN     VARCHAR2)
    RETURN BOOLEAN IS

      -- Domylna zwracana warto
      retval                  BOOLEAN := FALSE;

      -- Kursor zwraca pojedynczy wiersz po znalezieniu sekwencji
      CURSOR find_sequence IS
        SELECT   null
        FROM     user_objects
        WHERE    object_name = sequence_name_in;
      
    BEGIN

      -- Ptla for ustawia warto logiczn po znalezieniu sekwencji
      FOR i IN find_sequence LOOP
        retval := TRUE;
      END LOOP;

      -- Zwraca warto logiczn
      RETURN retval;

    END verify_sequence;

  BEGIN

    -- Jeli sekwencja istnieje, naley j usun
    IF verify_sequence(sequence_name) = TRUE THEN

      -- Tworzy dynamiczn instrukcj SQL
      statement := 'DROP SEQUENCE '||sequence_name;

      -- Zastosowanie NDS do uruchomienia instrukcji
      EXECUTE IMMEDIATE statement;

      -- Wywietlenie nazwy moduu
      dbms_output.put_line(
        '-> nds_tutorial.drop_sequence');

      -- Wywietlenie wiersza rozdzielajcego
      dbms_output.put_line(sline);

      -- Wywietlenie danych wyjciowych
      dbms_output.put_line(
        'Usunieto sekwencje <'||sequence_name||'>');

    ELSE

      -- Wywietlenie nazwy moduu
      dbms_output.put_line(
        '-> nds_tutorial.drop_sequence');

      -- Wywietlenie wiersza rozdzielajcego
      dbms_output.put_line(sline);

      -- Wywietlenie danych wyjciowych
      dbms_output.put_line(
        'Sekwencja <'||sequence_name||'> nie istnieje');

    END IF;

  END drop_sequence;

  /*
  || ------------------------------------------------------------------
  */

  -- Procedura usuwajca tabel przy uyciu czenia
  PROCEDURE drop_table
    ( table_name              IN     VARCHAR2) IS

    -- Definicja zmiennej lokalnej
    statement                 VARCHAR2(2000);

    -- Definicja funkcji lokalnej do sprawdzania, czy tabela istnieje
    FUNCTION verify_table
      ( object_name_in        IN     VARCHAR2)
    RETURN BOOLEAN IS

      -- Domylna zwracana warto
      retval                  BOOLEAN := FALSE;

      -- Kursor zwraca pojedynczy wiersz po znalezieniu tabeli
      CURSOR find_object IS
        SELECT   null
        FROM     user_objects
        WHERE    object_name = object_name_in;
      
    BEGIN

      -- Ptla for ustawia warto logiczn po znalezieniu tabeli
      FOR i IN find_object LOOP
        retval := TRUE;
      END LOOP;

      -- Zwraca warto logiczn
      RETURN retval;

    END verify_table;

  BEGIN

    IF verify_table(table_name) = TRUE THEN

      -- Tworzy dynamiczn instrukcj SQL
      statement := 'DROP TABLE '||table_name;

      -- Zastosowanie NDS do uruchomienia instrukcji
      EXECUTE IMMEDIATE statement;

      -- Wywietlenie nazwy moduu
      dbms_output.put_line(
        '-> nds_tutorial.drop_table');

      -- Wywietlenie wiersza rozdzielajcego
      dbms_output.put_line(sline);

      -- Wywietlenie danych wyjciowych
      dbms_output.put_line(
        'Usunieto tabele <'||table_name||'>');

    ELSE

      -- Wywietlenie nazwy moduu
      dbms_output.put_line(
        '-> nds_tutorial.drop_table');

      -- Wywietlenie wiersza rozdzielajcego
      dbms_output.put_line(sline);

      -- Wywietlenie danych wyjciowych
      dbms_output.put_line(
        'Obiekt <'||table_name||'> nie istnieje');

    END IF;

  END drop_table;

  /*
  || ------------------------------------------------------------------
  */

  -- Procedura ukrywajca blok SELECT-INTO jzyka PL/SQL
  PROCEDURE increment_sequence
    ( sequence_name           IN     VARCHAR2
    , sequence_value          IN OUT NUMBER  ) IS

    -- Definicja zmiennej lokalnej NDS
    statement                 VARCHAR2(2000);

    -- Definicja funkcji lokalnej do sprawdzania, czy sekwencja istnieje
    FUNCTION verify_sequence
      ( sequence_name_in      IN     VARCHAR2)
    RETURN BOOLEAN IS

      -- Domylna zwracana warto
      retval                  BOOLEAN := FALSE;

      -- Kursor zwraca pojedynczy wiersz po znalezieniu sekwencji
      CURSOR find_sequence IS
        SELECT   null
        FROM     user_objects
        WHERE    object_name = sequence_name_in;
      
    BEGIN

      -- Ptla for ustawia warto logiczn po znalezieniu sekwencji
      FOR i IN find_sequence LOOP
        retval := TRUE;
      END LOOP;

      -- Zwraca warto logiczn
      RETURN retval;

    END verify_sequence;

  BEGIN

    -- Sprawdza, czy sekwencja istnieje
    IF verify_sequence(sequence_name) = TRUE THEN

      -- Tworzy dynamiczn instrukcj SQL jako blok anonimowy jzyka PL/SQL
      statement := 'BEGIN'                                 ||CHR(10)
                || '  SELECT   PLSQL.'||sequence_name||'.nextval'||CHR(10)
                || '  INTO     :retval'                    ||CHR(10)
                || '  FROM     DUAL;'                      ||CHR(10)
                || 'END;';

      -- Wykonanie dynamicznej instrukcji SQL
      EXECUTE IMMEDIATE statement
        USING OUT sequence_value; 

      -- Wywietlenie nazwy moduu
      dbms_output.put_line(
        '-> nds_tutorial.increment_sequence');

      -- Wywietlenie wiersza rozdzielajcego
      dbms_output.put_line(sline);

      -- Wywietlenie danych wyjciowych
      dbms_output.put     ('Sekwencja <'||sequence_name||'> ');
      dbms_output.put_line('Wartosc <'||sequence_value||'>');

    ELSE

      -- Wywietlanie nazwy moduu
      dbms_output.put_line(
        '-> dbms_sql_tutorial.increment_sequence');

      -- Wywietlenie wiersza rozdzielajcego
      dbms_output.put_line(sline);

      -- Wywietlenie danych wyjciowych
      dbms_output.put_line(
        'Sekwencja <'||sequence_name||'> nie istnieje');

    END IF;

  END increment_sequence;

  /*
  || ------------------------------------------------------------------
  */

  -- Procedura demonstrujca instrukcj DML bez zmiennych powizanych
  PROCEDURE insert_into_table
    ( table_name              IN     VARCHAR2
    , table_column_value1     IN     NUMBER
    , table_column_value2     IN     VARCHAR2
    , table_column_value3     IN     VARCHAR2) IS

    -- Zmienna lokalna
    statement                 VARCHAR2(2000);

    -- Definicja funkcji lokalnej do sprawdzania, czy tabela istnieje
    FUNCTION verify_table
      ( object_name_in        IN     VARCHAR2)
    RETURN BOOLEAN IS

      -- Domylna zwracana warto
      retval                  BOOLEAN := FALSE;

      -- Kursor zwraca pojedynczy wiersz po znalezieniu tabeli
      CURSOR find_object IS
        SELECT   null
        FROM     user_objects
        WHERE    object_name = object_name_in;
      
    BEGIN

      -- Ptla for ustawia warto logiczn po znalezieniu tabeli
      FOR i IN find_object LOOP
        retval := TRUE;
      END LOOP;

      -- Zwraca warto logiczn
      RETURN retval;

    END verify_table;

  BEGIN

    -- Jeli tabela istnieje, naley wstawi do niej dane
    IF verify_table(table_name) = TRUE THEN

      -- Tworzy dynamiczn instrukcj SQL
      statement := 'INSERT '
                || 'INTO '||table_name||' '
                || 'VALUES ('
                || ''''||table_column_value1||''','
                || ''''||table_column_value2||''','
                || ''''||table_column_value3||''')';

      -- Wykonanie instrukcji NDS
      EXECUTE IMMEDIATE statement;

      -- Zatwierdzenie rekordw
      commit;

      -- Wywietlenie nazwy moduu
      dbms_output.put_line(
        '-> nds_tutorial.insert_into_table');

      -- Wywietlenie wiersza rozdzielajcego
      dbms_output.put_line(sline);

      -- Wywietlenie danych wyjciowych
      dbms_output.put_line(
        'Wstawiona wartosc <'||table_column_value1||'>');
      dbms_output.put_line(
        'Wstawiona wartosc <'||table_column_value2||'>');
      dbms_output.put_line(
        'Wstawiona wartosc <'||table_column_value3||'>');

    ELSE

      -- Wywietlenie nazwy moduu
      dbms_output.put_line(
        '-> nds_tutorial.insert_into_table');

      -- Wywietlenie wiersza rozdzielajcego
      dbms_output.put_line(sline);

      -- Wywietlenie danych wyjciowych
      dbms_output.put_line(
        'Obiekt <'||table_name||'> nie istnieje');

    END IF;

  END insert_into_table;

  /*
  || ------------------------------------------------------------------
  */

  -- Procedura demonstrujca instrukcj DML z uporzdkowanymi zmiennymi powizanymi
  PROCEDURE inserts_into_table
    ( table_name              IN     VARCHAR2
    , table_column_value1     IN     NUMBER
    , table_column_value2     IN     VARCHAR2
    , table_column_value3     IN     VARCHAR2) IS

    -- Zmienna lokalna
    statement                 VARCHAR2(2000);

    -- Definicja funkcji lokalnej do sprawdzania, czy tabela istnieje
    FUNCTION verify_table
      ( object_name_in        IN     VARCHAR2)
    RETURN BOOLEAN IS

      -- Domylna zwracana warto
      retval                  BOOLEAN := FALSE;

      -- Kursor zwraca pojedynczy wiersz po znalezieniu tabeli
      CURSOR find_object IS
        SELECT   null
        FROM     user_objects
        WHERE    object_name = object_name_in;
      
    BEGIN

      -- Ptla for ustawia warto logiczn po znalezieniu tabeli
      FOR i IN find_object LOOP
        retval := TRUE;
      END LOOP;

      -- Zwraca warto logiczn
      RETURN retval;

    END verify_table;

  BEGIN

    -- Jeli tabela istnieje, naley wstawi do niej dane
    IF verify_table(table_name) = TRUE THEN

      -- Tworzy dynamiczn instrukcj SQL
      statement := 'INSERT '
                || 'INTO '||table_name||' '
                || 'VALUES (:col_one, :col_two, :col_three)';

      -- Wykonanie instrukcji NDS
      EXECUTE IMMEDIATE statement
        USING table_column_value1
        ,     table_column_value2
        ,     table_column_value3;

      -- Zatwierdzenie rekordw
      commit;

      -- Wywietlenie nazwy moduu
      dbms_output.put_line(
        '-> nds_tutorial.insert_into_table');

      -- Wywietlenie wiersza rozdzielajcego
      dbms_output.put_line(sline);

      -- Wywietlenie danych wyjciowych
      dbms_output.put_line(
        'Wstawiona wartosc <'||table_column_value1||'>');
      dbms_output.put_line(
        'Wstawiona wartosc <'||table_column_value2||'>');
      dbms_output.put_line(
        'Wstawiona wartosc <'||table_column_value3||'>');

    ELSE

      -- Wywietlenie nazwy moduu
      dbms_output.put_line(
        '-> nds_tutorial.insert_into_table');

      -- Wywietlenie wiersza rozdzielajcego
      dbms_output.put_line(sline);

      -- Wywietlenie danych wyjciowych
      dbms_output.put_line(
        'Obiekt <'||table_name||'> nie istnieje');

    END IF;

  END inserts_into_table;

  /*
  || ------------------------------------------------------------------
  */

  -- Procedura demonstrujca wielowierszowe polecenie DQL
  PROCEDURE multiple_row_return IS

    -- Zmienne lokalne
    statement                 VARCHAR2(2000);
    value_out                 VARCHAR2_TABLE1;

  BEGIN

      -- Tworzy dynamiczn instrukcj SQL
    statement := 'BEGIN '
              || 'SELECT ''A'' '
              || 'BULK COLLECT INTO :col_val '
              || 'FROM DUAL;'
              || 'END;';

    -- Uycie masowej instrukcji NDS do skierowania zapytania do statycznego acucha
      EXECUTE IMMEDIATE statement
        USING OUT value_out;

      -- Wywietlenie nazwy moduu
      dbms_output.put_line(
        '-> nds_tutorial.multiple_row_return');

      -- Wywietlenie wiersza rozdzielajcego
      dbms_output.put_line(sline);

    -- Uycie ptli bazujcej na przedziale do wczytania wartoci
    FOR i IN 1..value_out.COUNT LOOP

      -- Wywietlenie danych wyjciowych
      dbms_output.put_line(
        'Wartosc z COLUMN_VALUE <'||value_out(i)||'>');

    END LOOP;

  END multiple_row_return;

  /*
  || ------------------------------------------------------------------
  */

  -- Procedura demonstrujca wielowierszowe polecenie DQL
  PROCEDURE multiple_row_return
    ( table_name    VARCHAR2
    , column_name1  VARCHAR2
    , column_name2  VARCHAR2
    , column_name3  VARCHAR2 )IS

    -- Zmienne lokalne NDS
    statement                 VARCHAR2(2000);
    cvalue_out1               CARD_NAME_VARRAY;
    cvalue_out2               CARD_SUIT_VARRAY;
    nvalue_out                CARD_NUMBER_VARRAY;

  BEGIN

      -- Tworzy dynamiczn instrukcj SQL
    statement := 'BEGIN '
              || 'SELECT '
              ||  column_name1 ||','
              ||  column_name2 ||','
              ||  column_name3 ||' '
              || 'BULK COLLECT INTO :col1, :col2, :col3 '
              || 'FROM '|| table_name ||';'
              || 'END;';

      -- Wykonanie instrukcji NDS
    EXECUTE IMMEDIATE statement
      USING OUT nvalue_out, OUT cvalue_out1, cvalue_out2;

      -- Wywietlenie nazwy moduu
      dbms_output.put_line('-> nds_tutorial.multiple_row_return');

      -- Wywietlenie wiersza rozdzielajcego
      dbms_output.put_line(sline);

      FOR i IN 1..nvalue_out.COUNT LOOP

      -- Wywietlenie danych wyjciowych
        dbms_output.put_line(
          'Wartosc z ['||column_name1||'] '||
          'to: ['||nvalue_out(i)||']');
        dbms_output.put_line(
          'Wartosc z ['||column_name1||'] '||
          'to: ['||SUBSTR(cvalue_out1(i),1,20)||']');
        dbms_output.put_line(
          'Wartosc z ['||column_name1||'] '||
          'to: ['||SUBSTR(cvalue_out2(i),1,30)||']');

      END LOOP;

  END multiple_row_return;

  /*
  || ------------------------------------------------------------------
  */

  -- Procedura demonstrujca jednowierszowe polecenie DQL
  PROCEDURE single_row_return IS

    -- Zmienne lokalne
    statement                 VARCHAR2(2000);
    value_out                 VARCHAR2(1);

  BEGIN

      -- Tworzy dynamiczn instrukcj SQL
    statement := 'SELECT ''A'' FROM DUAL';

    -- Uycie polecenia NDS do skierowania zapytania do statycznego acucha znakw
    EXECUTE IMMEDIATE statement
      INTO value_out;

      -- Wywietlenie nazwy moduu
    dbms_output.put_line('-> nds_tutorial.single_row_return');

      -- Wywietlenie wiersza rozdzielajcego
    dbms_output.put_line(sline);

      -- Wywietlenie danych wyjciowych
    dbms_output.put_line('Wartosc z COLUMN_VALUE <'||value_out||'>');

  END single_row_return;

  /*
  || ------------------------------------------------------------------
  */

  -- Procedura demonstrujca jednowierszowe polecenie DQL
  PROCEDURE single_row_return
    ( table_name    VARCHAR2
    , column_name1  VARCHAR2
    , column_name2  VARCHAR2
    , column_name3  VARCHAR2 ) IS

    -- Zmienne lokalne
    statement                 VARCHAR2(2000);
    cvalue_out1               VARCHAR2(20);
    cvalue_out2               VARCHAR2(30);
    nvalue_out                NUMBER;

  BEGIN

      -- Tworzy dynamiczn instrukcj SQL
    statement := 'SELECT '
              || column_name1 ||','
              || column_name2 ||','
              || column_name3 ||' '
              || 'FROM '|| table_name;

    EXECUTE IMMEDIATE statement
      INTO nvalue_out, cvalue_out1, cvalue_out2;

      -- Wywietlenie nazwy moduu
    dbms_output.put_line('-> nds_tutorial.single_row_return');

      -- Wywietlenie wiersza rozdzielajcego
    dbms_output.put_line(sline);

      -- Wywietlenie danych wyjciowych
    dbms_output.put_line(
      'Wartosc z COLUMN_VALUE <'||nvalue_out||'>');
    dbms_output.put_line(
      'Wartosc z COLUMN_VALUE <'||cvalue_out1||'>');
    dbms_output.put_line(
      'Wartosc z COLUMN_VALUE <'||cvalue_out2||'>');

  END single_row_return;

  /*
  || ------------------------------------------------------------------
  */

END nds_tutorial;
/

-- ==========================================================================
--  Zastosowano technik zarzdzania diagnozowaniem i dziennikami w celu
--  zapisu kompilowanego kodu i komunikatw o bdach.
--  Naley oznaczy ten fragment komentarzem, kiedy kod jest gotowy do udostpnienia,
--  a przy diagnozowaniu zmian ponownie usun oznaczenie komentarzem
-- ==========================================================================

SPOOL nds_tutorial_body.log

list

show errors

SPOOL OFF
