/*
 * dbms_sql.sql
 * Rozdzia 13, Oracle10g. Programowanie w jzyku PL/SQL
 * Ron Hardman, Mike McLaughlin i Scott Urman
 *
 * Cel:
 *   Dziaajcy samouczek dotyczcy pakietu wbudowanego
 *   DBMS_SQL z przykadami zastosowania najwaniejszych metod
 */

-- Konfiguracja rodowiska SQL*PLUS na potrzeby skryptu
SET SERVEROUTPUT ON SIZE 1000000
SET ECHO OFF
SET FEEDBACK OFF
SET LINESIZE 90
SET PAGESIZE 0

--  Umieszczenie pakietu w dzienniku
SELECT 'CREATE OR REPLACE PACKAGE dbms_sql_tutorial' FROM dual;

-- Tworzenie specyfikacji pakietu
CREATE OR REPLACE PACKAGE dbms_sql_tutorial AS

  /*
  || =========================================================================
  ||  PODSUMOWANIE:
  ||  -------------
  ||
  ||  Dziaajcy samouczek dotyczcy pakietu wbudowanego DBMS_SQL
  ||  z prostymi przykadami zastosowania rnych technik. Poniewa samouczek to
  ||  pojedynczy skrypt zawierajcy ciao i specyfikacj pakietu,
  ||  szczegowe informacje znajduj si tylko w specyfikacji
  ||
  ||  METODY        OPIS                               UYWANE PROGRAMY
  ||  -------       ------------------------------     -------------
  ||  Metoda 1      Instrukcje DDL i DML bez           EXECUTE
  ||                zmiennych powizanych ani          OPEN_CURSOR
  ||                polece DQL                        PARSE
  ||
  ||  Metoda 2      Instrukcje DML ze sta            BIND_VARIABLE
  ||                liczb zmiennych powizanych       EXECUTE
  ||                i bez polece DQL                  OPEN_CURSOR
  ||                                                   PARSE
  ||
  ||  Metoda 3      Polecenia DQL (zapytania) za       BIND_VARIABLE
  ||                sta liczb kolumn; ta metoda     COLUMN_VALUE
  ||                jest uywana zamiast               DEFINE_COLUMN
  ||                porednich i bezporednich         EXECUTE
  ||                kursorw                           FETCH_ROWS
  ||                                                   OPEN_CURSOR
  ||                                                   PARSE
  ||                                                   VARIABLE_VALUE
  ||
  ||  Metoda 4      Polecenia DQL (zapytania) ze       BIND_VARIABLE
  ||                zmienn liczb kolumn. W metodzie  COLUMN_VALUE
  ||                nie jest znana liczba ani          DEFINE_COLUMN
  ||                typy kolumn i zmiennych            EXECUTE
  ||                powizanych                        FETCH_ROWS
  ||                                                   OPEN_CURSOR
  ||                                                   PARSE
  ||                                                   VARIABLE_VALUE
  ||
  ||  ZAWARTO PAKIETU:
  ||  ----------------
  ||  Zawarto pakietu jest rozbita na zmienne globalne, funkcje
  ||  i procedury. Jeli dana procedura uywa metody DBMS_SQL,
  ||  wymieniony jest jej numer
  ||
  ||  NAZWA PROCEDURY              METODA  OPIS
  ||  --------------               ------  -----------
  ||  close_open_cursor                    Sprawdza, czy kursor jest otwarty,
  ||                                       przed uruchomieniem instrukcji DBMS_SQL.CLOSE_CURSOR(c).
  ||
  ||  create_sequence                #1    Demonstruje polecenie DDL (tworzenie
  ||                                       sekwencji) przy uyciu czenia
  ||
  ||  create_table                   #1    Demonstruje polecenie DDL (tworzenie
  ||                                       tabeli) przy uyciu czenia
  ||
  ||  drop_sequence                  #1    Demonstruje polecenie DDL (usuwanie
  ||                                       sekwencji) przy uyciu czenia
  ||
  ||  drop_table                     #1    Demonstruje polecenie DDL (usuwanie
  ||                                       tabeli) przy uyciu czenia
  ||
  ||  insert_into_table              #2    Demonstruje polecenie DML - wstawianie danych
  ||                                       do tabeli za pomoc uporzdkowanych
  ||                                       zmiennych powizanych
  ||
  ||  increment_sequence             #2    Demonstruje zwracanie zmiennych
  ||                                       z kursora przy uyciu struktury
  ||                                       SELECT-INTO jzyka PL/SQL. Ta
  ||                                       struktura jest ukryta w bloku jzyka
  ||                                       PL/SQL, a warto jest zwracana
  ||                                       przy uyciu zmiennej powizanej.
  ||
  ||  multiple_row_return            #3    Demonstruje zwracanie wielu wierszy
  ||                                       za pomoc jednostek programowych
  ||                                       DEFINE_COLUMN i COLUMN_VALUE, podobnie
  ||                                       jak przy uyciu kursorw bezporednich.
  ||
  ||  single_row_return              #3    Demonstruje zwracanie jednego wiersza
  ||                                       za pomoc jednostek programowych
  ||                                       DEFINE_COLUMN i COLUMN_VALUE, podobnie
  ||                                       jak przy uyciu kursorw bezporednich.
  ||
  || 
  || ------------------------------------------------------------------------
  ||  Standardowe lub konwencjonalne nazwy zmiennych w pakiecie DBMS_SQL i
  ||  dokumentacji Oracle oraz ksikach wydawnictw Oracle Press i O'Reilly
  || ------------------------------------------------------------------------
  ||  <c>         kursor przekazywany do DBMS_SQL
  ||  <fdbk>      liczba cakowita zwracana przez DBMS_SQL.EXECUTE lub
  ||                                          DBMS_SQL.EXECUTE_AND_FETCH
  ||  <retval>    warto zwracana przez funkcje
  ||  <statement> instrukcja SQL przekazywana do DBMS_SQL
  || =========================================================================
  */

  -- Definicje zmiennych formatujcych
  dline               VARCHAR2(80) := 
   '============================================================';
  sline               VARCHAR2(80) := 
   '------------------------------------------------------------';

  -- Procedura zamykajca otwarty kursor DBMS_SQL
  PROCEDURE close_open_cursor
    ( c                       IN OUT INTEGER);

  -- Procedura tworzca sekwencj przy uyciu czenia
  PROCEDURE create_sequence
    ( sequence_name           IN     VARCHAR2);

  -- Procedura tworzca tabel przy uyciu czenia
  PROCEDURE create_table
    ( table_name              IN     VARCHAR2
    , table_definition        IN     VARCHAR2);

  -- Procedura usuwajca sekwencj przy uyciu czenia
  PROCEDURE drop_sequence
    ( sequence_name           IN     VARCHAR2);

  -- Procedura usuwajca tabel przy uyciu czenia
  PROCEDURE drop_table
    ( table_name              IN     VARCHAR2);

  -- Procedura ukrywajca blok SELECT-INTO jzyka PL/SQL
  PROCEDURE increment_sequence
    ( sequence_name           IN     VARCHAR2
    , sequence_value          IN OUT NUMBER);

  -- Procedura demonstrujca instrukcj DML bez zmiennych powizanych
  PROCEDURE insert_into_table
    ( table_name              IN     VARCHAR2
    , table_column_value1     IN     NUMBER
    , table_column_value2     IN     VARCHAR2
    , table_column_value3     IN     VARCHAR2);

  -- Procedura demonstrujca instrukcj DML z uporzdkowanymi zmiennymi powizanymi
  PROCEDURE inserts_into_table
    ( table_name              IN     VARCHAR2
    , table_column_values1    IN     DBMS_SQL.NUMBER_TABLE
    , table_column_values2    IN     DBMS_SQL.VARCHAR2_TABLE
    , table_column_values3    IN     DBMS_SQL.VARCHAR2_TABLE);

  -- Procedura demonstrujca wielowierszowe polecenie DQL
  PROCEDURE multiple_row_return;

  -- Procedura demonstrujca wielowierszowe polecenie DQL
  PROCEDURE multiple_row_return
    ( table_name    VARCHAR2
    , column_name1  VARCHAR2
    , column_name2  VARCHAR2
    , column_name3  VARCHAR2 );

  -- Procedura demonstrujca jednowierszowe polecenie DQL
  PROCEDURE single_row_return;

  -- Procedura demonstrujca jednowierszowe polecenie DQL
  PROCEDURE single_row_return
    ( table_name    VARCHAR2
    , column_name1  VARCHAR2
    , column_name2  VARCHAR2
    , column_name3  VARCHAR2 );

END dbms_sql_tutorial;
/

-- ==========================================================================
--  Zastosowano technik zarzdzania diagnozowaniem i dziennikami w celu
--  zapisu kompilowanego kodu i komunikatw o bdach.
--  Naley oznaczy ten fragment komentarzem, kiedy kod jest gotowy do udostpnienia,
--  a przy diagnozowaniu zmian ponownie usun oznaczenie komentarzem
-- ==========================================================================

SPOOL dbms_sql_spec.log

list

show errors

SPOOL OFF

--  Umieszczanie ciaa pakietu w dzienniku
SELECT 'CREATE OR REPLACE PACKAGE BODY dbms_sql_tutorial' FROM dual;

-- Tworzenie ciaa pakietu
CREATE OR REPLACE PACKAGE BODY dbms_sql_tutorial IS

  /*
  || =========================================================================
  ||  PODSUMOWANIE:
  ||  -------
  ||  Dziaajcy samouczek dotyczcy pakietu wbudowanego DBMS_SQL
  ||  z prostymi przykadami zastosowania rnych technik.
  ||
  ||  Ten fragment ma kierowa programistw do specyfikacji pakietu,
  ||  jeli poszukuj szczegowych informacji. Poniewa programistom
  ||  zarzdzajcym duymi pakietami czsto trudno si po nich porusza,
  ||  procedury i funkcje s rozdzielone pustym wierszem
  || =========================================================================
  */

  /*
  || ------------------------------------------------------------------
  */

  -- Procedura zamykajca otwarty kursor DBMS_SQL
  PROCEDURE close_open_cursor
    ( c                       IN OUT INTEGER) IS

  BEGIN

    /*
    || Jeli kursor jest otwarty, naley go zamkn
    */

    IF dbms_sql.is_open(c) THEN
      dbms_sql.close_cursor(c);
    END IF;

  END close_open_cursor;

  /*
  || ------------------------------------------------------------------
  */

  -- Procedura tworzca sekwencj przy uyciu czenia
  PROCEDURE create_sequence
    ( sequence_name           IN     VARCHAR2) IS

    -- Definicja zmiennych lokalnych pakietu DBMS_SQL
    c               INTEGER := dbms_sql.open_cursor;
    fdbk            INTEGER;
    statement       VARCHAR2(2000);

    -- Definicja funkcji lokalnej do sprawdzania, czy sekwencja nie istnieje
    FUNCTION verify_not_sequence
      ( sequence_name_in      IN     VARCHAR2)
    RETURN BOOLEAN IS

      -- Domylna zwracana warto
      retval                  BOOLEAN := TRUE;

      -- Kursor zwraca pojedynczy wiersz po znalezieniu sekwencji
      CURSOR find_sequence IS
        SELECT   null
        FROM     user_objects
        WHERE    object_name = sequence_name_in;
      
    BEGIN

      -- Ptla for ustawia warto logiczn po znalezieniu sekwencji
      FOR i IN find_sequence LOOP
        retval := FALSE;
      END LOOP;

      -- Zwraca warto logiczn
      RETURN retval;

    END verify_not_sequence;

  BEGIN

    -- Jeli sekwencja nie istnieje, trzeba j utworzy
    IF verify_not_sequence(sequence_name) = TRUE THEN

      -- Tworzy dynamiczn instrukcj SQL
      statement := 'CREATE SEQUENCE '||sequence_name||CHR(10)
                || '  INCREMENT BY   1'             ||CHR(10)
                || '  START WITH     1'             ||CHR(10)
                || '  CACHE          20'            ||CHR(10)
                || '  ORDER';

      -- Przetwarzanie i wykonywanie utworzonej instrukcji
      dbms_sql.parse(c,statement,dbms_sql.native);
      fdbk := dbms_sql.execute(c);
  
      -- Zamknicie otwartego kursora
      dbms_sql.close_cursor(c);

      -- Wywietlenie nazwy moduu
      dbms_output.put_line(
        '-> dbms_sql_tutorial.create_sequence');

      -- Wywietlenie wiersza rozdzielajcego
      dbms_output.put_line(sline);

      -- Wywietlenie danych wyjciowych
      dbms_output.put_line(
        'Utworzono sekwencj <'||sequence_name||'>');

    ELSE

      -- Wywietlenie nazwy moduu
      dbms_output.put_line(
        '-> dbms_sql_tutorial.create_sequence');

      -- Wywietlenie wiersza rozdzielajcego
      dbms_output.put_line(sline);

      -- Wywietlenie danych wyjciowych
      dbms_output.put_line(
        'Sekwencja <'||sequence_name||'> ju istnieje');

    END IF;

  END create_sequence;

  /*
  || ------------------------------------------------------------------
  */

  -- Procedura tworzca tabel przy uyciu czenia
  PROCEDURE create_table
    ( table_name              IN     VARCHAR2
    , table_definition        IN     VARCHAR2) IS

    -- Definicja zmiennych lokalnych DBMS_SQL
    c                         INTEGER := dbms_sql.open_cursor;
    fdbk                      INTEGER;
    statement                 VARCHAR2(2000);

    -- Definicja funkcji lokalnej do sprawdzania, czy tabela nie istnieje
    FUNCTION verify_not_table
      ( object_name_in        IN     VARCHAR2)
    RETURN BOOLEAN IS

      -- Definicja domylnie zwracanej wartoci
      retval                  BOOLEAN := TRUE;

      -- Kursor zwraca jeden wiersz po znalezieniu sekwencji
      CURSOR find_object IS
        SELECT   null
        FROM     user_objects
        WHERE    object_name = object_name_in;
      
    BEGIN

      -- Ptla for ustawia warto logiczn po znalezieniu sekwencji
      FOR i IN find_object LOOP
        retval := FALSE;
      END LOOP;

      -- Zwraca warto logiczn
      RETURN retval;

    END verify_not_table;

  BEGIN

    -- Jeli sekwencja nie istnieje, trzeba j utworzy
    IF verify_not_table(table_name) = TRUE THEN

      -- Tworzy dynamiczn instrukcj SQL
      statement := 'CREATE TABLE '||table_name||CHR(10)
                || table_definition;

      /*
      || Wskazwka diagnostyczna:
      || =============
      || Przetwarzanie instrukcji, co obejmuje wczytanie poczonego acucha
      || do zmiennej statement i wykonanie instrukcji. Moe wystpi
      || bd braku uprawnie, jeli wykonujcy uytkownik nie ma uprawnie
      || "CREATE TABLE". Mona je przyzna danemu uytkownikowi, uywajc konta
      || SYS. Bdy tego typu wystpuj, jeli schemat wykonujcy
      || dane polecenie DDL otrzyma uprawnienia poprzez rol,
      || a nie zostay mu one przyznane bezporednio
      || -------------------------------------------------------------------
      ||   ORA-01031: insufficient privileges 
      */

      -- Przetwarzanie i wykonywanie utworzonej instrukcji
      dbms_sql.parse(c,statement,dbms_sql.native);
      fdbk := dbms_sql.execute(c);
  
      -- Zamknicie otwartego kursora
      dbms_sql.close_cursor(c);

      -- Wywietlenie nazwy moduu
      dbms_output.put_line(
        '-> dbms_sql_tutorial.create_table');

      -- Wywietlenie wiersza rozdzielajcego
      dbms_output.put_line(sline);

      -- Wywietlenie danych wyjciowych
      dbms_output.put_line(
        'Utworzono tabel <'||table_name||'>');

    ELSE

      -- Wywietlenie nazwy moduu
      dbms_output.put_line(
        '-> dbms_sql_tutorial.create_table');

      -- Wywietlenie wiersza rozdzielajcego
      dbms_output.put_line(sline);

      -- Wywietlenie danych wyjciowych
      dbms_output.put_line(
        'Obiekt <'||table_name||'> ju istnieje');

    END IF;

  END create_table;

  /*
  || ------------------------------------------------------------------
  */

  -- Procedura usuwajca sekwencj przy uyciu czenia
  PROCEDURE drop_sequence
    ( sequence_name           IN     VARCHAR2) IS

    --Definicje zmiennych lokalnych pakietu DBMS_SQL
    c                         INTEGER := dbms_sql.open_cursor;
    fdbk                      INTEGER;
    statement                 VARCHAR2(2000);

    -- Definicja funkcji lokalnej do sprawdzania, czy sekwencja istnieje
    FUNCTION verify_sequence
      ( sequence_name_in      IN     VARCHAR2)
    RETURN BOOLEAN IS

      -- Domylna zwracana warto
      retval                  BOOLEAN := FALSE;

      -- Kursor zwraca jeden wiersz po znalezieniu sekwencji
      CURSOR find_sequence IS
        SELECT   null
        FROM     user_objects
        WHERE    object_name = sequence_name_in;
      
    BEGIN

      -- Ptla for ustawia warto logiczn po znalezieniu sekwencji
      FOR i IN find_sequence LOOP
        retval := TRUE;
      END LOOP;

      -- Zwraca warto logiczn
      RETURN retval;

    END verify_sequence;

  BEGIN

    -- Jeli sekwencja istnieje, trzeba j usun
    IF verify_sequence(sequence_name) = TRUE THEN

      -- Tworzy dynamiczn instrukcj SQL
      statement := 'DROP SEQUENCE '||sequence_name;

      -- Przetwarzanie i wykonywanie utworzonej instrukcji
      dbms_sql.parse(c,statement,dbms_sql.native);
      fdbk := dbms_sql.execute(c);

      -- Zamknicie otwartego kursora
      dbms_sql.close_cursor(c);

      -- Wywietlenie nazwy moduu
      dbms_output.put_line(
        '-> dbms_sql_tutorial.drop_sequence');

      -- Wywietlenie wiersza rozdzielajcego
      dbms_output.put_line(sline);

      -- Wywietlenie danych wyjciowych
      dbms_output.put_line(
        'Usunito sekwencj <'||sequence_name||'>');

    ELSE

      -- Wywietlenie nazwy moduu
      dbms_output.put_line(
        '-> dbms_sql_tutorial.drop_sequence');

      -- Wywietlenie wiersza rozdzielajcego
      dbms_output.put_line(sline);

      -- Wywietlenie danych wyjciowych
      dbms_output.put_line(
        'Sekwencja <'||sequence_name||'> nie istnieje');

    END IF;

  END drop_sequence;

  /*
  || ------------------------------------------------------------------
  */

  -- Procedure drops a table using concatenation.
  PROCEDURE drop_table
    ( table_name              IN     VARCHAR2) IS

    -- Definicja zmiennych lokalnych pakietu DBMS_SQL
    c                         INTEGER := dbms_sql.open_cursor;
    fdbk                      INTEGER;
    statement                 VARCHAR2(2000);

    -- Definicja funkcji lokalnej do sprawdzania, czy tabela istnieje
    FUNCTION verify_table
      ( object_name_in        IN     VARCHAR2)
    RETURN BOOLEAN IS

      -- Domylna zwracana warto
      retval                  BOOLEAN := FALSE;

      -- Kursor zwraca jeden wiersz po znalezieniu tabeli
      CURSOR find_object IS
        SELECT   null
        FROM     user_objects
        WHERE    object_name = object_name_in;
      
    BEGIN

      -- Ptla for ustawia warto logiczn po znalezieniu tabeli
      FOR i IN find_object LOOP
        retval := TRUE;
      END LOOP;

      -- Zwraca warto logiczn
      RETURN retval;

    END verify_table;

  BEGIN

    IF verify_table(table_name) = TRUE THEN

      -- Tworzy dynamiczn instrukcj SQL
      statement := 'DROP TABLE '||table_name;

      -- Przetwarzanie i wykonywanie utworzonej instrukcji
      dbms_sql.parse(c,statement,dbms_sql.native);
      fdbk := dbms_sql.execute(c);

      -- Zamknicie otwartego kursora
      dbms_sql.close_cursor(c);

      -- Wywietlenie nazwy moduu
      dbms_output.put_line('
        -> dbms_sql_tutorial.drop_table');

      -- Wywietlenie wiersza rozdzielajcego
      dbms_output.put_line(sline);

      -- Wywietlenie danych wyjciowych
      dbms_output.put_line(
        'Usunito tabel <'||table_name||'>');

    ELSE

      -- Wywietlenie nazwy moduu
      dbms_output.put_line(
        '-> dbms_sql_tutorial.drop_table');

      -- Wywietlenie wiersza rozdzielajcego
      dbms_output.put_line(sline);

      -- Wywietlenie danych wyjciowych
      dbms_output.put_line(
        'Obiekt <'||table_name||'> nie istnieje');

    END IF;

  END drop_table;

  /*
  || ------------------------------------------------------------------
  */

  -- Ta procedura ukrywa blok SELECT-INTO jzyka PL/SQL
  PROCEDURE increment_sequence
    ( sequence_name           IN     VARCHAR2
    , sequence_value          IN OUT NUMBER  ) IS

    -- Definicja zmiennych lokalnych pakietu DBMS_SQL
    c                         INTEGER := dbms_sql.open_cursor;
    fdbk                      INTEGER;
    statement                 VARCHAR2(2000);

    -- Definicja funkcji lokalnej do sprawdzania, czy sekwencja istnieje
    FUNCTION verify_sequence
      ( sequence_name_in      IN     VARCHAR2)
    RETURN BOOLEAN IS

      -- Domylna zwracana warto
      retval                  BOOLEAN := FALSE;

      -- Kursor zwraca jeden wiersz po znalezieniu sekwencji
      CURSOR find_sequence IS
        SELECT   null
        FROM     user_objects
        WHERE    object_name = sequence_name_in;
      
    BEGIN

      -- Ptla for ustawia warto logiczn po znalezieniu sekwencji
      FOR i IN find_sequence LOOP
        retval := TRUE;
      END LOOP;

      -- Zwraca warto logiczn
      RETURN retval;

    END verify_sequence;

  BEGIN

    IF verify_sequence(sequence_name) = TRUE THEN

      /*
      || Wskazwka diagnostyczna:
      || =============
      || Uywajc w pakiecie DBMS_SQL bloku SELECT-INTO-FROM, ktra to skadnia
      || jest specyficzna dla PL/SQL i nie jest bezporednio obsugiwana w DBMS_SQL,
      || naley zastosowa nakadk w jzyku PL/SQL.  W takiej nakadce
      || trzeba uy rednikw w instrukcji i bloku jzyka
      || PL/SQL, poniewa DBMS_SQL automatycznie dodaje pojedynczy rednik, aby
      || uruchomi blok kodu PL/SQL. Jeli zapomnisz ukry kod SQL w
      || nakadce jzyka PL/SQL, wystpi nastpujcy bd:
      || -------------------------------------------------------------------
      || ORA-01006: bind variable does not exist
      */

      -- Tworzy dynamiczn instrukcj SQL jako blok anonimowy jzyka PL/SQL
      statement := 'BEGIN'||CHR(10)
                || ' SELECT PLSQL.'||sequence_name||'.nextval'||CHR(10)
                || ' INTO   :retval'||CHR(10)
                || ' FROM   DUAL;'||CHR(10)
                || 'END;';

      -- Przetwarzanie instrukcji
      dbms_sql.parse(c,statement,dbms_sql.native);

      /*
      || Uwaga techniczna:
      || ==============
      || Procedura BIND_VARIABLE zwraca warto typu NUMBER
      || i nie wymaga czwartego parametru
      */

      -- Wizanie zmiennej retval z wyjciow wartoci sekwencji
      dbms_sql.bind_variable(c,'retval',sequence_value);

      -- Wykonywanie dynamicznego kursora
      fdbk := dbms_sql.execute(c);

      -- Kopiowanie wartoci ze zmiennej powizanej
      dbms_sql.variable_value(c,'retval',sequence_value);

      -- Zamknicie otwartego kursora
      dbms_sql.close_cursor(c);

      -- Wywietlenie nazwy moduu
      dbms_output.put(
        'Sekwencja <'||sequence_name||'> ');

      -- Wywietlenie danych wyjciowych
      dbms_output.put_line(
        'Warto <'||sequence_value||'>');

    ELSE

      -- Wywietlenie nazwy moduu
      dbms_output.put_line(
        '-> dbms_sql_tutorial.increment_sequence');

      -- Wywietlenie wiersza rozdzielajcego
      dbms_output.put_line(sline);

      -- Wywietlenie danych wyjciowych
      dbms_output.put_line(
        'Sekwencja <'||sequence_name||'> nie istnieje');

    END IF;

  END increment_sequence;

  /*
  || ------------------------------------------------------------------
  */

  -- Procedura demonstrujca polecenie DML z uporzdkowanymi zmiennymi powizanymi
  PROCEDURE insert_into_table
    ( table_name              IN     VARCHAR2
    , table_column_value1     IN     NUMBER
    , table_column_value2     IN     VARCHAR2
    , table_column_value3     IN     VARCHAR2) IS

    -- Definicja zmiennych lokalnych pakietu DBMS_SQL
    c                         INTEGER := dbms_sql.open_cursor;
    fdbk                      INTEGER;
    statement                 VARCHAR2(2000);

    -- Definicja funkcji lokalnej do sprawdzania, czy tabela istnieje
    FUNCTION verify_table
      ( object_name_in        IN     VARCHAR2)
    RETURN BOOLEAN IS

      -- Domylna zwracana warto
      retval                  BOOLEAN := FALSE;

      -- Kursor zwraca jeden wiersz po znalezieniu tabeli
      CURSOR find_object IS
        SELECT   null
        FROM     user_objects
        WHERE    object_name = object_name_in;
      
    BEGIN

      -- Ptla for ustawia warto logiczn po znalezieniu tabeli
      FOR i IN find_object LOOP
        retval := TRUE;
      END LOOP;

      -- Zwraca warto logiczn
      RETURN retval;

    END verify_table;

  BEGIN

    -- Jeli tabela istnieje, naley wstawi do niej dane
    IF verify_table(table_name) = TRUE THEN

      /*
      || Wskazwka diagnostyczna:
      || =============
      || acuchy z instrukcj kocz si symbolem koca wiersza CHR(10), co
      || gwarantuje odstp pomidzy czonymi fragmentami. Uywanie
      || zmiennych powizanych poprawia wydajno instrukcji SQL,
      || poniewa pozwala unikn ich ponownego przetwarzania.  Dlatego
      || naley ich uywa w opisanyc poniej sposb:
      ||
      ||   INSTRUKCJE SQL    PREDYKATY
      ||   --------------    ----------
      ||   SELECT            WHERE
      ||   UPDATE            SET
      ||                     WHERE
      ||   DELETE            WHERE
      ||
      || Objanienia bdw:
      || ------------------
      || 1. W zmiennych VARCHAR2 zawsze trzeba jawnie okreli rozmiar,
      ||    a w procedurach przecionych ten rozmiar znajduje
      ||    si na czwartej pozycji. Poniej okrelony jest wyjciowy rozmiar,
      ||    co demonstruje t technik
      || 2. Komunikat o niepoprawnej zmiennej powizanej zwykle oznacza, e identyfikator
      ||    znajduje si poza acuchem typu VARCHAR2 i jest traktowany jako niezdefiniowana
      ||    zmienna powizana z poziomu sesji
      || 3. Bd "missing SELECT keyword" moe wystpi wtedy, kiedy w klauzuli INTO
      ||    w instrukcji INSERT zostanie umieszczona zmienna powizana dla
      ||    nazw kolumn
      || 4. Jeli wok zmiennej powizane typu VARCHAR2 znajduj si cudzysowy,
      ||    moe wystpi bd "bind variable does not exist".  Jeli trzeba uy
      ||    tej skadni, mona ukry polecenie DML w
      ||    nakadce jzyka PLSQL
      || -------------------------------------------------------------------
      || 1. ORA-06502: PL/SQL: numeric or value error
      || 2. PLS-00049: bad bind variable
      || 3. ORA-00928: missing SELECT keyword
      || 4. ORA-01006: bind variable does not exist
      */

      -- Tworzenie dynamicznej instrukcji SQL
      statement := 'INSERT '
                || 'INTO '||table_name||' '
                || 'VALUES '
                || '( :table_column_value1'
                || ', :table_column_value2'
                || ', :table_column_value3)';

      -- Przetwarzanie instrukcji
      dbms_sql.parse(c,statement,dbms_sql.native);

      -- Wizanie zmiennych powizanych
      dbms_sql.bind_variable(c,'table_column_value1',table_column_value1);
      dbms_sql.bind_variable(c,'table_column_value2',table_column_value2);
      dbms_sql.bind_variable(c,'table_column_value3',table_column_value3);

      -- Wykonywanie dynamicznej instrukcji
      fdbk := dbms_sql.execute(c);

      -- Zamknicie otwartego kursora
      dbms_sql.close_cursor(c);

      -- Zatwierdzanie rekordw
      commit;

      -- Wywietlenie nazwy moduu
      dbms_output.put_line(
        '-> dbms_sql_tutorial.insert_into_table');

      -- Wywietlenie wiersza rozdzielajcego
      dbms_output.put_line(sline);

      -- Wywietlenie danych wyjciowych
      dbms_output.put_line(
        'Wstawiono warto <'||table_column_value1||'>');
      dbms_output.put_line(
        'Wstawiono warto <'||table_column_value2||'>');
      dbms_output.put_line(
        'Wstawiono warto <'||table_column_value3||'>');

    ELSE

      -- Wywietlenie nazwy moduu
      dbms_output.put_line(
        '-> dbms_sql_tutorial.insert_into_table');

      -- Wywietlenie wiersza rozdzielajcego
      dbms_output.put_line(sline);

      -- Wywietlenie danych wyjciowych
      dbms_output.put_line(
        'Obiekt <'||table_name||'> nie istnieje');

    END IF;

  END insert_into_table;

  /*
  || ------------------------------------------------------------------
  */

  -- Procedura demonstrujca polecenie DML z uporzdkowanymi zmiennymi powizanymi
  PROCEDURE inserts_into_table
    ( table_name              IN     VARCHAR2
    , table_column_values1    IN     DBMS_SQL.NUMBER_TABLE
    , table_column_values2    IN     DBMS_SQL.VARCHAR2_TABLE
    , table_column_values3    IN     DBMS_SQL.VARCHAR2_TABLE) IS

    -- Definicja zmiennych lokalnych pakietu DBMS_SQL
    c                         INTEGER := dbms_sql.open_cursor;
    fdbk                      INTEGER;
    statement                 VARCHAR2(2000);

    -- Definicja funkcji lokalnej do sprawdzania, czy tabela istnieje
    FUNCTION verify_table
      ( object_name_in        IN     VARCHAR2)
    RETURN BOOLEAN IS

      -- Domylna zwracana warto
      retval                  BOOLEAN := FALSE;

      -- Kursor zwraca jeden wiersz po znalezieniu tabeli
      CURSOR find_object IS
        SELECT   null
        FROM     user_objects
        WHERE    object_name = object_name_in;
      
    BEGIN

      -- Ptla for ustawia warto logiczn po znalezieniu tabeli
      FOR i IN find_object LOOP
        retval := TRUE;
      END LOOP;

      -- Zwraca warto logiczn
      RETURN retval;

    END verify_table;

  BEGIN

    -- Jeli tabela istnieje, trzeba wstawi do niej dane
    IF verify_table(table_name) = TRUE THEN

      /*
      || Wskazwka diagnostyczna:
      || =============
      || acuchy z instrukcj kocz si symbolem koca wiersza CHR(10), co
      || gwarantuje odstp pomidzy czonymi fragmentami. Uywanie
      || zmiennych powizanych poprawia wydajno instrukcji SQL,
      || poniewa pozwala unikn ich ponownego przetwarzania.  Dlatego
      || naley ich uywa w opisanyc poniej sposb:
      ||
      ||   INSTRUKCJE SQL    PREDYKATY
      ||   --------------    ----------
      ||   SELECT            WHERE
      ||   UPDATE            SET
      ||                     WHERE
      ||   DELETE            WHERE
      ||
      || Objanienia bdw:
      || ------------------
      || 1. W zmiennych VARCHAR2 zawsze trzeba jawnie okreli rozmiar,
      ||    a w procedurach przecionych ten rozmiar znajduje
      ||    si na czwartej pozycji. Poniej okrelony jest wyjciowy rozmiar,
      ||    co demonstruje t technik
      || 2. Komunikat o niepoprawnej zmiennej powizanej zwykle oznacza, e identyfikator
      ||    znajduje si poza acuchem typu VARCHAR2 i jest traktowany jako niezdefiniowana
      ||    zmienna powizana z poziomu sesji
      || 3. Bd "missing SELECT keyword" moe wystpi wtedy, kiedy w klauzuli INTO
      ||    w instrukcji INSERT zostanie umieszczona zmienna powizana dla
      ||    nazw kolumn
      || 4. Jeli wok zmiennej powizane typu VARCHAR2 znajduj si cudzysowy,
      ||    moe wystpi bd "bind variable does not exist".  Jeli trzeba uy
      ||    tej skadni, mona ukry polecenie DML w
      ||    nakadce jzyka PLSQL
      || -------------------------------------------------------------------
      || 1. ORA-06502: PL/SQL: numeric or value error
      || 2. PLS-00049: bad bind variable
      || 3. ORA-00928: missing SELECT keyword
      || 4. ORA-01006: bind variable does not exist
      */

      -- Tworzenie dynamicznych instrukcji SQL
      statement := 'INSERT '
                || 'INTO '||table_name||' '
                || '( card_number '
                || ', card_name '
                || ', card_suit)'
                || 'VALUES '
                || '( :card_number'
                || ', :card_name'
                || ', :card_suit)';

      -- Przetwarzanie instrukcji
      dbms_sql.parse(c,statement,dbms_sql.native);

      -- Wizanie zmiennych
      dbms_sql.bind_array(c,'card_number',table_column_values1);
      dbms_sql.bind_array(c,'card_name',table_column_values2);
      dbms_sql.bind_array(c,'card_suit',table_column_values3);

      -- Wykonywanie dynamicznej instrukcji
      fdbk := dbms_sql.execute(c);

      -- Wywietlanie liczby wstawionych rekordw
      dbms_output.put_line('Wstawiono ['||fdbk||'].');

      -- Zamknicie otwartego kursora
      dbms_sql.close_cursor(c);

      -- Zatwierdzanie rekordw
      commit;

      -- Wywietlenie nazwy moduu
      dbms_output.put_line('-> dbms_sql_tutorial.inserts_into_table');

      -- Wywietlenie wiersza rozdzielajcego
      dbms_output.put_line(sline);

      -- Uywanie ptli for do wywietlania wartoci
      FOR i IN 1..table_column_values1.COUNT LOOP

        -- Wywietlenie danych wyjciowych
        dbms_output.put_line(
          'Wstawiono warto <'||table_column_values1(i)||'>');
        dbms_output.put_line(
          'Wstawiono warto <'||table_column_values2(i)||'>');
        dbms_output.put_line(
          'Wstawiono warto <'||table_column_values3(i)||'>');

      END LOOP;

    ELSE

      -- Wywietlenie nazwy moduu
      dbms_output.put_line(
        '-> dbms_sql_tutorial.inserts_into_table');

      -- Wywietlenie wiersza rozdzielajcego
      dbms_output.put_line(sline);

      -- Wywietlenie danych wyjciowych
      dbms_output.put_line(
        'Obiekt <'||table_name||'> nie istnieje');

    END IF;

  END inserts_into_table;

  /*
  || ------------------------------------------------------------------
  */

  -- Ta procedura demonstruje wielowierszowe polecenie DQL
  PROCEDURE multiple_row_return IS

    -- Definicja zmiennych lokalnych pakietu DBMS_SQL
    c                         INTEGER := dbms_sql.open_cursor;
    fdbk                      INTEGER;
    statement                 VARCHAR2(2000);
    value_out                 VARCHAR2(1);

  BEGIN

    --Tworzenie dynamicznej instrukcji SQL
    statement := 'SELECT ''A'' FROM DUAL';

    -- Przetwarzanie dynamicznej instrukcji SQL
    dbms_sql.parse(c,statement,dbms_sql.native);

    /*
    || Wskazwka diagnostyczna:
    || =============
    || Zdefiniuj wartoci kolumn i NIE zapomniej przypisa parametru okrelajcego
    || rozmiar acuchowych typw danych, takich jak VARCHAR2; jeli tego nie zrobisz,
    || wystpi nastpujcy bd:
    || -------------------------------------------------------------------
    || PLS-00307: too many declarations of 'DEFINE_COLUMN' match this call
    */

    -- Odwzorowanie kolumny na zmienn value_out
    dbms_sql.define_column(c,1,value_out,1);

    -- Wykonywanie dynamicznej instrukcji
    fdbk := dbms_sql.execute(c);

    -- Wczytanie wszystkich wierszy w ptli
    LOOP

      -- Koczenie dziaania, kiedy nie mona ju pobra wicej wierszy
      EXIT WHEN dbms_sql.fetch_rows(c) = 0;

      -- Kopiowanie zawartoci kolumny 1. do zmiennej value_out
      dbms_sql.column_value(c,1,value_out);

      -- Wywietlenie nazwy moduu
      dbms_output.put_line('-> dbms_sql_tutorial.multiple_row_return');

      -- Wywietlenie wiersza rozdzielajcego
      dbms_output.put_line(sline);

      -- Wywietlenie danych wyjciowych
      dbms_output.put_line('Warto z kolumny COLUMN_VALUE <'||value_out||'>');

    END LOOP;

    -- Zamknicie otwartego kursora
    dbms_sql.close_cursor(c);

  END multiple_row_return;

  /*
  || ------------------------------------------------------------------
  */

  -- Pobieranie wielu wierszy za pomoc polecenia DQL
  PROCEDURE multiple_row_return
    ( table_name    VARCHAR2
    , column_name1  VARCHAR2
    , column_name2  VARCHAR2
    , column_name3  VARCHAR2 )IS

    -- Definicja zmiennych lokalnych pakietu DBMS_SQL
    c                         INTEGER := dbms_sql.open_cursor;
    fdbk                      INTEGER;
    statement                 VARCHAR2(2000);
    cvalue_out1               VARCHAR2(2000);
    cvalue_out2               VARCHAR2(2000);
    nvalue_out                NUMBER;

  BEGIN

    -- Tworzenie dynamicznej instrukcji SQL
    statement := 'SELECT '
              ||  column_name1 ||','
              ||  column_name2 ||','
              ||  column_name3 ||' '
              || 'FROM '|| table_name;

    -- Przetwarzanie dynamicznej instrukcji SQL
    dbms_sql.parse(c,statement,dbms_sql.native);

    /*
    || Wskazwka diagnostyczna:
    || =============
    || Zdefiniuj wartoci kolumn i NIE zapomniej przypisa parametru okrelajcego
    || rozmiar acuchowych typw danych, takich jak VARCHAR2; jeli tego nie zrobisz,
    || wystpi nastpujcy bd:
    || -------------------------------------------------------------------
    || PLS-00307: too many declarations of 'DEFINE_COLUMN' match this call
    */

    -- Definiowanie kolumn odpowiadajcych wartociom zmiennych value_out
    dbms_sql.define_column(c,1,nvalue_out);
    dbms_sql.define_column(c,2,cvalue_out1,2000);
    dbms_sql.define_column(c,3,cvalue_out2,2000);

    -- Wykonywanie dynamicznej instrukcji
    fdbk := dbms_sql.execute(c);

    -- Uycie ptli do wczytania wszystkich wierszy
    LOOP

      -- Koczenie dziaania, kiedy nie ma ju wicej wierszy
      EXIT WHEN dbms_sql.fetch_rows(c) = 0;

      -- Kopiowanie zawartoci kolumny 1. do zmiennej value_out
      dbms_sql.column_value(c,1,nvalue_out);
      dbms_sql.column_value(c,2,cvalue_out1);
      dbms_sql.column_value(c,3,cvalue_out2);

      -- Wywietlenie nazwy moduu
      dbms_output.put_line(
        '-> dbms_sql_tutorial.multiple_row_return');

      -- Wywietlenie wiersza rozdzielajcego
      dbms_output.put_line(sline);

      -- Wywietlenie danych wyjciowych
      dbms_output.put_line(
        'Wartosc z ['||column_name1||'] '||
        'to: ['||nvalue_out||']');
      dbms_output.put_line(
        'Wartosc z ['||column_name1||'] '||
        'to: ['||SUBSTR(cvalue_out1,1,20)||']');
      dbms_output.put_line(
        'Wartosc z ['||column_name1||'] '||
        'to: ['||SUBSTR(cvalue_out2,1,30)||']');

    END LOOP;

    -- Zamknicie otwartego kursora
    dbms_sql.close_cursor(c);

  END multiple_row_return;

  /*
  || ------------------------------------------------------------------
  */

  /*
  || Zwracanie jednego wiersza przy uyciu jednostek programowych
  || DEFINE_COLUMN i COLUMN_VALUE, podobnie jak w przypadku kursorw bezporednich
  */
  -- Procedura demonstrujca jednowierszowe polecenie DQL
  PROCEDURE single_row_return IS

    -- Definicja zmiennych lokalnych pakietu DBMS_SQL
    c                         INTEGER := dbms_sql.open_cursor;
    fdbk                      INTEGER;
    statement                 VARCHAR2(2000);
    value_out                 VARCHAR2(1);

  BEGIN

    -- Tworzenie dynamicznej instrukcji SQL
    statement := 'SELECT ''A'' FROM DUAL';

    -- Przetwarzanie dynamicznej instrukcji SQL
    dbms_sql.parse(c,statement,dbms_sql.native);

    /*
    || Wskazwka diagnostyczna:
    || =============
    || Zdefiniuj wartoci kolumn i NIE zapomniej przypisa parametru okrelajcego
    || rozmiar acuchowych typw danych, takich jak VARCHAR2; jeli tego nie zrobisz,
    || wystpi nastpujcy bd:
    || -------------------------------------------------------------------
    || PLS-00307: too many declarations of 'DEFINE_COLUMN' match this call
    ||
    || Ten komunikat zostanie zwrcony, poniewa procedura DEFINE_COLUMN
    || jest przeciona i nie moe niejawnie zrzutowa typw bez
    || wartoci argumentu OUT_VALUE_SIZE. Tylko typy CHAR, RAW i VARCHAR2 obsuguj
    || czwarty argument
    */

    -- Odwzorowanie kolumny na zmienn value_out
    dbms_sql.define_column(c,1,value_out,1);

    -- Wykonywanie dynamicznej instrukcji
    fdbk := dbms_sql.execute_and_fetch(c);

    -- Kopiowanie zawartoci kolumny 1. do zmiennej value_out
    dbms_sql.column_value(c,1,value_out);

    -- Wywietlenie nazwy moduu
    dbms_output.put_line(
      '-> dbms_sql_tutorial.single_row_return');

    -- Wywietlenie wiersza rozdzielajcego
    dbms_output.put_line(sline);

    -- Wywietlenie danych wyjciowych
    dbms_output.put_line(
      'Warto z COLUMN_VALUE <'||value_out||'>');

    -- Zamknicie otwartego kursora
    dbms_sql.close_cursor(c);

  END single_row_return;

  /*
  || ------------------------------------------------------------------
  */

  -- Ta procedura demonstruje jednowierszowe polecenie DQL
  PROCEDURE single_row_return
    ( table_name    VARCHAR2
    , column_name1  VARCHAR2
    , column_name2  VARCHAR2
    , column_name3  VARCHAR2 ) IS

    -- Definicja zmiennych lokalnych pakietu DBMS_SQL
    c                         INTEGER := dbms_sql.open_cursor;
    fdbk                      INTEGER;
    statement                 VARCHAR2(2000);
    cvalue_out1               VARCHAR2(20);
    cvalue_out2               VARCHAR2(30);
    nvalue_out                NUMBER;

  BEGIN

    -- Tworzenie dynamicznej instrukcji SQL
    statement := 'SELECT '
              ||  column_name1 ||','
              ||  column_name2 ||','
              ||  column_name3 ||' '
              || 'FROM '|| table_name;

    -- Przetwarzanie dynamicznej instrukcji SQL
    dbms_sql.parse(c,statement,dbms_sql.native);

    /*
    || Wskazwka diagnostyczna:
    || =============
    || Zdefiniuj wartoci kolumn i NIE zapomniej przypisa parametru okrelajcego
    || rozmiar acuchowych typw danych, takich jak VARCHAR2; jeli tego nie zrobisz,
    || wystpi nastpujcy bd:
    || -------------------------------------------------------------------
    || PLS-00307: too many declarations of 'DEFINE_COLUMN' match this call
    ||
    || Ten komunikat zostanie zwrcony, poniewa procedura DEFINE_COLUMN
    || jest przeciona i nie moe niejawnie zrzutowa typw bez
    || wartoci argumentu OUT_VALUE_SIZE. Tylko typy CHAR, RAW i VARCHAR2 obsuguj
    || czwarty argument
    */

    -- Definiowanie kolumn odpowiadajcych wartociom zmiennych value_out
    dbms_sql.define_column(c,1,nvalue_out);
    dbms_sql.define_column(c,2,cvalue_out1,20);
    dbms_sql.define_column(c,3,cvalue_out2,30);

    -- Wykonywanie dynamicznej instrukcji
    fdbk := dbms_sql.execute_and_fetch(c);

    -- Kopiowanie zawartoci kolumny 1. do zmiennej value_out
    dbms_sql.column_value(c,1,nvalue_out);
    dbms_sql.column_value(c,2,cvalue_out1);
    dbms_sql.column_value(c,3,cvalue_out2);

    -- Wywietlenie nazwy moduu
    dbms_output.put_line(
      '-> dbms_sql_tutorial.single_row_return');

    -- Wywietlenie wiersza rozdzielajcego
    dbms_output.put_line(sline);

    -- Wywietlenie danych wyjciowych
    dbms_output.put_line(
      'Warto z COLUMN_VALUE <'||nvalue_out||'>');
    dbms_output.put_line(
      'Warto z COLUMN_VALUE <'||cvalue_out1||'>');
    dbms_output.put_line(
      'Warto z COLUMN_VALUE <'||cvalue_out2||'>');

    -- Zamknicie otwartego kursora
    dbms_sql.close_cursor(c);

  END single_row_return;

  /*
  || ------------------------------------------------------------------
  */

END dbms_sql_tutorial;
/

-- ==========================================================================
--  Zastosowano technik zarzdzania diagnozowaniem i dziennikami w celu
--  zapisu kompilowanego kodu i komunikatw o bdach.
--  Naley oznaczy ten fragment komentarzem, kiedy kod jest gotowy do udostpnienia,
--  a przy diagnozowaniu zmian ponownie usun oznaczenie komentarzem
-- ==========================================================================

SPOOL dbms_sql_body.log

list

show errors

SPOOL OFF
