/*
 * InsteadBooksAuthors.sql
 * Rozdzia 10, Oracle10g. Programowanie w jzyku PL/SQL
 * Ron Hardman, Mike McLaughlin i Scott Urman
 *
 * Ten skrypt demonstruje wyzwalacze zdefiniowane przez uytkownika
 */

SET ECHO ON

-- Usuwanie wyzwalaczy z poprzednich przykadw
DROP TRIGGER InsertBooksAuthors;
DROP TRIGGER GenerateAuthorID;

-- (Ponowne) utworzenie widoku, jeli to konieczne
CREATE OR REPLACE VIEW books_authors AS
  SELECT b.isbn, b.title, a.first_name, a.last_name
    FROM books b, authors a
    WHERE b.author1 = a.id
       OR b.author2 = a.id
       OR b.author3 = a.id;

CREATE OR REPLACE TRIGGER InsteadBooksAuthors
  INSTEAD OF INSERT OR UPDATE OR DELETE ON books_authors
  FOR EACH ROW
DECLARE

  v_Book books%ROWTYPE;
  v_NewAuthorID authors.ID%TYPE;
  v_OldAuthorID authors.ID%TYPE;
  
  -- Funkcja lokalna, ktra zwraca identyfikatory nowych autorw.
  -- Jeli w tabeli authors nie ma autora o podanym imieniu i nazwisku,
  -- funkcja generuje nowy identyfikator na podstawie sekwencji author_sequence
  FUNCTION getID(p_FirstName IN authors.first_name%TYPE,
                 p_LastName IN authors.last_name%TYPE)
    RETURN authors.ID%TYPE IS
    v_AuthorID authors.ID%TYPE;
  BEGIN
    -- Sprawdza, czy okrelone jest imi i nazwisko 
    IF p_FirstName IS NULL or p_LastName IS NULL THEN
      RAISE_APPLICATION_ERROR(-20004,
        'Nalezy okreslic imie i nazwisko');
    END IF;
    
    -- Blok zagniedony suy do przechwytywania wyjtkw NO_DATA_FOUND 
    BEGIN
      SELECT id
        INTO v_AuthorID
        FROM authors
        WHERE first_name = p_FirstName
          AND last_name = p_LastName;
    EXCEPTION
      WHEN NO_DATA_FOUND THEN
        -- Nie znaleziono autora - naley utworzy nowego
        INSERT INTO authors (id, first_name, last_name)
          VALUES (author_sequence.NEXTVAL, p_FirstName, p_LastName)
            RETURNING ID INTO v_AuthorID;
    END getID;

      -- Teraz zmienna  v_AuthorID zawiera poprawny identyfikator i mona go zwrci
      RETURN v_AuthorID;
    END getID;

  -- Funkcja lokalna zwracajca wiersz na podstawie tytuu
  -- lub numeru ISBN 
  FUNCTION getBook(p_ISBN IN books.ISBN%TYPE,
                   p_Title IN books.title%TYPE)
    RETURN books%ROWTYPE IS
    
    v_Book books%ROWTYPE;
  BEGIN
    -- Gwarantuje, e podany zosta przynajmniej jeden element (numer ISBN lub tytu)
    IF p_ISBN IS NULL AND p_Title IS NULL THEN
      RAISE_APPLICATION_ERROR(-20001,
        'Nalezy podac albo numer ISBN, albo tytul');
    ELSIF p_ISBN IS NOT NULL AND p_Title IS NOT NULL THEN
      -- Podano tytu i numer ISBN, dlatego w zapytaniu mona uy obu elementw
      SELECT *
        INTO v_Book
        FROM books
        WHERE isbn = p_ISBN
          AND title = p_Title;  
    ELSE
      -- Podano tylko jeden element, dlatego w zapytaniu trzeba uy albo numeru ISBN, albo tytuu
      SELECT *
        INTO v_Book
        FROM books
        WHERE isbn = p_ISBN
           OR title = p_Title;  
    END IF;
    
    RETURN v_Book;
  EXCEPTION
    WHEN NO_DATA_FOUND THEN
      RAISE_APPLICATION_ERROR(-20002,
        'Nie mozna znalezc ksiazki o podanym numerze ISBN lub tytule');
    WHEN TOO_MANY_ROWS THEN
      RAISE_APPLICATION_ERROR(-20003,
        'Numer ISBN lub tytul musza pasowac do jednej ksiazki');
  END getBook;


BEGIN  /* Pocztek ciaa wyzwalacza */
  IF INSERTING THEN
    -- Pobieranie danych o ksice i autorze
    v_Book := getBook(:new.ISBN, :new.title);
    v_NewAuthorID := getID(:new.first_name, :new.last_name);

    -- Sprawdzanie powtarzajcych si danych
    IF v_Book.author1 = v_NewAuthorID OR
       v_Book.author2 = v_NewAuthorID THEN
      RAISE_APPLICATION_ERROR(-20006,
        'Autor nie moze sie powtarzac');
    END IF;
    
    -- Sprawdzanie, czy ksika ma ju 1 lub 2 autorw, i
    -- odpowiednia aktualizacja danych
    IF v_Book.author2 IS NULL THEN
      UPDATE books
        SET author2 = v_NewAuthorID
        WHERE ISBN = v_Book.ISBN;
    ELSIF v_Book.author3 IS NULL THEN
      UPDATE books
        SET author3 = v_NewAuthorID
        WHERE ISBN = v_Book.ISBN;
    ELSE
      -- Too many authors, cannot insert
      RAISE_APPLICATION_ERROR(-20005,
        v_Book.title || ' ma juz 3 autorow');
    END IF;

  ELSIF UPDATING THEN
    -- Najpierw sprawdza, czy pola ISBN i title nie
    -- s modyfikowane
    IF (:new.ISBN != :old.ISBN OR
                     :new.title != :old.title) THEN
      RAISE_APPLICATION_ERROR(-20007,
        'Nie mozna modyfikowac pol ISBN i title w tabeli books_authors');
    END IF;
    
    -- Pobieranie danych o ksice i autorze
    v_Book := getBook(:new.ISBN, :new.title);
    v_NewAuthorID := getID(:new.first_name, :new.last_name);
    v_OldAuthorID := getID(:old.first_name, :old.last_name);

    -- Sprawdzanie, ktrego autora naley zmodyfikowa (author1, author2 lub author3) 
    -- i wprowadzanie odpowiednich zmian
    IF v_Book.author1 = v_OldAuthorID THEN
      UPDATE books
        SET author1 = v_NewAuthorID
        WHERE ISBN = v_Book.ISBN;
    ELSIF v_Book.author2 = v_OldAuthorID THEN
      UPDATE books
        SET author2 = v_NewAuthorID
        WHERE ISBN = v_Book.ISBN;
    ELSE
      UPDATE BOOKS
        SET author3 = v_NewAuthorID
        WHERE ISBN = v_Book.ISBN;
    END IF;
  ELSE
    -- Pobieranie informacji o ksice i autorze
    v_Book := getBook(:old.ISBN, :old.title);
    v_OldAuthorID := getID(:old.first_name, :old.last_name);
    
    -- Sprawdzanie, ktrego autora naley zmodyfikowa (author1, author2 lub author3) 
    -- i wprowadzanie odpowiednich zmian. Warto zauway, e jeli powoduje to 
    -- usunicie wszystkich autorw z tabeli, ograniczenie NOT NULL
    -- zwizane z polem author1 spowoduje bd
    IF v_Book.author1 = v_OldAuthorID THEN
      -- Ustawia author1 = author2, author2 = author3
      v_Book.Author1 := v_Book.Author2;
      v_Book.Author2 := v_Book.Author3;
    ELSIF v_Book.author2 = v_OldAuthorID THEN
      -- Ustawia author2 = author 3
      v_Book.Author2 := v_Book.Author3;
    ELSE
      -- Usuwa dane z pola author3
      v_Book.Author3 := NULL;
    END IF;
    
    UPDATE BOOKS
      SET author1 = v_Book.Author1,
          author2 = v_Book.Author2,
          author3 = v_Book.Author3
        WHERE ISBN = v_Book.ISBN;
  END IF;

  
END InsteadBooksAuthors;
/
show errors

-- To wywoanie powinno spowodowa wyjtek
INSERT INTO books_authors (first_name, last_name)
  VALUES ('Dorthy', 'Doolitle');

-- To wywoanie take powinno spowodowa wyjtek
INSERT INTO books_authors(ISBN, title, first_name, last_name)
  VALUES ('72223855', 'Wrong title', 'Esther', 'Elegant');

-- To wywoanie powinno si powie
INSERT INTO books_authors(ISBN, title, first_name, last_name)
  VALUES ('72223855', 'Oracle 9i New Features', 'Esther', 'Elegant');

-- To nie powinno dziaa, poniewa obecnie ksika ma dwch autorw
UPDATE books_authors
  SET first_name = 'Rose', last_name = 'Riznit'
  WHERE ISBN = '72223855';

-- To powinno dziaa prawidowo
UPDATE books_authors
  SET first_name = 'Rose', last_name = 'Riznit'
  WHERE ISBN = '72223855'
  AND last_name = 'Elegant';
  
-- To powinno dziaa prawidowo
DELETE FROM books_authors
  WHERE ISBN = '72223855'
  AND last_name = 'Riznit';

-- Tp wywoanie nie powinno dziaa
DELETE FROM books_authors
  WHERE ISBN = '72223855';
