/*
 * execute.sql
 * Rozdzia 9, Oracle10g. Programowanie w jzyku PL/SQL
 * Ron Hardman, Mike McLaughlin i Scott Urman
 *
 * Ten skrypt demonstruje dziaanie uprawnie systemowych EXECUTE
 */

set echo on
set serveroutput on

-- Najpierw kod tworzy uytkownikw userA i userB z potrzebnymi
-- obiektami. Trzeba poczy si z kontem o niezbdnych uprawnieniach,
-- na przykad SYSTEM, aby to zrobi.
-- Moesz te chcie zmieni uprawnienia UNLIMITED TABLESPACE
-- przedstawione poniej, aby jawnie przyzna limity na przestrzenie
-- tabel w uywanej bazie danych
connect system/manager
DROP USER UserA CASCADE;
CREATE USER UserA IDENTIFIED BY UserA;
GRANT CREATE SESSION, CREATE TABLE, CREATE PROCEDURE,
      UNLIMITED TABLESPACE, CREATE ROLE, DROP ANY ROLE TO UserA;

DROP USER UserB CASCADE;
CREATE USER UserB IDENTIFIED BY UserB;
GRANT CREATE SESSION, CREATE TABLE, CREATE PROCEDURE,
      UNLIMITED TABLESPACE TO UserB;

connect UserA/UserA

-- ***********************************
-- Sytuacja z rysunku 9.8: Wszystkie obiekty nale do uytkownika UserA.
-- ***********************************

-- Najpierw naley utworzy tabel books. Kod nie tworzy tabeli authors,
-- dlatego nie naley uywa ogranicze spjnoci
CREATE TABLE books (
  isbn      CHAR(10) PRIMARY KEY,
  category  VARCHAR2(20),
  title     VARCHAR2(100),
  num_pages NUMBER,
  price     NUMBER,
  copyright NUMBER(4),
  author1   NUMBER,
  author2   NUMBER,
  author3   NUMBER 
);

-- Wstawianie danych. Jest to tylko podzbir wierszy z kompletnej
-- tabeli books.Zwr uwag na to, e jedna ksika ma trzech
-- autorw, a pozostae - tylko dwch

INSERT
INTO books
(isbn
,category
,title
,num_pages
,price
,copyright
,author1
,author2
,author3)
VALUES
('007212606X'
,'Oracle Basics'
,'Oracle PL/SQL 101'
,420
,39.99
,2000
,1
,2
,3);

INSERT
INTO books
(isbn
,category
,title
,num_pages
,price
,copyright
,author1
,author2
,author3)
VALUES
('0072131454'
,'Oracle Basics'
,'Oracle Performance Tuning 101'
,404
,39.99
,2001
,1
,2
,3);

INSERT
INTO books
(isbn
,category
,title
,num_pages
,price
,copyright
,author1
,author2
,author3)
VALUES
('72121203'
,'Oracle Basics'
,'Oracle DBA 101'
,563
,39.99
,1999
,1
,2
,3);

INSERT
INTO books
(isbn
,category
,title
,num_pages
,price
,copyright
,author1
,author2)
VALUES
('72122048'
,'Oracle Basics'
,'Oracle8i: A Beginner''s Guide'
,765
,44.99
,1999
,4
,5);

-- Tworzenie tabeli temp_table
CREATE TABLE temp_table (
  num_col    NUMBER,
  char_col   VARCHAR2(60)
);

-- Potrzebna jest take funkcja ThreeAuthors:
CREATE OR REPLACE FUNCTION ThreeAuthors(p_ISBN IN books.isbn%TYPE)
  RETURN BOOLEAN AS

  v_Author3 books.author3%TYPE;
BEGIN
  -- Pobieranie trzeciego autora danej ksiki do zmiennej v_Author3
  SELECT author3
    INTO v_Author3
    FROM books
    WHERE isbn = p_ISBN;

  -- Jeli zmienna v_Author3 to NULL, oznacza to, e ksika ma mniej ni
  -- trzech autorw, dlatego mona zwrci warto false.  W przeciwnym 
  -- razie trzeba zwrci true
  IF v_Author3 IS NULL THEN
    RETURN FALSE;
  ELSE
    RETURN TRUE;
  END IF;
END ThreeAuthors;
/
 
-- Procedura RecordThreeAuthors
CREATE OR REPLACE PROCEDURE RecordThreeAuthors AS
  CURSOR c_Books IS
    SELECT *
      FROM books;
BEGIN
  FOR v_BookRecord in c_Books LOOP
    -- Rejestrowanie w tabeli temp_table
    -- wszystkich ksiek majcych trzech autorw
    IF ThreeAuthors(v_BookRecord.ISBN) THEN
      INSERT INTO temp_table (char_col) VALUES
        (v_BookRecord.title || ' ma trzech autorow');
    END IF;
  END LOOP;
END RecordThreeAuthors;
/
show errors

-- Teraz, po utworzeniu wszystkich obiektw, naley przyzna uprawnienia EXECUTE
-- do procedury RecordThreeAuthors uytkowikowi UserB.  Warto zauway, e UserB
-- nie ma innych uprawnie do obiektw uytkownika UserA
GRANT EXECUTE ON RecordThreeAuthors to UserB;

-- Poczenie jako UserB i uruchomienie RecordThreeAuthors. Wyniki
-- zostan zapisane w tabeli temp_table uytkownika UserA, poniewa
-- jest to jedyna kopia tej tabeli
connect UserB/UserB
BEGIN
  UserA.RecordThreeAuthors;
  COMMIT;
END;
/

-- Zapytanie do tabeli temp_table jako UserA w celu sprawdzenia wynikw
connect UserA/UserA
SELECT * FROM temp_table;

-- ***********************************
-- Sytuacja z rysunku 9.9 - uytkownik UserB take
--                          ma kopi tabeli temp_table.
-- ***********************************

-- Tworzenie UserB.temp_table
connect UserB/UserB
CREATE TABLE temp_table (
  num_col    NUMBER,
  char_col   VARCHAR2(60)
);

-- Wywoanie UserA.RecordThreeAuthors powoduje zmian w
-- tabeli temp_table uytkownika UserA
BEGIN
  UserA.RecordThreeAuthors;
  COMMIT;
END;
/

-- Zapytanie do tabeli uytkownika UserB - nie powinno zwrci wierszy
SELECT * FROM temp_table;

-- Zapytanie do tabeli uytkownika UserA - powinno zwrci dwa wiersze, po
-- jednym dla kadego wywoania
connect UserA/UserA
SELECT * FROM temp_table;

-- ***********************************
-- Sytuacja z rysunku 9.10 - uytkownik UserB jest wacicielem
--                           temp_table i RecordThreeAuthors, a uprawnienia
--                           zostay przyznane bezporednio
-- ***********************************

-- Najpierw usunicie danych z tabeli uytkownika UserA:
connect UserA/UserA
DROP TABLE temp_table;
DROP PROCEDURE RecordThreeAuthors;

-- Teraz trzeba przyzna uprawnienia do procedury ThreeAuthors i tabeli books
-- uytkownikowi UserB:
GRANT SELECT ON books TO UserB;
GRANT EXECUTE ON ThreeAuthors TO UserB;

-- UserB jest ju wacicielem tabeli temp_table, dlatego trzeba tylko
-- utworzy procedur. Warto zauway, e wywoywana jest procedura
-- ThreeAuthors schematu UserA poprzez zastosowanie notacji z kropk.
-- Bez wczeniejszego przyznania uprawnie ten kod nie skompiluje si
connect UserB/UserB
CREATE OR REPLACE PROCEDURE RecordThreeAuthors AS
  CURSOR c_Books IS
    SELECT *
      FROM UserA.books;
BEGIN
  FOR v_BookRecord in c_Books LOOP
    -- Zapisywanie w tabeli temp_table wszystkich
    -- ksiek majcych trzech autorw
    IF UserA.ThreeAuthors(v_BookRecord.ISBN) THEN
      INSERT INTO temp_table (char_col) VALUES
        (v_BookRecord.title || ' ma trzech autorow');
    END IF;
  END LOOP;
END RecordThreeAuthors;
/

-- Wykonanie procedury RecordThreeAuthors spowoduje zapisanie wynikw
-- w tabeli temp_table uytkownika UserB
BEGIN
  RecordThreeAuthors;
  COMMIT;
END;
/

-- To wywoanie powinno zwrci jeden wiersz
SELECT * FROM temp_table;

-- ***********************************
-- Sytuacja z rysunku 9.11: Uytkownik UserB jest wacicielem
--                          temp_table i RecordThreeAuthors, ale
--                          uprawnienia zostay przyznane przez rol
-- ***********************************

connect UserA/UserA
-- Odebranie wczeniejszych uprawnie
REVOKE SELECT ON books FROM UserB;
REVOKE EXECUTE ON ThreeAuthors FROM UserB;

-- Tworzenie roli
DROP ROLE UserA_Role;
CREATE ROLE UserA_Role;
GRANT SELECT ON books TO UserA_Role;
GRANT EXECUTE ON ThreeAuthors TO UserA_Role;
GRANT UserA_Role TO UserB;

-- Prba utworzenia procedury RecordThreeAuthors z konta UserB spowoduje
-- bdy ORA-942 i PLS-201
connect UserB/UserB
CREATE OR REPLACE PROCEDURE RecordThreeAuthors AS
  CURSOR c_Books IS
    SELECT *
      FROM UserA.books;
BEGIN
  FOR v_BookRecord in c_Books LOOP
    -- Zapisywanie w tabeli temp_table wszystkich
    -- ksiek majcych trzech autorw
    IF UserA.ThreeAuthors(v_BookRecord.ISBN) THEN
      INSERT INTO temp_table (char_col) VALUES
        (v_BookRecord.title || ' ma trzech autorow');
    END IF;
  END LOOP;
END RecordThreeAuthors;
/
show errors
