/*
 * utlRef.sql
 * Rozdzia 15, Oracle10g. Programowanie w jzyku PL/SQL
 * Ron Hardman, Mike McLaughlin i Scott Urman
 *
 * Ten skrypt demonstruje pakiet UTL_REF
 */

SET SERVEROUTPUT ON SIZE 1000000

exec clean_schema.synonyms
exec clean_schema.tables
exec clean_schema.objects

CREATE OR REPLACE TYPE discount_price_obj AS OBJECT (
   discount_rate   NUMBER (10, 4),
   price           NUMBER (10, 2),
   MEMBER FUNCTION discount_price
      RETURN NUMBER
)
INSTANTIABLE FINAL;
/

CREATE OR REPLACE TYPE BODY discount_price_obj
AS
   MEMBER FUNCTION discount_price
      RETURN NUMBER
   IS
   BEGIN
      RETURN (SELF.price * (1 - SELF.discount_rate));
   END discount_price;
END;
/

-- Typ address_obj znajduje si na kracu acucha, na ktrym nie wystpuj zalenoci

CREATE OR REPLACE TYPE address_obj AS OBJECT (
   address1   VARCHAR2 (30 CHAR),
   address2   VARCHAR2 (30 CHAR),
   city       VARCHAR2 (30 CHAR),
   state      CHAR (2 CHAR)
)
INSTANTIABLE FINAL;
/

-- person_obj take znajduje si na tym kracu acucha

CREATE OR REPLACE TYPE person_obj AS OBJECT (
   first_name   VARCHAR2 (20),
   last_name    VARCHAR2 (20)
)
INSTANTIABLE FINAL;
/

-- contact_obj czy dwa pierwsze typy obiektowe
--   i ma jeszcze jeden atrybut

CREATE OR REPLACE TYPE contact_obj AS OBJECT (
   NAME      person_obj,
   address   address_obj,
   phone     NUMBER (10)
)
INSTANTIABLE FINAL;
/

-- W ostatnim typie znajduje si atrybut publisher_obj.contact_info,
--   ktrego typ danych to contact_obj

CREATE OR REPLACE TYPE publisher_obj AS OBJECT (
   pub_name       VARCHAR2 (30),
   contact_info   contact_obj,
   MEMBER PROCEDURE show_contact
)
INSTANTIABLE FINAL;
/

CREATE OR REPLACE TYPE BODY publisher_obj
AS
   MEMBER PROCEDURE show_contact
   IS
   BEGIN
      DBMS_OUTPUT.put_line ('DANE KONTAKTOWE');
      DBMS_OUTPUT.put_line ('===============');
      DBMS_OUTPUT.put_line (SELF.pub_name);
      DBMS_OUTPUT.put_line (   SELF.contact_info.NAME.first_name
                            || ' '
                            || SELF.contact_info.NAME.last_name
                           );
      DBMS_OUTPUT.put_line (SELF.contact_info.address.address1);
      DBMS_OUTPUT.put_line (SELF.contact_info.address.city);
      DBMS_OUTPUT.put_line (SELF.contact_info.address.state);
      DBMS_OUTPUT.put_line (SELF.contact_info.phone);
      RETURN;
   END show_contact;
END;
/

-- Tabela obiektowa bazujca na typie publisher_obj
CREATE TABLE publisher_tbl OF publisher_obj;

-- Specyfikacja typu inventory_obj

CREATE OR REPLACE TYPE inventory_obj AS OBJECT (
   item_id          NUMBER (10),
   num_in_stock     NUMBER (10),
   reorder_status   VARCHAR2 (20 CHAR),
   price            NUMBER (10, 2),
   CONSTRUCTOR FUNCTION inventory_obj (
      item_id        IN   NUMBER,
      num_in_stock   IN   NUMBER,
      price          IN   NUMBER
   )
      RETURN SELF AS RESULT,
   MEMBER PROCEDURE print_inventory,
   MEMBER PROCEDURE print_status,
   MEMBER PROCEDURE print_price
)
INSTANTIABLE NOT FINAL;
/

CREATE OR REPLACE TYPE BODY inventory_obj
AS
   CONSTRUCTOR FUNCTION inventory_obj (
      item_id        IN   NUMBER,
      num_in_stock   IN   NUMBER,
      price          IN   NUMBER
   )
      RETURN SELF AS RESULT
   IS
   BEGIN
      SELF.item_id := item_id;
      SELF.num_in_stock := num_in_stock;
      SELF.price := price;
      RETURN;
   END;
   MEMBER PROCEDURE print_inventory
   IS
   BEGIN
      DBMS_OUTPUT.put_line ('ASORTYMENT KSIEGARNI1');
      DBMS_OUTPUT.put_line ('=====================');
      DBMS_OUTPUT.put_line (   'Liczba artykulow o numerze '
                            || SELF.item_id
                            || ' to '
                            || SELF.num_in_stock
                            || ' .'
                           );
   END print_inventory;
   MEMBER PROCEDURE print_status
   IS
      v_status   VARCHAR2 (20);
   BEGIN
      IF SELF.num_in_stock > 0
      THEN
         v_status := 'DOSTEPNY';
      ELSE
         v_status := 'NIEDOSTEPNY';
      END IF;

      DBMS_OUTPUT.put_line ('STAN MAGAZYNU KSIEGARNI1');
      DBMS_OUTPUT.put_line ('========================');
      DBMS_OUTPUT.put_line ('Produkt o numerze ' || SELF.item_id || ' jest '
                            || v_status
                           );
   END print_status;
   MEMBER PROCEDURE print_price
   IS
      v_discount_price   discount_price_obj
                                       := discount_price_obj (.1, SELF.price);
   BEGIN
      DBMS_OUTPUT.put_line ('CENY W KSIEGARNI1');
      DBMS_OUTPUT.put_line ('=================');
      DBMS_OUTPUT.put_line ('Produkt o numerze ' || SELF.item_id);
      DBMS_OUTPUT.put_line ('Cena detaliczna: ' || SELF.price || ' zlotych');
      DBMS_OUTPUT.put_line (   'NASZA BARDZO - BARDZO - NISKA CENA: '
                            || v_discount_price.price
                            || ' zlotych'
                           );
   END print_price;
END;
/

CREATE OR REPLACE TYPE book_obj
UNDER inventory_obj (
   isbn        CHAR (10 CHAR),
   CATEGORY    VARCHAR2 (20 CHAR),
   title       VARCHAR2 (100 CHAR),
   num_pages   NUMBER,
   publisher   publisher_obj,
   CONSTRUCTOR FUNCTION book_obj (
      item_id        NUMBER,
      num_in_stock   NUMBER,
      price          NUMBER,
      isbn           CHAR,
      title          VARCHAR2,
      num_pages      NUMBER
   )
      RETURN SELF AS RESULT,
   MEMBER PROCEDURE print_book_information,
   OVERRIDING MEMBER PROCEDURE print_price
)
INSTANTIABLE NOT FINAL;
/

CREATE OR REPLACE TYPE BODY book_obj
IS
   CONSTRUCTOR FUNCTION book_obj (
      item_id        NUMBER,
      num_in_stock   NUMBER,
      price          NUMBER,
      isbn           CHAR,
      title          VARCHAR2,
      num_pages      NUMBER
   )
      RETURN SELF AS RESULT
   IS
   BEGIN
      SELF.item_id := item_id;
      SELF.num_in_stock := num_in_stock;
      SELF.price := price;
      SELF.isbn := isbn;
      SELF.title := title;
      SELF.num_pages := num_pages;
      RETURN;
   END book_obj;
   MEMBER PROCEDURE print_book_information
   IS
   BEGIN
      DBMS_OUTPUT.put_line ('INFORMACJE O KSIAZCE');
      DBMS_OUTPUT.put_line ('====================');
      DBMS_OUTPUT.put_line ('Tytul: '
                            || SELF.title);
      DBMS_OUTPUT.put_line ('Liczba stron: ' 
                            || SELF.num_pages);
      DBMS_OUTPUT.put_line ('Na stanie: ' 
                            || SELF.num_in_stock);
   END print_book_information;
   OVERRIDING MEMBER PROCEDURE print_price
   IS
   BEGIN
      DBMS_OUTPUT.put_line ('CENY W KSIEGARNI1');
      DBMS_OUTPUT.put_line ('================');
      DBMS_OUTPUT.put_line ('Tytul: '
                            || SELF.title);
      DBMS_OUTPUT.put_line ('Zawsze niska cena: ' 
                            || SELF.price);
   END print_price;
END;
/

CREATE TABLE inventory_tbl OF inventory_obj;


INSERT INTO inventory_tbl
     VALUES (book_obj (2,
                       13,
                       'DOSTEPNY',
                       54.95,
                       '72121203',
                       'TECHNICAL',
                       'Oracle DBA 101',
                       563,
                       publisher_obj ('Oracle Press',
                                      contact_obj (person_obj ('Zuzanna',
                                                               'Wydawca'
                                                              ),
                                                   address_obj ('Prosta 10',
                                                                '2',
                                                                'Moje Miasto',
                                                                'MAZ'
                                                               ),
                                                   '5555555555'
                                                  )
                                     )
                      ));

commit;

DECLARE
   CURSOR c_inventory
   IS
      SELECT REF (i)
        FROM inventory_tbl i
       WHERE VALUE (i) IS OF (book_obj);

   v_inventoryref   REF inventory_obj;
   v_inventory      inventory_obj;
   v_status         VARCHAR2 (200 CHAR);
   v_book           book_obj;
BEGIN
-- Usuwanie jednego rekordu z tabeli inventory_tbl
   SELECT REF (i)
     INTO v_inventoryref
     FROM inventory_tbl i
    WHERE item_id = 2;

   UTL_REF.delete_object (v_inventoryref);
   DBMS_OUTPUT.put_line ('	');
   DBMS_OUTPUT.put_line ('DELETE_OBJECT');
   DBMS_OUTPUT.put_line ('=============');

   -- Sprawdza, czy wiersz zosta usunity
   SELECT    'Usunieto wiersz za pomoc funkcji DELETE_OBJECT, dlatego '
          || 'wartosc REF jest ZAWIESZONA. Trzeba anulowac te operacje, aby '
          || 'kontynuowac dzialanie.'
     INTO v_status
     FROM DUAL
    WHERE v_inventoryref IS DANGLING;

   DBMS_OUTPUT.put_line (v_status);
   DBMS_OUTPUT.put_line ('	');
   ROLLBACK;
   DBMS_OUTPUT.put_line ('SELECT_OBJECT');
   DBMS_OUTPUT.put_line ('=============');
   DBMS_OUTPUT.PUT_LINE
            ('Uycie procedury SELECT_OBJECT do zwrcenia obiektu i ');
   DBMS_OUTPUT.PUT_LINE
             ('uycie funkcji TREAT na obiekcie tabeli inventory w celu uzyskania ');
   DBMS_OUTPUT.PUT_LINE ('dostpu do metody print_book_information typu pochodnego.');
   DBMS_OUTPUT.put_line ('	');
   
   -- Pobranie obiektu za pomoc wczeniej uzyskanej referencji
   UTL_REF.select_object (v_inventoryref, v_inventory);

   SELECT TREAT (v_inventory AS book_obj)
     INTO v_book
     FROM DUAL;

   v_book.print_book_information;
END;
/
