/*
 * create_user.sql
 * Rozdzia 13, Oracle10g. Programowanie w jzyku PL/SQL
 * Ron Hardman, Mike McLaughlin i Scott Urman
 *
 * Ten skrypt tworzy uytkownika PLSQL. Trzeba uruchomi ten skrypt z konta SYSTEM
 *   lub konta uytkownika z uprawnieniami roli DBA oraz EXECUTE do pakietu DBMS_PIPE
 *   z moliwoci przyznawania ich. Znajduj si tu take instrukcje GRANT
 *   nadajce uprawnienia niezbdne do korzystania z pakietu DBMS_SQL.
 *
 * Jeli ten skrypt nie zadziaa w koncie SYSTEM, prawdopodobnie brakuje
 *  jednego z poniszych uprawnie, ktre mona przyzna z konta SYS:
 *
 *   GRANT EXECUTE ON dbms_sys_sql TO SYSTEM WITH GRANT OPTION;
 *   GRANT EXECUTE ON dbms_pipe TO system WITH GRANT OPTION;
 *   GRANT EXECUTE ON dbms_alert TO system WITH GRANT OPTION;
 *   GRANT EXECUTE ON dbms_sql TO system WITH GRANT OPTION;
 *   GRANT EXECUTE ON dbms_java TO system;
 */

-- Jeli chcesz zdiagnozowa kod, usu oznaczenie komentarzem:
-- SET ECHO ON
-- SET FEEDBACK ON

SET SERVEROUTPUT ON SIZE 1000000

DECLARE

  -- Definicja wyjtku
  wrong_schema EXCEPTION;
  PRAGMA EXCEPTION_INIT(wrong_schema,-20001);

  -- Definicja zwracanej zmiennej
  retval VARCHAR2(1 CHAR);

  /*
  || Definicja kursora do sprawdzania, czy biecy uytkownik to SYS
  || lub uytkownik z uprawnieniami roli DBA
  */
  CURSOR privs IS
    SELECT   DISTINCT null
    FROM     user_role_privs
    WHERE    username = 'SYSTEM'
    OR       granted_role = 'DBA';

BEGIN

  -- Odwarcie kursora i odczyt danych
  OPEN privs;
  LOOP

    -- Wczytywanie wiersza
    FETCH privs INTO retval;

    -- Sprawdzanie, czy kursor dziaa
    IF privs%NOTFOUND THEN

      -- Zgaszanie wyjtku
      RAISE wrong_schema;

    ELSE

      -- Sprawdza, czy uytkownik PLSQL istnieje. Jeli tak - usuwa go
      FOR i IN (SELECT null FROM dba_users WHERE username = 'PLSQL') LOOP
        EXECUTE IMMEDIATE 'DROP USER plsql CASCADE';
      END LOOP;

      -- Tworzenie uytkownika i przyznawanie uprawnie
      EXECUTE IMMEDIATE 'CREATE USER plsql IDENTIFIED BY plsql';
      EXECUTE IMMEDIATE 'GRANT connect TO plsql';
      EXECUTE IMMEDIATE 'GRANT resource TO plsql';
      EXECUTE IMMEDIATE 'GRANT create library TO plsql';
      EXECUTE IMMEDIATE 'GRANT execute ON dbms_pipe TO plsql';
      EXECUTE IMMEDIATE 'GRANT execute ON dbms_alert TO plsql';
      EXECUTE IMMEDIATE 'GRANT execute ON sys.dbms_sys_sql TO plsql';
      EXECUTE IMMEDIATE 'GRANT execute ON dbms_sql TO plsql';
      EXECUTE IMMEDIATE 'GRANT create table TO plsql';
      EXECUTE IMMEDIATE 'GRANT create sequence TO plsql';

      -- Przyznawanie uprawnie do jzyka Java do wykonywania operacji wejcia-wyjcia na plikach
      DBMS_JAVA.GRANT_PERMISSION('PLSQL'
                                ,'SYS:java.io.FilePermission'
                                ,'/tmp/file.txt'
                                ,'read');

      -- Wywietlanie informacji o powodzeniu
      DBMS_OUTPUT.PUT_LINE(CHR(10)||'Utworzono uzytkownika PLSQL.');

    END IF; 

    -- Wyjcie z ptli
    EXIT;

  END LOOP;

  -- Zamyka kursor
  CLOSE privs;

EXCEPTION

  -- Obsuga zdefiniowanego wyjtku
  WHEN wrong_schema THEN
    DBMS_OUTPUT.PUT_LINE('Ten skrypt musi zosta wykonany z konta SYSTEM i '
    ||                   'przy uzyciu schematu <'||user||'> lub '
    ||                   'z konta z uprawnieniami roli DBA.');

  -- Oglna obsuga wyjtkw
  WHEN others THEN
    DBMS_OUTPUT.PUT_LINE(SQLERRM);
    RETURN;

END;
/

-- Formatowanie w SQL*Plus
COL grantee          FORMAT A8
COL granted_role     FORMAT A30
COL grantor          FORMAT A8
COL privilege        FORMAT A12
COL owner            FORMAT A4
COL table_name       FORMAT A30

-- Zapytanie do rl utworzonych przez uytkownika
SELECT   grantee
,        granted_role
FROM     dba_role_privs
WHERE    grantee = 'PLSQL';

-- Query resources.
SELECT   grantor
,        owner
,        table_name
,        grantee
,        privilege
FROM     dba_tab_privs
WHERE    grantee = 'PLSQL';

COL admin_option     FORMAT A3
COL privilege        FORMAT A30
COL username         FORMAT A10

-- Query user system privileges.
SELECT   grantee
,        privilege
,        admin_option
FROM     dba_sys_privs
WHERE    grantee = 'PLSQL';
