//////////////////////////////////////////////////////////////////////
// (c) Janusz Ganczarski
// http://www.januszg.hg.pl
// JanuszG@enter.net.pl
//////////////////////////////////////////////////////////////////////

#include <string>
#include <sstream>
#include <iomanip>
#include <iostream>
#include <glm/glm.hpp>
#include <glm/gtc/matrix_transform.hpp>
#include <glm/gtc/type_ptr.hpp>
#include "shaders.h"
#include "teapot.h"
#include "sphere.h"
#include "materials.h"
#include "text.h"

//////////////////////////////////////////////////////////////////////
// rozmiary bryy obcinania
//////////////////////////////////////////////////////////////////////
GLfloat left = -4.0f;
GLfloat right = 4.0f;
GLfloat bottom = -4.0f;
GLfloat top = 4.0f;
GLfloat near = 6.0f;
GLfloat far = 14.0f;

//////////////////////////////////////////////////////////////////////
// wektory normalne wierzchokw trjktw tworzcych ciany pokoju
//////////////////////////////////////////////////////////////////////
GLfloat roomNormal[5*6*3] =
{
    // podoga
    0.0f, 1.0f, 0.0f,
    0.0f, 1.0f, 0.0f,
    0.0f, 1.0f, 0.0f,
    0.0f, 1.0f, 0.0f,
    0.0f, 1.0f, 0.0f,
    0.0f, 1.0f, 0.0f,

    // lewa ciana
    1.0f, 0.0f, 0.0f,
    1.0f, 0.0f, 0.0f,
    1.0f, 0.0f, 0.0f,
    1.0f, 0.0f, 0.0f,
    1.0f, 0.0f, 0.0f,
    1.0f, 0.0f, 0.0f,

    // prawa ciana
    -1.0f, 0.0f, 0.0f,
    -1.0f, 0.0f, 0.0f,
    -1.0f, 0.0f, 0.0f,
    -1.0f, 0.0f, 0.0f,
    -1.0f, 0.0f, 0.0f,
    -1.0f, 0.0f, 0.0f,

    // sufit
    0.0f, -1.0f, 0.0f,
    0.0f, -1.0f, 0.0f,
    0.0f, -1.0f, 0.0f,
    0.0f, -1.0f, 0.0f,
    0.0f, -1.0f, 0.0f,
    0.0f, -1.0f, 0.0f,

    // tylna ciana
    0.0f, 0.0f, 1.0f,
    0.0f, 0.0f, 1.0f,
    0.0f, 0.0f, 1.0f,
    0.0f, 0.0f, 1.0f,
    0.0f, 0.0f, 1.0f,
    0.0f, 0.0f, 1.0f
};

//////////////////////////////////////////////////////////////////////
// wsprzdne wierzchokw trjktw tworzcych ciany pokoju;
// wsprzdne w przestrzeni obserwatora
//////////////////////////////////////////////////////////////////////
GLfloat roomPosition[5*6*4] =
{
    // podoga
    left, bottom, -near, 1.0f,
    right, bottom, -near, 1.0f,
    right, bottom, -far + 1.0f, 1.0f,
    left, bottom, -near, 1.0f,
    right, bottom, -far + 1.0f, 1.0f,
    left, bottom, -far + 1.0f, 1.0f,

    // lewa ciana
    left, bottom, -near, 1.0f,
    left, bottom, -far + 1.0f, 1.0f,
    left, top, -far + 1.0f, 1.0f,
    left, bottom, -near, 1.0f,
    left, top, -far + 1.0f, 1.0f,
    left, top, -near, 1.0f,

    // prawa ciana
    right, bottom, -near, 1.0f,
    right, top, -near, 1.0f,
    right, top, -far + 1.0f, 1.0f,
    right, bottom, -near, 1.0f,
    right, top, -far + 1.0f, 1.0f,
    right, bottom, -far + 1.0f, 1.0f,

    // sufit
    left, top, -near, 1.0f,
    left, top, -far + 1.0f, 1.0f,
    right, top, -far + 1.0f, 1.0f,
    left, top, -near, 1.0f,
    right, top, -far + 1.0f, 1.0f,
    right, top, -near, 1.0f,

    // tylna ciana
    left, bottom, -far + 1.0f, 1.0f,
    right, bottom, -far + 1.0f, 1.0f,
    right, top, -far + 1.0f, 1.0f,
    left, bottom, -far + 1.0f, 1.0f,
    right, top, -far + 1.0f, 1.0f,
    left, top, -far + 1.0f, 1.0f
};

//////////////////////////////////////////////////////////////////////
// macierz rzutowania
//////////////////////////////////////////////////////////////////////
glm::mat4 projectionMatrix;

//////////////////////////////////////////////////////////////////////
// macierze modelowania i rzutowania powstae w efekcie
// przeksztacenia sceny na widok z punktu widzenia rda wiata
//////////////////////////////////////////////////////////////////////
glm::mat4 shadowModelViewMatrix;
glm::mat4 shadowProjectionMatrix;

//////////////////////////////////////////////////////////////////////
// wspczynniki skalowania obiektu
//////////////////////////////////////////////////////////////////////
GLfloat scale = 0.75f;

//////////////////////////////////////////////////////////////////////
// kty obrotu obiektu
//////////////////////////////////////////////////////////////////////
GLfloat rotateX = 0.0f;
GLfloat rotateY = 0.0f;

//////////////////////////////////////////////////////////////////////
// przesunicie obiektu
//////////////////////////////////////////////////////////////////////
GLfloat translateX = 0.0f;
GLfloat translateY = 0.0f;

//////////////////////////////////////////////////////////////////////
// pooenie rda wiata punktowego we wsprzdnych obserwatora
//////////////////////////////////////////////////////////////////////
glm::vec4 lightPosition( 3.0f, 3.0f, 3.0f - (near+far)/2.0f, 1.0f );

//////////////////////////////////////////////////////////////////////
// wybrany materia
//////////////////////////////////////////////////////////////////////
int material = MTL_BRASS;

//////////////////////////////////////////////////////////////////////
// numeracja obiektw programu
//////////////////////////////////////////////////////////////////////
enum
{
    GENERATE_SHADOW_MAP,        // generowanie mapy cienia
    LIGHT_POINT,                // pooenie rda wiata
    SHADOW_MAP_LIGHT,           // ciany - brak cienia
    SHADOW_MAP_STD,             // czajnik i ciany - mapa cienia algorytm podstawowy
    SHADOW_MAP_PCF_3X3_AVERAGE, // czajnik i ciany - mapa cienia algorytm PCF 3x3 z filtracj uredniajc
    SHADOW_MAP_PCF_3X3_GAUSS,   // czajnik i ciany - mapa cienia algorytm PCF 3x3 z filtracj Gaussa
    SHADOW_MAP_PCF_5X5_AVERAGE, // czajnik i ciany - mapa cienia algorytm PCF 5x5 z filtracj uredniajc
    SHADOW_MAP_PCF_5X5_GAUSS,   // czajnik i ciany - mapa cienia algorytm PCF 5x5 z filtracj Gaussa
    SHADOW_MAP_PCF_5X5_RANDOM,  // czajnik i ciany - mapa cienia algorytm PCF 5x5 z losowym
                                // przemieszczeniem wsprzdnych prbkowania
    SHADOW_MAP_PCF_5X5_POISSON, // czajnik i ciany - mapa cienia algorytm PCF 5x5 z przemieszczeniem
                                // wsprzdnych prbkowania przy uyciu rozkadu Poissona
    RECT_SHADOW,                // mapa gbokoci
    PROGRAM_SIZE
};

//////////////////////////////////////////////////////////////////////
// identyfikatory obiektw programu
//////////////////////////////////////////////////////////////////////
GLuint program[PROGRAM_SIZE];

//////////////////////////////////////////////////////////////////////
// numeracja obiektw bufora wierzchokw
//////////////////////////////////////////////////////////////////////
enum
{
    TEAPOT_POSITION,
    TEAPOT_NORMAL,
    LIGHT_POINT_POSITION,
    ROOM_POSITION,
    ROOM_NORMAL,
    POSITION_RECT,
    TEXCOORD_RECT,
    VERTEX_BUFFER_SIZE
};

//////////////////////////////////////////////////////////////////////
// identyfikatory obiektw bufora z danymi tablic wierzchokw
//////////////////////////////////////////////////////////////////////
GLuint vertexBuffer[VERTEX_BUFFER_SIZE];

//////////////////////////////////////////////////////////////////////
// numeracja obiektw tablic wierzchokw
//////////////////////////////////////////////////////////////////////
enum
{
    TEAPOT,
    // LIGHT_POINT,
    ROOM = 2,
    RECTANGLE,
    VERTEX_ARRAY_SIZE
};

//////////////////////////////////////////////////////////////////////
// identyfikatory obiektw tablic wierzchokw
//////////////////////////////////////////////////////////////////////
GLuint vertexArray[VERTEX_ARRAY_SIZE];

//////////////////////////////////////////////////////////////////////
// numeracja obiektw bufora wierzchokw
//////////////////////////////////////////////////////////////////////
enum
{
    TEAPOT_INDICES,
    LIGHT_POINT_INDICES,
    INDICES_BUFFER_SIZE
};

//////////////////////////////////////////////////////////////////////
// identyfikatory obiektw bufora z danymi tablic
// indeksw wierzchokw obiektw
//////////////////////////////////////////////////////////////////////
GLuint indicesBuffer[INDICES_BUFFER_SIZE];

//////////////////////////////////////////////////////////////////////
// numeracja obiektw bufora ramki i obiektw tekstury
//////////////////////////////////////////////////////////////////////
enum
{
    FRAME_BUFFER_256_X_256,
    FRAME_BUFFER_512_X_512,
    FRAME_BUFFER_1024_X_1024,
    FRAME_BUFFER_2048_X_2048,
    FRAME_BUFFER_SIZE,
    RENDER_TEXTURE_SIZE = FRAME_BUFFER_SIZE
};

//////////////////////////////////////////////////////////////////////
// identyfikatory obiektw bufora ramki
//////////////////////////////////////////////////////////////////////
GLuint frameBuffer[FRAME_BUFFER_SIZE];

//////////////////////////////////////////////////////////////////////
// rozmiary obszarw renderingu do tekstury
//////////////////////////////////////////////////////////////////////
const int FRAME_WIDTH[FRAME_BUFFER_SIZE] = { 256, 512, 1024, 2048 };
const int FRAME_HEIGHT[FRAME_BUFFER_SIZE] = { 256, 512, 1024, 2048 };

//////////////////////////////////////////////////////////////////////
// identyfikatory obiektu tekstury
//////////////////////////////////////////////////////////////////////
GLuint renderTexture[RENDER_TEXTURE_SIZE];

//////////////////////////////////////////////////////////////////////
// wybrana mapa cienia
//////////////////////////////////////////////////////////////////////
int depthMap = FRAME_BUFFER_512_X_512;

//////////////////////////////////////////////////////////////////////
// filtr powikszajcy/pomniejszajcy tekstury gbokoci
//////////////////////////////////////////////////////////////////////
GLenum textureFilter = GL_NEAREST;

//////////////////////////////////////////////////////////////////////
// znacznik informujcy czy naley wygenerow map cienia
//////////////////////////////////////////////////////////////////////
bool generateMap = true;

//////////////////////////////////////////////////////////////////////
// numeracja trybw renderingu
//////////////////////////////////////////////////////////////////////
enum
{
    RENDER_SCENE,       // rendering sceny
    RENDER_SHADOW_MAP   // rendering mapy gbokoci
};

//////////////////////////////////////////////////////////////////////
// tryb renderingu
//////////////////////////////////////////////////////////////////////
int renderMode = RENDER_SCENE;

//////////////////////////////////////////////////////////////////////
// algorytm filtracji mapy cienia
//////////////////////////////////////////////////////////////////////
int shadowMode = SHADOW_MAP_STD;

//////////////////////////////////////////////////////////////////////
// rozmiary obszaru renderingu okna programu
//////////////////////////////////////////////////////////////////////
GLuint windowWidth, windowHeight;

//////////////////////////////////////////////////////////////////////
// wsprzdne wierzchokw trjktw skadajcych si na kwadrat
//////////////////////////////////////////////////////////////////////
GLfloat positionRect[2*4] =
{
    -1.0f, -1.0f,
     1.0f, -1.0f,
    -1.0f,  1.0f,
     1.0f,  1.0f
};

//////////////////////////////////////////////////////////////////////
// wsprzdne wierzchokw trjktw skadajcych si na kwadrat
//////////////////////////////////////////////////////////////////////
GLfloat texCoord[2*4] =
{
    0.0f, 0.0f,
    1.0f, 0.0f,
    0.0f, 1.0f,
    1.0f, 1.0f
};

//////////////////////////////////////////////////////////////////////
// numery pooenia poszczeglnych atrybutw wierzchokw
//////////////////////////////////////////////////////////////////////
#define POSITION  0
#define NORMAL    1
#define TEX_COORD 2

//////////////////////////////////////////////////////////////////////
// rysowanie czajnika
// prg - numer obiektu programu
// projectionMatrix - macierz rzutowania
// modelViewMatrix - macierz modelu-widoku
// textureMatrix - macierz tekstury
//////////////////////////////////////////////////////////////////////
void DrawTeapot( const GLuint prg,
                 const glm::mat4x4 &projectionMatrix,
                 const glm::mat4x4 &modelViewMatrix,
                 const glm::mat4x4 &textureMatrix )
{
    // odwrcona macierz modelu-widoku niezbdna do przeksztace
    // do ukadu wsprzdnych obiektu
    glm::mat4x4 modelViewMatrixInverse( glm::inverse( modelViewMatrix ) );

    // przeksztacenie pooenia obserwatora do ukadu wsprzdnych obiektu
    glm::vec4 eyePosition( 0.0f, 0.0f, 0.0f, 1.0f );
    eyePosition = modelViewMatrixInverse * eyePosition;

    // przeksztacenie pooenia rda wiata do ukadu wsprzdnych obiektu
    glm::vec4 inverseLightPosition = modelViewMatrixInverse * lightPosition;

    // wczenie obiektu tablic wierzchokw
    glBindVertexArray( vertexArray[TEAPOT] );

    // wczenie programu
    glUseProgram( program[prg] );

    // zaadowanie zmiennej jednorodnej - iloczynu macierzy modelu-widoku i rzutowania
    glm::mat4 modelViewProjectionMatrix = projectionMatrix * modelViewMatrix;
    glUniformMatrix4fv( glGetUniformLocation( program[prg], "modelViewProjectionMatrix" ), 1, GL_FALSE, glm::value_ptr( modelViewProjectionMatrix ) );

    // zaadowanie pooenia rda wiata i pooenia obserwatora w ukadzie wsprzdnych obiektu
    glUniform4fv( glGetUniformLocation( program[prg], "lightSource[0].position" ), 1, glm::value_ptr( inverseLightPosition ) );
    glUniform4fv( glGetUniformLocation( program[prg], "eyePosition" ), 1, glm::value_ptr( eyePosition ) );

    // zaadowanie numeru wybranego materiau
    glUniform1i( glGetUniformLocation( program[prg], "material" ), material );

    // zmiana wartoci zmiennej jednorodnej - numeru jednostki teksturujcej
    glUniform1i( glGetUniformLocation( program[prg],"shadowTex" ), 0 );

    // zaadowanie zmiennej jednorodnej - macierzy tekstury
    glUniformMatrix4fv( glGetUniformLocation( program[prg], "textureMatrix" ), 1, GL_FALSE, glm::value_ptr( textureMatrix ) );

    // narysowanie danych zawartych w tablicach wierzchokw
    glDrawElements( GL_TRIANGLES, TEAPOT_INDICES_COUNT * 3, GL_UNSIGNED_INT, NULL );

    // wyczenie programu
    glUseProgram( 0 );

    // wyczenie obiektu tablic wierzchokw
    glBindVertexArray( 0 );
}

//////////////////////////////////////////////////////////////////////
// rysowanie cian pokoju
// prg - numer obiektu programu
// projectionMatrix - macierz rzutowania
// modelViewMatrix - macierz modelu-widoku
// textureMatrix - macierz tekstury
// clipPlane - wsprzdne paszczyzny obcinania
//////////////////////////////////////////////////////////////////////
void DrawRoom( const GLuint prg,
               const glm::mat4x4 &projectionMatrix,
               const glm::mat4x4 &modelViewMatrix,
               const glm::mat4x4 &textureMatrix,
               const glm::vec4 &clipPlane = glm::vec4( 0.0f, 0.0f, 0.0f, 0.0f ) )
{
    // wczenie obiektu tablic wierzchokw
    glBindVertexArray( vertexArray[ROOM] );

    // wczenie programu
    glUseProgram( program[prg] );

    // zaadowanie zmiennej jednorodnej - iloczynu macierzy modelu-widoku i rzutowania
    glm::mat4x4 modelViewProjectionMatrix = projectionMatrix * modelViewMatrix;
    glUniformMatrix4fv( glGetUniformLocation( program[prg], "modelViewProjectionMatrix" ), 1, GL_FALSE, glm::value_ptr( modelViewProjectionMatrix ) );

    // pooenie obserwatora w ukadzie wsprzdnych obserwatora
    GLfloat eyePosition[4] = { 0.0f, 0.0f, 0.0f, 1.0f };

    // zaadowanie pooenia rda wiata i pooenia obserwatora w ukadzie wsprzdnych obserwatora
    glUniform4fv( glGetUniformLocation( program[prg], "lightSource[0].position" ), 1, glm::value_ptr( lightPosition ) );
    glUniform4fv( glGetUniformLocation( program[prg], "eyePosition" ), 1, eyePosition );

    // zaadowanie numeru wybranego materiau
    glUniform1i( glGetUniformLocation( program[prg], "material" ), MTL_DEFAULT );

    // zmiana wartoci zmiennej jednorodnej - numeru jednostki teksturujcej
    glUniform1i( glGetUniformLocation( program[prg], "shadowTex" ), 0 );

    // zaadowanie zmiennej jednorodnej - macierzy tekstury
    glUniformMatrix4fv( glGetUniformLocation( program[prg], "textureMatrix" ), 1, GL_FALSE, glm::value_ptr( textureMatrix ) );

    // zaadowanie zmiennej jednorodnej - wsprzdnych paszczyzny obcinania
    glUniform4fv( glGetUniformLocation( program[prg], "clipPlane[0]" ), 1, glm::value_ptr( clipPlane ) );

    // narysowanie danych zawartych w tablicach wierzchokw
    glDrawArrays( GL_TRIANGLES, 0, 30 );

    // wyczenie programu
    glUseProgram( 0 );

    // wyczenie obiektu tablic wierzchokw
    glBindVertexArray( 0 );
}

//////////////////////////////////////////////////////////////////////
// rysowanie rda wiata
//////////////////////////////////////////////////////////////////////
void DrawLight()
{
    // transformacje pooenia rda wiata punktowego
    glm::mat4x4 modelViewMatrix = glm::mat4x4( 1.0 );
    modelViewMatrix = glm::translate( modelViewMatrix, glm::vec3( lightPosition ) );
    modelViewMatrix = glm::scale( modelViewMatrix, glm::vec3( 0.2f, 0.2f, 0.2f ) );

    // wczenie obiektu tablic wierzchokw
    glBindVertexArray( vertexArray[LIGHT_POINT] );

    // wczenie programu
    glUseProgram( program[LIGHT_POINT] );

    // zaadowanie zmiennej jednorodnej - iloczynu macierzy modelu-widoku i rzutowania
    glm::mat4x4 modelViewProjectionMatrix = projectionMatrix * modelViewMatrix;
    glUniformMatrix4fv( glGetUniformLocation( program[LIGHT_POINT], "modelViewProjectionMatrix" ), 1, GL_FALSE, glm::value_ptr( modelViewProjectionMatrix ) );

    // narysowanie danych zawartych w tablicach wierzchokw
    glDrawElements( GL_TRIANGLES, SPHERE_LOW_INDICES_COUNT * 3, GL_UNSIGNED_INT, NULL );

    // wyczenie programu
    glUseProgram( 0 );

    // wyczenie obiektu tablic wierzchokw
    glBindVertexArray( 0 );
}

//////////////////////////////////////////////////////////////////////
// generowanie mapy cienia
// modelViewMatrix - macierz modelu-widoku
//////////////////////////////////////////////////////////////////////
void GenerateShadowMap( glm::mat4x4 &modelViewMatrix )
{
    // bufor ramki do zapisu = obiekt bufora ramki
    glBindFramebuffer( GL_DRAW_FRAMEBUFFER, frameBuffer[depthMap] );

    // obszar renderingu - caa powierzchnia bufora ramki
    glViewport( 0, 0, FRAME_WIDTH[depthMap], FRAME_HEIGHT[depthMap] );

    // sprawdzenie, czy rdo wiata owietla przestrze pokoju
    bool lightRoom = (lightPosition[0] >= left && lightPosition[0] <= right &&
                      lightPosition[1] >= bottom && lightPosition[1] <= top &&
                      lightPosition[2] >= -far + 1.0f) || lightPosition[2] > -near;

    // konfiguracja macierzy modelu-widoku mapy cienia
    shadowModelViewMatrix = glm::mat4( 1.0f );
    if( lightRoom )
        shadowModelViewMatrix = glm::lookAt( glm::vec3( lightPosition[0], lightPosition[1], lightPosition[2] ),
                                             glm::vec3( 0.0f, 0.0f, 0.0f - (near+far)/2.0f ),
                                             glm::vec3( 0.0f, 1.0f, 0.0f ) );
    else
        shadowModelViewMatrix = glm::lookAt( glm::vec3( 0.0f, 9.0f, 1.0f - (near+far)/2.0f ),
                                             glm::vec3( 0.0f, 0.0f, 0.0f - (near+far)/2.0f ),
                                             glm::vec3( 0.0f, 1.0f, 0.0f ) );

    // konfiguracja macierzy rzutowania mapy cienia
    shadowProjectionMatrix = glm::mat4( 1.0f );

    // kt patrzenia kamery z punktu widzenia rda wiata
    const GLdouble angle = lightRoom ? 105.0 : 90.0;
    shadowProjectionMatrix = glm::perspective( angle, 1.0, 1.0, 500.0 );

    // czyszczenie zawartoci bufora gbokoci
    glClear( GL_DEPTH_BUFFER_BIT );

    // przesunicie wartoci gbokoci
    glPolygonOffset( 6.0, 0.0 );
    glEnable( GL_POLYGON_OFFSET_FILL );

    // rysowanie cian pokoju
    DrawRoom( GENERATE_SHADOW_MAP, shadowProjectionMatrix, shadowModelViewMatrix, glm::mat4( 1.0f ) );

    // rysowanie czajnika
    if( lightRoom )
        DrawTeapot( GENERATE_SHADOW_MAP, shadowProjectionMatrix, shadowModelViewMatrix * modelViewMatrix, glm::mat4( 1.0f ) );

    // wyczenie przesuwania wartoci gbokoci
    glDisable( GL_POLYGON_OFFSET_FILL );
}

//////////////////////////////////////////////////////////////////////
// rendering mapy cienia
//////////////////////////////////////////////////////////////////////
void RenderShadowMap()
{
    // wyczenie testu gbokoci
    glDisable( GL_DEPTH_TEST );

    // wybr tekstury 2D
    glBindTexture( GL_TEXTURE_2D, renderTexture[depthMap] );

    // tryb porwnywania - zwyka tekstura
    glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_COMPARE_MODE, GL_NONE );

    // bufor ramki do zapisu = domylny bufor ramki
    glBindFramebuffer( GL_DRAW_FRAMEBUFFER, 0 );

    // obszar renderingu - cae okno
    glViewport( 0, 0, windowWidth, windowHeight );

    // czyszczenie bufora koloru i bufora gbokoci
    glClear( GL_COLOR_BUFFER_BIT );

    // wczenie obiektu tablic wierzchokw
    glBindVertexArray( vertexArray[RECTANGLE] );

    // wczenie programu
    glUseProgram( program[RECT_SHADOW] );

    // zaadowanie zmiennej jednorodnej - uchwyt tekstury
    glUniform1i( glGetUniformLocation( program[RECT_SHADOW], "tex" ), 0 );

    // narysowanie danych zawartych w tablicach wierzchokw
    glDrawArrays( GL_TRIANGLE_STRIP, 0, 4 );

    // wyczenie programu
    glUseProgram( 0 );

    // wyczenie obiektu tablic wierzchokw
    glBindVertexArray( 0 );

    // tryb porwnywania - tekstura gbokoci
    glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_COMPARE_MODE, GL_COMPARE_REF_TO_TEXTURE );

    // wyczenie testu gbokoci
    glEnable( GL_DEPTH_TEST );
}

//////////////////////////////////////////////////////////////////////
// rendering sceny
// modelViewMatrix - macierz modelu-widoku
//////////////////////////////////////////////////////////////////////
void RenderScene( glm::mat4x4 &modelViewMatrix )
{
    // bufor ramki do zapisu = domylny bufor ramki
    glBindFramebuffer( GL_DRAW_FRAMEBUFFER, 0 );

    // obszar renderingu - cae okno
    glViewport( 0, 0, windowWidth, windowHeight );

    // czyszczenie bufora koloru i bufora gbokoci
    glClear( GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT );

    // konfiguracja podstawowej macierzy przeksztacenia wsprzdnych tekstury cienia
    glm::mat4x4 textureMatrix = glm::mat4x4( 1.0 );
    textureMatrix = glm::translate( textureMatrix, glm::vec3( 0.5f, 0.5f, 0.5f ) );
    textureMatrix = glm::scale( textureMatrix, glm::vec3( 0.5f, 0.5f, 0.5f ) );
    textureMatrix *= shadowProjectionMatrix;
    textureMatrix *= shadowModelViewMatrix;

    // sprawdzenie, czy rdo wiata znajduje si wewntrz pokoju
    // oraz innej pozycji wymagajcej dwuetapowego renderingu
    bool insideRoom = lightPosition[0] >= left && lightPosition[0] <= right &&
                      lightPosition[1] >= bottom && lightPosition[1] <= top &&
                      lightPosition[2] >= -far + 1.0f && lightPosition[2] <= -near + 1.5;

    // wiato wewntrz pokoju
    if( insideRoom )
    {
        // wczenie pierwszej paszczyzny obcinania
        glEnable( GL_CLIP_DISTANCE0 );

        // obliczenie wsprzdnych paszczyzny obcinania przechodzcej
        // przez punkt, w ktrym znajduje si rdo wiata o wektorze
        // normalnym okrelonym przez kierunek wiata
        glm::vec4 clipPlane = glm::vec4( 0.0f, 0.0f, -(near + far)/2.0f, 0.0f ) - lightPosition;
        clipPlane = glm::normalize( clipPlane );
        clipPlane[3] = -glm::dot( glm::vec3( clipPlane ), glm::vec3( lightPosition ) );

        // rysowanie cian pokoju; z uwagi na to, e wsprzdne wierzchokw
        // trjktw tworzcych ciany pokoju s ju w przestrzeni obserwatora,
        // macierz modelu-widoku jest macierz jednostkow
        DrawRoom( shadowMode, projectionMatrix, glm::mat4( 1.0f ), textureMatrix, clipPlane );
        DrawRoom( SHADOW_MAP_LIGHT, projectionMatrix, glm::mat4( 1.0f ), textureMatrix, -clipPlane );

        // wyczenie pierwszej paszczyzny obcinania
        glDisable( GL_CLIP_DISTANCE0 );
    }
    else
        // wiato poza pokojem
        DrawRoom( shadowMode, projectionMatrix, glm::mat4( 1.0f ), textureMatrix );

    // rysowanie czajnika
    DrawTeapot( shadowMode, projectionMatrix, modelViewMatrix, textureMatrix * modelViewMatrix );

    // rysowanie rda wiata
    DrawLight();
}

//////////////////////////////////////////////////////////////////////
// funkcja generujca scen 3D
//////////////////////////////////////////////////////////////////////
void DisplayScene()
{
    // macierz modelu-widoku = macierz jednostkowa
    glm::mat4x4 modelViewMatrix = glm::mat4x4( 1.0 );

    // przesunicie obserwatora tak, aby ukad wsprzdnych obiektu by w rodku bryy obcinania
    modelViewMatrix = glm::translate( modelViewMatrix, glm::vec3( 0.0f, 0.0f, -(near+far)/2.0f ) );

    // skalowanie obiektu
    modelViewMatrix = glm::scale( modelViewMatrix, glm::vec3( scale, scale, scale ) );

    // obroty obiektu
    modelViewMatrix = glm::rotate( modelViewMatrix, rotateX, glm::vec3( 1.0f, 0.0f, 0.0f ) );
    modelViewMatrix = glm::rotate( modelViewMatrix, rotateY, glm::vec3( 0.0f, 1.0f, 0.0f ) );

    // aktywny obiekt tekstury - wybr mapy gbokoci
    glBindTexture( GL_TEXTURE_2D, renderTexture[depthMap] );

    // filtr powikszajcy/pomniejszajcy tekstury gbokoci
    glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, textureFilter );
    glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, textureFilter );

    // generowanie mapy cienia (tylko w przypadku zmiany
    // pooenia obiektw sceny lub kierunku rda wiata)
    if( generateMap == true )
    {
        GenerateShadowMap( modelViewMatrix );
        generateMap = false;
    }

    // tryb renderingu: scena/mapa cieni
    switch( renderMode )
    {
        case RENDER_SCENE:
            RenderScene( modelViewMatrix );
            break;
        case RENDER_SHADOW_MAP:
            RenderShadowMap();
            break;
    }

    // wyczenie testu gbokoci
    glDisable( GL_DEPTH_TEST );

    // wypisanie pooenia rda wiata punktowego
    const glm::vec4 clGreen( 0.000000f, 0.501961f, 0.000000f, 1.000000f );
    std::ostringstream txt;
    txt << std::setprecision( 1 ) << std::fixed
        << "pooenie rda wiata: (" << lightPosition[0] << ";"
        << lightPosition[1] << ";" << lightPosition[2] + (near+far)/2.0f << ")";
    DrawText8x16( 3, 3, txt.str(), clGreen );

    // wypisanie nazwy materiau
    DrawText8x16( 3, 21, std::string( "materia: " ) + GetMaterialName( material ), clGreen );

    // wypisanie rozmiarw tekstury z map cienia
    txt.str( "" );
    txt << "rozmiar mapy cienia: " << FRAME_WIDTH[depthMap] << "" << FRAME_HEIGHT[depthMap];
    DrawText8x16( 3, 39, txt.str(), clGreen );

    // wypisanie filtra powikszajcy/pomniejszajcy tekstury gbokoci
    if( textureFilter == GL_NEAREST )
        DrawText8x16( 3, 56, "filtr tekstury gbokoci: GL_NEAREST", clGreen );
    else
        DrawText8x16( 3, 56, "filtr tekstury gbokoci: GL_LINEAR (PCF 22)", clGreen );

    // wypisanie filtracji mapy cienia
    switch( shadowMode )
    {
        case SHADOW_MAP_STD:
            DrawText8x16( 3, 73, "filtracja mapy cienia: brak", clGreen );
            break;
        case SHADOW_MAP_PCF_3X3_AVERAGE:
            DrawText8x16( 3, 73, "filtracja mapy cienia: PCF 3x3 z filtracj uredniajc", clGreen );
            break;
        case SHADOW_MAP_PCF_3X3_GAUSS:
            DrawText8x16( 3, 73, "filtracja mapy cienia: PCF 3x3 z filtracj Gaussa", clGreen );
            break;
        case SHADOW_MAP_PCF_5X5_AVERAGE:
            DrawText8x16( 3, 73, "filtracja mapy cienia: PCF 5x5 z filtracj uredniajc", clGreen );
            break;
        case SHADOW_MAP_PCF_5X5_GAUSS:
            DrawText8x16( 3, 73, "filtracja mapy cienia: PCF 5x5 z filtracj Gaussa", clGreen );
            break;
        case SHADOW_MAP_PCF_5X5_RANDOM:
            DrawText8x16( 3, 73, "filtracja mapy cienia: PCF 5x5 z losowym pooeniem prbek", clGreen );
            break;
        case SHADOW_MAP_PCF_5X5_POISSON:
            DrawText8x16( 3, 73, "filtracja mapy cienia: PCF 55 z rozkadem Poissona pooenia prbek", clGreen );
            break;
    }

    // wczenie testu gbokoci
    glEnable( GL_DEPTH_TEST );
}

//////////////////////////////////////////////////////////////////////
// zmiana wielkoci okna
//////////////////////////////////////////////////////////////////////
void Reshape( int width, int height )
{
    // zapisanie rozmiarw obszaru okna renderingu
    windowWidth = width;
    windowHeight = height;

    // parametry bryy obcinania - rzutowanie perspektywiczne
    projectionMatrix = glm::frustum( left, right, bottom, top, near, far );
}

//////////////////////////////////////////////////////////////////////
// inicjalizacja staych elementw maszyny stanu OpenGL
//////////////////////////////////////////////////////////////////////
void InitScene()
{
    // kolor ta - zawarto bufora koloru
    glClearColor( 1.0f, 1.0f, 1.0f, 1.0f );

    // wczytanie shaderw i przygotowanie obsugi programu
    program[SHADOW_MAP_LIGHT] = glCreateProgram();
    glAttachShader( program[SHADOW_MAP_LIGHT], LoadShader( GL_VERTEX_SHADER, "mapy_cieni_vs.glsl" ) );
    glAttachShader( program[SHADOW_MAP_LIGHT], LoadShader( GL_FRAGMENT_SHADER, "../../common/materials_static.glsl" ) );
    glAttachShader( program[SHADOW_MAP_LIGHT], LoadShader( GL_FRAGMENT_SHADER, "../../common/light_model_static.glsl" ) );
    glAttachShader( program[SHADOW_MAP_LIGHT], LoadShader( GL_FRAGMENT_SHADER, "../../common/blinn_phong_light.glsl" ) );
    glAttachShader( program[SHADOW_MAP_LIGHT], LoadShader( GL_FRAGMENT_SHADER, "mapy_cieni_swiatlo_fs.glsl" ) );
    LinkValidateProgram( program[SHADOW_MAP_LIGHT] );

    // wczytanie shaderw i przygotowanie obsugi programu
    program[SHADOW_MAP_STD] = glCreateProgram();
    glAttachShader( program[SHADOW_MAP_STD], LoadShader( GL_VERTEX_SHADER, "mapy_cieni_vs.glsl" ) );
    glAttachShader( program[SHADOW_MAP_STD], LoadShader( GL_FRAGMENT_SHADER, "../../common/materials_static.glsl" ) );
    glAttachShader( program[SHADOW_MAP_STD], LoadShader( GL_FRAGMENT_SHADER, "../../common/light_model_static.glsl" ) );
    glAttachShader( program[SHADOW_MAP_STD], LoadShader( GL_FRAGMENT_SHADER, "../../common/blinn_phong_light.glsl" ) );
    glAttachShader( program[SHADOW_MAP_STD], LoadShader( GL_FRAGMENT_SHADER, "../../common/lambert_light.glsl" ) );
    glAttachShader( program[SHADOW_MAP_STD], LoadShader( GL_FRAGMENT_SHADER, "mapy_cieni_std_fs.glsl" ) );
    LinkValidateProgram( program[SHADOW_MAP_STD] );

    // wczytanie shaderw i przygotowanie obsugi programu
    program[SHADOW_MAP_PCF_3X3_AVERAGE] = glCreateProgram();
    glAttachShader( program[SHADOW_MAP_PCF_3X3_AVERAGE], LoadShader( GL_VERTEX_SHADER, "mapy_cieni_vs.glsl" ) );
    glAttachShader( program[SHADOW_MAP_PCF_3X3_AVERAGE], LoadShader( GL_FRAGMENT_SHADER, "../../common/materials_static.glsl" ) );
    glAttachShader( program[SHADOW_MAP_PCF_3X3_AVERAGE], LoadShader( GL_FRAGMENT_SHADER, "../../common/light_model_static.glsl" ) );
    glAttachShader( program[SHADOW_MAP_PCF_3X3_AVERAGE], LoadShader( GL_FRAGMENT_SHADER, "../../common/blinn_phong_light.glsl" ) );
    glAttachShader( program[SHADOW_MAP_PCF_3X3_AVERAGE], LoadShader( GL_FRAGMENT_SHADER, "../../common/lambert_light.glsl" ) );
    glAttachShader( program[SHADOW_MAP_PCF_3X3_AVERAGE], LoadShader( GL_FRAGMENT_SHADER, "../../common/convolution_filters_3x3.glsl" ) );
    glAttachShader( program[SHADOW_MAP_PCF_3X3_AVERAGE], LoadShader( GL_FRAGMENT_SHADER, "mapy_cieni_pcf_3x3_average_fs.glsl" ) );
    LinkValidateProgram( program[SHADOW_MAP_PCF_3X3_AVERAGE] );

    // wczytanie shaderw i przygotowanie obsugi programu
    program[SHADOW_MAP_PCF_3X3_GAUSS] = glCreateProgram();
    glAttachShader( program[SHADOW_MAP_PCF_3X3_GAUSS], LoadShader( GL_VERTEX_SHADER, "mapy_cieni_vs.glsl" ) );
    glAttachShader( program[SHADOW_MAP_PCF_3X3_GAUSS], LoadShader( GL_FRAGMENT_SHADER, "../../common/materials_static.glsl" ) );
    glAttachShader( program[SHADOW_MAP_PCF_3X3_GAUSS], LoadShader( GL_FRAGMENT_SHADER, "../../common/light_model_static.glsl" ) );
    glAttachShader( program[SHADOW_MAP_PCF_3X3_GAUSS], LoadShader( GL_FRAGMENT_SHADER, "../../common/blinn_phong_light.glsl" ) );
    glAttachShader( program[SHADOW_MAP_PCF_3X3_GAUSS], LoadShader( GL_FRAGMENT_SHADER, "../../common/lambert_light.glsl" ) );
    glAttachShader( program[SHADOW_MAP_PCF_3X3_GAUSS], LoadShader( GL_FRAGMENT_SHADER, "../../common/convolution_filters_3x3.glsl" ) );
    glAttachShader( program[SHADOW_MAP_PCF_3X3_GAUSS], LoadShader( GL_FRAGMENT_SHADER, "mapy_cieni_pcf_3x3_gauss_fs.glsl" ) );
    LinkValidateProgram( program[SHADOW_MAP_PCF_3X3_GAUSS] );

    // wczytanie shaderw i przygotowanie obsugi programu
    program[SHADOW_MAP_PCF_5X5_AVERAGE] = glCreateProgram();
    glAttachShader( program[SHADOW_MAP_PCF_5X5_AVERAGE], LoadShader( GL_VERTEX_SHADER, "mapy_cieni_vs.glsl" ) );
    glAttachShader( program[SHADOW_MAP_PCF_5X5_AVERAGE], LoadShader( GL_FRAGMENT_SHADER, "../../common/materials_static.glsl" ) );
    glAttachShader( program[SHADOW_MAP_PCF_5X5_AVERAGE], LoadShader( GL_FRAGMENT_SHADER, "../../common/light_model_static.glsl" ) );
    glAttachShader( program[SHADOW_MAP_PCF_5X5_AVERAGE], LoadShader( GL_FRAGMENT_SHADER, "../../common/blinn_phong_light.glsl" ) );
    glAttachShader( program[SHADOW_MAP_PCF_5X5_AVERAGE], LoadShader( GL_FRAGMENT_SHADER, "../../common/lambert_light.glsl" ) );
    glAttachShader( program[SHADOW_MAP_PCF_5X5_AVERAGE], LoadShader( GL_FRAGMENT_SHADER, "mapy_cieni_pcf_5x5_average_fs.glsl" ) );
    LinkValidateProgram( program[SHADOW_MAP_PCF_5X5_AVERAGE] );

    // wczytanie shaderw i przygotowanie obsugi programu
    program[SHADOW_MAP_PCF_5X5_GAUSS] = glCreateProgram();
    glAttachShader( program[SHADOW_MAP_PCF_5X5_GAUSS], LoadShader( GL_VERTEX_SHADER, "mapy_cieni_vs.glsl" ) );
    glAttachShader( program[SHADOW_MAP_PCF_5X5_GAUSS], LoadShader( GL_FRAGMENT_SHADER, "../../common/materials_static.glsl" ) );
    glAttachShader( program[SHADOW_MAP_PCF_5X5_GAUSS], LoadShader( GL_FRAGMENT_SHADER, "../../common/light_model_static.glsl" ) );
    glAttachShader( program[SHADOW_MAP_PCF_5X5_GAUSS], LoadShader( GL_FRAGMENT_SHADER, "../../common/blinn_phong_light.glsl" ) );
    glAttachShader( program[SHADOW_MAP_PCF_5X5_GAUSS], LoadShader( GL_FRAGMENT_SHADER, "../../common/lambert_light.glsl" ) );
    glAttachShader( program[SHADOW_MAP_PCF_5X5_GAUSS], LoadShader( GL_FRAGMENT_SHADER, "mapy_cieni_pcf_5x5_gauss_fs.glsl" ) );
    LinkValidateProgram( program[SHADOW_MAP_PCF_5X5_GAUSS] );

    // wczytanie shaderw i przygotowanie obsugi programu
    program[SHADOW_MAP_PCF_5X5_RANDOM] = glCreateProgram();
    glAttachShader( program[SHADOW_MAP_PCF_5X5_RANDOM], LoadShader( GL_VERTEX_SHADER, "mapy_cieni_vs.glsl" ) );
    glAttachShader( program[SHADOW_MAP_PCF_5X5_RANDOM], LoadShader( GL_FRAGMENT_SHADER, "../../common/materials_static.glsl" ) );
    glAttachShader( program[SHADOW_MAP_PCF_5X5_RANDOM], LoadShader( GL_FRAGMENT_SHADER, "../../common/light_model_static.glsl" ) );
    glAttachShader( program[SHADOW_MAP_PCF_5X5_RANDOM], LoadShader( GL_FRAGMENT_SHADER, "../../common/blinn_phong_light.glsl" ) );
    glAttachShader( program[SHADOW_MAP_PCF_5X5_RANDOM], LoadShader( GL_FRAGMENT_SHADER, "../../common/lambert_light.glsl" ) );
    glAttachShader( program[SHADOW_MAP_PCF_5X5_RANDOM], LoadShader( GL_FRAGMENT_SHADER, "mapy_cieni_pcf_5x5_random_fs.glsl" ) );
    LinkValidateProgram( program[SHADOW_MAP_PCF_5X5_RANDOM] );

    // wczytanie shaderw i przygotowanie obsugi programu
    program[SHADOW_MAP_PCF_5X5_POISSON] = glCreateProgram();
    glAttachShader( program[SHADOW_MAP_PCF_5X5_POISSON], LoadShader( GL_VERTEX_SHADER, "mapy_cieni_vs.glsl" ) );
    glAttachShader( program[SHADOW_MAP_PCF_5X5_POISSON], LoadShader( GL_FRAGMENT_SHADER, "../../common/materials_static.glsl" ) );
    glAttachShader( program[SHADOW_MAP_PCF_5X5_POISSON], LoadShader( GL_FRAGMENT_SHADER, "../../common/light_model_static.glsl" ) );
    glAttachShader( program[SHADOW_MAP_PCF_5X5_POISSON], LoadShader( GL_FRAGMENT_SHADER, "../../common/blinn_phong_light.glsl" ) );
    glAttachShader( program[SHADOW_MAP_PCF_5X5_POISSON], LoadShader( GL_FRAGMENT_SHADER, "../../common/lambert_light.glsl" ) );
    glAttachShader( program[SHADOW_MAP_PCF_5X5_POISSON], LoadShader( GL_FRAGMENT_SHADER, "mapy_cieni_pcf_5x5_poisson_fs.glsl" ) );
    LinkValidateProgram( program[SHADOW_MAP_PCF_5X5_POISSON] );

    // wczytanie shaderw i przygotowanie obsugi programu
    program[LIGHT_POINT] = glCreateProgram();
    glAttachShader( program[LIGHT_POINT], LoadShader( GL_VERTEX_SHADER, "punkt_swiatla_vs.glsl" ) );
    glAttachShader( program[LIGHT_POINT], LoadShader( GL_FRAGMENT_SHADER, "punkt_swiatla_fs.glsl" ) );
    LinkValidateProgram( program[LIGHT_POINT] );

    // wczytanie shaderw i przygotowanie obsugi programu
    program[GENERATE_SHADOW_MAP] = glCreateProgram();
    glAttachShader( program[GENERATE_SHADOW_MAP], LoadShader( GL_VERTEX_SHADER, "generowanie_mapy_cienia_vs.glsl" ) );
    glAttachShader( program[GENERATE_SHADOW_MAP], LoadShader( GL_FRAGMENT_SHADER, "generowanie_mapy_cienia_fs.glsl" ) );
    LinkValidateProgram( program[GENERATE_SHADOW_MAP] );

    // wczytanie shaderw i przygotowanie obsugi programu
    program[RECT_SHADOW] = glCreateProgram();
    glAttachShader( program[RECT_SHADOW], LoadShader( GL_VERTEX_SHADER, "prostakat_vs.glsl" ) );
    glAttachShader( program[RECT_SHADOW], LoadShader( GL_FRAGMENT_SHADER, "prostakat_fs.glsl" ) );
    LinkValidateProgram( program[RECT_SHADOW] );

    // utworzenie poszczglnych obiektw bufora ramki razem z teksturami gbokoci
    for( int fbo = 0; fbo < FRAME_BUFFER_SIZE; fbo++ )
    {
        // utworzenie obiektu tekstury na dane bufora gbokoci
        glGenTextures( 1, &renderTexture[fbo] );
        glBindTexture( GL_TEXTURE_2D, renderTexture[fbo] );
        glTexImage2D( GL_TEXTURE_2D, 0, GL_DEPTH_COMPONENT32F, FRAME_WIDTH[fbo], FRAME_HEIGHT[fbo], 0, GL_DEPTH_COMPONENT, GL_FLOAT, NULL );

        // parametry tekstury gbokoci
        glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, textureFilter );
        glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, textureFilter );
        glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_BORDER );
        glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_BORDER );
        glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_COMPARE_MODE, GL_COMPARE_REF_TO_TEXTURE );
        GLfloat border[4] = { 1.0f, 0.0f, 0.0f, 0.0f };
        glTexParameterfv( GL_TEXTURE_2D, GL_TEXTURE_BORDER_COLOR, border );

        // utworzenie obiektu bufora ramki i doczenie obiektu tekstury
        glGenFramebuffers( 1, &frameBuffer[fbo] );
        glBindFramebuffer( GL_DRAW_FRAMEBUFFER, frameBuffer[fbo] );
        glFramebufferTexture( GL_DRAW_FRAMEBUFFER, GL_DEPTH_ATTACHMENT, renderTexture[fbo], 0 );

        // sprawdzenie poprawnoci i kompletnoci obiektu bufora ramki
        GLenum error = glCheckFramebufferStatus( GL_DRAW_FRAMEBUFFER );
        if( error != GL_FRAMEBUFFER_COMPLETE )
        {
            std::cout << "Niepoprawny obiekt bufora renderingu" << std::endl;
            exit( 0 );
        }
    }

    // generowanie identyfikatora obiektu tablic wierzchokw
    glGenVertexArrays( 1, &vertexArray[TEAPOT] );

    // utworzenie obiektu tablic wierzchokw
    glBindVertexArray( vertexArray[TEAPOT] );

    // utworzenie obiektu bufora wierzchokw (VBO) i zaadowanie danych
    glGenBuffers( 1, &vertexBuffer[TEAPOT_POSITION] );
    glBindBuffer( GL_ARRAY_BUFFER, vertexBuffer[TEAPOT_POSITION] );
    glBufferData( GL_ARRAY_BUFFER, sizeof( teapotPosition ), teapotPosition, GL_STATIC_DRAW );
    glVertexAttribPointer( POSITION, 3, GL_FLOAT, GL_FALSE, 0, NULL );

    // utworzenie obiektu bufora wierzchokw (VBO) i zaadowanie danych
    glGenBuffers( 1, &vertexBuffer[TEAPOT_NORMAL] );
    glBindBuffer( GL_ARRAY_BUFFER, vertexBuffer[TEAPOT_NORMAL] );
    glBufferData( GL_ARRAY_BUFFER, sizeof( teapotNormal ), teapotNormal, GL_STATIC_DRAW );
    glVertexAttribPointer( NORMAL, 3, GL_FLOAT, GL_FALSE, 0, NULL );

    // wczenie tablic wierzchokw
    glEnableVertexAttribArray( POSITION );
    glEnableVertexAttribArray( NORMAL );

    // utworzenie obiektu bufora indeksw wierzchokw i zaadowanie danych
    glGenBuffers( 1, &indicesBuffer[TEAPOT_INDICES] );
    glBindBuffer( GL_ELEMENT_ARRAY_BUFFER, indicesBuffer[TEAPOT_INDICES] );
    glBufferData( GL_ELEMENT_ARRAY_BUFFER, sizeof( teapotIndices ), teapotIndices, GL_STATIC_DRAW );

    // wyczenie obiektu tablic wierzchokw
    glBindVertexArray( 0 );

    // generowanie identyfikatora obiektu tablic wierzchokw
    glGenVertexArrays( 1, &vertexArray[LIGHT_POINT] );

    // utworzenie obiektu tablic wierzchokw
    glBindVertexArray( vertexArray[LIGHT_POINT] );

    // utworzenie obiektu bufora wierzchokw (VBO) i zaadowanie danych
    glGenBuffers( 1, &vertexBuffer[LIGHT_POINT_POSITION] );
    glBindBuffer( GL_ARRAY_BUFFER, vertexBuffer[LIGHT_POINT_POSITION] );
    glBufferData( GL_ARRAY_BUFFER, sizeof( sphereLowPositionNormal ), sphereLowPositionNormal, GL_STATIC_DRAW );
    glVertexAttribPointer( POSITION, 3, GL_FLOAT, GL_FALSE, 0, NULL );

    // wczenie tablic wierzchokw
    glEnableVertexAttribArray( POSITION );

    // utworzenie obiektu bufora indeksw wierzchokw i zaadowanie danych
    glGenBuffers( 1, &indicesBuffer[LIGHT_POINT_INDICES] );
    glBindBuffer( GL_ELEMENT_ARRAY_BUFFER, indicesBuffer[LIGHT_POINT_INDICES] );
    glBufferData( GL_ELEMENT_ARRAY_BUFFER, sizeof( sphereLowIndices ), sphereLowIndices, GL_STATIC_DRAW );

    // wyczenie obiektu tablic wierzchokw
    glBindVertexArray( 0 );

    // generowanie identyfikatora obiektu tablic wierzchokw
    glGenVertexArrays( 1, &vertexArray[ROOM] );

    // utworzenie obiektu tablic wierzchokw
    glBindVertexArray( vertexArray[ROOM] );

    // utworzenie obiektu bufora wierzchokw (VBO) i zaadowanie danych
    glGenBuffers( 1, &vertexBuffer[ROOM_POSITION] );
    glBindBuffer( GL_ARRAY_BUFFER, vertexBuffer[ROOM_POSITION] );
    glBufferData( GL_ARRAY_BUFFER, sizeof( roomPosition ), roomPosition, GL_STATIC_DRAW );
    glVertexAttribPointer( POSITION, 4, GL_FLOAT, GL_FALSE, 0, NULL );

    // utworzenie obiektu bufora wierzchokw (VBO) i zaadowanie danych
    glGenBuffers( 1, &vertexBuffer[ROOM_NORMAL] );
    glBindBuffer( GL_ARRAY_BUFFER, vertexBuffer[ROOM_NORMAL] );
    glBufferData( GL_ARRAY_BUFFER, sizeof( roomNormal ), roomNormal, GL_STATIC_DRAW );
    glVertexAttribPointer( NORMAL, 3, GL_FLOAT, GL_FALSE, 0, NULL );

    // wczenie tablic wierzchokw
    glEnableVertexAttribArray( POSITION );
    glEnableVertexAttribArray( NORMAL );

    // wyczenie obiektu tablic wierzchokw
    glBindVertexArray( 0 );

    // utworzenie obiektu tablic wierzchokw
    glGenVertexArrays( 1, &vertexArray[RECTANGLE] );
    glBindVertexArray( vertexArray[RECTANGLE] );

    // utworzenie obiektu bufora wierzchokw (VBO) i zaadowanie danych
    glGenBuffers( 1, &vertexBuffer[POSITION_RECT] );
    glBindBuffer( GL_ARRAY_BUFFER, vertexBuffer[POSITION_RECT] );
    glBufferData( GL_ARRAY_BUFFER, sizeof( positionRect ), positionRect, GL_STATIC_DRAW );
    glVertexAttribPointer( POSITION, 2, GL_FLOAT, GL_FALSE, 0, NULL );

    // utworzenie obiektu bufora wierzchokw (VBO) i zaadowanie danych
    glGenBuffers( 1, &vertexBuffer[TEXCOORD_RECT] );
    glBindBuffer( GL_ARRAY_BUFFER, vertexBuffer[TEXCOORD_RECT] );
    glBufferData( GL_ARRAY_BUFFER, sizeof( texCoord ), texCoord, GL_STATIC_DRAW );
    glVertexAttribPointer( TEX_COORD, 2, GL_FLOAT, GL_FALSE, 0, NULL );

    // wczenie tablic wierzchokw
    glEnableVertexAttribArray( POSITION );
    glEnableVertexAttribArray( TEX_COORD );

    // wyczenie obiektu tablic wierzchokw
    glBindVertexArray( 0 );

    // wczenie testu bufora gbokoci
    glEnable( GL_DEPTH_TEST );

    // wczenie mechanizmw uywanych podczas renderingu tekstu
    InitDrawText();
}

//////////////////////////////////////////////////////////////////////
// usunicie obiektw OpenGL
//////////////////////////////////////////////////////////////////////
void DeleteScene()
{
    // porzdki
    for( int i = 0; i < PROGRAM_SIZE; i++ )
        glDeleteProgram( program[i] );
    glDeleteBuffers( VERTEX_BUFFER_SIZE, vertexBuffer );
    glDeleteVertexArrays( VERTEX_ARRAY_SIZE, vertexArray );
    glDeleteBuffers( INDICES_BUFFER_SIZE, indicesBuffer );
    glDeleteTextures( RENDER_TEXTURE_SIZE, renderTexture );
    glDeleteFramebuffers( FRAME_BUFFER_SIZE, frameBuffer );

    // usunicie mechanizmw uywanych podczas renderingu tekstu
    DeleteDrawText();
}
