//////////////////////////////////////////////////////////////////////
// (c) Janusz Ganczarski
// http://www.januszg.hg.pl
// JanuszG@enter.net.pl
//////////////////////////////////////////////////////////////////////

#include <iostream>
#include <glm/glm.hpp>
#include <glm/gtc/matrix_transform.hpp>
#include <glm/gtc/type_ptr.hpp>
#include "shaders.h"
#include "text.h"
#include "textures.h"

//////////////////////////////////////////////////////////////////////
// rozmiary bryy obcinania
//////////////////////////////////////////////////////////////////////
GLfloat left = -2.0f;
GLfloat right = 2.0f;
GLfloat bottom = -2.0f;
GLfloat top = 2.0f;
GLfloat near = 3.0f;
GLfloat far = 19.0f;

//////////////////////////////////////////////////////////////////////
// macierz rzutowania
//////////////////////////////////////////////////////////////////////
glm::mat4x4 projectionMatrix;

//////////////////////////////////////////////////////////////////////
// wspczynniki skalowania obiektu
//////////////////////////////////////////////////////////////////////
GLfloat scale = 1.0f;

//////////////////////////////////////////////////////////////////////
// kty obrotu obiektu
//////////////////////////////////////////////////////////////////////
GLfloat rotateX = 15.0f;
GLfloat rotateY = 0.0f;

//////////////////////////////////////////////////////////////////////
// przesunicie obiektu
//////////////////////////////////////////////////////////////////////
GLfloat translateX = 0.0f;
GLfloat translateZ = 0.0f;

//////////////////////////////////////////////////////////////////////
// numeracja obiektw programu, obiektw tablic wierzchokw
// i obiektw tekstury
//////////////////////////////////////////////////////////////////////
enum
{
    FLOOR,          // podoe
    TREE,           // drzewa
    PROGRAM_SIZE,
    VERTEX_ARRAY_SIZE = PROGRAM_SIZE,
    TEXTURE_SIZE = PROGRAM_SIZE
};

//////////////////////////////////////////////////////////////////////
// identyfikatory obiektw programu
//////////////////////////////////////////////////////////////////////
GLuint program[PROGRAM_SIZE];

//////////////////////////////////////////////////////////////////////
// numeracja obiektw bufora wierzchokw
//////////////////////////////////////////////////////////////////////
enum
{
    POSITION_FLOOR,
    TEX_COORD_FLOOR,
    POSITION_TREE,
    TEX_COORD_TREE,
    VERTEX_BUFFER_SIZE
};

//////////////////////////////////////////////////////////////////////
// identyfikatory obiektw bufora z danymi tablic wierzchokw
//////////////////////////////////////////////////////////////////////
GLuint vertexBuffer[VERTEX_BUFFER_SIZE];

//////////////////////////////////////////////////////////////////////
// identyfikatory obiektw tablic wierzchokw
//////////////////////////////////////////////////////////////////////
GLuint vertexArray[VERTEX_ARRAY_SIZE];

//////////////////////////////////////////////////////////////////////
// wsprzdne wierzchokw trjktw skadajcych si na obiekt
//////////////////////////////////////////////////////////////////////
GLfloat positionFloor[4*3] =
{
    -10.0f, -1.0f, -10.0f,
    10.0f, -1.0f, -10.0f,
    -10.0f, -1.0f, 10.0f,
    10.0f, -1.0f, 10.0f
};

//////////////////////////////////////////////////////////////////////
// wsprzdne tekstury
//////////////////////////////////////////////////////////////////////
GLfloat texCoordFloor[4*2] =
{
    0.0f, 0.0f,
    5.0f, 0.0f,
    0.0f, 5.0f,
    5.0f, 5.0f
};

//////////////////////////////////////////////////////////////////////
// wsprzdne wierzchokw trjktw skadajcych si na obiekt
//////////////////////////////////////////////////////////////////////
GLfloat positionTree[12*3] =
{
    -1.5f, -1.0f, 0.0f,
    1.5f, -1.0f, 0.0f,
    1.5f, 2.0f, 0.0f,
    -1.5f, -1.0f, 0.0f,
    1.5f, 2.0f, 0.0f,
    -1.5f, 2.0f, 0.0f,

    0.0f, -1.0f, -1.5f,
    0.0f, -1.0f, 1.5f,
    0.0f, 2.0f, 1.5f,
    0.0f, -1.0f, -1.5f,
    0.0f, 2.0f, 1.5f,
    0.0f, 2.0f, -1.5f
};

//////////////////////////////////////////////////////////////////////
// wsprzdne tekstury
//////////////////////////////////////////////////////////////////////
GLfloat texCoordTree[12*2] =
{
    0.0f, 1.0f,
    1.0f, 1.0f,
    1.0f, 0.0f,
    0.0f, 1.0f,
    1.0f, 0.0f,
    0.0f, 0.0f,

    0.0f, 1.0f,
    1.0f, 1.0f,
    1.0f, 0.0f,
    0.0f, 1.0f,
    1.0f, 0.0f,
    0.0f, 0.0f
};

//////////////////////////////////////////////////////////////////////
// numery indeksw poszczeglnych atrybutw wierzchokw
//////////////////////////////////////////////////////////////////////
enum
{
    POSITION,
    TEX_COORD
};

//////////////////////////////////////////////////////////////////////
// identyfikatory obiektw tekstury
//////////////////////////////////////////////////////////////////////
GLuint textures[TEXTURE_SIZE];

//////////////////////////////////////////////////////////////////////
// zmienne konfigurujce tryb renderingu:
// - zastpowanie wartoci pokrycia skadow A
// - mieszanie kolorw
// - test skadowej A
//////////////////////////////////////////////////////////////////////
GLboolean alphaToCoverage = GL_TRUE;
GLboolean colorBlend = GL_FALSE;
GLboolean alphaTest = GL_FALSE;

//////////////////////////////////////////////////////////////////////
// funkcja generujca scen 3D
//////////////////////////////////////////////////////////////////////
void DisplayScene()
{
    // czyszczenie bufora koloru i bufora gbokoci
    glClear( GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT );

    // macierz widoku - macierz jednostkowa
    glm::mat4x4 modelViewMatrix = glm::mat4x4( 1.0 );

    // przesunicie obserwatora tak, aby ukad wsprzdnych obiektu by w rodku bryy obcinania
    modelViewMatrix = glm::translate( modelViewMatrix, glm::vec3( translateX, 0.0f, translateZ-(near+far)/2.0f ) );

    // skalowanie obiektu
    modelViewMatrix = glm::scale( modelViewMatrix, glm::vec3( scale, scale, scale ) );

    // obroty obiektu
    modelViewMatrix = glm::rotate( modelViewMatrix, rotateX, glm::vec3( 1.0f, 0.0f, 0.0f ) );
    modelViewMatrix = glm::rotate( modelViewMatrix, rotateY, glm::vec3( 0.0f, 1.0f, 0.0f ) );

    // wczenie obiektu tablic wierzchokw
    glBindVertexArray( vertexArray[FLOOR] );

    // wybr tekstury 2D
    glBindTexture( GL_TEXTURE_2D, textures[FLOOR] );

    // wczenie programu
    glUseProgram( program[FLOOR] );

    // zaadowanie zmiennej jednorodnej - iloczynu macierzy modelu-widoku i rzutowania
    glm::mat4x4 modelViewProjectionMatrix = projectionMatrix * modelViewMatrix;
    glUniformMatrix4fv( glGetUniformLocation( program[FLOOR], "modelViewProjectionMatrix" ), 1, GL_FALSE, glm::value_ptr( modelViewProjectionMatrix ) );

    // zmiana wartoci zmiennej jednorodnej - numeru jednostki teksturujcej
    glUniform1i( glGetUniformLocation( program[FLOOR] ,"tex" ), 0 );

    // narysowanie danych zawartych w tablicach wierzchokw
    glDrawArrays( GL_TRIANGLE_STRIP, 0, 4 );

    // wyczenie programu
    glUseProgram( 0 );

    // wyczenie tekstury 2D
    glBindTexture( GL_TEXTURE_2D, 0 );

    // wyczenie obiektu tablic wierzchokw
    glBindVertexArray( 0 );

    // wczenie/wyczenie mieszania kolorw
    if( colorBlend == GL_TRUE )
        glEnable( GL_BLEND );
    else
        glDisable( GL_BLEND );

    // wczenie/wyczenie zastpowania wartoci pokrycia prbki skadow A
    if( alphaToCoverage == GL_TRUE )
        glEnable( GL_SAMPLE_ALPHA_TO_COVERAGE );
    else
        glDisable( GL_SAMPLE_ALPHA_TO_COVERAGE );

    // wczenie obiektu tablic wierzchokw
    glBindVertexArray( vertexArray[TREE] );

    // wybr tekstury 2D
    glBindTexture( GL_TEXTURE_2D, textures[TREE] );

    // wczenie programu
    glUseProgram( program[TREE] );

    // zaadowanie zmiennej jednorodnej - iloczynu macierzy modelu-widoku i rzutowania
    glUniformMatrix4fv( glGetUniformLocation( program[TREE], "modelViewProjectionMatrix" ), 1, GL_FALSE, glm::value_ptr( modelViewProjectionMatrix ) );

    // zmiana wartoci zmiennej jednorodnej - numeru jednostki teksturujcej
    glUniform1i( glGetUniformLocation( program[TREE] ,"tex" ), 0 );

    // zmiana wartoci zmiennej jednorodnej - znacznik testu skadowej A
    glUniform1i( glGetUniformLocation( program[TREE], "alphaTest" ), alphaTest );

    // narysowanie danych zawartych w tablicach wierzchokw
    glDrawArraysInstanced( GL_TRIANGLES, 0, 12, 14 );

    // wyczenie programu
    glUseProgram( 0 );

    // wyczenie tekstury 2D
    glBindTexture( GL_TEXTURE_2D, 0 );

    // wyczenie obiektu tablic wierzchokw
    glBindVertexArray( 0 );

    // wywietlenie informacji o konfiguracji trybu renderingu
    glDisable( GL_DEPTH_TEST );
    const glm::vec4 yellowGreen( 0.603922f, 0.803922f, 0.196078f, 1.000000f );
    if( alphaToCoverage == GL_TRUE )
        DrawText8x16( 3, 3, "GL_SAMPLE_ALPHA_TO_COVERAGE = GL_TRUE", yellowGreen );
    else
        DrawText8x16( 3, 3, "GL_SAMPLE_ALPHA_TO_COVERAGE = GL_FALSE", yellowGreen );
    if( colorBlend == GL_TRUE )
        DrawText8x16( 3, 20, "GL_BLEND = GL_TRUE", yellowGreen );
    else
        DrawText8x16( 3, 20, "GL_BLEND = GL_FALSE", yellowGreen );
    if( alphaTest == GL_TRUE )
        DrawText8x16( 3, 37, "test alfa = GL_TRUE", yellowGreen );
    else
        DrawText8x16( 3, 37, "test alfa = GL_FALSE", yellowGreen );
    glEnable( GL_DEPTH_TEST );
}

//////////////////////////////////////////////////////////////////////
// zmiana wielkoci okna
//////////////////////////////////////////////////////////////////////
void Reshape( int width, int height )
{
    // obszar renderingu - cae okno
    glViewport( 0, 0, width, height );

    // parametry bryy obcinania - rzutowanie perspektywiczne
    // wysoko okna wiksza od szerokoci okna
    if( width < height && width > 0 )
         projectionMatrix = glm::frustum( left, right, bottom*height/width, top*height/width, near, far );
    else
        // szeroko okna wiksza lub rwna wysokoci okna
        if (width >= height && height > 0)
            projectionMatrix = glm::frustum( left*width/height, right*width/height, bottom, top, near, far );
        else
            projectionMatrix = glm::frustum( left, right, bottom, top, near, far );
}

//////////////////////////////////////////////////////////////////////
// inicjalizacja staych elementw maszyny stanu OpenGL
//////////////////////////////////////////////////////////////////////
void InitScene()
{
    // kolor ta - zawarto bufora koloru
    glClearColor( 0.9f, 0.9f, 1.0f, 1.0f );

    // generowanie dwch identyfikatorw obiektw tekstury
    glGenTextures( TEXTURE_SIZE, textures );

    // utworzenie pierwszego obiektu tekstury
    glBindTexture( GL_TEXTURE_2D, textures[FLOOR] );

    // wczytanie tekstury
    if( !LoadTexture( "../../media/developer.nvidia.com/grass.png", GL_TEXTURE_2D ) )
    {
        std::cout << "Niepoprawny odczyt pliku " << "../../media/developer.nvidia.com/grass.png" << std::endl;
        exit( 0 );
    }

    // filtr pomniejszajcy
    glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR );

    // tryby zawijania tekstury
    glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT );
    glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT );

    // utworzenie drugiego obiektu tekstury
    glBindTexture( GL_TEXTURE_2D, textures[TREE] );

    // wczytanie tekstury
    if( !LoadTexture( "../../media/developer.nvidia.com/Palm.png", GL_TEXTURE_2D ) )
    {
        std::cout << "Niepoprawny odczyt pliku " << "../../media/developer.nvidia.com/Palm.png" << std::endl;
        exit( 0 );
    }

    // filtr pomniejszajcy
    glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR );

    // tryby zawijania tekstury
    glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE );
    glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE );

    // przeczenie na tekstur domyln
    glBindTexture( GL_TEXTURE_2D, 0 );

    // wczytanie shaderw i przygotowanie obsugi programu
    program[FLOOR] = glCreateProgram();
    glAttachShader( program[FLOOR], LoadShader( GL_VERTEX_SHADER, "wieloprobkowanie_z_alfa_ziemia_vs.glsl" ) );
    glAttachShader( program[FLOOR], LoadShader( GL_FRAGMENT_SHADER, "wieloprobkowanie_z_alfa_fs.glsl" ) );
    LinkValidateProgram( program[FLOOR] );

    // wczytanie shaderw i przygotowanie obsugi programu
    program[TREE] = glCreateProgram();
    glAttachShader( program[TREE], LoadShader( GL_VERTEX_SHADER, "wieloprobkowanie_z_alfa_drzewo_vs.glsl" ) );
    glAttachShader( program[TREE], LoadShader( GL_FRAGMENT_SHADER, "wieloprobkowanie_z_alfa_fs.glsl" ) );
    LinkValidateProgram( program[TREE] );

    // generowanie identyfikatora obiektu tablic wierzchokw
    glGenVertexArrays( 1, &vertexArray[FLOOR] );

    // utworzenie pierwszego obiektu tablic wierzchokw
    glBindVertexArray( vertexArray[FLOOR] );

    // utworzenie obiektu bufora wierzchokw (VBO) i zaadowanie danych
    glGenBuffers( 1, &vertexBuffer[POSITION_FLOOR] );
    glBindBuffer( GL_ARRAY_BUFFER, vertexBuffer[POSITION_FLOOR] );
    glBufferData( GL_ARRAY_BUFFER, sizeof( positionFloor ), positionFloor, GL_STATIC_DRAW );
    glVertexAttribPointer( POSITION, 3, GL_FLOAT, GL_FALSE, 0, NULL );

    // utworzenie obiektu bufora wierzchokw (VBO) i zaadowanie danych
    glGenBuffers( 1, &vertexBuffer[TEX_COORD_FLOOR] );
    glBindBuffer( GL_ARRAY_BUFFER, vertexBuffer[TEX_COORD_FLOOR] );
    glBufferData( GL_ARRAY_BUFFER, sizeof( texCoordFloor ), texCoordFloor, GL_STATIC_DRAW );
    glVertexAttribPointer( TEX_COORD, 2, GL_FLOAT, GL_FALSE, 0, NULL );

    // wczenie tablic wierzchokw
    glEnableVertexAttribArray( POSITION );
    glEnableVertexAttribArray( TEX_COORD );

    // wyczenie obiektu tablic wierzchokw
    glBindVertexArray( 0 );

    // generowanie identyfikatora obiektu tablic wierzchokw
    glGenVertexArrays( 1, &vertexArray[TREE] );

    // utworzenie pierwszego obiektu tablic wierzchokw
    glBindVertexArray( vertexArray[TREE] );

    // utworzenie obiektu bufora wierzchokw (VBO) i zaadowanie danych
    glGenBuffers( 1, &vertexBuffer[POSITION_TREE] );
    glBindBuffer( GL_ARRAY_BUFFER, vertexBuffer[POSITION_TREE] );
    glBufferData( GL_ARRAY_BUFFER, sizeof( positionTree ), positionTree, GL_STATIC_DRAW );
    glVertexAttribPointer( POSITION, 3, GL_FLOAT, GL_FALSE, 0, NULL );

    // utworzenie obiektu bufora wierzchokw (VBO) i zaadowanie danych
    glGenBuffers( 1, &vertexBuffer[TEX_COORD_TREE] );
    glBindBuffer( GL_ARRAY_BUFFER, vertexBuffer[TEX_COORD_TREE] );
    glBufferData( GL_ARRAY_BUFFER, sizeof( texCoordTree ), texCoordTree, GL_STATIC_DRAW );
    glVertexAttribPointer( TEX_COORD, 2, GL_FLOAT, GL_FALSE, 0, NULL );

    // wczenie tablic wierzchokw
    glEnableVertexAttribArray( POSITION );
    glEnableVertexAttribArray( TEX_COORD );

    // wyczenie obiektu tablic wierzchokw
    glBindVertexArray( 0 );

    // wczenie mechanizmw uywanych podczas renderingu tekstu
    InitDrawText();

    // wczenie testu bufora gbokoci
    glEnable( GL_DEPTH_TEST );

    // wczenie wieloprbkowania
    glEnable( GL_MULTISAMPLE );

    // rwnanie mieszania kolorw
    glBlendFunc( GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA );
}

//////////////////////////////////////////////////////////////////////
// usunicie obiektw OpenGL
//////////////////////////////////////////////////////////////////////
void DeleteScene()
{
    // porzdki
    glDeleteProgram( program[FLOOR] );
    glDeleteProgram( program[TREE] );
    glDeleteBuffers( VERTEX_BUFFER_SIZE, vertexBuffer );
    glDeleteVertexArrays( VERTEX_ARRAY_SIZE, vertexArray );
    glDeleteTextures( TEXTURE_SIZE, textures );

    // usunicie mechanizmw uywanych podczas renderingu tekstu
    DeleteDrawText();
}
