//////////////////////////////////////////////////////////////////////
// (c) Janusz Ganczarski
// http://www.januszg.hg.pl
// JanuszG@enter.net.pl
//////////////////////////////////////////////////////////////////////

#include <string>
#include <sstream>
#include <iostream>
#include <cstdlib>
#include <glm/glm.hpp>
#include <glm/gtc/matrix_transform.hpp>
#include <glm/gtc/type_ptr.hpp>
#include "shaders.h"
#include "text.h"
#include "textures.h"

//////////////////////////////////////////////////////////////////////
// rozmiary bryy obcinania
//////////////////////////////////////////////////////////////////////
GLfloat left = -2.0f;
GLfloat right = 2.0f;
GLfloat bottom = -2.0f;
GLfloat top = 2.0f;
GLfloat near = 4.0f;
GLfloat far = 100.0f;

//////////////////////////////////////////////////////////////////////
// macierz rzutowania
//////////////////////////////////////////////////////////////////////
glm::mat4x4 projectionMatrix;

//////////////////////////////////////////////////////////////////////
// kty obrotu obiektu
//////////////////////////////////////////////////////////////////////
GLfloat rotateX = 0.0f;
GLfloat rotateY = 0.0f;

//////////////////////////////////////////////////////////////////////
// przesunicie obiektu
//////////////////////////////////////////////////////////////////////
GLfloat translateX = 0.0f;
GLfloat translateZ = 10.0f;

//////////////////////////////////////////////////////////////////////
// identyfikator obiektu programu
//////////////////////////////////////////////////////////////////////
GLuint program;

//////////////////////////////////////////////////////////////////////
// numeracja obiektw bufora wierzchokw
//////////////////////////////////////////////////////////////////////
enum
{
    POSITION,
    TEX_COORD,
    VERTEX_BUFFER_SIZE
};

//////////////////////////////////////////////////////////////////////
// identyfikatory obiektw bufora z danymi tablic wierzchokw
//////////////////////////////////////////////////////////////////////
GLuint vertexBuffer[VERTEX_BUFFER_SIZE];

//////////////////////////////////////////////////////////////////////
// identyfikator obiektu bufora z danymi tablicy
// indeksw wierzchokw obiektu
//////////////////////////////////////////////////////////////////////
GLuint indicesBuffer;

//////////////////////////////////////////////////////////////////////
// identyfikator obiektu tablic wierzchokw
//////////////////////////////////////////////////////////////////////
GLuint vertexArray;

//////////////////////////////////////////////////////////////////////
// wsprzdne wierzchokw trjktw skadajcych si na szecian
//////////////////////////////////////////////////////////////////////
GLfloat position[8*3] =
{
    20.0f, 20.0f, 20.0f,
    -20.0f, 20.0f, 20.0f,
    -20.0f, -20.0f, 20.0f,
    20.0f, -20.0f, 20.0f,
    20.0f, 20.0f, -20.0f,
    -20.0f, 20.0f, -20.0f,
    -20.0f, -20.0f, -20.0f,
    20.0f, -20.0f, -20.0f
};

//////////////////////////////////////////////////////////////////////
// skadowe wsprzdnych tekstury szeciennej
//////////////////////////////////////////////////////////////////////
const GLfloat texCoord[8*3] =
{
    1.0f, 1.0f, 1.0f,
    -1.0f, 1.0f, 1.0f,
    -1.0f, -1.0f, 1.0f,
    1.0f, -1.0f, 1.0f,
    1.0f, 1.0f, -1.0f,
    -1.0f, 1.0f, -1.0f,
    -1.0f, -1.0f, -1.0f,
    1.0f, -1.0f, -1.0f
};

//////////////////////////////////////////////////////////////////////
// dane indeksw wierzchokw trjktw skadajcych si na szecian
//////////////////////////////////////////////////////////////////////
GLuint indices[12*3] =
{
    5, 0, 1,
    5, 4, 0,
    2, 0, 3,
    2, 1, 0,
    7, 0, 4,
    7, 3, 0,
    3, 6, 2,
    3, 7, 6,
    1, 2, 6,
    1, 6, 5,
    4, 5, 6,
    4, 6, 7
};

//////////////////////////////////////////////////////////////////////
// numeracja obiektw tekstury
//////////////////////////////////////////////////////////////////////
enum
{
    CUBEMAP_MASKONAIVE,
    CUBEMAP_TEIDE,
    CUBEMAP_TENERIFE,
    CUBEMAP_VASA,
    TEXTURE_SIZE
};

//////////////////////////////////////////////////////////////////////
// identyfikatory obiektw tekstury
//////////////////////////////////////////////////////////////////////
GLuint texture[TEXTURE_SIZE];

//////////////////////////////////////////////////////////////////////
// numer wybranego obiektu tekstury
//////////////////////////////////////////////////////////////////////
int texNumber = CUBEMAP_MASKONAIVE;

//////////////////////////////////////////////////////////////////////
// parametry biecej tekstury
//////////////////////////////////////////////////////////////////////
GLint minFilter = GL_NEAREST_MIPMAP_LINEAR;
GLint magFilter = GL_NEAREST;
GLint textureWrap = GL_REPEAT;
GLboolean cubeMapSeamless = GL_FALSE;

//////////////////////////////////////////////////////////////////////
// funkcja generujca scen 3D
//////////////////////////////////////////////////////////////////////
void DisplayScene()
{
    // czyszczenie bufora gbokoci
    glClear( GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT );

    // macierz widoku - macierz jednostkowa
    glm::mat4x4 modelViewMatrix = glm::mat4x4( 1.0 );

    // przesunicie obserwatora do punktu pocztkowego wewntrz skybox'a
    modelViewMatrix = glm::translate( modelViewMatrix, glm::vec3( translateX, 0.0f, translateZ ) );

    // obroty obiektu
    modelViewMatrix = glm::rotate( modelViewMatrix, rotateX, glm::vec3( 1.0f, 0.0f, 0.0f ) );
    modelViewMatrix = glm::rotate( modelViewMatrix, rotateY, glm::vec3( 0.0f, 1.0f, 0.0f ) );

    // wybr obiektu tekstury
    glBindTexture( GL_TEXTURE_CUBE_MAP, texture[texNumber] );

    // bezszwowa filtracja tekstury szeciennej
    if( cubeMapSeamless )
        glEnable( GL_TEXTURE_CUBE_MAP_SEAMLESS );
    else
        glDisable( GL_TEXTURE_CUBE_MAP_SEAMLESS );

    // filtr powikszajcy i pomniejszajcy
    glTexParameteri( GL_TEXTURE_CUBE_MAP, GL_TEXTURE_MAG_FILTER, magFilter );
    glTexParameteri( GL_TEXTURE_CUBE_MAP, GL_TEXTURE_MIN_FILTER, minFilter );

    // tryb zawijania
    glTexParameteri( GL_TEXTURE_CUBE_MAP, GL_TEXTURE_WRAP_S, textureWrap );
    glTexParameteri( GL_TEXTURE_CUBE_MAP, GL_TEXTURE_WRAP_T, textureWrap );
    glTexParameteri( GL_TEXTURE_CUBE_MAP, GL_TEXTURE_WRAP_R, textureWrap );

    // wczenie obiektu tablic wierzchokw
    glBindVertexArray( vertexArray );

    // wczenie programu
    glUseProgram( program );

    // zaadowanie zmiennej jednorodnej - iloczynu macierzy modelu-widoku i rzutowania
    glm::mat4x4 modelViewProjectionMatrix = projectionMatrix * modelViewMatrix;
    glUniformMatrix4fv( glGetUniformLocation( program, "modelViewProjectionMatrix" ), 1, GL_FALSE, glm::value_ptr( modelViewProjectionMatrix ) );

    // zaadowanie zmiennej jednorodnej - numeru jednostki teksturujcej
    glUniform1i( glGetUniformLocation( program, "tex" ), 0 );

    // narysowanie danych zawartych w tablicach wierzchokw
    glDrawElements( GL_TRIANGLES, 12 * 3, GL_UNSIGNED_INT, NULL );

    // wyczenie programu
    glUseProgram( 0 );

    // wyczenie obiektu tablic wierzchokw
    glBindVertexArray( 0 );

    // pobranie i wywietlenie parametrw renderingu biecej tekstury szeciennej
    if( glIsEnabled( GL_TEXTURE_CUBE_MAP_SEAMLESS ) )
        DrawText8x16( 3, 3, "GL_TEXTURE_CUBE_MAP_SEAMLESS = GL_TRUE", glm::vec4( 1.0f ) );
    else
        DrawText8x16( 3, 3, "GL_TEXTURE_CUBE_MAP_SEAMLESS = GL_FALSE", glm::vec4( 1.0f ) );
    GLint magFilter;
    glGetTexParameteriv( GL_TEXTURE_CUBE_MAP, GL_TEXTURE_MAG_FILTER, &magFilter );
    if( magFilter == GL_NEAREST )
        DrawText8x16( 3, 21, "GL_TEXTURE_MAG_FILTER = GL_NEAREST", glm::vec4( 1.0f ) );
    else if( magFilter == GL_LINEAR )
        DrawText8x16( 3, 21, "GL_TEXTURE_MAG_FILTER = GL_LINEAR", glm::vec4( 1.0f ) );
    GLint minFilter;
    glGetTexParameteriv( GL_TEXTURE_CUBE_MAP, GL_TEXTURE_MIN_FILTER, &minFilter );
    switch( minFilter )
    {
        case GL_NEAREST: DrawText8x16( 3, 39, "GL_TEXTURE_MIN_FILTER = GL_NEAREST", glm::vec4( 1.0f ) ); break;
        case GL_LINEAR: DrawText8x16( 3, 39, "GL_TEXTURE_MIN_FILTER = GL_LINEAR", glm::vec4( 1.0f ) ); break;
        case GL_NEAREST_MIPMAP_NEAREST: DrawText8x16( 3, 39, "GL_TEXTURE_MIN_FILTER = GL_NEAREST_MIPMAP_NEAREST", glm::vec4( 1.0f ) ); break;
        case GL_LINEAR_MIPMAP_NEAREST: DrawText8x16( 3, 39, "GL_TEXTURE_MIN_FILTER = GL_LINEAR_MIPMAP_NEAREST", glm::vec4( 1.0f ) ); break;
        case GL_NEAREST_MIPMAP_LINEAR: DrawText8x16( 3, 39, "GL_TEXTURE_MIN_FILTER = GL_NEAREST_MIPMAP_LINEAR", glm::vec4( 1.0f ) ); break;
        case GL_LINEAR_MIPMAP_LINEAR: DrawText8x16( 3, 39, "GL_TEXTURE_MIN_FILTER = GL_LINEAR_MIPMAP_LINEAR", glm::vec4( 1.0f ) ); break;
    }
    GLint wrapS;
    glGetTexParameteriv( GL_TEXTURE_CUBE_MAP, GL_TEXTURE_WRAP_S, &wrapS );
    switch( wrapS )
    {
        case GL_CLAMP_TO_EDGE: DrawText8x16( 3, 57, "GL_TEXTURE_WRAP_S/T/R = GL_CLAMP_TO_EDGE", glm::vec4( 1.0f ) ); break;
        case GL_REPEAT: DrawText8x16( 3, 57, "GL_TEXTURE_WRAP_S/T/R = GL_REPEAT", glm::vec4( 1.0f ) ); break;
        case GL_CLAMP_TO_BORDER: DrawText8x16( 3, 57, "GL_TEXTURE_WRAP_S/T/R = GL_CLAMP_TO_BORDER", glm::vec4( 1.0f ) ); break;
        case GL_MIRRORED_REPEAT: DrawText8x16( 3, 57, "GL_TEXTURE_WRAP_S/T/R = GL_MIRRORED_REPEAT", glm::vec4( 1.0f ) ); break;
        case GL_MIRROR_CLAMP_TO_EDGE: DrawText8x16( 3, 57, "GL_TEXTURE_WRAP_S/T/R = GL_MIRROR_CLAMP_TO_EDGE", glm::vec4( 1.0f ) ); break;
    }

    // wyczenie obiektu tekstury
    glBindTexture( GL_TEXTURE_CUBE_MAP, 0 );
}

//////////////////////////////////////////////////////////////////////
// zmiana wielkoci okna
//////////////////////////////////////////////////////////////////////
void Reshape( int width, int height )
{
    // obszar renderingu - cae okno
    glViewport( 0, 0, width, height );

    // parametry bryy obcinania - rzutowanie perspektywiczne
    // wysoko okna wiksza od szerokoci okna
    // wysoko okna wiksza od szerokoci okna
    if( width < height && width > 0 )
         projectionMatrix = glm::frustum( left, right, bottom*height/width, top*height/width, near, far );
    else
        // szeroko okna wiksza lub rwna wysokoci okna
        if (width >= height && height > 0)
            projectionMatrix = glm::frustum( left*width/height, right*width/height, bottom, top, near, far );
        else
            projectionMatrix = glm::frustum( left, right, bottom, top, near, far );
}

//////////////////////////////////////////////////////////////////////
// inicjalizacja staych elementw maszyny stanu OpenGL
//////////////////////////////////////////////////////////////////////
void InitScene()
{
    // wczytanie shaderw i przygotowanie obsugi programu
    program = glCreateProgram();
    glAttachShader( program, LoadShader( GL_VERTEX_SHADER, "skybox_vs.glsl" ) );
    glAttachShader( program, LoadShader( GL_FRAGMENT_SHADER, "skybox_fs.glsl" ) );
    LinkValidateProgram( program );

    // utworzenie obiektu tablic wierzchokw
    glGenVertexArrays( 1, &vertexArray );
    glBindVertexArray( vertexArray );

    // utworzenie obiektu bufora wierzchokw (VBO) i zaadowanie danych
    glGenBuffers( 1, &vertexBuffer[POSITION] );
    glBindBuffer( GL_ARRAY_BUFFER, vertexBuffer[POSITION] );
    glBufferData( GL_ARRAY_BUFFER, sizeof( position ), position, GL_STATIC_DRAW );
    glVertexAttribPointer( POSITION, 3, GL_FLOAT, GL_FALSE, 0, NULL );

    // utworzenie obiektu bufora wierzchokw (VBO) i zaadowanie danych
    glGenBuffers( 1, &vertexBuffer[TEX_COORD] );
    glBindBuffer( GL_ARRAY_BUFFER, vertexBuffer[TEX_COORD] );
    glBufferData( GL_ARRAY_BUFFER, sizeof( texCoord ), texCoord, GL_STATIC_DRAW );
    glVertexAttribPointer( TEX_COORD, 3, GL_FLOAT, GL_FALSE, 0, NULL );

    // wczenie tablic wierzchokw
    glEnableVertexAttribArray( POSITION );
    glEnableVertexAttribArray( TEX_COORD );

    // utworzenie obiektu bufora indeksw wierzchokw i zaadowanie danych
    glGenBuffers( 1, &indicesBuffer );
    glBindBuffer( GL_ELEMENT_ARRAY_BUFFER, indicesBuffer );
    glBufferData( GL_ELEMENT_ARRAY_BUFFER, sizeof( indices ), indices, GL_STATIC_DRAW );

    // wyczenie obiektu tablic wierzchokw
    glBindVertexArray( 0 );

    // nazwy plikw z obrazami trzech tekstur szeciennych
    // dane pochodz ze strony WWW: http://www.humus.name
    const std::string fileNames[6*TEXTURE_SIZE] =
    {
        "../../media/www.humus.name/maskonaive/posx.jpg",
        "../../media/www.humus.name/maskonaive/negx.jpg",
        "../../media/www.humus.name/maskonaive/posy.jpg",
        "../../media/www.humus.name/maskonaive/negy.jpg",
        "../../media/www.humus.name/maskonaive/posz.jpg",
        "../../media/www.humus.name/maskonaive/negz.jpg",

        "../../media/www.humus.name/teide/posx.jpg",
        "../../media/www.humus.name/teide/negx.jpg",
        "../../media/www.humus.name/teide/posy.jpg",
        "../../media/www.humus.name/teide/negy.jpg",
        "../../media/www.humus.name/teide/posz.jpg",
        "../../media/www.humus.name/teide/negz.jpg",

        "../../media/www.humus.name/tenerife/posx.jpg",
        "../../media/www.humus.name/tenerife/negx.jpg",
        "../../media/www.humus.name/tenerife/posy.jpg",
        "../../media/www.humus.name/tenerife/negy.jpg",
        "../../media/www.humus.name/tenerife/posz.jpg",
        "../../media/www.humus.name/tenerife/negz.jpg",

        "../../media/www.humus.name/vasa/posx.jpg",
        "../../media/www.humus.name/vasa/negx.jpg",
        "../../media/www.humus.name/vasa/posy.jpg",
        "../../media/www.humus.name/vasa/negy.jpg",
        "../../media/www.humus.name/vasa/posz.jpg",
        "../../media/www.humus.name/vasa/negz.jpg"
    };

    // kolejne obrazy tekstury szeciennej
    const GLenum targets[6] =
    {
        GL_TEXTURE_CUBE_MAP_POSITIVE_X,
        GL_TEXTURE_CUBE_MAP_NEGATIVE_X,
        GL_TEXTURE_CUBE_MAP_POSITIVE_Y,
        GL_TEXTURE_CUBE_MAP_NEGATIVE_Y,
        GL_TEXTURE_CUBE_MAP_POSITIVE_Z,
        GL_TEXTURE_CUBE_MAP_NEGATIVE_Z
    };

    // generowanie trzech identyfikatorw obiektw tekstury
    glGenTextures( TEXTURE_SIZE, texture );

    // utworzenie trzech tekstur szeciennych
    for( int i = 0; i < TEXTURE_SIZE; i++ )
    {
        // utworzenie obiektu tekstury
        glBindTexture( GL_TEXTURE_CUBE_MAP, texture[i] );

        // odczyt kolejnych plikw z tekstur i ewentualny komunikat o bdzie
        for( int j = 0; j < 6; j++ )
            if( !LoadTexture( fileNames[i*6+j].c_str(), targets[j] ) )
            {
                std::cout << "Niepoprawny odczyt pliku " << fileNames[i*6+j] << std::endl;
                exit( 0 );
            }

        // generownanie mipmap
        glGenerateMipmap( GL_TEXTURE_CUBE_MAP );
    }

    // przeczenie na tekstur domyln
    glBindTexture( GL_TEXTURE_CUBE_MAP, 0 );

    // wczenie testu bufora gbokoci
    glEnable( GL_DEPTH_TEST );

    // wczenie mechanizmw uywanych podczas renderingu tekstu
    InitDrawText();
}

//////////////////////////////////////////////////////////////////////
// usunicie obiektw OpenGL
//////////////////////////////////////////////////////////////////////
void DeleteScene()
{
    // porzdki
    glDeleteProgram( program );
    glDeleteBuffers( VERTEX_BUFFER_SIZE, vertexBuffer );
    glDeleteBuffers( 1, &indicesBuffer );
    glDeleteVertexArrays( 1, &vertexArray );
    glDeleteTextures( TEXTURE_SIZE, texture );

    // usunicie mechanizmw uywanych podczas renderingu tekstu
    DeleteDrawText();
}
