//////////////////////////////////////////////////////////////////////
// (c) Janusz Ganczarski
// http://www.januszg.hg.pl
// JanuszG@enter.net.pl
//////////////////////////////////////////////////////////////////////

#include <string>
#include <sstream>
#include <iomanip>
#include <cstring>
#include <glm/glm.hpp>
#include <glm/gtc/matrix_transform.hpp>
#include <glm/gtc/type_ptr.hpp>
#include "shaders.h"
#include "teapot.h"
#include "teapot_low.h"
#include "teapot_high.h"
#include "materials.h"
#include "text.h"

//////////////////////////////////////////////////////////////////////
// rozmiary bryy obcinania
//////////////////////////////////////////////////////////////////////
GLfloat left = -4.0f;
GLfloat right = 4.0f;
GLfloat bottom = -4.0f;
GLfloat top = 4.0f;
GLfloat near = 10.0f;
GLfloat far = 18.0f;

//////////////////////////////////////////////////////////////////////
// macierz rzutowania
//////////////////////////////////////////////////////////////////////
glm::mat4x4 projectionMatrix;

//////////////////////////////////////////////////////////////////////
// wspczynniki skalowania obiektu
//////////////////////////////////////////////////////////////////////
GLfloat scale = 1.0f;

//////////////////////////////////////////////////////////////////////
// kty obrotu obiektu
//////////////////////////////////////////////////////////////////////
GLfloat rotateX = 0.0f;
GLfloat rotateY = 0.0f;

//////////////////////////////////////////////////////////////////////
// przesunicie obiektu
//////////////////////////////////////////////////////////////////////
GLfloat translateX = 0.0f;
GLfloat translateY = 0.0f;

//////////////////////////////////////////////////////////////////////
// kty obrotu kierunku wiata
//////////////////////////////////////////////////////////////////////
GLfloat rotateLightX = 0.0f;
GLfloat rotateLightY = 0.0f;

//////////////////////////////////////////////////////////////////////
// wybrany materia
//////////////////////////////////////////////////////////////////////
int material = MTL_BRASS;

//////////////////////////////////////////////////////////////////////
// numeracja obiektw programu
//////////////////////////////////////////////////////////////////////
enum
{
    FLAT_TESS,          // teselacja paska
    PN_TRINAGLES_TESS,  // algorytm PN Trinagles
    PHONG_TESS,         // algorytm teselacji Phonga
    LIGHT_VECTOR,       // kierunek wiata
    PROGRAM_SIZE
};

//////////////////////////////////////////////////////////////////////
// identyfikatory obiektw programu
//////////////////////////////////////////////////////////////////////
GLuint program[PROGRAM_SIZE];

//////////////////////////////////////////////////////////////////////
// rodzaj modelu owietlenia Phonga/Blinna-Phonga/Lamberta
//////////////////////////////////////////////////////////////////////
enum
{
    PHONG_LIGHT,
    BLINN_PHONG_LIGHT,
    LAMBERT_LIGHT
};
int lightType = PHONG_LIGHT;

//////////////////////////////////////////////////////////////////////
// numeracja obiektw bufora wierzchokw
// numery pooenia poszczeglnych atrybutw wierzchokw
//////////////////////////////////////////////////////////////////////
enum
{
    POSITION,
    NORMAL,
    POSITION_TEAPOT = POSITION,
    POSITION_TEAPOT_LOW,
    POSITION_TEAPOT_HIGH,
    NORMAL_TEAPOT,
    NORMAL_TEAPOT_LOW,
    NORMAL_TEAPOT_HIGH,
    POSITION_LIGHT,
    VERTEX_BUFFER_SIZE
};

//////////////////////////////////////////////////////////////////////
// identyfikatory obiektw bufora z danymi tablic wierzchokw:
// wsprzdnymi wierzchokw i wektorw normalnych
//////////////////////////////////////////////////////////////////////
GLuint vertexBuffer[VERTEX_BUFFER_SIZE];

//////////////////////////////////////////////////////////////////////
// numeracja obiektw bufora indeksw wierzchokw, obiektw
// tablic wierzchokw oraz rodzajw czajnikw
//////////////////////////////////////////////////////////////////////
enum
{
    TEAPOT,
    TEAPOT_LOW,
    TEAPOT_HIGH,
    INDICES_BUFFER_SIZE,
    LIGHT = INDICES_BUFFER_SIZE,
    VERTEX_ARRAY_SIZE
};

//////////////////////////////////////////////////////////////////////
// identyfikatory obiektw bufora z danymi tablic
// indeksw wierzchokw obiektw
//////////////////////////////////////////////////////////////////////
GLuint indicesBuffer[INDICES_BUFFER_SIZE];

//////////////////////////////////////////////////////////////////////
// identyfikatory obiektw tablic wierzchokw
//////////////////////////////////////////////////////////////////////
GLuint vertexArray[VERTEX_ARRAY_SIZE];

//////////////////////////////////////////////////////////////////////
// numery nazwanych blokw zmiennych jednorodnych
//////////////////////////////////////////////////////////////////////
enum
{
    MATERIALS,
    LIGHT_MODEL,
    UBO_SIZE
};

//////////////////////////////////////////////////////////////////////
// rozmiary blokw zmiennych jednorodnych
//////////////////////////////////////////////////////////////////////
GLint size[UBO_SIZE];

//////////////////////////////////////////////////////////////////////
// przesunicia pooenia blokw zmiennych jednorodnych
//////////////////////////////////////////////////////////////////////
GLint offset[UBO_SIZE];

//////////////////////////////////////////////////////////////////////
// numeracja modyfikowanych zmiennych z bloku jednorodnego
//////////////////////////////////////////////////////////////////////
enum
{
    LIGHT_SOURCE_0_AMBIENT,
    LIGHT_SOURCE_0_DIFFUSE,
    LIGHT_SOURCE_0_SPECULAR,
    LIGHT_SOURCE_0_POSITION,
    LIGHT_SOURCE_0_CONSTANT_ATTENUATION,
    LIGHT_SOURCE_0_LINEAR_ATTENUATION,
    LIGHT_SOURCE_0_QUADRATIC_ATTENUATION,
    LIGHT_MODEL_AMBIENT,
    EYE_POSITION,
    UNIFORMS_SIZE
};

//////////////////////////////////////////////////////////////////////
// nazwy modyfikowanych zmiennych z bloku jednorodnego
//////////////////////////////////////////////////////////////////////
const GLchar *uniformsName[UNIFORMS_SIZE] =
{
    "lightSource[0].ambient",
    "lightSource[0].diffuse",
    "lightSource[0].specular",
    "lightSource[0].position",
    "lightSource[0].constantAttenuation",
    "lightSource[0].linearAttenuation",
    "lightSource[0].quadraticAttenuation",
    "lightModel.ambient",
    "eyePosition"
};

//////////////////////////////////////////////////////////////////////
// wartoci przesunicia (pooenia) zmiennych jednorodnych w bloku
//////////////////////////////////////////////////////////////////////
GLint uniformsOffset[UNIFORMS_SIZE];

//////////////////////////////////////////////////////////////////////
// identyfikator obiektu bufora bloku zmiennych jednorodnych
//////////////////////////////////////////////////////////////////////
GLuint uniformBuffer;

//////////////////////////////////////////////////////////////////////
// liczba danych w tablicach indeksw wierzchokw
//////////////////////////////////////////////////////////////////////
const GLuint indicesCount[3] =
{
    TEAPOT_INDICES_COUNT * 3,
    TEAPOT_LOW_INDICES_COUNT * 3,
    TEAPOT_HIGH_INDICES_COUNT * 3
};

//////////////////////////////////////////////////////////////////////
// numer rysowanego czajnika
//////////////////////////////////////////////////////////////////////
int teapot = TEAPOT;

//////////////////////////////////////////////////////////////////////
// wewntrzny i zewntrzny poziom teselacji
//////////////////////////////////////////////////////////////////////
float tessLevel = 1.0;

//////////////////////////////////////////////////////////////////////
// stopie "puszystoci" powierzchni w teselacji Phonga
//////////////////////////////////////////////////////////////////////
GLfloat alpha = 0.75f;

//////////////////////////////////////////////////////////////////////
// wybr obiektu programu - rodzaju teselacji
//////////////////////////////////////////////////////////////////////
GLuint tesselation = FLAT_TESS;

//////////////////////////////////////////////////////////////////////
// funkcja generujca scen 3D
//////////////////////////////////////////////////////////////////////
void DisplayScene()
{
    // czyszczenie bufora koloru i bufora gbokoci
    glClear( GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT );

    // macierz modelu-widoku = macierz jednostkowa
    glm::mat4x4 modelViewMatrix = glm::mat4x4( 1.0 );

    // przesunicie obserwatora tak, aby ukad wsprzdnych obiektu by w rodku bryy obcinania
    modelViewMatrix = glm::translate( modelViewMatrix, glm::vec3( 0.0f, 0.0f, -(near+far)/2.0f ) );

    // skalowanie obiektu
    modelViewMatrix = glm::scale( modelViewMatrix, glm::vec3( scale, scale, scale ) );

    // przesunicie obiektu
    modelViewMatrix = glm::translate( modelViewMatrix, glm::vec3( translateX, translateY, 0.0f ) );

    // obroty obiektu
    modelViewMatrix = glm::rotate( modelViewMatrix, rotateX, glm::vec3( 1.0f, 0.0f, 0.0f ) );
    modelViewMatrix = glm::rotate( modelViewMatrix, rotateY, glm::vec3( 0.0f, 1.0f, 0.0f ) );

    // obroty wektora kierunku wiata
    glm::mat4x4 rotateLightDir = glm::mat4x4( 1.0 );
    rotateLightDir = glm::rotate( rotateLightDir, rotateLightX, glm::vec3( 1.0f, 0.0f, 0.0f ) );
    rotateLightDir = glm::rotate( rotateLightDir, rotateLightY, glm::vec3( 0.0f, 1.0f, 0.0f ) );
    glm::vec4 lightPosition( 0.0f, 0.0f, 1.0f, 0.0f );
    lightPosition = rotateLightDir * lightPosition;
    lightPosition = glm::normalize( lightPosition );

    // odwrcona macierz modelu-widoku niezbdna do przeksztace
    // do ukadu wsprzdnych obiektu
    glm::mat4x4 modelViewMatrixInverse( glm::inverse( modelViewMatrix ) );

    // transformacja kierunku wiata do ukadu wsprzdnych obiektu
    glm::vec4 inverseLightPosition( modelViewMatrixInverse * lightPosition );
    inverseLightPosition = glm::normalize( inverseLightPosition );

    // przeksztacenie pooenia obserwatora do ukadu wsprzdnych obiektu
    glm::vec4 eyePosition( 0.0f, 0.0f, 0.0f, 1.0f );
    eyePosition = modelViewMatrixInverse * eyePosition;

    // wczenie obiektu tablic wierzchokw
    glBindVertexArray( vertexArray[teapot] );

    // wczenie programu
    glUseProgram( program[tesselation] );

    // zaadowanie zmiennej jednorodnej - iloczynu macierzy modelu-widoku i rzutowania
    glm::mat4x4 modelViewProjectionMatrix = projectionMatrix * modelViewMatrix;
    glUniformMatrix4fv( glGetUniformLocation( program[tesselation], "modelViewProjectionMatrix" ), 1, GL_FALSE, glm::value_ptr( modelViewProjectionMatrix ) );

    // zaadowanie zmiennej jednorodnej - rodzaj materiau
    glUniform1i( glGetUniformLocation( program[tesselation], "material" ), material );

    // wczenie UBO
    glBindBufferRange( GL_UNIFORM_BUFFER, LIGHT_MODEL, uniformBuffer, offset[LIGHT_MODEL], size[LIGHT_MODEL] );

    // zaadowanie zmiennej jednorodnej z kierunkiem wiata
    glBufferSubData( GL_UNIFORM_BUFFER, uniformsOffset[LIGHT_SOURCE_0_POSITION], 16, glm::value_ptr( inverseLightPosition ) );

    // zaadowanie zmiennej jednorodnej z pooeniem obserwatora
    glBufferSubData( GL_UNIFORM_BUFFER, uniformsOffset[EYE_POSITION], 16, glm::value_ptr( eyePosition ) );

    // wyczenie UBO
    glBindBuffer( GL_UNIFORM_BUFFER, 0 );

    // pobranie indeksu funkcji obsugujcej wybrany rodzaj modelu owietlenia
    const char *lightName[3] = { "Phong", "BlinnPhong", "Lambert" };
    GLuint light = glGetSubroutineIndex( program[tesselation], GL_FRAGMENT_SHADER, lightName[lightType] );

    // zaadowanie zmiennej jednorodnej podprogramu - rodzaj modelu owietlenia
    glUniformSubroutinesuiv( GL_FRAGMENT_SHADER, 1, &light );

    // zaadowanie zmiennej jednorodnej z wewntrznym i zewntrznym poziomem teselacji
    glUniform1f( glGetUniformLocation( program[tesselation], "tessLevel" ), tessLevel );

    // zaadowanie zmiennej jednorodnej ze stopniem "puszystoci" powierzchni
    if( tesselation == PHONG_TESS )
        glUniform1f( glGetUniformLocation( program[tesselation], "alpha" ), alpha );

    // narysowanie danych zawartych w tablicach wierzchokw
    glDrawElements( GL_PATCHES, indicesCount[teapot], GL_UNSIGNED_INT, NULL );

    // wyczenie programu
    glUseProgram( 0 );

    // wyczenie obiektu tablic wierzchokw
    glBindVertexArray( 0 );

    // transformacje pooenia wektora kierunku wiata
    modelViewMatrix = glm::mat4x4( 1.0 );
    modelViewMatrix = glm::translate( modelViewMatrix, glm::vec3( 0.0f, 0.0f, -(near+far)/2.0f ) );
    modelViewMatrix *= rotateLightDir;

    // wczenie obiektu tablic wierzchokw
    glBindVertexArray( vertexArray[LIGHT] );

    // wczenie programu
    glUseProgram( program[LIGHT_VECTOR] );

    // zaadowanie zmiennej jednorodnej - iloczynu macierzy modelu-widoku i rzutowania
    modelViewProjectionMatrix = projectionMatrix * modelViewMatrix;
    glUniformMatrix4fv( glGetUniformLocation( program[LIGHT_VECTOR], "modelViewProjectionMatrix" ), 1, GL_FALSE, glm::value_ptr( modelViewProjectionMatrix ) );

    // narysowanie danych zawartych w tablicach wierzchokw
    glDrawArrays( GL_LINES, 0, 2 );

    // wyczenie programu
    glUseProgram( 0 );

    // wyczenie obiektu tablic wierzchokw
    glBindVertexArray( 0 );

    // wypisanie wektora kierunku rda wiata
    std::ostringstream txt;
    txt << std::setprecision( 4 ) << std::fixed
        << "Wektor kierunku wiata: (" << lightPosition[0] << ";" << lightPosition[1] << ";" << lightPosition[2] << ")";
    DrawText8x16( 3, 3, txt.str() );

    // wypisanie nazwy materiau
    DrawText8x16( 3, 21, std::string( "Materia: " ) + GetMaterialName( material ) );

    // wypisanie rodzaju modelu owietlenia
    const char* ligtTypeStr[] = { "owietlenie Phonga", "owietlenie Blinna-Phonga", "owietlenie Lamberta" };
    DrawText8x16( 3, 39, ligtTypeStr[lightType] );

    // wypisanie rodzaju i poziomi teselacji
    txt.str( "" );
    const char* tessTypeStr[] = { "teselacja paska", "teselacja PN Triangles", "teselacja Phonga" };
    txt << std::setprecision( 1 ) << std::fixed;
    txt << tessTypeStr[tesselation] << ", poziom = " << tessLevel;
    if( tesselation == PHONG_TESS )
        txt << std::setprecision( 2 ) << std::fixed << ", alfa = " << alpha;
    DrawText8x16( 3, 58, txt.str() );
}

//////////////////////////////////////////////////////////////////////
// zmiana wielkoci okna
//////////////////////////////////////////////////////////////////////
void Reshape( int width, int height )
{
    // obszar renderingu - cae okno
    glViewport( 0, 0, width, height );

    // parametry bryy obcinania - rzutowanie perspektywiczne
    // wysoko okna wiksza od szerokoci okna
    if( width < height && width > 0 )
         projectionMatrix = glm::frustum( left, right, bottom*height/width, top*height/width, near, far );
    else
        // szeroko okna wiksza lub rwna wysokoci okna
        if (width >= height && height > 0)
            projectionMatrix = glm::frustum( left*width/height, right*width/height, bottom, top, near, far );
        else
            projectionMatrix = glm::frustum( left, right, bottom, top, near, far );
}

//////////////////////////////////////////////////////////////////////
// inicjalizacja staych elementw maszyny stanu OpenGL
//////////////////////////////////////////////////////////////////////
void InitScene()
{
    // kolor ta - zawarto bufora koloru
    glClearColor( 1.0f, 1.0f, 1.0f, 1.0f );

    // wczytanie shaderw i przygotowanie obsugi programu
    program[FLAT_TESS] = glCreateProgram();
    glAttachShader( program[FLAT_TESS], LoadShader( GL_VERTEX_SHADER, "oswietlenie_phonga_teselacja_vs.glsl" ) );
    glAttachShader( program[FLAT_TESS], LoadShader( GL_TESS_CONTROL_SHADER, "oswietlenie_phonga_teselacja_plaska_tcs.glsl" ) );
    glAttachShader( program[FLAT_TESS], LoadShader( GL_TESS_EVALUATION_SHADER, "oswietlenie_phonga_teselacja_plaska_tes.glsl" ) );
    glAttachShader( program[FLAT_TESS], LoadShader( GL_FRAGMENT_SHADER, "../../common/light_model.glsl" ) );
    glAttachShader( program[FLAT_TESS], LoadShader( GL_FRAGMENT_SHADER, "../../common/lambert_light.glsl" ) );
    glAttachShader( program[FLAT_TESS], LoadShader( GL_FRAGMENT_SHADER, "../../common/blinn_phong_light.glsl" ) );
    glAttachShader( program[FLAT_TESS], LoadShader( GL_FRAGMENT_SHADER, "../../common/phong_light.glsl" ) );
    glAttachShader( program[FLAT_TESS], LoadShader( GL_FRAGMENT_SHADER, "../../common/materials.glsl" ) );
    glAttachShader( program[FLAT_TESS], LoadShader( GL_FRAGMENT_SHADER, "oswietlenie_phonga_teselacja_fs.glsl" ) );
    LinkValidateProgram( program[FLAT_TESS] );

    // wczytanie shaderw i przygotowanie obsugi programu
    program[PN_TRINAGLES_TESS] = glCreateProgram();
    glAttachShader( program[PN_TRINAGLES_TESS], LoadShader( GL_VERTEX_SHADER, "oswietlenie_phonga_teselacja_vs.glsl" ) );
    glAttachShader( program[PN_TRINAGLES_TESS], LoadShader( GL_TESS_CONTROL_SHADER, "oswietlenie_phonga_teselacja_PNT_tcs.glsl" ) );
    glAttachShader( program[PN_TRINAGLES_TESS], LoadShader( GL_TESS_EVALUATION_SHADER, "oswietlenie_phonga_teselacja_PNT_tes.glsl" ) );
    glAttachShader( program[PN_TRINAGLES_TESS], LoadShader( GL_FRAGMENT_SHADER, "../../common/light_model.glsl" ) );
    glAttachShader( program[PN_TRINAGLES_TESS], LoadShader( GL_FRAGMENT_SHADER, "../../common/lambert_light.glsl" ) );
    glAttachShader( program[PN_TRINAGLES_TESS], LoadShader( GL_FRAGMENT_SHADER, "../../common/blinn_phong_light.glsl" ) );
    glAttachShader( program[PN_TRINAGLES_TESS], LoadShader( GL_FRAGMENT_SHADER, "../../common/phong_light.glsl" ) );
    glAttachShader( program[PN_TRINAGLES_TESS], LoadShader( GL_FRAGMENT_SHADER, "../../common/materials.glsl" ) );
    glAttachShader( program[PN_TRINAGLES_TESS], LoadShader( GL_FRAGMENT_SHADER, "oswietlenie_phonga_teselacja_fs.glsl" ) );
    LinkValidateProgram( program[PN_TRINAGLES_TESS] );

    // wczytanie shaderw i przygotowanie obsugi programu
    program[PHONG_TESS] = glCreateProgram();
    glAttachShader( program[PHONG_TESS], LoadShader( GL_VERTEX_SHADER, "oswietlenie_phonga_teselacja_vs.glsl" ) );
    glAttachShader( program[PHONG_TESS], LoadShader( GL_TESS_CONTROL_SHADER, "oswietlenie_phonga_teselacja_Phong_tcs.glsl" ) );
    glAttachShader( program[PHONG_TESS], LoadShader( GL_TESS_EVALUATION_SHADER, "oswietlenie_phonga_teselacja_Phong_tes.glsl" ) );
    glAttachShader( program[PHONG_TESS], LoadShader( GL_FRAGMENT_SHADER, "../../common/light_model.glsl" ) );
    glAttachShader( program[PHONG_TESS], LoadShader( GL_FRAGMENT_SHADER, "../../common/lambert_light.glsl" ) );
    glAttachShader( program[PHONG_TESS], LoadShader( GL_FRAGMENT_SHADER, "../../common/blinn_phong_light.glsl" ) );
    glAttachShader( program[PHONG_TESS], LoadShader( GL_FRAGMENT_SHADER, "../../common/phong_light.glsl" ) );
    glAttachShader( program[PHONG_TESS], LoadShader( GL_FRAGMENT_SHADER, "../../common/materials.glsl" ) );
    glAttachShader( program[PHONG_TESS], LoadShader( GL_FRAGMENT_SHADER, "oswietlenie_phonga_teselacja_fs.glsl" ) );
    LinkValidateProgram( program[PHONG_TESS] );

    // wczytanie shaderw i przygotowanie obsugi programu
    program[LIGHT_VECTOR] = glCreateProgram();
    glAttachShader( program[LIGHT_VECTOR], LoadShader( GL_VERTEX_SHADER, "wektor_swiatla_vs.glsl" ) );
    glAttachShader( program[LIGHT_VECTOR], LoadShader( GL_FRAGMENT_SHADER, "wektor_swiatla_fs.glsl" ) );
    LinkValidateProgram( program[LIGHT_VECTOR] );

    // generowanie identyfikatorw obiektw tablic wierzchokw
    glGenVertexArrays( VERTEX_ARRAY_SIZE, vertexArray );

    // utworzenie pierwszego obiektu tablic wierzchokw
    glBindVertexArray( vertexArray[TEAPOT] );

    // utworzenie obiektu bufora wierzchokw (VBO) i zaadowanie danych
    glGenBuffers( 1, &vertexBuffer[POSITION_TEAPOT] );
    glBindBuffer( GL_ARRAY_BUFFER, vertexBuffer[POSITION_TEAPOT] );
    glBufferData( GL_ARRAY_BUFFER, sizeof( teapotPosition ), teapotPosition, GL_STATIC_DRAW );
    glVertexAttribPointer( POSITION, 3, GL_FLOAT, GL_FALSE, 0, NULL );

    // utworzenie obiektu bufora wierzchokw (VBO)
    glGenBuffers( 1, &vertexBuffer[NORMAL_TEAPOT] );
    glBindBuffer( GL_ARRAY_BUFFER, vertexBuffer[NORMAL_TEAPOT] );
    glBufferData( GL_ARRAY_BUFFER, sizeof( teapotNormal ), teapotNormal, GL_STATIC_DRAW );
    glVertexAttribPointer( NORMAL, 3, GL_FLOAT, GL_FALSE, 0, NULL );

    // wczenie tablic wierzchokw
    glEnableVertexAttribArray( POSITION );
    glEnableVertexAttribArray( NORMAL );

    // utworzenie obiektu bufora indeksw wierzchokw i zaadowanie danych
    glGenBuffers( 1, &indicesBuffer[TEAPOT] );
    glBindBuffer( GL_ELEMENT_ARRAY_BUFFER, indicesBuffer[TEAPOT] );
    glBufferData( GL_ELEMENT_ARRAY_BUFFER, sizeof( teapotIndices ), teapotIndices, GL_STATIC_DRAW );

    // wyczenie obiektu tablic wierzchokw
    glBindVertexArray( 0 );

    // utworzenie drugiego obiektu tablic wierzchokw
    glBindVertexArray( vertexArray[TEAPOT_LOW] );

    // utworzenie obiektu bufora wierzchokw (VBO) i zaadowanie danych
    glGenBuffers( 1, &vertexBuffer[POSITION_TEAPOT_LOW] );
    glBindBuffer( GL_ARRAY_BUFFER, vertexBuffer[POSITION_TEAPOT_LOW] );
    glBufferData( GL_ARRAY_BUFFER, sizeof( teapotLowPosition ), teapotLowPosition, GL_STATIC_DRAW );
    glVertexAttribPointer( POSITION, 3, GL_FLOAT, GL_FALSE, 0, NULL );

    // utworzenie obiektu bufora wierzchokw (VBO) i zaadowanie danych
    glGenBuffers( 1, &vertexBuffer[NORMAL_TEAPOT_LOW] );
    glBindBuffer( GL_ARRAY_BUFFER, vertexBuffer[NORMAL_TEAPOT_LOW] );
    glBufferData( GL_ARRAY_BUFFER, sizeof( teapotLowNormal ), teapotLowNormal, GL_STATIC_DRAW );
    glVertexAttribPointer( NORMAL, 3, GL_FLOAT, GL_FALSE, 0, NULL );

    // wczenie tablic wierzchokw
    glEnableVertexAttribArray( POSITION );
    glEnableVertexAttribArray( NORMAL );

    // utworzenie obiektu bufora indeksw wierzchokw i zaadowanie danych
    glGenBuffers( 1, &indicesBuffer[TEAPOT_LOW] );
    glBindBuffer( GL_ELEMENT_ARRAY_BUFFER, indicesBuffer[TEAPOT_LOW] );
    glBufferData( GL_ELEMENT_ARRAY_BUFFER, sizeof( teapotLowIndices ), teapotLowIndices, GL_STATIC_DRAW );

    // wyczenie obiektu tablic wierzchokw
    glBindVertexArray( 0 );

    // utworzenie trzeciego obiektu tablic wierzchokw
    glBindVertexArray( vertexArray[TEAPOT_HIGH] );

    // utworzenie obiektu bufora wierzchokw (VBO) i zaadowanie danych
    glGenBuffers( 1, &vertexBuffer[POSITION_TEAPOT_HIGH] );
    glBindBuffer( GL_ARRAY_BUFFER, vertexBuffer[POSITION_TEAPOT_HIGH] );
    glBufferData( GL_ARRAY_BUFFER, sizeof( teapotHighPosition ), teapotHighPosition, GL_STATIC_DRAW );
    glVertexAttribPointer( POSITION, 3, GL_FLOAT, GL_FALSE, 0, NULL );

    // utworzenie obiektu bufora wierzchokw (VBO) i zaadowanie danych
    glGenBuffers( 1, &vertexBuffer[NORMAL_TEAPOT_HIGH] );
    glBindBuffer( GL_ARRAY_BUFFER, vertexBuffer[NORMAL_TEAPOT_HIGH] );
    glBufferData( GL_ARRAY_BUFFER, sizeof( teapotHighNormal ), teapotHighNormal, GL_STATIC_DRAW );
    glVertexAttribPointer( NORMAL, 3, GL_FLOAT, GL_FALSE, 0, NULL );

    // wczenie tablic wierzchokw
    glEnableVertexAttribArray( POSITION );
    glEnableVertexAttribArray( NORMAL );

    // utworzenie obiektu bufora indeksw wierzchokw i zaadowanie danych
    glGenBuffers( 1, &indicesBuffer[TEAPOT_HIGH] );
    glBindBuffer( GL_ELEMENT_ARRAY_BUFFER, indicesBuffer[TEAPOT_HIGH] );
    glBufferData( GL_ELEMENT_ARRAY_BUFFER, sizeof( teapotHighIndices ), teapotHighIndices, GL_STATIC_DRAW );

    // wyczenie obiektu tablic wierzchokw
    glBindVertexArray( 0 );

    // utworzenie czwartego obiektu tablic wierzchokw
    glBindVertexArray( vertexArray[LIGHT] );

    // utworzenie obiektu bufora wierzchokw (VBO) i zaadowanie danych
    GLfloat lightPosition[3*2] =
    {
        0.0f, 0.0f, 0.0f,
        0.0f, 0.0f, 5.0f,
    };
    glGenBuffers( 1, &vertexBuffer[POSITION_LIGHT] );
    glBindBuffer( GL_ARRAY_BUFFER, vertexBuffer[POSITION_LIGHT] );
    glBufferData( GL_ARRAY_BUFFER, sizeof( lightPosition ), lightPosition, GL_STATIC_DRAW );
    glVertexAttribPointer( POSITION, 3, GL_FLOAT, GL_FALSE, 0, NULL );

    // wczenie tablic wierzchokw
    glEnableVertexAttribArray( POSITION );

    // wyczenie obiektu tablic wierzchokw
    glBindVertexArray( 0 );

    // pobranie indeksw blokw jednorodnych
    GLuint index[UBO_SIZE];
    index[MATERIALS] = glGetUniformBlockIndex( program[FLAT_TESS], "Materials" );
    index[LIGHT_MODEL] = glGetUniformBlockIndex( program[FLAT_TESS], "LightModel" );

    // pobranie rozmiarw blokw jednorodnych
    glGetActiveUniformBlockiv( program[FLAT_TESS], index[MATERIALS], GL_UNIFORM_BLOCK_DATA_SIZE, &size[MATERIALS] );
    glGetActiveUniformBlockiv( program[FLAT_TESS], index[LIGHT_MODEL], GL_UNIFORM_BLOCK_DATA_SIZE, &size[LIGHT_MODEL] );

    // okrelenie rozmiaru blokw z uwzgldnieniem wyrwnywania
    GLint alignment;
    glGetIntegerv( GL_UNIFORM_BUFFER_OFFSET_ALIGNMENT, &alignment );
    size[MATERIALS] += alignment - size[MATERIALS] % alignment;
    size[LIGHT_MODEL] += alignment - size[LIGHT_MODEL] % alignment;

    // okrelenie przesuni blokw zmiennych jednorodnych,
    // ktre zale od numeru indeksu bloku
    offset[MATERIALS] = index[MATERIALS] > index[LIGHT_MODEL] ? size[LIGHT_MODEL] : 0;
    offset[LIGHT_MODEL] = index[LIGHT_MODEL] > index[MATERIALS] ? size[MATERIALS] : 0;

    // utworzenie obiektu bufora bloku jednorodnego (UBO) bez definiowania danych
    glGenBuffers( 1, &uniformBuffer );
    glBindBuffer( GL_UNIFORM_BUFFER, uniformBuffer );
    glBufferData( GL_UNIFORM_BUFFER, size[MATERIALS] + size[LIGHT_MODEL], NULL, GL_DYNAMIC_DRAW );

    // powizanie bloku jednorodnego do punktw wizania
    glUniformBlockBinding( program[FLAT_TESS], index[MATERIALS], MATERIALS );
    glUniformBlockBinding( program[PN_TRINAGLES_TESS], index[MATERIALS], MATERIALS );
    glUniformBlockBinding( program[PHONG_TESS], index[MATERIALS], MATERIALS );
    glBindBufferRange( GL_UNIFORM_BUFFER, MATERIALS, uniformBuffer, offset[MATERIALS], size[MATERIALS] );

    // zaadowanie danych bloku jednorodnego
    GLintptr offsetBuf = offset[MATERIALS];
    const GLsizeiptr sizeMat = 4 * sizeof( GLfloat );
    for( unsigned int material = MTL_DEFAULT; material <= MTL_YELLOW_RUBBER; material++ )
    {
        glBufferSubData( GL_UNIFORM_BUFFER, offsetBuf + 0 * sizeMat, sizeMat, GetMaterial( material, MTL_AMBIENT ) );
        glBufferSubData( GL_UNIFORM_BUFFER, offsetBuf + 1 * sizeMat, sizeMat, GetMaterial( material, MTL_DIFFUSE ) );
        glBufferSubData( GL_UNIFORM_BUFFER, offsetBuf + 2 * sizeMat, sizeMat, GetMaterial( material, MTL_SPECULAR ) );
        glBufferSubData( GL_UNIFORM_BUFFER, offsetBuf + 3 * sizeMat, sizeMat/4, GetMaterial( material, MTL_SHININESS ) );
        offsetBuf += 4 * sizeMat;
    }

    // powizanie bloku jednorodnego do punktw wizania
    glUniformBlockBinding( program[FLAT_TESS], index[LIGHT_MODEL], LIGHT_MODEL );
    glUniformBlockBinding( program[PN_TRINAGLES_TESS], index[LIGHT_MODEL], LIGHT_MODEL );
    glUniformBlockBinding( program[PHONG_TESS], index[LIGHT_MODEL], LIGHT_MODEL );
    glBindBufferRange( GL_UNIFORM_BUFFER, LIGHT_MODEL, uniformBuffer, offset[LIGHT_MODEL], size[LIGHT_MODEL] );

    // pobranie indeksw i wartoci przesuni wybranych zmiennych jednorodnych z bloku jednorodnego
    GLuint uniformsIndex[UNIFORMS_SIZE];
    glGetUniformIndices( program[FLAT_TESS], UNIFORMS_SIZE, uniformsName, uniformsIndex );
    glGetActiveUniformsiv( program[FLAT_TESS], UNIFORMS_SIZE, uniformsIndex, GL_UNIFORM_OFFSET, uniformsOffset );

    // modyfikacja przesuni
    for( int i = 0; i < UNIFORMS_SIZE; i++ )
        uniformsOffset[i] += offset[LIGHT_MODEL];

    // domylne parametry rda wiata nr 0
    const GLfloat lightSource0ambient[4] = { 0.0f, 0.0f, 0.0f, 1.0f };
    const GLfloat lightSource0diffuse[4] = { 1.0f, 1.0f, 1.0f, 1.0f };
    const GLfloat lightSource0specular[4] = { 1.0f, 1.0f, 1.0f, 1.0f };
    const GLfloat lightSource0position[4] = { 0.0f, 0.0f, 1.0f, 0.0f };
    const GLfloat lightSource0constantAttenuation = 1.0f;
    const GLfloat lightSource0linearAttenuation = 0.0f;
    const GLfloat lightSource0quadraticAttenuation = 0.0f;

    // zaadowanie domylnych parametrw rda wiata nr 0 do LigthSource[0]
    glBufferSubData( GL_UNIFORM_BUFFER, uniformsOffset[LIGHT_SOURCE_0_AMBIENT], 16, lightSource0ambient );
    glBufferSubData( GL_UNIFORM_BUFFER, uniformsOffset[LIGHT_SOURCE_0_DIFFUSE], 16, lightSource0diffuse );
    glBufferSubData( GL_UNIFORM_BUFFER, uniformsOffset[LIGHT_SOURCE_0_SPECULAR], 16, lightSource0specular );
    glBufferSubData( GL_UNIFORM_BUFFER, uniformsOffset[LIGHT_SOURCE_0_POSITION], 16, lightSource0position );
    glBufferSubData( GL_UNIFORM_BUFFER, uniformsOffset[LIGHT_SOURCE_0_CONSTANT_ATTENUATION], 4, &lightSource0constantAttenuation );
    glBufferSubData( GL_UNIFORM_BUFFER, uniformsOffset[LIGHT_SOURCE_0_LINEAR_ATTENUATION], 4, &lightSource0linearAttenuation );
    glBufferSubData( GL_UNIFORM_BUFFER, uniformsOffset[LIGHT_SOURCE_0_QUADRATIC_ATTENUATION], 4, &lightSource0quadraticAttenuation );

    // zaadowanie globalnego wiata otoczenia do lightModel.ambient
    const GLfloat lightModelAmbient[4] = { 0.2f, 0.2f, 0.2f, 1.0f };
    glBufferSubData( GL_UNIFORM_BUFFER, uniformsOffset[LIGHT_MODEL_AMBIENT], 16, lightModelAmbient );

    // zaadowanie wsprzdnych pooenia obserwatora
    const GLfloat eyePosition[4] = { 0.0f, 0.0f, 0.0f, 1.0f };
    glBufferSubData( GL_UNIFORM_BUFFER, uniformsOffset[EYE_POSITION], 16, eyePosition );

    // wyczenie powizania
    glBindBuffer( GL_UNIFORM_BUFFER, 0 );

    // szeroko linii
    glLineWidth( 3.0f );

    // wczenie testu bufora gbokoci
    glEnable( GL_DEPTH_TEST );

    // wielko wejciowego pata wierzchokw
    glPatchParameteri( GL_PATCH_VERTICES, 3 );

    // wczenie mechanizmw uywanych podczas renderingu tekstu
    InitDrawText();
}

//////////////////////////////////////////////////////////////////////
// usunicie obiektw OpenGL
//////////////////////////////////////////////////////////////////////
void DeleteScene()
{
    // porzdki
    for( int i = 0; i < PROGRAM_SIZE; i++ )
    {
        glDeleteProgram( program[i] );
    }
    glDeleteBuffers( VERTEX_BUFFER_SIZE, vertexBuffer );
    glDeleteBuffers( INDICES_BUFFER_SIZE, indicesBuffer );
    glDeleteVertexArrays( VERTEX_ARRAY_SIZE, vertexArray );
    glDeleteBuffers( 1, &uniformBuffer );

    // usunicie mechanizmw uywanych podczas renderingu tekstu
    DeleteDrawText();
}
