//////////////////////////////////////////////////////////////////////
// (c) Janusz Ganczarski
// http://www.januszg.hg.pl
// JanuszG@enter.net.pl
//////////////////////////////////////////////////////////////////////

#include <iostream>
#include <sstream>
#include <glm/glm.hpp>
#include <glm/gtc/matrix_transform.hpp>
#include <glm/gtc/type_ptr.hpp>
#include "shaders.h"
#include "teapot.h"
#include "materials.h"

//////////////////////////////////////////////////////////////////////
// rozmiary bryy obcinania
//////////////////////////////////////////////////////////////////////
GLfloat left = -2.0f;
GLfloat right = 2.0f;
GLfloat bottom = -2.0f;
GLfloat top = 2.0f;
GLfloat near = 3.0f;
GLfloat far = 7.0f;

//////////////////////////////////////////////////////////////////////
// macierz rzutowania
//////////////////////////////////////////////////////////////////////
glm::mat4x4 projectionMatrix;

//////////////////////////////////////////////////////////////////////
// wspczynniki skalowania obiektu
//////////////////////////////////////////////////////////////////////
GLfloat scale = 0.5f;

//////////////////////////////////////////////////////////////////////
// kty obrotu obiektu
//////////////////////////////////////////////////////////////////////
GLfloat rotateX = 0.0f;
GLfloat rotateY = 0.0f;

//////////////////////////////////////////////////////////////////////
// przesunicie obiektu
//////////////////////////////////////////////////////////////////////
GLfloat translateX = 0.0f;
GLfloat translateY = 0.0f;

//////////////////////////////////////////////////////////////////////
// wybrany materia
//////////////////////////////////////////////////////////////////////
int material = MTL_BRASS;

//////////////////////////////////////////////////////////////////////
// numeracja obiektw programu
//////////////////////////////////////////////////////////////////////
enum
{
    OBJECT,         // obiekt - czajnik
    OVERSAMPLING,   // nadprbkowanie - postprocesing
    PROGRAM_SIZE
};

//////////////////////////////////////////////////////////////////////
// identyfikatory obiektw programu
//////////////////////////////////////////////////////////////////////
GLuint program[PROGRAM_SIZE];

//////////////////////////////////////////////////////////////////////
// numeracja obiektw bufora wierzchokw
//////////////////////////////////////////////////////////////////////
enum
{
    POSITION_TEAPOT,
    NORMAL_TEAPOT,
    POSITION_RECT,
    TEXCOORD_RECT,
    VERTEX_BUFFER_SIZE
};

//////////////////////////////////////////////////////////////////////
// identyfikatory obiektw bufora z danymi tablic wierzchokw:
// wsprzdnymi wierzchokw, wektorw normalnych i tekstury
//////////////////////////////////////////////////////////////////////
GLuint vertexBuffer[VERTEX_BUFFER_SIZE];

//////////////////////////////////////////////////////////////////////
// numery indeksw poszczeglnych atrybutw wierzchokw
//////////////////////////////////////////////////////////////////////
enum
{
    POSITION,
    NORMAL,
    TEX_COORD
};

//////////////////////////////////////////////////////////////////////
// identyfikator obiektu bufora z danymi tablicy
// indeksw wierzchokw obiektu
//////////////////////////////////////////////////////////////////////
GLuint indicesBuffer;

//////////////////////////////////////////////////////////////////////
// numeracja obiektw tablic wierzchokw
//////////////////////////////////////////////////////////////////////
enum
{
    TEAPOT,
    RECTANGLE,
    VERTEX_ARRAY_SIZE
};

//////////////////////////////////////////////////////////////////////
// identyfikatory obiektw tablic wierzchokw
//////////////////////////////////////////////////////////////////////
GLuint vertexArray[VERTEX_ARRAY_SIZE];

//////////////////////////////////////////////////////////////////////
// wsprzdne wierzchokw trjktw skadajcych si na kwadrat
//////////////////////////////////////////////////////////////////////
GLfloat positionRect [4*2] =
{
    -1.0f, -1.0f,
    1.0f, -1.0f,
    -1.0f, 1.0f,
    1.0f, 1.0f
};

//////////////////////////////////////////////////////////////////////
// wsprzdne tekstury w wierzchokach trjktw
// skadajcych si na kwadrat
//////////////////////////////////////////////////////////////////////
GLfloat texCoordRect [4*2] =
{
    0.0f, 0.0f,
    1.0f, 0.0f,
    0.0f, 1.0f,
    1.0f, 1.0f
};

//////////////////////////////////////////////////////////////////////
// identyfikator obiektu bufora ramki
//////////////////////////////////////////////////////////////////////
GLuint frameBuffer;

//////////////////////////////////////////////////////////////////////
// rozmiary obszarw renderingu do tekstury
//////////////////////////////////////////////////////////////////////
const int FRAME_WIDTH = 4*256;
const int FRAME_HEIGHT = 4*256;

//////////////////////////////////////////////////////////////////////
// numeracja obiektw tekstury
//////////////////////////////////////////////////////////////////////
enum
{
    COLOR0,
    DEPTH,
    RENDER_TEXTURE_SIZE
};

//////////////////////////////////////////////////////////////////////
// identyfikatory obiektw tekstury - bufor koloru i gbokoci
//////////////////////////////////////////////////////////////////////
GLuint renderTexture[RENDER_TEXTURE_SIZE];

//////////////////////////////////////////////////////////////////////
// funkcja generujca scen 3D
//////////////////////////////////////////////////////////////////////
void DisplayScene()
{
    // bufor ramki do zapisu = obiekt bufora ramki
    glBindFramebuffer( GL_DRAW_FRAMEBUFFER, frameBuffer );

    // czyszczenie bufora koloru i bufora gbokoci
    glClear( GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT );

    // macierz modelu-widoku = macierz jednostkowa
    glm::mat4x4 modelViewMatrix = glm::mat4x4( 1.0 );

    // przesunicie ukadu wsprzdnych obiektu do rodka bryy obcinania
    modelViewMatrix = glm::translate( modelViewMatrix, glm::vec3( 0.0f, 0.0f, -(near+far)/2.0f ) );

    // skalowanie obiektu
    modelViewMatrix = glm::scale( modelViewMatrix, glm::vec3( scale, scale, scale ) );

    // przesunicie obiektu
    modelViewMatrix = glm::translate( modelViewMatrix, glm::vec3( translateX, translateY, 0.0f ) );

    // obroty obiektu
    modelViewMatrix = glm::rotate( modelViewMatrix, rotateX, glm::vec3( 1.0f, 0.0f, 0.0f ) );
    modelViewMatrix = glm::rotate( modelViewMatrix, rotateY, glm::vec3( 0.0f, 1.0f, 0.0f ) );

    // odwrcona macierz modelu-widoku niezbdna do przeksztace
    // do ukadu wsprzdnych obiektu
    glm::mat4x4 modelViewMatrixInverse( glm::inverse( modelViewMatrix ) );

    // transformacja kierunku wiata do ukadu wsprzdnych obiektu
    glm::vec4 lightPosition( 0.0f, 0.0f, 1.0f, 0.0f );
    lightPosition = modelViewMatrixInverse * lightPosition;
    lightPosition = glm::normalize( lightPosition );

    // przeksztacenie pooenia obserwatora do ukadu wsprzdnych obiektu
    glm::vec4 eyePosition( 0.0f, 0.0f, 0.0f, 1.0f );
    eyePosition = modelViewMatrixInverse * eyePosition;

    // wczenie obiektu tablic wierzchokw
    glBindVertexArray( vertexArray[TEAPOT] );

    // wczenie programu
    glUseProgram( program[OBJECT] );

    // zaadowanie zmiennej jednorodnej - iloczynu macierzy modelu-widoku i rzutowania
    glm::mat4x4 modelViewProjectionMatrix = projectionMatrix * modelViewMatrix;
    glUniformMatrix4fv( glGetUniformLocation( program[OBJECT], "modelViewProjectionMatrix" ), 1, GL_FALSE, glm::value_ptr( modelViewProjectionMatrix ) );

    // zaadowanie zmiennej jednorodnej - rodzaj materiau
    glUniform1i( glGetUniformLocation( program[OBJECT], "material" ), material );

    // zaadowanie kierunku rda wiata i pooenia obserwatora w ukadzie wsprzdnych obiektu
    glUniform4fv( glGetUniformLocation( program[OBJECT], "lightSource[0].position" ), 1, glm::value_ptr( lightPosition ) );
    glUniform4fv( glGetUniformLocation( program[OBJECT], "eyePosition" ), 1, glm::value_ptr( eyePosition ) );

    // narysowanie danych zawartych w tablicach wierzchokw
    glDrawElements( GL_TRIANGLES, TEAPOT_INDICES_COUNT * 3, GL_UNSIGNED_INT, NULL );

    // wyczenie programu
    glUseProgram( 0 );

    // wyczenie obiektu tablic wierzchokw
    glBindVertexArray( 0 );

    // bufor ramki do zapisu = domylny bufor ramki
    glBindFramebuffer( GL_DRAW_FRAMEBUFFER, 0 );

    // czyszczenie bufora koloru
    glClear( GL_COLOR_BUFFER_BIT );

    // wyczenie testu bufora gbokoci
    glDisable( GL_DEPTH_TEST );

    // wybr tekstury 2D
    glBindTexture( GL_TEXTURE_2D, renderTexture[COLOR0] );

    // wczenie obiektu tablic wierzchokw
    glBindVertexArray( vertexArray[RECTANGLE] );

    // wczenie programu
    glUseProgram( program[OVERSAMPLING] );

    // zaadowanie zmiennej jednorodnej - uchwyt tekstury
    glUniform1i( glGetUniformLocation( program[OVERSAMPLING], "tex" ), 0 );

    // narysowanie danych zawartych w tablicach wierzchokw
    glDrawArrays( GL_TRIANGLE_STRIP, 0, 4 );

    // wyczenie programu
    glUseProgram( 0 );

    // wyczenie obiektu tablic wierzchokw
    glBindVertexArray( 0 );

    // wyczenie tekstury 2D
    glBindTexture( GL_TEXTURE_2D, 0 );

    // wczenie testu bufora gbokoci
    glEnable( GL_DEPTH_TEST );
}

//////////////////////////////////////////////////////////////////////
// zmiana wielkoci okna
//////////////////////////////////////////////////////////////////////
void Reshape( int width, int height )
{
    // obszar renderingu - cae okno
    glViewportIndexedf( 0, 0.0, 0.0, static_cast<GLfloat>( width ), static_cast<GLfloat>( height ) );

    // parametry bryy obcinania - rzutowanie perspektywiczne
    // wysoko okna wiksza od szerokoci okna
    if( width < height && width > 0 )
         projectionMatrix = glm::frustum( left, right, bottom*height/width, top*height/width, near, far );
    else
        // szeroko okna wiksza lub rwna wysokoci okna
        if (width >= height && height > 0)
            projectionMatrix = glm::frustum( left*width/height, right*width/height, bottom, top, near, far );
        else
            projectionMatrix = glm::frustum( left, right, bottom, top, near, far );
}

//////////////////////////////////////////////////////////////////////
// inicjalizacja staych elementw maszyny stanu OpenGL
//////////////////////////////////////////////////////////////////////
void InitScene()
{
    // kolor ta - zawarto bufora koloru
    glClearColor( 1.0f, 1.0f, 1.0f, 1.0f );

    // utworzenie obiektw tekstur - na dane bufora koloru i bufora gbokoci
    glGenTextures( RENDER_TEXTURE_SIZE, renderTexture );
    glBindTexture( GL_TEXTURE_2D, renderTexture[COLOR0] );
    glTexImage2D( GL_TEXTURE_2D, 0, GL_RGBA, FRAME_WIDTH, FRAME_HEIGHT, 0, GL_RGBA, GL_UNSIGNED_BYTE, NULL );
    glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR );
    glBindTexture( GL_TEXTURE_2D, renderTexture[DEPTH] );
    glTexImage2D( GL_TEXTURE_2D, 0, GL_DEPTH_COMPONENT, FRAME_WIDTH, FRAME_HEIGHT, 0, GL_DEPTH_COMPONENT, GL_FLOAT, NULL );
    glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR );

    // utworzenie obiektu bufora ramki i doczenie obiektw tekstury
    glGenFramebuffers( 1, &frameBuffer );
    glBindFramebuffer( GL_DRAW_FRAMEBUFFER, frameBuffer );
    glFramebufferTexture( GL_DRAW_FRAMEBUFFER, GL_COLOR_ATTACHMENT0, renderTexture[COLOR0], 0 );
    glFramebufferTexture( GL_DRAW_FRAMEBUFFER, GL_DEPTH_ATTACHMENT, renderTexture[DEPTH], 0 );

    // sprawdzenie poprawnoci i kompletnoci obiektu bufora ramki
    GLenum error = glCheckFramebufferStatus( GL_DRAW_FRAMEBUFFER );
    if( error != GL_FRAMEBUFFER_COMPLETE )
    {
        std::cout << "Niepoprawny obiekt bufora renderingu" << std::endl;
        exit( 0 );
    }

    // obszar renderingu - caa powierzchnia bufora ramki
    glViewportIndexedf( 1, 0.0, 0.0, static_cast<GLfloat>( FRAME_WIDTH ), static_cast<GLfloat>( FRAME_HEIGHT ) );

    // wczytanie shaderw i przygotowanie obsugi programu
    program[OBJECT] = glCreateProgram();
    glAttachShader( program[OBJECT], LoadShader( GL_VERTEX_SHADER, "obiekt_vs.glsl" ) );
    glAttachShader( program[OBJECT], LoadShader( GL_GEOMETRY_SHADER, "obiekt_gs.glsl" ) );
    glAttachShader( program[OBJECT], LoadShader( GL_FRAGMENT_SHADER, "../../common/light_model_static.glsl" ) );
    glAttachShader( program[OBJECT], LoadShader( GL_FRAGMENT_SHADER, "../../common/materials_static.glsl" ) );
    glAttachShader( program[OBJECT], LoadShader( GL_FRAGMENT_SHADER, "../../common/blinn_phong_light.glsl" ) );
    glAttachShader( program[OBJECT], LoadShader( GL_FRAGMENT_SHADER, "obiekt_fs.glsl" ) );
    LinkValidateProgram( program[OBJECT] );

    // wczytanie shaderw i przygotowanie obsugi programu
    program[OVERSAMPLING] = glCreateProgram();
    glAttachShader( program[OVERSAMPLING], LoadShader( GL_VERTEX_SHADER, "nadprobkowanie_vs.glsl" ) );
    glAttachShader( program[OVERSAMPLING], LoadShader( GL_FRAGMENT_SHADER, "../../common/convolution_filters_3x3.glsl" ) );
    glAttachShader( program[OVERSAMPLING], LoadShader( GL_FRAGMENT_SHADER, "nadprobkowanie_fs.glsl" ) );
    LinkValidateProgram( program[OVERSAMPLING] );

    // generowanie identyfikatora obiektu tablic wierzchokw
    glGenVertexArrays( 1, &vertexArray[TEAPOT] );

    // utworzenie pierwszego obiektu tablic wierzchokw
    glBindVertexArray( vertexArray[TEAPOT] );

    // utworzenie obiektu bufora wierzchokw (VBO) i zaadowanie danych
    glGenBuffers( 1, &vertexBuffer[POSITION_TEAPOT] );
    glBindBuffer( GL_ARRAY_BUFFER, vertexBuffer[POSITION_TEAPOT] );
    glBufferData( GL_ARRAY_BUFFER, sizeof( teapotPosition ), teapotPosition, GL_STATIC_DRAW );
    glVertexAttribPointer( POSITION, 3, GL_FLOAT, GL_FALSE, 0, NULL );

    // utworzenie obiektu bufora wierzchokw (VBO) i zaadowanie danych
    glGenBuffers( 1, &vertexBuffer[NORMAL_TEAPOT] );
    glBindBuffer( GL_ARRAY_BUFFER, vertexBuffer[NORMAL_TEAPOT] );
    glBufferData( GL_ARRAY_BUFFER, sizeof( teapotNormal ), teapotNormal, GL_STATIC_DRAW );
    glVertexAttribPointer( NORMAL, 3, GL_FLOAT, GL_FALSE, 0, NULL );

    // wczenie tablic wierzchokw
    glEnableVertexAttribArray( POSITION );
    glEnableVertexAttribArray( NORMAL );

    // utworzenie obiektu bufora indeksw wierzchokw i zaadowanie danych
    glGenBuffers( 1, &indicesBuffer );
    glBindBuffer( GL_ELEMENT_ARRAY_BUFFER, indicesBuffer );
    glBufferData( GL_ELEMENT_ARRAY_BUFFER, sizeof( teapotIndices ), teapotIndices, GL_STATIC_DRAW );

    // wyczenie obiektu tablic wierzchokw
    glBindVertexArray( 0 );

    // utworzenie obiektu tablic wierzchokw
    glGenVertexArrays( 1, &vertexArray[RECTANGLE] );
    glBindVertexArray( vertexArray[RECTANGLE] );

    // utworzenie obiektu bufora wierzchokw (VBO) i zaadowanie danych
    glGenBuffers( 1, &vertexBuffer[POSITION_RECT] );
    glBindBuffer( GL_ARRAY_BUFFER, vertexBuffer[POSITION_RECT] );
    glBufferData( GL_ARRAY_BUFFER, sizeof( positionRect ), positionRect, GL_STATIC_DRAW );
    glVertexAttribPointer( POSITION, 2, GL_FLOAT, GL_FALSE, 0, NULL );

    // utworzenie obiektu bufora wierzchokw (VBO) i zaadowanie danych
    glGenBuffers( 1, &vertexBuffer[TEXCOORD_RECT] );
    glBindBuffer( GL_ARRAY_BUFFER, vertexBuffer[TEXCOORD_RECT] );
    glBufferData( GL_ARRAY_BUFFER, sizeof( texCoordRect ), texCoordRect, GL_STATIC_DRAW );
    glVertexAttribPointer( TEX_COORD, 2, GL_FLOAT, GL_FALSE, 0, NULL );

    // wczenie tablic wierzchokw
    glEnableVertexAttribArray( POSITION );
    glEnableVertexAttribArray( TEX_COORD );

    // wyczenie obiektu tablic wierzchokw
    glBindVertexArray( 0 );

    // wczenie testu bufora gbokoci
    glEnable( GL_DEPTH_TEST );
}

//////////////////////////////////////////////////////////////////////
// usunicie obiektw OpenGL
//////////////////////////////////////////////////////////////////////
void DeleteScene()
{
    // porzdki
    glDeleteProgram( program[OBJECT] );
    glDeleteProgram( program[OVERSAMPLING] );
    glDeleteBuffers( VERTEX_BUFFER_SIZE, vertexBuffer );
    glDeleteBuffers( 1, &indicesBuffer );
    glDeleteVertexArrays( VERTEX_ARRAY_SIZE, vertexArray );
    glDeleteTextures( RENDER_TEXTURE_SIZE, renderTexture );
    glDeleteFramebuffers( 1, &frameBuffer );
}
