//////////////////////////////////////////////////////////////////////
// (c) Janusz Ganczarski
// http://www.januszg.hg.pl
// JanuszG@enter.net.pl
//////////////////////////////////////////////////////////////////////

#include <string>
#include <sstream>
#include <iomanip>
#include <ctime>
#include <glm/glm.hpp>
#include <glm/gtc/matrix_transform.hpp>
#include <glm/gtc/type_ptr.hpp>
#include "shaders.h"
#include "teapot_high.h"
#include "text.h"

//////////////////////////////////////////////////////////////////////
// rozmiary bryy obcinania
//////////////////////////////////////////////////////////////////////
GLfloat left = -2.0f;
GLfloat right = 2.0f;
GLfloat bottom = -2.0f;
GLfloat top = 2.0f;
GLfloat near = 3.0f;
GLfloat far = 7.0f;

//////////////////////////////////////////////////////////////////////
// macierz rzutowania
//////////////////////////////////////////////////////////////////////
glm::mat4x4 projectionMatrix;

//////////////////////////////////////////////////////////////////////
// wspczynniki skalowania obiektu
//////////////////////////////////////////////////////////////////////
GLfloat scale = 0.35f;

//////////////////////////////////////////////////////////////////////
// kty obrotu obiektu
//////////////////////////////////////////////////////////////////////
GLfloat rotateX = 0.0f;
GLfloat rotateY = 0.0f;

//////////////////////////////////////////////////////////////////////
// przesunicie obiektu
//////////////////////////////////////////////////////////////////////
GLfloat translateX = 0.0f;
GLfloat translateY = 0.0f;

//////////////////////////////////////////////////////////////////////
// numeracja obiektw programu
//////////////////////////////////////////////////////////////////////
enum
{
    OBJECT,         // obiekty sceny
    BOUNDING_BOX,   // brya otaczajca
    PROGRAM_SIZE
};

//////////////////////////////////////////////////////////////////////
// identyfikatory obiektw programu
//////////////////////////////////////////////////////////////////////
GLuint program[PROGRAM_SIZE];

//////////////////////////////////////////////////////////////////////
// numeracja obiektw bufora wierzchokw
//////////////////////////////////////////////////////////////////////
enum
{
    POSITION_TEAPOT,
    NORMAL_TEAPOT,
    POSITION_WALL,
    NORMAL_WALL,
    POSITION_BOUNDING_BOX,
    VERTEX_BUFFER_SIZE
};

//////////////////////////////////////////////////////////////////////
// identyfikatory obiektw bufora z danymi tablic wierzchokw
//////////////////////////////////////////////////////////////////////
GLuint vertexBuffer[VERTEX_BUFFER_SIZE];

//////////////////////////////////////////////////////////////////////
// numeracja obiektw bufora indeksw wierzchokw
//////////////////////////////////////////////////////////////////////
enum
{
    INDICES_TEAPOT,
    INDICES_BOUNDING_BOX,
    INDICES_BUFFER_SIZE
};

//////////////////////////////////////////////////////////////////////
// identyfikatory obiektw bufora z danymi tablic indeksw wierzchokw
//////////////////////////////////////////////////////////////////////
GLuint indicesBuffer[INDICES_BUFFER_SIZE];

//////////////////////////////////////////////////////////////////////
// numeracja obiektw tablic wierzchokw
//////////////////////////////////////////////////////////////////////
enum
{
    TEAPOT,
    // BOUNDING_BOX
    WALL = 2,
    VERTEX_ARRAY_SIZE
};

//////////////////////////////////////////////////////////////////////
// identyfikatory obiektw tablic wierzchokw
//////////////////////////////////////////////////////////////////////
GLuint vertexArray[VERTEX_ARRAY_SIZE];

//////////////////////////////////////////////////////////////////////
// wsprzdne wierzchokw trjktw skadajcych si na mur
//////////////////////////////////////////////////////////////////////
GLfloat positionWall[2*3*3] =
{
    -6.0f, -5.0f, -0.0f,
     6.0f, -5.0f, -0.0f,
     6.0f,  5.0f, -0.0f,

    -6.0f, -5.0f, -0.0f,
     6.0f,  5.0f, -0.0f,
    -6.0f,  5.0f, -0.0f
};

//////////////////////////////////////////////////////////////////////
// wsprzdne wektorw normalnych trjktw skadajcych si na mur
//////////////////////////////////////////////////////////////////////
GLfloat normalWall[2*3*3] =
{
    0.0f, 0.0f, 1.0f,
    0.0f, 0.0f, 1.0f,
    0.0f, 0.0f, 1.0f,

    0.0f, 0.0f, 1.0f,
    0.0f, 0.0f, 1.0f,
    0.0f, 0.0f, 1.0f
};

//////////////////////////////////////////////////////////////////////
// wsprzdne wierzchokw trjktw skadajcych si na szecian
//////////////////////////////////////////////////////////////////////
GLfloat positionCube[8*3] =
{
    3.5f, 2.0f, 2.2f,
    -3.5f, 2.0f, 2.2f,
    -3.5f, -2.0f, 2.2f,
    3.5f, -2.0f, 2.2f,
    3.5f, 2.0f, -2.2f,
    -3.5f, 2.0f, -2.2f,
    -3.5f, -2.0f, -2.2f,
    3.5f, -2.0f, -2.2f
};

//////////////////////////////////////////////////////////////////////
// dane indeksw wierzchokw trjktw skadajcych si na szecian
//////////////////////////////////////////////////////////////////////
GLuint indicesCube[12*3] =
{
    5, 1, 0,
    5, 0, 4,
    2, 3, 0,
    2, 0, 1,
    7, 4, 0,
    7, 0, 3,
    3, 2, 6,
    3, 6, 7,
    1, 6, 2,
    1, 5, 6,
    4, 6, 5,
    4, 7, 6
};

//////////////////////////////////////////////////////////////////////
// numery indeksw poszczeglnych atrybutw wierzchokw
//////////////////////////////////////////////////////////////////////
enum
{
    POSITION,
    NORMAL
};

//////////////////////////////////////////////////////////////////////
// numeracja obiektw zapytania
//////////////////////////////////////////////////////////////////////
enum
{
    ANY_SAMPLES_BOUNDING_BOX_0,
    ANY_SAMPLES_BOUNDING_BOX_1,
    SAMPLES_BOUNDING_BOX_0,
    SAMPLES_BOUNDING_BOX_1,
    ANY_SAMPLES_CONSERVATIVE_BOUNDING_BOX_0,
    ANY_SAMPLES_CONSERVATIVE_BOUNDING_BOX_1,
    TIMER,
    QUERY_SIZE
};

//////////////////////////////////////////////////////////////////////
// identyfikatory obiektw zapytania
//////////////////////////////////////////////////////////////////////
GLuint query[QUERY_SIZE];

//////////////////////////////////////////////////////////////////////
// rodzaj testu zasaniania
//////////////////////////////////////////////////////////////////////
enum
{
    ANY_SAMPLES_PASSED,
    SAMPLES_PASSED,
    ANY_SAMPLES_PASSED_CONSERVATIVE,
    NONE
};
GLuint occlusionTest = ANY_SAMPLES_PASSED;
GLuint occlusionTestID[3] = { GL_ANY_SAMPLES_PASSED, GL_SAMPLES_PASSED, GL_ANY_SAMPLES_PASSED_CONSERVATIVE };

//////////////////////////////////////////////////////////////////////
// licznik ramek
//////////////////////////////////////////////////////////////////////
int frames = 0;

//////////////////////////////////////////////////////////////////////
// licznik czasu
//////////////////////////////////////////////////////////////////////
GLint64 framesTime = 0;

//////////////////////////////////////////////////////////////////////
// uredniony czas rysowania ramki
//////////////////////////////////////////////////////////////////////
double framesMS = 0.0;

//////////////////////////////////////////////////////////////////////
// rysowanie obiektu sceny
// object - numer obiektu tablic wierzchokw
// zBias - dodatkowe przesunicie obiektu w kierunku osi OZ
//         wzgldem pocztku ukadu wsprzdnych
//////////////////////////////////////////////////////////////////////
void DrawObject( const GLuint object, const GLfloat zBias )
{
    // macierz modelu-widoku = macierz jednostkowa
    glm::mat4x4 modelViewMatrix = glm::mat4x4( 1.0 );

    // przesunicie obserwatora tak, aby ukad wsprzdnych obiektu by w rodku bryy obcinania
    modelViewMatrix = glm::translate( modelViewMatrix, glm::vec3( translateX, translateY, -(near+far)/2.0f ) );

    // skalowanie obiektu
    modelViewMatrix = glm::scale( modelViewMatrix, glm::vec3( scale, scale, scale ) );

    // obroty obiektu
    modelViewMatrix = glm::rotate( modelViewMatrix, rotateX, glm::vec3( 1.0f, 0.0f, 0.0f ) );
    modelViewMatrix = glm::rotate( modelViewMatrix, rotateY, glm::vec3( 0.0f, 1.0f, 0.0f ) );

    // dodatkowe przesunicie obiektu
    modelViewMatrix = glm::translate( modelViewMatrix, glm::vec3( 0.0f, 0.0f, zBias ) );

    // odwrcona macierz modelu-widoku niezbdna do przeksztace
    // do ukadu wsprzdnych obiektu
    glm::mat4x4 modelViewMatrixInverse( glm::inverse( modelViewMatrix ) );

    // transformacja kierunku wiata do ukadu wsprzdnych obiektu
    glm::vec4 lightPosition( 0.0f, 0.0f, 1.0f, 0.0f );
    lightPosition = modelViewMatrixInverse * lightPosition;
    lightPosition = glm::normalize( lightPosition );

    // przeksztacenie pooenia obserwatora do ukadu wsprzdnych obiektu
    glm::vec4 eyePosition( 0.0f, 0.0f, 0.0f, 1.0f );
    eyePosition = modelViewMatrixInverse * eyePosition;

    // wczenie obiektu tablic wierzchokw
    glBindVertexArray( vertexArray[object] );

    // wczenie programu
    glUseProgram( program[OBJECT] );

    // zaadowanie zmiennej jednorodnej - iloczynu macierzy modelu-widoku i rzutowania
    glm::mat4x4 modelViewProjectionMatrix = projectionMatrix * modelViewMatrix;
    glUniformMatrix4fv( glGetUniformLocation( program[OBJECT], "modelViewProjectionMatrix" ), 1, GL_FALSE, glm::value_ptr( modelViewProjectionMatrix ) );

    // zaadowanie kierunku rda wiata i pooenia obserwatora w ukadzie wsprzdnych obiektu
    glUniform4fv( glGetUniformLocation( program[OBJECT], "lightSource[0].position" ), 1, glm::value_ptr( lightPosition ) );
    glUniform4fv( glGetUniformLocation( program[OBJECT], "eyePosition" ), 1, glm::value_ptr( eyePosition ) );

    // narysowanie danych zawartych w tablicach wierzchokw
    if( object == WALL )
        glDrawArrays( GL_TRIANGLES, 0, 6 );
    else
        glDrawElementsInstanced( GL_TRIANGLES, TEAPOT_HIGH_INDICES_COUNT * 3, GL_UNSIGNED_INT, NULL, 20 );

    // wyczenie programu
    glUseProgram( 0 );

    // wyczenie obiektu tablic wierzchokw
    glBindVertexArray( 0 );
}

//////////////////////////////////////////////////////////////////////
// rysowanie bryy ograniczajcej (prostopadocianu) dla obiektu
// objtego testem zasaniania
// zBias - dodatkowe przesunicie obiektu w kierunku osi OZ
//         wzgldem pocztku ukadu wsprzdnych
//////////////////////////////////////////////////////////////////////
void DrawBoundingBox( const GLfloat zBias )
{
    // macierz modelu-widoku = macierz jednostkowa
    glm::mat4x4 modelViewMatrix = glm::mat4x4( 1.0 );

    // przesunicie obserwatora tak, aby ukad wsprzdnych obiektu by w rodku bryy obcinania
    modelViewMatrix = glm::translate( modelViewMatrix, glm::vec3( translateX, translateY, -(near+far)/2.0f ) );

    // skalowanie obiektu
    modelViewMatrix = glm::scale( modelViewMatrix, glm::vec3( scale, scale, scale ) );

    // obroty obiektu
    modelViewMatrix = glm::rotate( modelViewMatrix, rotateX, glm::vec3( 1.0f, 0.0f, 0.0f ) );
    modelViewMatrix = glm::rotate( modelViewMatrix, rotateY, glm::vec3( 0.0f, 1.0f, 0.0f ) );

    // dodatkowe przesunicie obiektu
    modelViewMatrix = glm::translate( modelViewMatrix, glm::vec3( 0.0f, 0.0f, zBias ) );

    // wyczenie zapisu do bufora gbokoci
    glDepthMask( GL_FALSE );

    // wyczenie zapisu skadowych RGBA do bufora kolorw
    glColorMask( GL_FALSE, GL_FALSE, GL_FALSE, GL_FALSE );

    // wczenie obiektu tablic wierzchokw
    glBindVertexArray( vertexArray[BOUNDING_BOX] );

    // wczenie programu
    glUseProgram( program[BOUNDING_BOX] );

    // zaadowanie zmiennej jednorodnej - iloczynu macierzy modelu-widoku i rzutowania
    glm::mat4x4 modelViewProjectionMatrix = projectionMatrix * modelViewMatrix;
    glUniformMatrix4fv( glGetUniformLocation( program[BOUNDING_BOX], "modelViewProjectionMatrix" ), 1, GL_FALSE, glm::value_ptr( modelViewProjectionMatrix ) );

    // narysowanie danych zawartych w tablicach wierzchokw
    glDrawElements( GL_TRIANGLES, 12 * 3, GL_UNSIGNED_INT, NULL );

    // wyczenie programu
    glUseProgram( 0 );

    // wyczenie obiektu tablic wierzchokw
    glBindVertexArray( 0 );

    // wczenie zapisu skadowych RGBA do bufora kolorw
    glColorMask( GL_TRUE, GL_TRUE, GL_TRUE, GL_TRUE );

    // wczenie zapisu do bufora gbokoci
    glDepthMask( GL_TRUE );
}

//////////////////////////////////////////////////////////////////////
// funkcja generujca scen 3D
//////////////////////////////////////////////////////////////////////
void DisplayScene()
{
    // licznik ramek
    frames++;

    // aktywacja zapytania asynchronicznego
    // z licznikiem czasu
    glBeginQuery( GL_TIME_ELAPSED, query[TIMER] );

    // czyszczenie bufora koloru i bufora gbokoci
    glClear( GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT );

    // narysowanie elementw sceny nieobjtych testem zasaniania
    DrawObject( WALL, 0.0f );

    // test zasaniania
    if( occlusionTest != NONE )
    {
        // aktywacja pierwszego zapytania asynchronicznego
        glBeginQuery( occlusionTestID[occlusionTest], query[occlusionTest*2+0] );

        // narysowanie bryy ograniczajcej
        DrawBoundingBox( -3.0f );

        // zakoczenie zapytania asynchronicznego
        glEndQuery( occlusionTestID[occlusionTest] );

        // aktywacja drugiego zapytania asynchronicznego
        glBeginQuery( occlusionTestID[occlusionTest], query[occlusionTest*2+1] );

        // narysowanie bryy ograniczajcej
        DrawBoundingBox( 3.0f );

        // zakoczenie zapytania asynchronicznego
        glEndQuery( occlusionTestID[occlusionTest] );

        // rendering warunkowy pierwszego czajnika
        glBeginConditionalRender( query[occlusionTest*2+0], GL_QUERY_BY_REGION_WAIT );
        DrawObject( TEAPOT, -3.0f );
        glEndConditionalRender();

        // rendering warunkowy drugiego czajnika
        glBeginConditionalRender( query[occlusionTest*2+1], GL_QUERY_BY_REGION_WAIT );
        DrawObject( TEAPOT, 3.0f );
        glEndConditionalRender();
    }
    else
    {
        // bez testu zasaniania
        DrawObject( TEAPOT, -3.0f );
        DrawObject( TEAPOT, 3.0f );
    }

    // zakoczenie zapytania asynchronicznego
    glEndQuery( GL_TIME_ELAPSED );

    // sprawdzenie, czy dostpny jest wynik zapytania
    // asynchronicznego z licznikiem czasu
    GLint available;
    do
    {
        glGetQueryObjectiv( query[TIMER], GL_QUERY_RESULT_AVAILABLE, &available );
    }
    while( !available );

    // pobranie wyniku zapytania asynchronicznego
    GLint64 frameTime;
    glGetQueryObjecti64v( query[TIMER], GL_QUERY_RESULT, &frameTime );

    // zliczanie cznego czasu
    framesTime += frameTime;

    // okresowe zliczanie redniego czasu rysowania sceny
    const double CLOCK = 100;
    if( frames == CLOCK )
    {
        framesMS = (double)framesTime / (1000000.0 * CLOCK);
        frames = 0;
        framesTime = 0;
    }

    // komunikat o czasie rysowanie sceny
    std::ostringstream txt;
    txt << std::setprecision( 4 ) << std::fixed << "czas renderingu sceny: " << framesMS << " ms.";
    DrawText8x16( 3, 20, txt.str() );

    // komunikat o tecie zasaniania
    switch( occlusionTest )
    {
        case ANY_SAMPLES_PASSED:
            DrawText8x16( 3, 3, "test zasaniania: GL_ANY_SAMPLES_PASSED" );
            break;
        case SAMPLES_PASSED:
            DrawText8x16( 3, 3, "test zasaniania: GL_SAMPLES_PASSED" );
            break;
        case ANY_SAMPLES_PASSED_CONSERVATIVE:
            DrawText8x16( 3, 3, "test zasaniania: GL_ANY_SAMPLES_PASSED_CONSERVATIVE" );
            break;
        case NONE:
            DrawText8x16( 3, 3, "test zasaniania: wyczony" );
            break;
    }
}

//////////////////////////////////////////////////////////////////////
// zmiana wielkoci okna
//////////////////////////////////////////////////////////////////////
void Reshape( int width, int height )
{
    // obszar renderingu - cae okno
    glViewport( 0, 0, width, height );

    // parametry bryy obcinania - rzutowanie perspektywiczne
    // wysoko okna wiksza od szerokoci okna
    if( width < height && width > 0 )
         projectionMatrix = glm::frustum( left, right, bottom*height/width, top*height/width, near, far );
    else
        // szeroko okna wiksza lub rwna wysokoci okna
        if (width >= height && height > 0)
            projectionMatrix = glm::frustum( left*width/height, right*width/height, bottom, top, near, far );
        else
            projectionMatrix = glm::frustum( left, right, bottom, top, near, far );
}

//////////////////////////////////////////////////////////////////////
// inicjalizacja staych elementw maszyny stanu OpenGL
//////////////////////////////////////////////////////////////////////
void InitScene()
{
    // kolor ta - zawarto bufora koloru
    glClearColor( 1.0f, 1.0f, 1.0f, 1.0f );

    // wczytanie shaderw i przygotowanie obsugi programu
    program[OBJECT] = glCreateProgram();
    glAttachShader( program[OBJECT], LoadShader( GL_VERTEX_SHADER, "rendering_warunkowy_obiekt_vs.glsl" ) );
    glAttachShader( program[OBJECT], LoadShader( GL_FRAGMENT_SHADER, "../../common/light_model_static.glsl" ) );
    glAttachShader( program[OBJECT], LoadShader( GL_FRAGMENT_SHADER, "../../common/materials_static.glsl" ) );
    glAttachShader( program[OBJECT], LoadShader( GL_FRAGMENT_SHADER, "../../common/blinn_phong_light.glsl" ) );
    glAttachShader( program[OBJECT], LoadShader( GL_FRAGMENT_SHADER, "rendering_warunkowy_obiekt_fs.glsl" ) );
    LinkValidateProgram( program[OBJECT] );

    // wczytanie shaderw i przygotowanie obsugi programu
    program[BOUNDING_BOX] = glCreateProgram();
    glAttachShader( program[BOUNDING_BOX], LoadShader( GL_VERTEX_SHADER, "bryla_otaczajaca_vs.glsl" ) );
    glAttachShader( program[BOUNDING_BOX], LoadShader( GL_FRAGMENT_SHADER, "bryla_otaczajaca_fs.glsl" ) );
    LinkValidateProgram( program[BOUNDING_BOX] );

    // generowanie identyfikatora obiektu tablic wierzchokw
    glGenVertexArrays( 1, &vertexArray[TEAPOT] );

    // utworzenie pierwszego obiektu tablic wierzchokw
    glBindVertexArray( vertexArray[TEAPOT] );

    // utworzenie obiektu bufora wierzchokw (VBO) i zaadowanie danych
    glGenBuffers( 1, &vertexBuffer[POSITION_TEAPOT] );
    glBindBuffer( GL_ARRAY_BUFFER, vertexBuffer[POSITION_TEAPOT] );
    glBufferData( GL_ARRAY_BUFFER, sizeof( teapotHighPosition ), teapotHighPosition, GL_STATIC_DRAW );
    glVertexAttribPointer( POSITION, 3, GL_FLOAT, GL_FALSE, 0, NULL );

    // utworzenie obiektu bufora wierzchokw (VBO) i zaadowanie danych
    glGenBuffers( 1, &vertexBuffer[NORMAL_TEAPOT] );
    glBindBuffer( GL_ARRAY_BUFFER, vertexBuffer[NORMAL_TEAPOT] );
    glBufferData( GL_ARRAY_BUFFER, sizeof( teapotHighNormal ), teapotHighNormal, GL_STATIC_DRAW );
    glVertexAttribPointer( NORMAL, 3, GL_FLOAT, GL_FALSE, 0, NULL );

    // wczenie tablic wierzchokw
    glEnableVertexAttribArray( POSITION );
    glEnableVertexAttribArray( NORMAL );

    // utworzenie obiektu bufora indeksw wierzchokw i zaadowanie danych
    glGenBuffers( 1, &indicesBuffer[INDICES_TEAPOT] );
    glBindBuffer( GL_ELEMENT_ARRAY_BUFFER, indicesBuffer[INDICES_TEAPOT] );
    glBufferData( GL_ELEMENT_ARRAY_BUFFER, sizeof( teapotHighIndices ), teapotHighIndices, GL_STATIC_DRAW );

    // wyczenie obiektu tablic wierzchokw
    glBindVertexArray( 0 );

    // utworzenie obiektu tablic wierzchokw
    glGenVertexArrays( 1, &vertexArray[WALL] );
    glBindVertexArray( vertexArray[WALL] );

    // utworzenie obiektu bufora wierzchokw (VBO) i zaadowanie danych
    glGenBuffers( 1, &vertexBuffer[POSITION_WALL] );
    glBindBuffer( GL_ARRAY_BUFFER, vertexBuffer[POSITION_WALL] );
    glBufferData( GL_ARRAY_BUFFER, sizeof( positionWall ), positionWall, GL_STATIC_DRAW );
    glVertexAttribPointer( POSITION, 3, GL_FLOAT, GL_FALSE, 0, NULL );

    // utworzenie obiektu bufora wierzchokw (VBO) i zaadowanie danych
    glGenBuffers( 1, &vertexBuffer[NORMAL_WALL] );
    glBindBuffer( GL_ARRAY_BUFFER, vertexBuffer[NORMAL_WALL] );
    glBufferData( GL_ARRAY_BUFFER, sizeof( normalWall ), normalWall, GL_STATIC_DRAW );
    glVertexAttribPointer( NORMAL, 3, GL_FLOAT, GL_FALSE, 0, NULL );

    // wczenie tablic wierzchokw
    glEnableVertexAttribArray( POSITION );
    glEnableVertexAttribArray( NORMAL );

    // wyczenie obiektu tablic wierzchokw
    glBindVertexArray( 0 );

    // utworzenie obiektu tablic wierzchokw
    glGenVertexArrays( 1, &vertexArray[BOUNDING_BOX] );
    glBindVertexArray( vertexArray[BOUNDING_BOX] );

    // utworzenie obiektu bufora wierzchokw (VBO) i zaadowanie danych
    glGenBuffers( 1, &vertexBuffer[POSITION_BOUNDING_BOX] );
    glBindBuffer( GL_ARRAY_BUFFER, vertexBuffer[POSITION_BOUNDING_BOX] );
    glBufferData( GL_ARRAY_BUFFER, sizeof( positionCube ), positionCube, GL_STATIC_DRAW );
    glVertexAttribPointer( POSITION, 3, GL_FLOAT, GL_FALSE, 0, NULL );

    // wczenie tablic wierzchokw
    glEnableVertexAttribArray( POSITION );

    // utworzenie obiektu bufora indeksw wierzchokw i zaadowanie danych
    glGenBuffers( 1, &indicesBuffer[INDICES_BOUNDING_BOX] );
    glBindBuffer( GL_ELEMENT_ARRAY_BUFFER, indicesBuffer[INDICES_BOUNDING_BOX] );
    glBufferData( GL_ELEMENT_ARRAY_BUFFER, sizeof( indicesCube ), indicesCube, GL_STATIC_DRAW );

    // wyczenie obiektu tablic wierzchokw
    glBindVertexArray( 0 );

    // generowanie identyfikatorw obiektw zapytania
    glGenQueries( QUERY_SIZE, query );

    // wczenie mechanizmw uywanych podczas renderingu tekstu
    InitDrawText();

    // wczenie testu bufora gbokoci
    glEnable( GL_DEPTH_TEST );
}

//////////////////////////////////////////////////////////////////////
// usunicie obiektw OpenGL
//////////////////////////////////////////////////////////////////////
void DeleteScene()
{
    // porzdki
    glDeleteProgram( program[OBJECT] );
    glDeleteProgram( program[BOUNDING_BOX] );
    glDeleteBuffers( VERTEX_BUFFER_SIZE, vertexBuffer );
    glDeleteBuffers( INDICES_BUFFER_SIZE, indicesBuffer );
    glDeleteVertexArrays( VERTEX_ARRAY_SIZE, vertexArray );
    glDeleteQueries( QUERY_SIZE, query );

    // usunicie mechanizmw uywanych podczas renderingu tekstu
    DeleteDrawText();
}
