//////////////////////////////////////////////////////////////////////
// (c) Janusz Ganczarski
// http://www.januszg.hg.pl
// JanuszG@enter.net.pl
//////////////////////////////////////////////////////////////////////
// na podstawie: ModelAndTextureLoader.cpp i ModelAndTextureLoader.hpp
// AMD FirePro Technology SDK
// http://developer.amd.com/tools-and-sdks/graphics-development/firepro-sdk/
// AMD-FirePro-SDK-v0.9.2-Windows.exe
//////////////////////////////////////////////////////////////////////

#include <iostream>
#include "models.h"
#include <assimp/postprocess.h>
#include "textures.h"

//////////////////////////////////////////////////////////////////////
// bryy otaczajce dla poszczeglnych wzw modelu
//////////////////////////////////////////////////////////////////////
void ModelAndTextureLoader::GetBoundingBoxForNode( const aiNode *nd, aiVector3D &min, aiVector3D &max, aiMatrix4x4 &trafo )
{
    // zapamitanie macierzy transformacji
    aiMatrix4x4 prev;
    prev = trafo;

    // macierz transformacji
    aiMultiplyMatrix4( &trafo, &nd -> mTransformation );

    // przegld wierzchokw wza
    for ( unsigned int n = 0; n < nd -> mNumMeshes; ++n )
    {
        const aiMesh *mesh = assimpScene -> mMeshes[nd -> mMeshes[n]];
        for( unsigned int t = 0; t < mesh -> mNumVertices; ++t ) 
        {
            // transformacja biecej siatki
            aiVector3D tmp = mesh -> mVertices[t];
            aiTransformVecByMatrix4( &tmp, &trafo );

            // minimum
            min.x = std::min( min.x, tmp.x );
            min.y = std::min( min.y, tmp.y );
            min.z = std::min( min.z, tmp.z );

            // maksimum
            max.x = std::max( max.x, tmp.x );
            max.y = std::max( max.y, tmp.y );
            max.z = std::max( max.z, tmp.z );
        }
    }

    // rekursja po kolejnych wzach modelu
    for( unsigned int n = 0; n < nd->mNumChildren; ++n )
    {
        GetBoundingBoxForNode( nd -> mChildren[n], min, max, trafo );
    }

    // przywrcenie macierzy transformacji
    trafo = prev;
}

//////////////////////////////////////////////////////////////////////
// pobranie rozmiarw bryy otaczajcej model (prostopadocianu)
//////////////////////////////////////////////////////////////////////
void ModelAndTextureLoader::GetBoundingBox( aiVector3D &min, aiVector3D &max )
{
    // wartoci pocztkowe
    min.x = min.y = min.z =  1e10f;
    max.x = max.y = max.z = -1e10f;

    // pocztkowa macierz transformacji - macierz jednostkowa
    aiMatrix4x4 trafo;
    aiIdentityMatrix4( &trafo );

    // bryy otaczajce dla poszczeglnych wzw modelu
    GetBoundingBoxForNode( assimpScene -> mRootNode, min, max, trafo );
}

/////////////////////////////////////////////////////////////////////////////////////////////
// sprawdzenie, czy s (p)przezroczyste materiay
/////////////////////////////////////////////////////////////////////////////////////////////
void ModelAndTextureLoader::IsOpacity( const aiNode* nd, unsigned int &currentMesh )
{
    // ptla po siatkach biecego wza
    for( unsigned int n = 0; n < nd -> mNumMeshes; ++n )
    {
        // bieca siatka
        const aiMesh *mesh = assimpScene -> mMeshes[nd -> mMeshes[n]];

        // biecy materia
        const aiMaterial *material = assimpScene -> mMaterials[mesh -> mMaterialIndex];

        // tekstury
        MATERIAL_TEXTUREID textureIDs = textureOfEachMaterial[mesh -> mMaterialIndex];

        // wspczynnik przezroczystoci
        aiColor4D opacityMat;
        if( !textureIDs.diffuse && aiGetMaterialColor( material, AI_MATKEY_OPACITY, &opacityMat ) == aiReturn_SUCCESS )
        {
            if( opacityMat[0] < 1.0f )
            {
                opacity = true;
                return;
            }
        }

        // nastpna siatka
        currentMesh++;
    }

    // rendering danych siatek z wzw potomnych
    for( unsigned int n = 0; n < nd -> mNumChildren; ++n )
    {
        IsOpacity( nd -> mChildren[n], currentMesh );
    }
}


//////////////////////////////////////////////////////////////////////
// zaadowanie obiektu z pliku
//////////////////////////////////////////////////////////////////////
bool ModelAndTextureLoader::Load( const char *fileName )
{
    // porzdki
    Delete();

    // test obecnoci nazwy pliku
    if( fileName == NULL )
        return false;

    // import danych z pliku
    assimpScene = aiImportFile( fileName, aiProcess_CalcTangentSpace |      // generowanie wektorw przestrzeni stycznej
                                          aiProcess_JoinIdenticalVertices | // zczenie duplikatw wierzchokw
                                          aiProcess_FlipUVs |               // odwrcenie wsprzdnych t tekstury
                                          aiProcess_Triangulate |           // triangulacja prymityww o wicej ni 3 wierzchokach
                                          aiProcess_GenSmoothNormals |      // generowanie urednionych wektorw normalnych
                                          aiProcess_PreTransformVertices ); // transformacja wierzchokw w oparciu
                                                                            // macierze transformacji zawarte w pliku

    // test poprawnoci odczytu pliku
    if( !assimpScene )
        return false;

    // obliczenie rodka modelu i jego rozmiarw
    aiVector3D scene_min, scene_max, scene_center;
    GetBoundingBox( scene_min, scene_max );
    center.x = (scene_min.x + scene_max.x) / 2.0f;
    center.y = (scene_min.y + scene_max.y) / 2.0f;
    center.z = (scene_min.z + scene_max.z) / 2.0f;
    size = scene_max.x - scene_min.x;
    size = std::max( scene_max.y - scene_min.y, size );
    size = std::max( scene_max.z - scene_min.z, size );

    // obliczenie cznej liczby siatek w pliku
    totalMesh = 0;
    CountTotalMesh( assimpScene -> mRootNode );

    // zaadowanie siatek
    allMeshes.resize( totalMesh );
    unsigned int iMesh = 0;
    MeshLoading( assimpScene -> mRootNode, iMesh );

    // tekstury do poszczeglnych materiaw
    textureOfEachMaterial.resize( assimpScene -> mNumMaterials );

    // pobranie nazw poszczeglnych tekstur
    for( unsigned int m = 0; m < assimpScene -> mNumMaterials; m++ )
    {
        int texIndex = 0;
        aiString path;
        while( assimpScene -> mMaterials[m] -> GetTexture( aiTextureType_AMBIENT, texIndex, &path ) == AI_SUCCESS )
        {
            textureIdMap[path.data] = 0;
            texIndex++;
        }
        texIndex = 0;
        while( assimpScene -> mMaterials[m] -> GetTexture( aiTextureType_DIFFUSE, texIndex, &path ) == AI_SUCCESS )
        {
            textureIdMap[path.data] = 0;
            texIndex++;
        }
        texIndex = 0;
        while( assimpScene -> mMaterials[m] -> GetTexture( aiTextureType_SPECULAR, texIndex, &path ) == AI_SUCCESS )
        {
            textureIdMap[path.data] = 0;
            texIndex++;
        }
        texIndex = 0;
        while( assimpScene -> mMaterials[m] -> GetTexture( aiTextureType_SHININESS, texIndex, &path ) == AI_SUCCESS )
        {
            textureIdMap[path.data] = 0;
            texIndex++;
        }
        texIndex = 0;
        while( assimpScene -> mMaterials[m] -> GetTexture( aiTextureType_NORMALS, texIndex, &path ) == AI_SUCCESS )
        {
            textureIdMap[path.data] = 0;
            texIndex++;
        }
        texIndex = 0;
        while( assimpScene -> mMaterials[m] -> GetTexture( aiTextureType_HEIGHT, texIndex, &path ) == AI_SUCCESS )
        {
            textureIdMap[path.data] = 0;
            texIndex++;
        }
    }

    // pobranie liczby tekstur
    int numTextures = int(textureIdMap.size());
    if( numTextures )
    {
        // utworzenie identyfikatorw obiektw tekstury
        textureIds.resize( numTextures );
        for( int i = 0; i< numTextures; i++ )
            glGenTextures( 1, &textureIds[i] );

        // pobranie cieki pliku
        std::string path;
        path.clear();
        std::string name = fileName;
        std::string::size_type offset = name.find_last_of( '\\' );
        if( offset != std::string::npos )
        {
            path = name.substr( 0, offset + 1 );
        }
        else
        {
            offset = name.find_last_of( '/' );
            if( offset != std::string::npos )
                path = name.substr( 0, offset + 1 );
        }

        // odczyt poszczeglnych tekstur
        glActiveTexture( GL_TEXTURE0 );
        std::map< std::string, GLuint >::iterator itr = textureIdMap.begin();
        for( int i = 0; i < numTextures; i++ )
        {
            // nazwa pliku z tekstur
            std::string texFileName = path + (*itr).first;
            (*itr).second = textureIds[i];

            // utworzenie obiektu tekstury i jej odczyt
            glBindTexture( GL_TEXTURE_2D, textureIds[i] );
            if( !LoadTexture( texFileName.c_str(), GL_TEXTURE_2D ) )
            {
                std::cout << "Niepoprawny odczyt pliku " << texFileName << std::endl;
                return false;
            }

            // filtr pomniejszajcy i powikszajcy
            glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR_MIPMAP_LINEAR );
            glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR );

            // generowanie mipmap
            glGenerateMipmap( GL_TEXTURE_2D );

            // przeczenie na tekstur domyln
            glBindTexture( GL_TEXTURE_2D, 0 );
            itr++;
        }

        // przyporzdkowanie identyfikatorw obiektw tekstury
        // do poszczeglnych rodzajw materiaw (w praktyce
        // pojedyncza tekstura moe obsugiwa kilka materiaw)
        for( unsigned int m = 0; m < assimpScene -> mNumMaterials; m++ )
        {
            aiString path;
            if( assimpScene -> mMaterials[m] -> GetTexture( aiTextureType_AMBIENT, 0, &path ) ==  AI_SUCCESS ) 
            {
                textureOfEachMaterial[m].ambient = textureIdMap[path.data];
            }
            else
                textureOfEachMaterial[m].ambient = 0;
            if( assimpScene -> mMaterials[m] -> GetTexture( aiTextureType_DIFFUSE, 0, &path ) ==  AI_SUCCESS ) 
            {
                textureOfEachMaterial[m].diffuse = textureIdMap[path.data];
            }
            else
                textureOfEachMaterial[m].diffuse = 0;
            if( assimpScene->mMaterials[m]->GetTexture( aiTextureType_SPECULAR, 0, &path ) ==  AI_SUCCESS ) 
            {
                textureOfEachMaterial[m].specular = textureIdMap[path.data];
            }
            else
                textureOfEachMaterial[m].specular = 0;
            if( assimpScene -> mMaterials[m] -> GetTexture( aiTextureType_SHININESS, 0, &path ) ==  AI_SUCCESS ) 
            {
                textureOfEachMaterial[m].shininess = textureIdMap[path.data] ;
            }
            else
                textureOfEachMaterial[m].shininess = 0;
            if( assimpScene -> mMaterials[m] -> GetTexture( aiTextureType_NORMALS, 0, &path ) ==  AI_SUCCESS ) 
            {
                textureOfEachMaterial[m].normal = textureIdMap[path.data];
            }
            else
                textureOfEachMaterial[m].normal = 0;
            if( assimpScene -> mMaterials[m] -> GetTexture( aiTextureType_HEIGHT, 0, &path ) ==  AI_SUCCESS ) 
            {
                textureOfEachMaterial[m].height = textureIdMap[path.data];
            }
            else
                textureOfEachMaterial[m].height = 0;
        }
    }

    // sprawdzenie wystepowania (p)przezroczystych materiaw
    iMesh = 0;
    IsOpacity( assimpScene -> mRootNode, iMesh );

    // sukces
    return true;
}

//////////////////////////////////////////////////////////////////////
// usunicie wszystkich elementw sceny
//////////////////////////////////////////////////////////////////////
void ModelAndTextureLoader::Delete()
{
    // usunicie obiektw tekstury
    for( unsigned int i = 0; i < textureIdMap.size(); i++ )
        glDeleteTextures( 1, &textureIds[i] );

    // usunicie poszczeglnych siatek sceny
    if(assimpScene )
    {
        unsigned int iMesh = 0;
        MeshDelete( assimpScene -> mRootNode, iMesh );
        aiReleaseImport( assimpScene );
    }

    // oprnienie kontenerw
    textureIdMap.clear();
    allMeshes.clear();
    textureIds.clear();
    textureOfEachMaterial.clear();

    // wartoci pocztkowe
    assimpScene = NULL;
    totalMesh = 0;
    size = 0.0;
    opacity = false;
    center = glm::vec3( 0.0f );
}

//////////////////////////////////////////////////////////////////////
// obliczenie cznej liczby siatek w wybranym wle
//////////////////////////////////////////////////////////////////////
void ModelAndTextureLoader::CountTotalMesh( const aiNode *nd )
{
    totalMesh += nd -> mNumMeshes;
    for( unsigned int n = 0; n < nd -> mNumChildren; ++n )
    {
        CountTotalMesh( nd -> mChildren[n] );
    }
}

//////////////////////////////////////////////////////////////////////
// zaadowanie danych siatek z wybranego wza
//////////////////////////////////////////////////////////////////////
void ModelAndTextureLoader::MeshLoading( const aiNode *nd, unsigned int &currentMesh )
{
    // ptla po siatkach biecego wza
    for( unsigned int n = 0; n < nd -> mNumMeshes; ++n )
    {
        // bieca siatka
        const aiMesh *mesh = assimpScene -> mMeshes[nd -> mMeshes[n]];

        // generowanie identyfikatorw obiektw VBO i EAB (indeksy)
        glGenBuffers( 1, &(allMeshes[currentMesh].position) );
        glGenBuffers( 1, &(allMeshes[currentMesh].normal) );
        glGenBuffers( 1, &(allMeshes[currentMesh].texCoord) );
        glGenBuffers( 1, &(allMeshes[currentMesh].tangent) );
        glGenBuffers( 1, &(allMeshes[currentMesh].bitangent) );
        glGenBuffers( 1, &(allMeshes[currentMesh].indices) );

        // generowanie identyfikatora obiektu VAO
        glGenVertexArrays( 1, &(allMeshes[currentMesh].vertexArray) );

        // bufory wierzchow
        glBindBuffer( GL_ARRAY_BUFFER, allMeshes[currentMesh].position );
        glBufferData( GL_ARRAY_BUFFER, mesh -> mNumVertices*3*sizeof(float), mesh -> mVertices, GL_STATIC_DRAW );
        glBindBuffer( GL_ARRAY_BUFFER, allMeshes[currentMesh].normal );
        glBufferData( GL_ARRAY_BUFFER, mesh -> mNumVertices*3*sizeof(float), mesh -> mNormals, GL_STATIC_DRAW );
        glBindBuffer( GL_ARRAY_BUFFER, allMeshes[currentMesh].texCoord );
        glBufferData( GL_ARRAY_BUFFER, mesh -> mNumVertices*3*sizeof(float), mesh -> mTextureCoords[0], GL_STATIC_DRAW );
        glBindBuffer( GL_ARRAY_BUFFER, allMeshes[currentMesh].tangent );
        glBufferData( GL_ARRAY_BUFFER, mesh -> mNumVertices*3*sizeof(float), mesh -> mTangents, GL_STATIC_DRAW );
        glBindBuffer( GL_ARRAY_BUFFER, allMeshes[currentMesh].bitangent );
        glBufferData( GL_ARRAY_BUFFER, mesh -> mNumVertices*3*sizeof(float), mesh -> mBitangents, GL_STATIC_DRAW );

        // zliczanie liczby indeksw wierzchokw
        allMeshes[currentMesh].pointsFaces = 0;
        for( unsigned int t = 0; t < mesh -> mNumFaces; ++t ) 
        {
            const aiFace *face = &mesh -> mFaces[t];
            if ( face -> mNumIndices == 3 ) 
            {
                allMeshes[currentMesh].pointsFaces++;
            }
        }

        // generowanie danych indeksw wierzchokw
        unsigned int *p3PointsFaces = new unsigned int[allMeshes[currentMesh].pointsFaces * 3];
        unsigned int pointFaceCounter = 0;
        for( unsigned int t = 0; t < mesh -> mNumFaces; ++t )
        {
            const aiFace *face = &mesh -> mFaces[t];
            if ( face -> mNumIndices == 3 )
            {
                p3PointsFaces[pointFaceCounter*3+0] = face -> mIndices[0];
                p3PointsFaces[pointFaceCounter*3+1] = face -> mIndices[1];
                p3PointsFaces[pointFaceCounter*3+2] = face -> mIndices[2];
                pointFaceCounter++;
            }
        }

        // bufor indeksw wierzchokw
        glBindBuffer( GL_ELEMENT_ARRAY_BUFFER, allMeshes[currentMesh].indices );
        glBufferData( GL_ELEMENT_ARRAY_BUFFER, allMeshes[currentMesh].pointsFaces*3*sizeof(unsigned int), p3PointsFaces, GL_STATIC_DRAW );

        // usuwanie danych indeksw wierzchokw
        delete[] p3PointsFaces;

        // wczenie obiektu tablic wierzchokw
        glBindVertexArray( allMeshes[currentMesh].vertexArray );

        // atrybuty wierzchokw
        glBindBuffer( GL_ARRAY_BUFFER, allMeshes[currentMesh].position );
        glVertexAttribPointer( POSITION, 3, GL_FLOAT, GL_FALSE, 0, 0 );
        glEnableVertexAttribArray( POSITION );

        // atrybuty wierzchokw
        glBindBuffer( GL_ARRAY_BUFFER, allMeshes[currentMesh].normal );
        glVertexAttribPointer( NORMAL, 3, GL_FLOAT, GL_FALSE, 0, 0 );
        glEnableVertexAttribArray( NORMAL );

        // atrybuty wierzchokw
        glBindBuffer( GL_ARRAY_BUFFER, allMeshes[currentMesh].texCoord );
        glVertexAttribPointer( TEX_COORD, 3, GL_FLOAT, GL_FALSE, 0, 0 );
        glEnableVertexAttribArray( TEX_COORD );

        // atrybuty wierzchokw
        glBindBuffer( GL_ARRAY_BUFFER, allMeshes[currentMesh].tangent );
        glVertexAttribPointer( TANGENT, 3, GL_FLOAT, GL_FALSE, 0, 0 );
        glEnableVertexAttribArray( TANGENT );

        // atrybuty wierzchokw
        glBindBuffer( GL_ARRAY_BUFFER, allMeshes[currentMesh].bitangent );
        glVertexAttribPointer( BITANGENT, 3, GL_FLOAT, GL_FALSE, 0, 0 );
        glEnableVertexAttribArray( BITANGENT );

        // indeksy wierzchokw
        glBindBuffer( GL_ELEMENT_ARRAY_BUFFER, allMeshes[currentMesh].indices );

        // wyczenie obiektu tablic wierzchokw
        glBindVertexArray( 0 );

        // nastpna siatka
        currentMesh++;
    }

    // zaadowanie danych siatek z wzw potomnych
    for( unsigned int n = 0; n < nd -> mNumChildren; ++n )
    {
        MeshLoading( nd -> mChildren[n], currentMesh );
    }
}

//////////////////////////////////////////////////////////////////////
// usunicie danych siatek z wybranego wza
//////////////////////////////////////////////////////////////////////
void ModelAndTextureLoader::MeshDelete( const aiNode *nd, unsigned int &currentMesh )
{
    // ptla po siatkach biecego wza
    for( unsigned int n = 0; n < nd -> mNumMeshes; ++n )
    {
        // bieca siatka
        const aiMesh *mesh = assimpScene -> mMeshes[nd -> mMeshes[n]];

        // usunicie obiektw VBO i EAB (indeksy)
        glDeleteBuffers( 1, &(allMeshes[currentMesh].position) );
        glDeleteBuffers( 1, &(allMeshes[currentMesh].normal) );
        glDeleteBuffers( 1, &(allMeshes[currentMesh].texCoord) );
        glDeleteBuffers( 1, &(allMeshes[currentMesh].tangent) );
        glDeleteBuffers( 1, &(allMeshes[currentMesh].bitangent) );
        glDeleteBuffers( 1, &(allMeshes[currentMesh].indices) );

        // usunicie obiektu VAO
        glDeleteVertexArrays( 1, &(allMeshes[currentMesh].vertexArray) );

        // nastpna siatka
        currentMesh++;
    }

    // usunicie danych siatek z wzw potomnych
    for( unsigned int n = 0; n < nd -> mNumChildren; ++n )
    {
        MeshDelete( nd -> mChildren[n], currentMesh );
    }
}

/////////////////////////////////////////////////////////////////////////////////////////////
// rendering siatek z wybranego wza
/////////////////////////////////////////////////////////////////////////////////////////////
void ModelAndTextureLoader::MeshRendering( const aiNode *nd, unsigned int &currentMesh,
                                           bool uniforms[4] )
{
    // definicje numerw pooenia zmiennych jednorodnych i powizania uchwytw tekstur
    enum
    {
        AMBIENT,
        DIFFUSE,
        SPECULAR,
        SHININESS,
        HEIGHT,
        NORMAL,
        TEXTURE_LOC = 4
    };

    // ptla po siatkach biecego wza
    for( unsigned int n = 0; n < nd -> mNumMeshes; ++n )
    {
        // bieca siatka
        const aiMesh *mesh = assimpScene -> mMeshes[nd -> mMeshes[n]];

        // biecy materia
        const aiMaterial *material = assimpScene -> mMaterials[mesh -> mMaterialIndex];

        // tekstury
        MATERIAL_TEXTUREID textureIDs = textureOfEachMaterial[mesh -> mMaterialIndex];

        // znaczniki dostpnoci poszczeglnych w tekstur
        GLint textures[6] = { textureIDs.ambient, textureIDs.diffuse, textureIDs.specular,
                              textureIDs.shininess, textureIDs.height, textureIDs.normal };
        glUniform1iv( TEXTURE_LOC, 6, textures );

        // wspczynnik odbicia wiata otoczenia; oznaczenie Ma
        if( textureIDs.ambient )
        {
            glActiveTexture( GL_TEXTURE0 + AMBIENT );
            glBindTexture( GL_TEXTURE_2D, textureIDs.ambient );
        }
        else
        {
            aiColor4D ambient;
            if( uniforms[AMBIENT] && aiGetMaterialColor( material, AI_MATKEY_COLOR_AMBIENT, &ambient ) == aiReturn_SUCCESS )
            {
                glUniform4f( AMBIENT, ambient[0], ambient[1], ambient[2], ambient[3] );
            }
        }

        // wspczynnik odbicia wiata rozproszonego; oznaczenie Md
        if( textureIDs.diffuse )
        {
            glActiveTexture( GL_TEXTURE0 + DIFFUSE );
            glBindTexture( GL_TEXTURE_2D, textureIDs.diffuse );
        }
        else
        {
            aiColor4D diffuse;
            if( uniforms[DIFFUSE] && aiGetMaterialColor( material, AI_MATKEY_COLOR_DIFFUSE, &diffuse ) == aiReturn_SUCCESS )
            {
                glUniform4f( DIFFUSE, diffuse[0], diffuse[1], diffuse[2], diffuse[3] );
            }
        }

        // wspczynnik odbicia wiata zwierciadlanego; oznaczenie Ms
        if( textureIDs.specular )
        {
            glActiveTexture( GL_TEXTURE0 + SPECULAR );
            glBindTexture( GL_TEXTURE_2D, textureIDs.specular );
        }
        else
        {
            aiColor4D specular;
            if( uniforms[SPECULAR] && aiGetMaterialColor( material, AI_MATKEY_COLOR_SPECULAR, &specular ) == aiReturn_SUCCESS )
            {
                glUniform4f( SPECULAR, specular[0], specular[1], specular[2], specular[3] );
            }
        }

        // wykadnik wspczynnika funkcji rozbysku; zakres wartoci <0;128>; oznaczenie Msh
        if( textureIDs.shininess )
        {
            glActiveTexture( GL_TEXTURE0 + SHININESS );
            glBindTexture( GL_TEXTURE_2D, textureIDs.shininess );
        }
        else
        {
            aiColor4D shininess;
            if( uniforms[SHININESS] && aiGetMaterialColor( material, AI_MATKEY_SHININESS, &shininess ) == aiReturn_SUCCESS )
            {
                glUniform1f( SHININESS, shininess[0] );
            }
        }

        // mapa wysokoci
        if( textureIDs.height )
        {
            glActiveTexture( GL_TEXTURE0 + HEIGHT );
            glBindTexture( GL_TEXTURE_2D, textureIDs.height );
        }

        // mapa wektorw normalnych
        if( textureIDs.normal )
        {
            glActiveTexture( GL_TEXTURE0 + NORMAL );
            glBindTexture( GL_TEXTURE_2D, textureIDs.normal );
        }

        // rendering biecej siatki
        glBindVertexArray( allMeshes[currentMesh].vertexArray );
        glDrawElements( IsTesselation() ? GL_PATCHES : GL_TRIANGLES, allMeshes[currentMesh].pointsFaces*3, GL_UNSIGNED_INT, NULL );
        glBindVertexArray( 0 );

        // nastpna siatka
        currentMesh++;
    }

    // rendering danych siatek z wzw potomnych
    for( unsigned int n = 0; n < nd -> mNumChildren; ++n )
    {
        MeshRendering( nd -> mChildren[n], currentMesh, uniforms );
    }
}

/////////////////////////////////////////////////////////////////////////////////////////////
// rendering siatek z wybranego wza
/////////////////////////////////////////////////////////////////////////////////////////////
void ModelAndTextureLoader::MeshRenderingOpacity( bool opacity, const aiNode *nd,
                                                  unsigned int &currentMesh, bool uniforms[4] )
{
    // definicje numerw pooenia zmiennych jednorodnych i powizania uchwytw tekstur
    enum
    {
        AMBIENT,
        DIFFUSE,
        SPECULAR,
        SHININESS,
        HEIGHT,
        NORMAL,
        TEXTURE_LOC = 4
    };

    // ptla po siatkach biecego wza
    for( unsigned int n = 0; n < nd -> mNumMeshes; ++n )
    {
        // bieca siatka
        const aiMesh *mesh = assimpScene -> mMeshes[nd -> mMeshes[n]];

        // biecy materia
        const aiMaterial *material = assimpScene -> mMaterials[mesh -> mMaterialIndex];

        // tekstury
        MATERIAL_TEXTUREID textureIDs = textureOfEachMaterial[mesh -> mMaterialIndex];

        // pobranie stopnia (p)przezroczystoci
        aiColor4D opacityMat;
        opacityMat[0] = 1.0f;
        if( !textureIDs.diffuse )
            aiGetMaterialColor( material, AI_MATKEY_OPACITY, &opacityMat );

        // test obiektw do renderingu (p)przezroczyste/nieprzezroczyste
        if( ( opacityMat[0] < 1.0f && opacity ) || ( opacityMat[0] >= 1.0f && !opacity ) )
        {
            // znaczniki dostpnoci poszczeglnych w tekstur
            GLint textures[6] = { textureIDs.ambient, textureIDs.diffuse, textureIDs.specular,
                                  textureIDs.shininess, textureIDs.height, textureIDs.normal };
            glUniform1iv( TEXTURE_LOC, 6, textures );

            // wspczynnik odbicia wiata otoczenia; oznaczenie Ma
            if( textureIDs.ambient )
            {
                glActiveTexture( GL_TEXTURE0 + AMBIENT );
                glBindTexture( GL_TEXTURE_2D, textureIDs.ambient );
            }
            else
            {
                aiColor4D ambient;
                if( uniforms[AMBIENT] && aiGetMaterialColor( material, AI_MATKEY_COLOR_AMBIENT, &ambient ) == aiReturn_SUCCESS )
                {
                    glUniform4f( AMBIENT, ambient[0], ambient[1], ambient[2], ambient[3] );
                }
            }

            // wspczynnik odbicia wiata rozproszonego; oznaczenie Md
            if( textureIDs.diffuse )
            {
                glActiveTexture( GL_TEXTURE0 + DIFFUSE );
                glBindTexture( GL_TEXTURE_2D, textureIDs.diffuse );
            }
            else
            {
                aiColor4D diffuse;
                if( uniforms[DIFFUSE] && aiGetMaterialColor( material, AI_MATKEY_COLOR_DIFFUSE, &diffuse ) == aiReturn_SUCCESS )
                {
                    glUniform4f( DIFFUSE, diffuse[0], diffuse[1], diffuse[2], opacityMat[0] );
                }
            }

            // wspczynnik odbicia wiata zwierciadlanego; oznaczenie Ms
            if( textureIDs.specular )
            {
                glActiveTexture( GL_TEXTURE0 + SPECULAR );
                glBindTexture( GL_TEXTURE_2D, textureIDs.specular );
            }
            else
            {
                aiColor4D specular;
                if( uniforms[SPECULAR] && aiGetMaterialColor( material, AI_MATKEY_COLOR_SPECULAR, &specular ) == aiReturn_SUCCESS )
                {
                    glUniform4f( SPECULAR, specular[0], specular[1], specular[2], specular[3] );
                }
            }

            // wykadnik wspczynnika funkcji rozbysku; zakres wartoci <0;128>; oznaczenie Msh
            if( textureIDs.shininess )
            {
                glActiveTexture( GL_TEXTURE0 + SHININESS );
                glBindTexture( GL_TEXTURE_2D, textureIDs.shininess );
            }
            else
            {
                aiColor4D shininess;
                if( uniforms[SHININESS] && aiGetMaterialColor( material, AI_MATKEY_SHININESS, &shininess ) == aiReturn_SUCCESS )
                {
                    glUniform1f( SHININESS, shininess[0] );
                }
            }

            // mapa wysokoci
            if( textureIDs.height )
            {
                glActiveTexture( GL_TEXTURE0 + HEIGHT );
                glBindTexture( GL_TEXTURE_2D, textureIDs.height );
            }

            // mapa wektorw normalnych
            if( textureIDs.normal )
            {
                glActiveTexture( GL_TEXTURE0 + NORMAL );
                glBindTexture( GL_TEXTURE_2D, textureIDs.normal );
            }

            // rendering biecej siatki
            glBindVertexArray( allMeshes[currentMesh].vertexArray );
            glDrawElements( IsTesselation() ? GL_PATCHES : GL_TRIANGLES, allMeshes[currentMesh].pointsFaces*3, GL_UNSIGNED_INT, NULL );
        }

        // nastpna siatka
        currentMesh++;
    }

    // rendering danych siatek z wzw potomnych
    for( unsigned int n = 0; n < nd -> mNumChildren; ++n )
    {
        MeshRenderingOpacity( opacity, nd -> mChildren[n], currentMesh, uniforms );
    }
}

/////////////////////////////////////////////////////////////////////////////////////////////
// sprawdzenie, czy jest aktywna teselacja
/////////////////////////////////////////////////////////////////////////////////////////////
bool ModelAndTextureLoader::IsTesselation()
{
    // pobranie identyfikatora biecego obiektu programu
    GLint program = 0;
    glGetIntegerv( GL_CURRENT_PROGRAM, &program );

    // jest obiekt programu
    if( program )
    {
        // pobranie liczby dowizanych obiektw shadera
        GLint attachedShaders = 0;
        glGetProgramiv( program, GL_ATTACHED_SHADERS, &attachedShaders );

        // s obiekty shadera
        if( attachedShaders )
        {
            // pobranie identyfikatorw dowizanych obiektw shadera
            GLuint *shaders = new GLuint[attachedShaders];
            glGetAttachedShaders( program, attachedShaders, NULL, shaders );

            // test rodzajw poszczeglnych obiektw shadera
            for( int i = 0; i < attachedShaders; i++ )
            {
                // pobranie rodzaju obiektu shadera
                GLint shaderType;
                glGetShaderiv( shaders[i], GL_SHADER_TYPE, &shaderType );

                // sprawdzenie, czy jest shader kontroli teselacji lub shader ewaluacjai teselacji
                if( shaderType == GL_TESS_CONTROL_SHADER || shaderType == GL_TESS_EVALUATION_SHADER )
                {
                    delete[] shaders;
                    return true;
                }
            }

            // porzdki
            delete[] shaders;
        }
    }

    // brak teselacji
    return false;
}

/////////////////////////////////////////////////////////////////////////////////////////////
// sprawdzenie, czy s dostepne zmienne jednorodne obsugujce parametry materiaw
/////////////////////////////////////////////////////////////////////////////////////////////
void ModelAndTextureLoader::IsUniforms( bool uniforms[4] )
{
    // definicje numerw pooenia zmiennych jednorodnych
    enum
    {
        AMBIENT,
        DIFFUSE,
        SPECULAR,
        SHININESS
    };

    // wartoci pocztkowe znacznikw
    uniforms[AMBIENT] = false;
    uniforms[DIFFUSE] = false;
    uniforms[SPECULAR] = false;
    uniforms[SHININESS] = false;

    // pobranie identyfikatora biecego obiektu programu
    GLint program = 0;
    glGetIntegerv( GL_CURRENT_PROGRAM, &program );

    // jest obiekt programu
    if( program )
    {
        // pobranie liczby aktywnych zmiennych jednorodnych
        GLint activeUniforms = 0;
        glGetProgramiv( program, GL_ACTIVE_UNIFORMS, &activeUniforms );

        // s aktywne zmienne jednorodne
        if( activeUniforms )
        {
            // maksymalna dugo nazwy zmiennej jednorodnej
            GLint maxLength;
            glGetProgramiv( program, GL_ACTIVE_UNIFORM_MAX_LENGTH, &maxLength );

            // pobranie danych kolejnych zmiennych jednorodnych
            GLchar *name = new GLchar[maxLength + 1];
            for( int i = 0; i < activeUniforms; i++ )
            {
                // pobranie nazwy i typu zmiennej
                GLint size;
                GLenum type;
                glGetActiveUniform( program, i, maxLength, NULL, &size, &type, name );

                // pobranie lokalizacji zmiennej i sprawdzenie zgodnoci typu
                switch( glGetUniformLocation( program, name ) )
                {
                    case AMBIENT:
                        if( type == GL_FLOAT_VEC4 ) uniforms[AMBIENT] = true;
                        break;
                    case DIFFUSE:
                        if( type == GL_FLOAT_VEC4 ) uniforms[DIFFUSE] = true;
                        break;
                    case SPECULAR:
                        if( type == GL_FLOAT_VEC4 ) uniforms[SPECULAR] = true;
                        break;
                    case SHININESS:
                        if( type == GL_FLOAT ) uniforms[SHININESS] = true;
                        break;
                }
            }

            // porzdki
            delete[] name;
        }
    }
}
