﻿using System;
using System.IO;
using System.Reflection;
using System.Runtime.ConstrainedExecution;
using System.Runtime.InteropServices;
using Microsoft.Win32.SafeHandles;
using ADODB;

namespace Wrox.DotNetFramework2.Samples
{
    public class Chapter11
    {
        public static void _title()
        {
            Console.WriteLine("Rozdział 11 - Interoperacyjność z kodem niezarządzanym");
            Console.WriteLine("======================================================");
        }

        /*** WSKAŹNIKI, UCHWYTY I ZASOBY ***/

        // Przykład 1: wskaźniki IntPtr i niebezpieczne ustawianie bitów
        private static unsafe void SetByte(IntPtr b, int offset, byte value)
        {
            if (b == IntPtr.Zero)
                throw new ArgumentOutOfRangeException();
            byte* pTarget = (byte*)b.ToPointer() + offset;
            *pTarget = value;
        }

        public static unsafe void ex01()
        {
            int a = 10;
            Console.WriteLine("a jest równe {0} ({0:X})", a);
            IntPtr ip = new IntPtr(&a);
            SetByte(ip, 1, 1);
            Console.WriteLine("a jest równe {0} ({0:X})", a);
        }

        // Przykład 2: abstrakcja pliku
        class MyFile : IDisposable
        {
            private IntPtr hFileHandle;

            public MyFile(string filename)
            {
                OFSTRUCT ofs;
                hFileHandle = OpenFile(filename, out ofs, OF_READWRITE);
            }

            ~MyFile()
            {
                Dispose();
            }

            public void Dispose()
            {
                if (hFileHandle != IntPtr.Zero)
                    CloseHandle(hFileHandle);
                hFileHandle = IntPtr.Zero;
                GC.SuppressFinalize(this);
            }

            public long ReadBytes(byte[] buffer)
            {
                // Najpierw upewniamy się, że plik nie jest zamknięty
                IntPtr hFile = hFileHandle;
                if (hFile == IntPtr.Zero)
                    throw new ObjectDisposedException("Plik jest zamknięty");

                uint read;
                if (!ReadFile(hFile, buffer,
                        (uint)buffer.Length, out read, IntPtr.Zero))
                    throw new Exception("Błąd " + Marshal.GetLastWin32Error());
                return (long)read;
            }

            private const int OF_READWRITE = 0x00000002;

            [DllImport("kernel32.dll")]
            static extern IntPtr OpenFile(string lpFileName, out OFSTRUCT lpReOpenBuff, uint uStyle);

            [DllImport("kernel32.dll")]
            [return: MarshalAs(UnmanagedType.Bool)]
            static extern bool CloseHandle(IntPtr hObject);

            [DllImport("kernel32.dll")]
            static extern bool ReadFile(IntPtr hFile, byte[] lpBuffer,
                uint nNumberOfBytesToRead, out uint lpNumberOfBytesRead, IntPtr lpOverlapped);
        }

        [System.Runtime.InteropServices.StructLayout(LayoutKind.Sequential)]
        public struct OFSTRUCT
        {
            public byte cBytes;
            public byte fFixedDisc;
            public UInt16 nErrCode;
            public UInt16 Reserved1;
            public UInt16 Reserved2;
            [MarshalAs(System.Runtime.InteropServices.UnmanagedType.ByValTStr, SizeConst = 128)] 
            public string szPathName;
        }

        // Przykład 3: używanie pamięci niezarządzanej
        public static void ex03()
        {
            // Alokujemy 1 kB pamięci
            IntPtr ptr = Marshal.AllocHGlobal(1024);
            try
            {
                // Robimy coś ze wskaźnikiem ptr
                Console.WriteLine("{0:X}", ptr.ToInt32());
            }
            finally
            {
                // Teraz zwalniamy pamięć
                if (ptr != IntPtr.Zero)
                {
                    Marshal.FreeHGlobal(ptr);
                    ptr = IntPtr.Zero;
                }
            }
        }

        // Przykład 4: przykładowy typ menedżera zasobów (bez bezpiecznego uchwytu).
        sealed class MyResourceManager : IDisposable
        {
            private IntPtr ptr;

            public MyResourceManager()
            {
                ptr = Marshal.AllocHGlobal(1024);
            }

            ~MyResourceManager()
            {
                Dispose(false);
            }

            public void Dispose()
            {
                Dispose(true);
                GC.SuppressFinalize(this);
            }

            private void Dispose(bool disposing)
            {
                if (ptr != IntPtr.Zero)
                {
                    Marshal.FreeHGlobal(ptr);
                    ptr = IntPtr.Zero;
                }
            }
        }

        public static void ex04()
        {
            using (MyResourceManager rm = new MyResourceManager())
            {
                // Używamy zasobu ...
            }
            // Automatycznie zwolniony w metodzie Dispose obiektu rm
        }

        // Przykład 5: prosty bezpieczny uchwyt do niezarządzanej pamięci
        class HGlobalMemorySafeHandle : SafeHandleZeroOrMinusOneIsInvalid
        {
            public HGlobalMemorySafeHandle(int bytes) : base(true)
            {
                this.SetHandle(Marshal.AllocHGlobal(bytes));
            }

            [ReliabilityContract(Consistency.WillNotCorruptState, Cer.Success)]
            protected override bool ReleaseHandle()
            {
                Marshal.FreeHGlobal(handle);
                return true;
            }
        }

        class MySafeHandleWrapper : IDisposable
        {
            private SafeHandle memoryHandle;

            public MySafeHandleWrapper()
            {
                memoryHandle = new HGlobalMemorySafeHandle(1024);
            }

            public void Dispose()
            {
                SafeHandle handle = memoryHandle;
                if (handle != null && !handle.IsClosed)
                    handle.Dispose();
            }
        }

        // Przykład 6: dodawanie presji pamięciowej do powyższej nakładki
        sealed class MyResourceManager2 : IDisposable
        {
            private IntPtr ptr;

            public MyResourceManager2()
            {
                ptr = Marshal.AllocHGlobal(1024);
                GC.AddMemoryPressure(1024);
            }

            ~MyResourceManager2()
            {
                Dispose(false);
            }

            public void Dispose()
            {
                Dispose(true);
                GC.SuppressFinalize(this);
            }

            private void Dispose(bool disposing)
            {
                if (ptr != IntPtr.Zero)
                {
                    Marshal.FreeHGlobal(ptr);
                    ptr = IntPtr.Zero;
                    GC.RemoveMemoryPressure(1024);
                }
            }
        }

        public static void ex06()
        {
            using (MyResourceManager2 rm = new MyResourceManager2())
            {
                // Używamy zasobu ...
            }
               // Automatycznie zwolniony w metodzie Dispose obiektu rm
        }

        /*** INTEROPERACYJNOŚC Z COM ***/

        // Przykład 7: współpraca z wywołaniami API ADO
        private const bool enableEx07 = false;
        public static void ex07()
        {
            if (enableEx07)
            {
                Connection cn = new ConnectionClass();
                try
                {
                    cn.Open("MyConnectionString", "foo", "bar", 0);
                    Command cmd = new CommandClass();
                    try
                    {
                        cmd.ActiveConnection = cn;
                        cmd.CommandText = "SELECT ...";
                        // ...
                    }
                    finally
                    {
                        Marshal.ReleaseComObject(cmd);
                    }
                }
                finally
                {
                    if (cn.State != 0)
                        cn.Close();
                    Marshal.ReleaseComObject(cn);
                }
            }
        }

        // Przykład 8: współpraca z ADO, późne wiązanie
        // (ten sam efekt co w przykładzie 7, ale kod jest wiązany późno, a nie statycznie).
        private const bool enableEx08 = false;
        public static void ex08()
        {
            if (enableEx08)
            {
                Type cnType = Type.GetTypeFromProgID("ADODB.Connection");
                object cn = Activator.CreateInstance(cnType);
                try
                {
                    object[] args = new object[] { "MyConnectionString", "foo", "bar", 0 };
                    cnType.InvokeMember("Open", BindingFlags.InvokeMethod, null, cn, args);

                    Type cmdType = Type.GetTypeFromProgID("ADODB.Command");
                    object cmd = Activator.CreateInstance(cmdType);
                    try
                    {
                        cmdType.InvokeMember("ActiveConnection", BindingFlags.SetProperty,
                            null, cmd, new object[] { cn });
                        cmdType.InvokeMember("CommandText", BindingFlags.SetProperty, null,
                            cmd, new object[] { "SELECT ..." });
                        // ...
                    }
                    finally
                    {
                        Marshal.ReleaseComObject(cmd);
                    }
                }
                finally
                {
                    if ((int)cnType.InvokeMember("State", BindingFlags.GetProperty, null, cn, null) != 0)
                        cnType.InvokeMember("Close", BindingFlags.InvokeMethod, null, cn, null);
                    Marshal.ReleaseComObject(cn);
                }
            }
        }

        /*** PRACA Z KODEM NIEZARZĄDZANYM ***/

        // Przykład 9: używanie P/Invoke (do sprawdzenia ilości wolnego miejsca na dysku)
        [DllImport("kernel32.dll", SetLastError = true, CharSet = CharSet.Auto)]
        static extern bool GetDiskFreeSpaceEx(string lpDirectoryName,
           out ulong lpFreeBytesAvailable,
           out ulong lpTotalNumberOfBytes,
           out ulong lpTotalNumberOfFreeBytes);

        public static void ex09()
        {
            ulong freeBytesAvail;
            ulong totalNumOfBytes;
            ulong totalNumOfFreeBytes;

            if (!GetDiskFreeSpaceEx(@"C:\", out freeBytesAvail, out totalNumOfBytes, out totalNumOfFreeBytes))
            {
                Console.Error.WriteLine("Wystąpił błąd: {0}",
                    Marshal.GetExceptionForHR(Marshal.GetLastWin32Error()).Message);
            }
            else
            {
                Console.WriteLine("Wolne miejsce na dysku:");
                Console.WriteLine("    Dostępna liczba bajtów : {0}", freeBytesAvail);
                Console.WriteLine("    Łączna liczba bajtów: {0}", totalNumOfBytes);
                Console.WriteLine("    Liczba wolnych bajtów: {0}", totalNumOfFreeBytes);
            }
        }

    }

}
