using System;
using System.Collections.Generic;
using System.IO;
using System.IO.IsolatedStorage;
using System.Net;
using System.Net.Sockets;
using System.Reflection;
using System.Security;
using System.Security.AccessControl;
using System.Security.Policy;
using System.Security.Permissions;
using System.Security.Principal;
using System.Text;
using System.Threading;

namespace Wrox.DotNetFramework2.Samples
{
    public class Chapter09
    {
        public static void _title()
        {
            Console.WriteLine("Rozdzia 9 - Bezpieczestwo");
            Console.WriteLine("===========================");
        }

        /*** ZABEZPIECZENIA DOSTPU DO KODU ***/

        // Inicjalizacja: tworzymy zamknit domen AppDomain do dalszych przykadw
        private static AppDomain sandboxAd;

        public static void __init()
        {
            Evidence ev = new Evidence();
            ev.AddAssembly(Assembly.GetExecutingAssembly());
            PermissionSet permSet = new PermissionSet(PermissionState.None);
            permSet.AddPermission(new SecurityPermission(PermissionState.Unrestricted));
            //permSet.AddPermission(new FileIOPermission(PermissionState.Unrestricted));
            //permSet.AddPermission(new ReflectionPermission(PermissionState.Unrestricted));
            sandboxAd = AppDomain.CreateDomain("Sandbox_ch09", ev, new AppDomainSetup(), permSet);
        }

        // Przykad 1: operacje chronione
        public static void ex01()
        {
            sandboxAd.DoCallBack(delegate
            {
                byte[] fileContents = File.ReadAllBytes(@"C:\foo\jakiswaznyplik.txt");
                Socket s = new Socket(AddressFamily.InterNetwork, SocketType.Stream, ProtocolType.IPv4);
                s.Connect("http://www.bluebytesoftware.com/", 666);
                s.Send(fileContents);
                s.Close();
            });
        }

        // Przykad 2: tworzenie i wywietlanie przykadowej polityki
        public static void ex02()
        {
            // Tworzymy polityk odnoszc si do grupy
            PolicyStatement policy = new PolicyStatement(
                new PermissionSet(PermissionState.Unrestricted), new PolicyStatementAttribute());

            // Konstruujemy grup macierzyst  wymagamy, aby czonkowie naleeli do strefy internetowej
            UnionCodeGroup group = new UnionCodeGroup(
                new ZoneMembershipCondition(SecurityZone.Internet), policy);
            group.Name = "MicrosoftInternet";

            // Teraz dodajemy grup potomn  wymagamy, aby czonkowie naleeli do domeny microsoft.com
            UnionCodeGroup childGroup = new UnionCodeGroup(
                new UrlMembershipCondition("http://microsoft.com/"), policy);
            group.AddChild(childGroup);

            Console.WriteLine(group.ToXml().ToString());
        }

        // Przykad 3: izolowana przestrze dyskowa
        public static void ex03()
        {
            using (TextWriter sw = new StreamWriter(
                new IsolatedStorageFileStream("Tekst.txt", FileMode.OpenOrCreate)))
            {
                // praca z obiektem zapisujcym tekst
            }
        }

        // Przykad 4: zbiory uprawnie
        public static void ex04()
        {
            NamedPermissionSet ps = 
                new NamedPermissionSet("SamplePermissionSet", PermissionState.None);
            ps.AddPermission(new FileIOPermission(
                FileIOPermissionAccess.AllAccess, @"C:\temp\"));
            ps.AddPermission(new ReflectionPermission(PermissionState.Unrestricted));
            ps.AddPermission(new SecurityPermission(SecurityPermissionFlag.Execution));

            Console.WriteLine(ps.ToXml().ToString());
        }

        // Przykad 5: wymagania w zabezpieczeniach deklaratywnych
        [FileIOPermission(SecurityAction.Demand, Read=@"C:\temp\")]
        private static void SecureOperationDeclarative()
        {
            // Wykonujemy chronion operacj:
            Console.WriteLine("Wywoano chronion operacj");
        }

        public static void ex05()
        {
            sandboxAd.DoCallBack(SecureOperationDeclarative);
        }

        // Przykad 6: wymagania w zabezpieczeniach imperatywnych
        private static void SecureOperationImperative()
        {
            // Sprawdzamy zezwolenia w sposb imperatywny. Zauwamy, e parametry mog si
            // rni w zalenoci od argumentw funkcji, inaczej ni w zabezpieczeniach
            // deklaratywnych.
            FileIOPermission p = new FileIOPermission(FileIOPermissionAccess.Read, @"C:\temp\");
            p.Demand();

            // Wykonujemy chronion operacj:
            Console.WriteLine("Wywoano chronion operacj");
        }

        public static void ex06()
        {
            sandboxAd.DoCallBack(SecureOperationImperative);
        }

        // Przykad 7: wymagania cza
        [ReflectionPermission(SecurityAction.LinkDemand)]
        private static object InvokePrivately(object o, string m)
        {
            return o.GetType().InvokeMember(m,
                BindingFlags.NonPublic|BindingFlags.Instance|BindingFlags.InvokeMethod,
                null, o, new object[0]);
        }

        public static void ex07()
        {
            // Gdyby przykad ex07 nie spenia wymaga cza, ten kod nie zostaby
            // skompilowany JIT. Jeli jednak "wrox20" jest w peni zaufany,
            // kod zadziaa.
            string s = "Witaj wiecie";
            object x = InvokePrivately(s, "IsAscii");
            Console.WriteLine(x);
        }

        // Przykad 8: wymagania dziedziczenia
        [ReflectionPermission(SecurityAction.InheritanceDemand)]
        class InheritanceDemandType {}

        // Gdyby "wrox20" nie mia odpowiednich zezwole, nie moglibymy
        // utworzy podklasy typu InheritanceDemandType
        class InheritanceDemandSubtype : InheritanceDemandType {}

        // Przykad 9: asercje
        interface IFoo
        {
            void Bar();
        }

        [SecurityPermission(SecurityAction.Deny, SkipVerification=true)]
        private static void F(IFoo f)
        {
            G();
            f.Bar();
        }

        private static void G()
        {
            SecurityPermission p = new SecurityPermission(SecurityPermissionFlag.UnmanagedCode);
            p.Assert();
            try
            {
                // Robimy co z kodem niezarzdzanym... na przykad wykonujemy wywoanie P/Invoke
            }
            finally
            {
                CodeAccessPermission.RevertAssert();
            }

            // Wykonujemy jakie inne operacje
        }

        /*** ZABEZPIECZENIA OPARTE NA TOSAMOCI UYTKOWNIKW ***/

        // Przykad 10: sprawdzanie rl
        public static void ex10()
        {
            IPrincipal user = Thread.CurrentPrincipal;
            if (!user.IsInRole("Administratorzy"))
                Console.WriteLine("Uytkownik nie jest administratorem");
            else
                Console.WriteLine("Uytkownik jest administratorem");
        }

        // Przykad 11: przybieranie tosamoci
        public static void ex11()
        {
            // Oczywicie metoda GetCurrent nie jest zbyt uyteczna. Jeli mamy
            // przybiera inn tosamo, powinnimy znale jak bardziej interesujc.
            WindowsIdentity identity = WindowsIdentity.GetCurrent();
            WindowsImpersonationContext context = identity.Impersonate();
            try
            {
                // Wykonujemy jakie uprzywilejowane informacje...
                Console.WriteLine("Dziaania z przybran tosamoci...");
            }
            finally
            {
                context.Undo();
                Console.WriteLine("Przywrcono poprzedni tosamo.");
            }
        }

        // Przykad 12: plikowe listy ACL
        public static void ex12()
        {
            FileSecurity acl = File.GetAccessControl(@"C:\Windows\");
            AuthorizationRuleCollection rules = acl.GetAccessRules(true, true, typeof(NTAccount));
            foreach (AuthorizationRule rule in rules)
            {
                Console.WriteLine(rule.IdentityReference);
                FileSystemAccessRule accessRule = rule as FileSystemAccessRule;
                if (accessRule != null)
                    Console.WriteLine("  ...{0}", accessRule.FileSystemRights);
            }
        }

        // Przykad 13: tworzenie nowego pliku z konkretnymi wpisami ACL
        public static void ex13()
        {
            string fileName = "c09_acl_02.txt";
            if (File.Exists(fileName))
                File.Delete(fileName);

            FileSecurity acl = new FileSecurity();
            acl.AddAccessRule(new FileSystemAccessRule(WindowsIdentity.GetCurrent().Name,
                FileSystemRights.CreateFiles | FileSystemRights.Modify | FileSystemRights.Delete,
                AccessControlType.Allow));
            using (FileStream file = File.Open(fileName, FileMode.OpenOrCreate))
            {
                // wypisywanie zawartoci pliku
            }
        }
    }
}
