﻿using System;
using System.Collections.Generic;
using System.Globalization;
using System.IO;
using System.Text;
using System.Security.Cryptography;

namespace Wrox.DotNetFramework2.Samples
{
    public class Chapter05
    {
        public void _title()
        {
            Console.WriteLine("Rozdział 5. Najważniejsze typy .NET");
            Console.WriteLine("===================================");
        }

        /*** TYPY PODSTAWOWE ***/

        // Przykład 1: równość typów referencyjnych
        class Ex01Person
        {
            public string Name;
            public int Age;

            public Ex01Person(string name, int age)
            {
                Name = name;
                Age = age;
            }

            public override bool Equals(object obj)
            {
                // Proste testy...
                if (obj == this)
                    return true;
                if (obj == null || !obj.GetType().Equals(typeof(Ex01Person)))
                    return false;

                // Teraz porównujemy poszczególne składowe:
                Ex01Person p = (Ex01Person)obj;
                return this.Name == p.Name && this.Age == p.Age;
            }
        }

        public static void ex01()
        {
            Ex01Person p1 = new Ex01Person("Bartek", 55);
            Ex01Person p2 = new Ex01Person("Bartek", 55);
            Console.WriteLine(p1 == p2); // wypisuje 'false'
            Console.WriteLine(p1.Equals(p2)); // wypisuje 'true'
        }

        // Przykład 2: równość typów wartościowych (działanie domyślne)
        struct Ex02Car
        {
            public string Make;
            public string Model;
            public uint Year;

            public Ex02Car(string make, string model, uint year)
            {
                Make = make;
                Model = model;
                Year = year;
            }
        }

        public static void ex02()
        {
            Ex02Car c1 = new Ex02Car("BMW", "330Ci", 2001);
            Ex02Car c2 = new Ex02Car("BMW", "330Ci", 2001);
            Console.WriteLine(c1.Equals(c2)); // wypisuje 'true'
        }

        // Przykład 3: równość typów wartoścuowych (przesłonięcie)
        struct Ex03Car
        {
            public string Make;
            public string Model;
            public uint Year;

            public Ex03Car(string make, string model, uint year)
            {
                Make = make;
                Model = model;
                Year = year;
            }

            public bool Equals(Ex03Car c)
            {
                return c.Make == this.Make &&
                    c.Model == this.Model &&
                    c.Year == this.Year;
            }

            public override bool Equals(object obj)
            {
                if (obj is Ex03Car)
                    return Equals((Ex03Car)obj);
                return false;
            }
        }

        public static void ex03()
        {
            Ex03Car c1 = new Ex03Car("Audi", "S4", 2005);
            Ex03Car c2 = new Ex03Car("Audi", "S4", 2005);
            Console.WriteLine(c1.Equals(c2)); // wypisuje 'true'
        }

        // Przykład 4: prosty wzorzec IDisposable
        class Ex04MyClass : IDisposable
        {
            private Stream myStream = null/*…*/;
            // …
            public void Dispose()
            {
                Stream s = myStream;
                if (s != null)
                    ((IDisposable)s).Dispose();
            }
        }

        public static void ex04()
        {
            using (Ex05MyClass mc = new Ex05MyClass())
            {
                // Używamy 'mc'
            }
        }

        // Przykład 5: złożony wzorzec IDisposable
        class Ex05MyClass : IDisposable
        {
            private IntPtr myHandle = IntPtr.Zero/*…*/;
            // …
            ~Ex05MyClass()
            {
                Dispose(false);
            }
            public void Dispose()
            {
                Dispose(true);
                GC.SuppressFinalize(this);
            }
            protected void Dispose(bool disposing)
            {
                IntPtr h = myHandle;
                if (h != IntPtr.Zero)
                {
                    //CloseHandle(h);
                    h = IntPtr.Zero;
                }
            }
        }

        public static void ex05()
        {
            using (Ex05MyClass mc = new Ex05MyClass())
            {
                // Używamy 'mc'
            }
        }

        // Przykład 6: tożsamość typu
        public static void ex06()
        {
            string s = "Instancja łańcucha...";
            Type t = s.GetType();
            Console.WriteLine(t.Name);              // "String"
            Console.WriteLine(t.Namespace);         // "System"
            Console.WriteLine(t.IsPublic);          // "True"
            Console.WriteLine(t == typeof(string)); // "True"
        }

        // Przykład 7: ToString
        class Ex07Person
        {
            public string Name;
            public int Age;

            public override string ToString()
            {
                return String.Format("{0}[Name={1}, Age={2}]",
                    base.ToString(), Name, Age);
            }
        }

        public static void ex07()
        {
            Ex07Person p = new Ex07Person();
            p.Name = "Janek";
            p.Age = 16;
            Console.WriteLine(p.ToString());
        }

        // Przykład 8: zakresy liczb całkowitych
        public static void ex08()
        {
            Console.WriteLine("{0}: bajty: {1}, zakres: [{2},{3}]",
                typeof(byte).ToString(), sizeof(byte), byte.MinValue, byte.MaxValue);
            Console.WriteLine("{0}: bajty: {1}, zakres: [{2},{3}]",
                typeof(char).ToString(), sizeof(char), (int)char.MinValue, (int)char.MaxValue);
            Console.WriteLine("{0}: bajty: {1}, zakres: [{2},{3}]",
                typeof(short).ToString(), sizeof(short), short.MinValue, short.MaxValue);
            Console.WriteLine("{0}: bajty: {1}, zakres: [{2},{3}]",
                typeof(int).ToString(), sizeof(int), int.MinValue, int.MaxValue);
            Console.WriteLine("{0}: bajty: {1}, zakres: [{2},{3}]",
                typeof(long).ToString(), sizeof(long), long.MinValue, long.MaxValue);
            Console.WriteLine("{0}: bajty: {1}, zakres: [{2},{3}]",
                typeof(sbyte).ToString(), sizeof(sbyte), sbyte.MinValue, sbyte.MaxValue);
            Console.WriteLine("{0}: bajty: {1}, zakres: [{2},{3}]",
                typeof(ushort).ToString(), sizeof(ushort), ushort.MinValue, ushort.MaxValue);
            Console.WriteLine("{0}: bajty: {1}, zakres: [{2},{3}]",
                typeof(uint).ToString(), sizeof(uint), uint.MinValue, uint.MaxValue);
            Console.WriteLine("{0}: bajty: {1}, zakres: [{2},{3}]",
                typeof(ulong).ToString(), sizeof(ulong), ulong.MinValue, ulong.MaxValue);
        }

        // Przykład 9: znaki specjalne
        public static void ex09()
        {
            Console.WriteLine("Apostrof \\': {0} ({1})", '\'', (int)'\'');
            Console.WriteLine("Cudzysłów \\\": {0} ({1})", '\"', (int)'\"');
            Console.WriteLine("Ukośnik lewy \\\\: {0} ({1})", '\\', (int)'\\');
            Console.WriteLine("Znak pusty \\0: {0} ({1})", '\0', (int)'\0');
            Console.WriteLine("Alarm \\a: {0} ({1})", '\a', (int)'\a');
            Console.WriteLine("Cofnięcie \\b: {0} ({1})", '\b', (int)'\b');
            Console.WriteLine("Tabulacja pozioma \\t: {0} ({1})", '\t', (int)'\t');
            Console.WriteLine("Nowy wiersz \\r: {0} ({1})", '\r', (int)'\r');
            Console.WriteLine("Tabulacja pionowa \\v: {0} ({1})", '\v', (int)'\v');
            Console.WriteLine("Wysuw arkusza \\f: {0} ({1})", '\f', (int)'\f');
            Console.WriteLine("Powrót karetki \\r: {0} ({1})", '\r', (int)'\r');
        }

        // Przykład 10: zakresy liczb zmiennopozycyjnych i dziesiętnych
        public static void ex10()
        {
            Console.WriteLine("{0}: bajty: {1}, zakres: [{2},{3}]",
                typeof(float).ToString(), sizeof(float), float.MinValue, float.MaxValue);
            Console.WriteLine("{0}: bajty: {1}, zakres: [{2},{3}]",
                typeof(double).ToString(), sizeof(double), double.MinValue, double.MaxValue);
            Console.WriteLine("{0}: bajty: {1}, zakres: [{2},{3}]",
                typeof(decimal).ToString(), sizeof(decimal), decimal.MinValue, decimal.MaxValue);
        }

        // Przykład 11: tworzenie łańcuchów
        public static void ex11()
        {
            string s1 = "Jestem łańcuchem, do tego dość efektownym.";
            string s2 = String.Empty;
            string s3 = "";

            Console.WriteLine("s1: {0}", s1);
            Console.WriteLine("s2: {0}", s2);
            Console.WriteLine("s3: {0}", s3);
        }

        // Przykład 12: konkatenacja
        public static void ex12()
        {
            string s = "Witaj, ";
            string w = "świecie";
            string sw = s + w;
            Console.WriteLine(sw);
        }

        // Przykład 13: konkatenacja z wartościami niełańcuchowymi
        public static void ex13()
        {
            int number = 10;
            string msg = "Janek ma " + number + " lat.";
            msg += " To wspaniale, prawda?";
            Console.WriteLine("msg: {0}", msg);

            String s1 = 10 + 5 + ": dwa plus trzy równa się " + 2 + 3;
            String s2 = 10 + 5 + ": dwa plus trzy równa się " + (2 + 3);
            Console.WriteLine("s1: {0}", s1);
            Console.WriteLine("s2: {0}", s2);
        }

        // Przykład 14: formatowanie
        public static void ex14()
        {
            {
                int age = 25;
                string name = "Janek";
                Console.WriteLine(String.Format("{0} ma {1} lat.", name, age));
            }

            {
                string name = "Hej";
                Console.WriteLine(String.Format("{0} tam. Powiedziałem: {0}! {0}???", name));
            }

            {
                int age = 50;
                string name = "Aleksander";
                Console.WriteLine("{0} ma {1} lat.", name, age);
            }

            Console.WriteLine("{0:X}", 29);
        }

        // Przykład 15: dostęp do zawartości łańcucha
        public static void ex15()
        {
            string s = "Witaj, świecie.";

            // Użycie indeksatora:
            for (int i = 0; i < s.Length; i++)
            {
                char c = s[i];
                // Robimy coś z 'c'...
            }

            // Użycie wyliczenia:
            foreach (char c in s)
            {
                // Robimy coś z 'c'...
            }

            // Uzyskiwanie tablicy char[] z łańcucha:
            char[] c1 = s.ToCharArray();
            char[] c2 = s.ToCharArray(0, 5);
        }

        // Przykład 16: porównywanie łańcuchów
        public static void ex16()
        {
            bool b1 = "witaj" == "witaj";      // True
            Console.WriteLine(b1);
            bool b2 = "witaj" == "hej";        // False
            Console.WriteLine(b2);
            bool b3 = "witaj".Equals("witaj"); // True
            Console.WriteLine(b3);
            bool b4 = "witaj".Equals("hej");   // False
            Console.WriteLine(b4);
            bool b5 = "WiTaJ".Equals("witaj"); // False
            Console.WriteLine(b5);
            bool b6 = "WiTaJ".Equals("witaj",  // True
                StringComparison.OrdinalIgnoreCase); 
            Console.WriteLine(b6);
        }

        // Przykład 17: metody StartsWith i EndsWith.
        public static void ex17()
        {
            // Uwaga: w poniższych łańcuchach używamy znaku unikowego '@'. Jest to
            // mechanizm języka C#, dzięki któremu ukośniki lewe (\) nie są traktowane
            // jak znaki unikowe. Przydaje się to do zapisu ścieżek.

            string path = @"C:\Program Files\Moja aplikacja\PewienPlik.XML";

            // StartsWith:
            bool b1 = path.StartsWith(@"C:\"); // True
            Console.WriteLine(b1);
            bool b2 = path.StartsWith(@"c:\"); // False
            Console.WriteLine(b2);
            bool b3 = path.StartsWith(@"c:\", true, null); // True
            Console.WriteLine(b3);

            // EndsWith:
            bool b4 = path.EndsWith(".XML"); // True
            Console.WriteLine(b4);
            bool b5 = path.EndsWith(".xml"); // False
            Console.WriteLine(b5);
            bool b6 = path.EndsWith(".xml", true, null); // True
            Console.WriteLine(b6);
        }

        // Przykład 18: zmiana wielkości liter
        public static void ex18()
        {
            string s = "Mój mały String.";
            Console.WriteLine(s);
            Console.WriteLine(s.ToUpper());
            Console.WriteLine(s.ToLower());
            Console.WriteLine(s);
        }

        // Przykład 19: zastępowanie lub usuwanie zawartości łańcucha
        public static void ex19()
        {
            string s1 = "Jeśli chcesz zrozumieć, musisz słuchać uważnie...";
            Console.WriteLine(s1.Replace('z', 's'));
            Console.WriteLine(s1.Replace("sz", "my"));

            string s2 = "Jestem dziś nieszczęśliwy!";
            Console.WriteLine(s2);
            Console.WriteLine(s2.Remove(6));
            Console.WriteLine(s2.Remove(12,3));
        }

        // Przykład 20: przycinanie łańcucha
        public static void ex20()
        {
            string s1 = "    Mój łańcuch jest otoczony spacjami!    ";
            Console.WriteLine(s1);
            Console.WriteLine(s1.Trim());

            string s2 = "__...,Hej tam, kolego!,..._";
            Console.WriteLine(s2);
            char[] trimChars = new char[] { '.', ',', '_' };
            Console.WriteLine(s2.Trim(trimChars));
            Console.WriteLine(s2.TrimStart(trimChars));
            Console.WriteLine(s2.TrimEnd(trimChars));
        }

        // Przykład 21: dopełnianie łańcuchów

        enum PrintJustification
        {
            Left,
            Right,
            Center
        }

        static void PrintJustified(string s, int width, PrintJustification just)
        {
            int diff = width - s.Length;
            if (diff > 0)
            {
                switch (just)
                {
                    case PrintJustification.Left:
                        Console.WriteLine(s.PadRight(width, '-'));
                        break;
                    case PrintJustification.Right:
                        Console.WriteLine(s.PadLeft(width, '-'));
                        break;
                    case PrintJustification.Center:
                        s = s.PadLeft(s.Length + (diff / 2), '-');
                        s = s.PadRight(width, '-');
                        Console.WriteLine(s);
                        break;
                }
            }
        }
        
        public static void ex21()
        {
            string s = "Dopełnijcie mnie, proszę";
            Console.WriteLine(s.PadRight(30, '.'));
            
            PrintJustified("Tekst wyrównany do prawej", 60, PrintJustification.Right);
            PrintJustified("Tekst wyrównany do lewej", 60, PrintJustification.Left);
            PrintJustified("Tekst wyrównany do środka", 60, PrintJustification.Center);
        }


        // Przykład 22: podłańcuchy
        public static void ex22()
        {
            string s = "Mój mały, śliczny, łańcuchowy łańcuch.";
            Console.WriteLine(s.Substring(10, 7));
            Console.WriteLine(s.Substring(19));
        }

        // Przykład 23: dzielenie łańcuchów
        public static void ex23()
        {
            string s = "Joe Duffy|Microsoft|Menedżer projektu|Zespół CLR";
            string[] pieces = s.Split('|');
            foreach (string piece in pieces)
            {
                Console.WriteLine(piece);
            }
        }

        // Przykład 24: łączenie łańcuchów
        public static void ex24()
        {
            string[] s = new string[] { "Joe Duffy", "Microsoft",
                "Menedżer projektu", "Zespół CLR" };
            string joined = String.Join("|", s);
            Console.WriteLine(joined);
        }

        // Przykład 25: tworzenie obiektu DateTime
        public static void ex25()
        {
            // Pobieranie bieżącej daty i czasu:
            DateTime localNow = DateTime.Now;
            Console.WriteLine("{0} – {1} ({2})", localNow, localNow.Kind,
                TimeZone.CurrentTimeZone.StandardName);
            DateTime utcNow = DateTime.UtcNow;
            Console.WriteLine("{0} – {1}", utcNow, utcNow.Kind);

            // Tworzenie określonych dat i czasów:
            DateTime dt1 = new DateTime(2004, 10, 19);
            Console.WriteLine("dt1: {0}", dt1);
            DateTime dt2 = new DateTime(2004, 10, 19, 22, 47, 35);
            Console.WriteLine("dt2: {0}", dt2);
            DateTime dt3 = new DateTime(2004, 10, 19, 22, 47, 35, 259);
            Console.WriteLine("dt3: {0}", dt3);
        }

        // Przykład 26: interesujące właściwości daty i czasu
        public static void ex26()
        {
            DateTime dt1 = new DateTime(2004, 10, 19, 22, 47, 35, 259);
            Console.WriteLine("Day: {0}", dt1.Day);
            Console.WriteLine("DayOfWeek: {0}", dt1.DayOfWeek);
            Console.WriteLine("DayOfYear: {0}", dt1.DayOfYear);
            Console.WriteLine("Hour: {0}", dt1.Hour);
            Console.WriteLine("Millisecond: {0}", dt1.Millisecond);
            Console.WriteLine("Minute: {0}", dt1.Minute);
            Console.WriteLine("Month: {0}", dt1.Month);
            Console.WriteLine("Second: {0}", dt1.Second);
            Console.WriteLine("Year: {0}", dt1.Year);
        }

        /*** POMOCNICZE KLASY BCL ***/

        // Przykład 27: specyfikatory formatowania liczbowego
        public static void ex27()
        {
            Console.WriteLine("C: {0}", 39.22M.ToString("C"));
            Console.WriteLine("D: {0}", 982L.ToString("D"));
            Console.WriteLine("E: {0}", 3399283712.382387D.ToString("E"));
            Console.WriteLine("F: {0}", .993F.ToString("F"));
            Console.WriteLine("G: {0}", 32.559D.ToString("G"));
            Console.WriteLine("N: {0}", 93823713.ToString("N"));
            Console.WriteLine("P: {0}", .59837.ToString("P"));
            Console.WriteLine("R: {0}", 99.33234D.ToString("R"));
            Console.WriteLine("X: {0}", 369329.ToString("X"));
            Console.WriteLine("Format własny: {0}", 23369.ToString("### ###.00 złotych"));
        }

        // Przykład 28: wbudowane specyfikatory daty i czasu
        public static void ex28()
        {
            DateTime dt1 = new DateTime(2004, 10, 9, 22, 47, 35, 259);
            Console.WriteLine(dt1.ToString("yyyy MMM"));
            DateTimeFormatInfo di = CultureInfo.CurrentCulture.DateTimeFormat;
            for (char c = 'a'; c <= 'z'; c++)
            {
                try
                {
                    foreach (string s in di.GetAllDateTimePatterns(c))
                    {
                        Console.WriteLine("'{0}': {1} - {2}/{3}", c, s,
                            dt1.ToString(c.ToString()), dt1.ToString(s));
                    }
                    char cUpper = Char.ToUpper(c);
                    foreach (string s in di.GetAllDateTimePatterns(cUpper))
                    {
                            Console.WriteLine("'{0}': {1} - {2}", cUpper, s,
                            dt1.ToString(cUpper.ToString()), dt1.ToString(s));
                    }
                }
                catch (ArgumentException)
                {
                    // Ignorujemy -- nie ma takiego specyfikatora
                }
            }
        }

        // Przykład 29: analiza składniowa
        public static void ex29()
        {
            int i = int.Parse("550");
            float f = float.Parse("21,99328");
            bool b = bool.Parse("True");
            int j = int.Parse("zły format"); // Zgłasza wyjątek FormatException…
        }

        // Przykład 30: analiza w starym stylu (.NET 1.0 i v1.1)
        public static void ex30()
        {
            string s = "zły format";
            int i;
            try
            {
                i = int.Parse(s);
                // Używamy 'i'...
            }
            catch (ArgumentException)
            {
                // Obsługujemy błąd, tzn. informujemy użytkownika, że wprowadził błędne dane
                Console.WriteLine("Błędne dane wejściowe");
            }
        }

        // Przykład 31: nowy styl analizy TryParse (.NET 2.0)
        public static void ex31()
        {
            string s = "zły format";
            int i;
            if (int.TryParse(s, out i))
            {
                // Używamy 'i'...
            }
            else
            {
                // Obsługujemy błąd, tzn. informujemy użytkownika, że wprowadził błędne dane
                Console.WriteLine("Błędne dane wejściowe");
            }
        }

        // Przykład 32: budowanie nowego łańcucha za pomocą klasy StringBuilder
        public static void ex32()
        {
            StringBuilder sb = new StringBuilder("Foo");
            sb.Append("Bar");
            sb.AppendFormat("...{0}", DateTime.Now);

            string final = sb.ToString();
            Console.WriteLine(final);
        }

        // Przykład 33: odśmiecanie
        public static void ex33()
        {
            Console.WriteLine("Łączna pamięć: {0}", GC.GetTotalMemory(false));
            Console.WriteLine("(wymuszone odśmiecanie...)");
            GC.Collect();
            Console.WriteLine("Łączna pamięć: {0}", GC.GetTotalMemory(false));
        }

        // Przykład 34: presja pamięciowa
        class UnmanagedWrapper : IDisposable
        {
            private IntPtr handle;
            private long size;

            public UnmanagedWrapper()
            {
                size = 100*1024*1024;
                handle = IntPtr.Zero/*AllocateUnmanagedMB(size)*/;
                GC.AddMemoryPressure(size);
            }

            ~UnmanagedWrapper()
            {
                Dispose(false);
            }

            public void Dispose()
            {
                Dispose(true);
                GC.SuppressFinalize(this);
            }

            protected void Dispose(bool disposing)
            {
                GC.RemoveMemoryPressure(size);
                /*FreeUnmanagedMB(handle, size);*/
            }
        }

        // Przykład 35: słabe referencje
        public static void ex35()
        {
            object target = new object();
            WeakReference wr = new WeakReference(target);
            //...
            Console.WriteLine("IsAlive: {0}", wr.IsAlive);
            GC.Collect();
            Console.WriteLine("IsAlive: {0}", wr.IsAlive);
        }

        // Przykład 36: wywołania matematyczne
        public static void ex36()
        {
            Console.WriteLine("Abs({0}) = {1}", -55, Math.Abs(-55));
            Console.WriteLine("Ceiling({0}) = {1}", 55.3, Math.Ceiling(55.3));
            Console.WriteLine("Pow({0},{1}) = {2}", 10.5, 3, Math.Pow(10.5, 3));
            Console.WriteLine("Round({0},{1}) = {2}",
                10.55358, 2, Math.Round(10.55358, 2));
            Console.WriteLine("Sin({0}) = {1}", 323.333, Math.Sin(323.333));
            Console.WriteLine("Cos({0}) = {1}", 323.333, Math.Cos(323.333));
            Console.WriteLine("Tan({0}) = {1}", 323.333, Math.Tan(323.333));
        }

        // Przykład 37: generowanie liczb losowych
        public static void ex37()
        {
            // Proste generowanie liczb losowych:
            Random r1 = new Random();
            Console.WriteLine(r1.Next());
            Console.WriteLine(r1.Next(150000));
            Console.WriteLine(r1.Next(9999, 100750));
            Console.WriteLine(r1.NextDouble());

            // Demonstracja przewidywalności (zawsze wypisuje te same liczby):
            Random r2 = new Random(99830123);
            Console.WriteLine(r2.Next());
            Console.WriteLine(r2.Next(150000));
            Console.WriteLine(r2.Next(9999, 100750));
            Console.WriteLine(r2.NextDouble());

            // Liczby losowe bezpieczne kryptograficznie:
            RandomNumberGenerator rng = RandomNumberGenerator.Create();
            byte[] b = new byte[1];
            rng.GetBytes(b);
            Console.WriteLine(b[0]);

            try
            {
                throw new ArgumentNullException("someObject");
            }
            catch (ArgumentNullException a)
            {
                Console.WriteLine(a.Message);
                Console.WriteLine(a.ParamName);
            }
        }

        /*** CZĘSTO UŻYWANE WYJĄTKI ***/

        // Przykład 38: niestandardowy typ wyjątku
        class CustomException : Exception
        {
            public CustomException() : base() {}
            public CustomException(string message) : base(message) {}
            public CustomException(string message, Exception innerException) :
                base(message, innerException) {}
            protected CustomException(System.Runtime.Serialization.SerializationInfo info,
                System.Runtime.Serialization.StreamingContext context) : base(info, context) {}
        }
    }
}