﻿using System;
using System.Collections.Generic;
using System.Reflection;
using System.Runtime.InteropServices;

namespace Wrox.DotNetFramework2.Samples
{
    public class Chapter02
    {
        public static void _title()
        {
            Console.WriteLine("Rozdział 02 - Wspólny system typów");
            Console.WriteLine("==================================");
        }

        /*** WPROWADZENIE DO SYSTEMU TYPÓW ***/

        // Przykład 1: uszkodzenie pamięci w niezarządzanym C
        //     (kod oznaczony komentarzem; wymaga oddzielnego kompilatora)
        /*
            #include <stdlib.h>
            #include <stdio.h>
            
            void fill_buffer(char*, int, char);
            
            int main()
            {
                int  x = 10;
                char buffer[16];
                // ...
                fill_buffer(buffer, 32, 'a');
                // ...
                printf("%d", x);
            }
            
            void fill_buffer(char* buffer, int size, char c)
            {
                int i;
                for (i = 0; i < size; i++)
                {
                    buffer[i] = c;
                }
            }
         */

        // Przykład 2: ciąg Fibonacciego w C#
        static int Fibonacci(int x)
        {
            if (x <= 1)
                return 1;
            return Fibonacci(x - 1) + Fibonacci(x - 2);
        }

        public static void ex02()
        {
            Console.WriteLine(Fibonacci(10));
        }

        // Przykład 3: ciąg Fibonacciego w F#
        //     (kod oznaczony komentarzem; wymaga oddzielnego kompilatora)
        /*
            let rec fibonacci x =
                match x with
                    0 -> 1
                | 1 -> 1
                | n -> fibonacci(x – 1) + fibonacci(x – 2);;
            
            fibonacci 10;;
         */

        // Przykład 4: ciąg Fibonacciego w VB.
        //     (kod oznaczony komentarzem; wymaga oddzielnego kompilatora)
        /*
            Option Explicit Off

            Class Program
                Shared Function Fibonacci(x)
                    If (x <= 1)
                        Return 1
                    End If

                    Return Fibonacci(x - 1) + Fibonacci(x - 2)
                End Function

                Shared Sub Main()
                    Console.WriteLine(Fibonacci(10))
                End Sub
            End Class
         */
          

        // Przykład 5: ciąg Fibonacciego w Pythonie
        //     (kod oznaczony komentarzem; wymaga oddzielnego kompilatora)
        /*
            def fib(i):
                if i <= 1:
                    return 1
                return fib(i-1) + fib(i-2)

            print fib(10)
         */

        // Przykład 6: ciąg Fibonacciego w Scheme
        //     (kod oznaczony komentarzem; wymaga oddzielnego kompilatora)
        /*
            (letrec ((fib (lambda (x)
                            (if (<= x 1)
                                1
                                (+ (fib (- x 1)) (fib (- x 2)))))))
                (fib 10))
         */

        /*** TYPY I OBIEKTY ***/

        // Przykład 7: typ referencyjny
        class Customer
        {
            public string name;
            public string address;
        }

        // Przykład 8: typ wartościowy
        struct Point2d
        {
            public int x;
            public int y;
        }

        // Przykład 9: typy referencyjne i wartościowe (układ w pamięci)
        class SampleClass
        {
            public int x;
            public int y;
        }

        struct SampleStruct
        {
            public int x;
            public int y;
        }

        // Przykład 10: opakowywanie i odpakowywanie
        public static void ex10()
        {
            int x = 10;
            object y = x;
            int z = (int)y;

            Console.WriteLine("{0}...{1}...{2}", x, y, z);
        }

        // Przykład 11: unifikacja wartości null
        public static void ex11()
        {
            int? x1 = null;
            int? x2 = new int?();
            int? y1 = 55;
            int? y2 = new int?(55);

            Console.WriteLine("{0}...{1}...{2}...{3}", x1, x2, y1, y2);

            Console.WriteLine("x1 == null? {0}", x1 == null);
            object x1o = x1; // opakowuje "x", zmieniając je w null
            Console.WriteLine("x1o == null? {0}", x1o == null);

            Console.WriteLine("{0}...{1}", x1.GetType(), x1o.GetType());
        }

        // Przykład 12: zagnieżdżanie klas
        internal class Outer
        {
            private static int state;

            internal class Inner
            {
                void Foo() { state++; }
            }
        }

        // Przykład 13: pola
        class FieldExample
        {
            private static int idCounter;

            protected int id;
            public string name;
            public int x;
            public int y;
            private System.DateTime createDate;
        }

        // Przykład 14: pola tylko do odczytu
        class ReadOnlyFieldExample
        {
            private readonly static int staticData; // Możemy tu ustawić to pole
            static ReadOnlyFieldExample()
            {
                // Pole staticData możemy również ustawić tutaj
            }

            private readonly int instanceData; // Możemy tu ustawić to pole
            public ReadOnlyFieldExample()
            {
                // Pole instanceData możemy również ustawić
                // w jednym z konstruktorów ReadOnlyFieldExample
            }
        }

        // Przykład 15: zmienne pola tylko do odczytu
        class ReadOnlyFieldBadExample
        {
            public static readonly int[] ConstantNumbers = new int[] {
                0, 1, 2, 3, 4, 5, 6, 7, 8, 9
            };
        }

        public static void ex15()
        {
            Console.Write("Przed: ");
            foreach (int c in ReadOnlyFieldBadExample.ConstantNumbers)
                Console.Write("{0} ", c);
            Console.WriteLine();

            // Jakiś szkodliwy kod może nadal zmieniać liczby w tablicy;
            // nie może tylko ustawić zmiennej ConstantNumbers na nową tablicę!
            ReadOnlyFieldBadExample.ConstantNumbers[0] = 9;
            ReadOnlyFieldBadExample.ConstantNumbers[1] = 8;

            Console.Write("Po: ");
            foreach (int c in ReadOnlyFieldBadExample.ConstantNumbers)
                Console.Write("{0} ", c);
            Console.WriteLine();
        }

        // Przykład 16: stałe
        class UsefulConstants
        {
            public const double PI = 3.1415926535897931;

            // Poniższe wyrażenie jest na tyle proste, że kompilator może zredukować je do 4:
            public const int TwoPlusTwo = 2 + 2;
        }

        // Przykład 17: układ jawny
        [StructLayout(LayoutKind.Explicit)]
        struct MyUnion
        {
            [FieldOffset(0)] string someText;
            [FieldOffset(7)] byte additionalData;
            [FieldOffset(6)] byte unionTag; // 0 = a, 1 = b
            [FieldOffset(4)] short unionA;
            [FieldOffset(4)] byte unionB1;
            [FieldOffset(5)] byte unionB2;
        }

        // Przykład 18: metody
        class Calculator
        {
            public int Add(int x, int y)
            {
                return x + y;
            }
        }

        public static void ex18()
        {
            Calculator calc = new Calculator();
            int sum = calc.Add(3, 5);
            Console.WriteLine("3 + 5 = {0}", sum);
        }

        // Przykład 19: jawne użycie this
        class Example19
        {
            int state;
            public void Foo()
            {
                state++;
                this.state++;
            }
        }

        // Przykład 20: zmienne lokalne
        class Example
        {
            public void Foo()
            {
                int x = 0;
                double y = 5.5;
                string s = "Witaj, świecie";
                // ...
            }
        }

        // Przykład 21: przeciążanie i rozwikływanie przeciążeń
        class Foo
        {
            internal void Bar() { Bar(10); /* wartość "domyślna" */ }
            internal void Bar(int x) { /*...*/ }
            internal void Bar(int x, int y) { /*...*/ }
            internal void Bar(object o) { /*...*/ }
            internal void Fooz(object o) { Console.WriteLine("Foo(object)"); }
            internal void Fooz(string s) { Console.WriteLine("Foo(string)"); }
        }

        public static void ex21()
        {
            Foo f = new Foo();
            Console.WriteLine("Wywołuję f.Fooz(\"Witaj, wiązanie!\")...");
            f.Fooz("Witaj, wiązanie!");
        }

        // Przykład 22: styl przekazywania argumentów
        static void Swap<T>(ref T x, ref T y)
        {
            T t = x;
            x = y;
            y = t;
        }

        public static void ex22()
        {
            string x = "Foo";
            string y = "Bar";

            // Wyświetlamy pierwotne wartości x i y:
            Console.WriteLine("x:{0}, y:{1}", x, y);

            // Przestawiamy wartości...
            Swap<string>(ref x, ref y);

            // Teraz wyświetlamy zmodyfikowane wartości x i y:
            Console.WriteLine("x:{0}, y:{1}", x, y);
        }

        // Przykład 23: argumenty wyjściowe
        static void Next3(int a, out int x, out int y, out int z)
        {
            x = a + 1;
            y = a + 2;
            z = a + 3;
        }

        public static void ex23()
        {
            int a, b, c;
            Next3(0, out a, out b, out c);
            Console.WriteLine("a={0}...b={1}...c={2}", a, b, c);
        }

        // Przykład 24: metody o zmiennej liczbie argumentów
        static void PrintOut(params object[] data)
        {
            foreach (object o in data)
                Console.WriteLine(o);
        }

        public static void ex24()
        {
            PrintOut("Witaj", 5, new DateTime(1999, 10, 10));
        }

        // Przykład 25: metody wirtualne i przeciążanie
        class Base25
        {
            public virtual void Foo()
            {
                Console.WriteLine("Base::Foo");
            }
        }

        class Derived25 : Base25
        {
            public override void Foo()
            {
                Console.WriteLine("Derived::Foo");
            }
        }

        public static void ex25()
        {
            // Konstruujemy kilka instancji:
            Base25 baseObj = new Base25();
            Derived25 derivedObj = new Derived25();
            Base25 derivedTypedAsBaseObj = new Derived25();

            // Teraz wywołujemy Foo na każdej z nich:
            baseObj.Foo();
            derivedObj.Foo();
            derivedTypedAsBaseObj.Foo();
        }

        // Przykład 26: nowe sloty
        class Base26
        {
            public virtual void Foo()
            {
                Console.WriteLine("Base::Foo");
            }
        }
        class Derived26 : Base26
        {
            public new void Foo()
            {
                Console.WriteLine("Derived::Foo");
            }
        }

        public static void ex26()
        {
            // Konstruujemy kilka instancji:
            Base26 baseObj = new Base26();
            Derived26 derivedObj = new Derived26();
            Base26 derivedTypedAsBaseObj = new Derived26();

            // Teraz wywołujemy Foo na każdej z nich:
            baseObj.Foo();
            derivedObj.Foo();
            derivedTypedAsBaseObj.Foo();
        }

        // Przykład 27: konstruktory
        class Customer27
        {
            static int idCounter;

            private int myId;
            private string name;

            public Customer27(string name)
            {
                if (name == null)
                    throw new ArgumentNullException("name");
                this.myId = ++idCounter;
                this.name = name;
            }
        }

        // Przykład 28: konstruktory domyślne
        class CtorExample28_1 {}

        class CtorExample28_2
        {
            private int value;
            public CtorExample28_2(int value)
            {
                this.value = value;
            }
        }

        // Przykład 29: łańcuchy konstruktorów
        class CtorExample29
        {
            private int value;
            public CtorExample29(int value)
            {
                this.value = value;
            }
        }

        class CtorExample29Derived : CtorExample29
        {
            private bool wasDefaultCalled;
            private DateTime initializedDate;

            public CtorExample29Derived() : this(0)
            {
                wasDefaultCalled = true;
            }

            public CtorExample29Derived(int value) : base(value * 2)
            {
                initializedDate = DateTime.Now;
            }
        }

        // Przykład 30: inicjalizacja pól
        class FieldInitExample
        {
            int x = 5;
            int y;

            public FieldInitExample() : this(5)
            {
            }

            public FieldInitExample(int y)
            {
                this.y = y;
            }
        }

        // Przykład 31: inicjalizacja pól statycznych
        class CctorExample
        {
            static DateTime classLoadTimestamp = DateTime.Now;
        }

        // Przykład 32: właściwości
        class PropertyExampleBad
        {
            public int X;
            public int Y;
        }

        class PropertyExampleGood
        {
            private int x;
            private int y;

            public int X
            {
                get { return x; }
                set { x = value; }
            }

            public int Y
            {
                get { return y; }
                set { y = value; }
            }
        }

        // Przykład 33: właściwości indeksujące
        class Order33 { /*...*/ }
        class Customer33
        {
            private Dictionary<int, Order33> orders = new Dictionary<int, Order33>();

            public Order33 this[int id]
            {
                get { return orders[id]; }
                set { orders[id] = value; }
            }
        }

        public static void ex33()
        {
            Customer33 c = new Customer33();
            Order33 o = c[10010]; // ten przykład zgłosi wyjątek, ponieważ tablica nie jest wypełniona
            Console.WriteLine(o);
        }

        // Przykład 24: dostępność w trybie mieszanym
        class PropertyExample34
        {
            private int x;
            private int y;

            public int X
            {
                get { return x; }
                protected set { x = value; }
            }

            public int Y
            {
                get { return y; }
                protected set { y = value; }
            }
        }

        // Przykład 35: zdarzenia
        class Foo35 : IDisposable
        {
            public event EventHandler OnInitialized;
            public event EventHandler OnDisposed;

            public void Init()
            {
                // Inicjalizujemy stan...
                EventHandler onInit = OnInitialized;
                if (onInit != null)
                    onInit(this, new EventArgs());
            }

            public void Dispose()
            {
                // Zwalniamy stan...
                EventHandler onDisp = OnDisposed;
                if (onDisp != null)
                    onDisp(this, new EventArgs());
            }
        }

        public static void ex35()
        {
            using (Foo35 f = new Foo35())
            {
                f.OnInitialized += delegate { Console.WriteLine("init"); };
                f.OnDisposed += delegate { Console.WriteLine("disposed"); };
                // ...
                f.Init();
                // ...
            }            
        }

        // Przykład 36: przeciążanie operatorów
        class State
        {
            internal State Combine(State other) { return this; }
        }
        class StatefulInt
        {
            private int value;
            private State state;

            public StatefulInt(int value, State state)
            {
                this.value = value;
                this.state = state;
            }

            public override string ToString()
            {
                return value.ToString();
            }

            public static StatefulInt operator+(StatefulInt i1, StatefulInt i2)
            {
                return new StatefulInt(i1.value + i2.value,
                    i1.state.Combine(i2.state));
            }

            public static StatefulInt operator+(StatefulInt i1, int i2)
            {
                return new StatefulInt(i1.value + i2, i1.state);
            }
        }

        public static void ex36()
        {
            StatefulInt i1 = new StatefulInt(10, new State());
            Console.WriteLine(i1);
            StatefulInt i2 = new StatefulInt(30, new State());
            Console.WriteLine(i2);
            StatefulInt i3 = i1 + i2; // value == 40
            Console.WriteLine(i3);
            StatefulInt i4 = i2 + 50; // value == 80
            Console.WriteLine(i4);
        }

        // Przykład 37: podklasy
        class A37 {}
        class B37 : A37 {}
        class C37 : A37 {}

        // Przykład 38: dziedziczenie implementacji
        class A38
        {
            public void Foo()
            {
                Console.WriteLine("A::Foo");
            }
        }

        class B38 : A38
        {
            public void Bar()
            {
                Console.WriteLine("B::Bar");
            }
        }

        // Przykład 39: klasy abstrakcyjne
        // Abstrakcyjna, nie zawiera składowych
        abstract class AbstractType1 {}

        // Abstrakcyjna, zawiera tylko abstrakcyjne składowe
        abstract class AbstractType2
        {
            public abstract void Foo();
            public abstract void Bar();
        }

        // Abstrakcyjna, zawiera tylko nieabstrakcyjne składowe
        abstract class AbstractType3
        {
            public void Foo()
            {
                Console.WriteLine("AbstractType3::Foo");
            }
            public void Bar()
            {
                Console.WriteLine("AbstractType3::Bar");
            }
        }

        // Abstrakcyjna, zawiera mieszankę składowych abstrakcyjnych i nieabstrakcyjnych
        abstract class AbstractType4
        {
            public void Foo()
            {
                Console.WriteLine("AbstractType4::Foo");
            }
            public abstract void Bar();
        }

        class ConcreteType : AbstractType4
        {
            public override void Bar()
            {
                // Musimy tu dostarczyć implementacji, a w przeciwnym razie
                // oznaczyć również klasę ConcreteType jako abstrakcyjną.
                Console.WriteLine("ConcreteType::Bar");
            }
        }

        // Przykład 40: interfejsy.
        interface MyICollection : System.Collections.IEnumerable
        {
            void CopyTo(Array array, int index);
            int Count { get; }
            bool IsSynchronized { get; }
            object SyncRoot { get; }
        }

        interface MyIComparable
        {
            int CompareTo(object obj);
        }

        struct MyInt32 : MyIComparable
        {
            private int value;
            public int CompareTo(object obj)
            {
                if (!(obj is MyInt32))
                    throw new ArgumentException();
                int num = ((MyInt32)obj).value;
                if (this.value < num)
                    return -1;
                else if (this.value > num)
                    return 1;
                return 0;
            }
            // ...
        }

        // Przykład 41: dziedziczenie wielu interfejsów
        interface IFoo
        {
            void Foo();
        }

        interface IBar
        {
            void Bar();
        }

        class Implementer : IFoo, IBar
        {
            public void Foo()
            {
                Console.WriteLine("Implementer::Foo");
            }

            public void Bar()
            {
                Console.WriteLine("Implementer::Bar");
            }
        }

        public static void ex41()
        {
            Implementer imp = new Implementer();
            IFoo fooImp = imp;
            fooImp.Foo();
            IBar barImp = imp;
            barImp.Bar();
        }

        // Przykład 42: prywatne dziedziczenie interfejsów
        class PrivateImplementer : IFoo
        {
            void IFoo.Foo()
            {
                Console.WriteLine("PrivateImplementer::IFoo.Foo");
            }
        }

        public static void ex42()
        {
            PrivateImplementer p = new PrivateImplementer();
            //p.Foo(); // Ten wiersz się nie skompiluje
            IFoo f = p;
            f.Foo();
        }

        // Przykład 43: dostępność w dziedziczeniu prywatnym
        class PrivateExtender : PrivateImplementer, IFoo
        {
            void IFoo.Foo()
            {
                //base.Foo(); // Ten wiersz się nie skompiluje
                Console.WriteLine("PrivateExtender::IFoo.Foo");
            }
        }

        // Przykład 44: reimplementacja interfejsu
        class FooBase : IFoo
        {
            public void Foo()
            {
                Console.WriteLine("FooBase::Foo");
            }
            public void Bar()
            {
                Console.WriteLine("FooBase::Bar");
            }
        }

        class FooDerived : FooBase, IFoo
        {
            public new void Foo()
            {
                Console.WriteLine("FooDerived:Foo");
            }
            public new void Bar()
            {
                Console.WriteLine("FooDerived::Bar");
            }
        }

        public static void ex44()
        {
            FooDerived d = new FooDerived();
            FooBase b = d;
            IFoo i = d;

            b.Foo(); // Wyświetla "FooBase::Foo"
            b.Bar(); // Wyświetla "FooBase::Bar"
            d.Foo(); // Wyświetla "FooDerived::Foo"
            d.Bar(); // Wyświetla "FooDerived::Bar"
            i.Foo(); // Wyświetla "FooDerived::Foo"
        }

        // Przykład 45: typy i metody zapieczętowane
        sealed class FooSealed {}
        class Base45
        {
            protected virtual void Bar() { /*...*/ }
        }

        class Derived45 : Base45
        {
            protected override sealed void Bar() { /* ... */ }
        }

        // Przykład 46: sprawdzanie typów podczas wykonania programu
        public static void ex46()
        {
            object o = "test";
            string s1 = (string)o; // Rzutowanie używa instrukcji castclass
            string s2 = o as string; // Słowo kluczowe as używa instrukcji isinst
            if (o is string) { // Słowo kluczowe is również używa instrukcji isinst
                // ...
            }
        }

        // Przykład 47: typy delegacyjne
        public delegate void MyDelegate(int x, int y);

        static void PrintPair(int a, int b)
        {
            Console.WriteLine("a = {0}", a);
            Console.WriteLine("b = {0}", b);
        }

        public static void ex47()
        {
            // W istocie 'new MyDelegate(this.PrintPair)':
            MyDelegate del = PrintPair;
            // W istocie 'Invoke':
            del(10, 20);
        }

        // Przykład 48: ko- i kontrawariancja
        class A48 { /* ... */ }
        class B48 : A48 { /* ... */ }
        class C48 : B48 { /* ... */ }

        static B48 Foo48(B48 b) { return b; }

        delegate B48 MyDelegate1(B48 b);
        delegate B48 MyDelegate2(C48 c);
        delegate A48 MyDelegate3(B48 b);
        delegate A48 MyDelegate4(C48 c);

        public static void ex48()
        {
            MyDelegate1 d1 = Foo48;
            MyDelegate1 d2 = Foo48;
            MyDelegate1 d3 = Foo48;
            MyDelegate1 d4 = Foo48;
        }

        // Przykład 49: metody anonimowe
        delegate int IntIntDelegate(int x);

        static void TransformUpTo(IntIntDelegate d, int max)
        {
            for (int i = 0; i <= max; i++)
                Console.WriteLine(d(i));
        }

        public static void ex49()
        {
            TransformUpTo(delegate(int x) { return x * x; }, 10);
        }

        // Przykład 50: przechwytywanie zmiennych lokalnych
        delegate void FooBar();

        static void Invoke3Times(FooBar d)
        {
            d(); d(); d();
        }

        public static void ex50()
        {
            int i = 0;
            Invoke3Times(delegate { i++; });
            Console.WriteLine(i);
        }

        // Przykład 51: atrybuty niestandardowe
        [AttributeUsage(AttributeTargets.Class | AttributeTargets.Method)]
        class MyAttribute : Attribute
        {
            private string myString;
            private int mySize;

            public MyAttribute(string myName)
            {
                this.myString = myName;
                this.mySize = 8; // wartość domyślna
            }

            public string MyName {
                get { return myString; }
            }
            public int MySize {
                get { return mySize; }
                set { mySize = value; }
            }
        }

        [MyAttribute("MyFoo")]
        class Foo51
        {
            [MyAttribute("MyFoo::MyBar", MySize = 16)]
            public void Bar()
            {
            }
        }

        public static void ex51()
        {
            Type fooType = typeof(Foo51);
            object[] myAttrOnType = fooType.GetCustomAttributes(
                typeof(MyAttribute), false);
            if (myAttrOnType.Length > 0)
            {
                // Robimy coś specjalnego... typ ma atrybut MyAttribute
                Console.WriteLine("Liczba atrybutów Foo51: {0}", myAttrOnType.Length);
            }

            MethodInfo fooBarMethod = fooType.GetMethod("Bar");
            foreach (object attr in fooBarMethod.GetCustomAttributes(false))
            {
                if (attr.GetType() == typeof(MyAttribute))
                {
                    // Ma atrybut MyAttribute, zróbmy coś z tym
                    Console.WriteLine("Foo51.Bar ma atrybut MyAttribute");
                }
            }
        }

        // Przykład 52: wyliczenia
        enum Color : byte
        {
            Czerwony,
            Zielony,
            Niebieski
        }

        class ColorPicker
        {
            internal static void Launch()
            {
                Color favorite = SelectFavoriteColor();
                RespondToFavoriteColor(favorite);
            }

            static Color SelectFavoriteColor()
            {
                Color favorite = (Color)(0xFF);

                // Program działa w pętli, dopóki nie zostanie wybrany prawidłowy kolor
                do
                {
                    // Wyświetlamy monit i listę prawidłiowych kolorów
                    Console.Write("Wpisz swój ulubiony kolor (");
                    foreach (string name in Enum.GetNames(typeof(Color)))
                        Console.Write(" {0} ", name);
                    Console.Write("): ");

                    string input = Console.In.ReadLine();
                    try
                    {
                        favorite = (Color)Enum.Parse(typeof(Color), input, true);
                    }
                    catch (ArgumentException)
                    {
                        // Dane wpisane przez użytkownika nie pasują do nazw w wyliczeniu
                        Console.WriteLine("Błędny kolor, wybierz jeszcze raz!");
                        Console.WriteLine();
                    }
                }
                while (favorite == (Color)(0xFF));

                return favorite;
            }

            static void RespondToFavoriteColor(Color c)
            {
                // Zauważmy, że w C# można stosować instrukcję switch na wyliczeniu
                switch (c)
                {
                    case Color.Czerwony:
                        // Robimy coś dla osób, które lubią kolor czerwony
                        Console.WriteLine("lubisz czerwony");
                        break;
                    case Color.Zielony:
                        // Robimy coś dla osób, które lubią kolor zielony
                        Console.WriteLine("lubisz zielony");
                        break;
                    case Color.Niebieski:
                        // Robimy coś dla osób, które lubią kolor niebieski
                        Console.WriteLine("lubisz niebieski");
                        break;
                    default:
                        // Wykryto nieznany kolor
                        // Jest to prawdopodobnie błąd, ale musimy go obsłużyć!
                        Console.WriteLine("błąd!");
                        break;
                }
            }
        }

        public static void ex52()
        {
            ColorPicker.Launch();
        }

        // Przykład 53: wyliczenia znacznikowe
        [Flags]
        enum FileAccess
        {
            Read = 1,
            Write = 2,
            ReadWrite = 3
        }

        public static void ex53()
        {
            FileAccess rw1 = FileAccess.Read | FileAccess.Write; // wartość 3
            Console.WriteLine("rw1 == {0}", rw1);
            FileAccess rw2 = FileAccess.ReadWrite; // wartość 3
            Console.WriteLine("rw2 == {0}", rw2);

            Console.WriteLine("rw1 == rw2? {0}", rw1 == rw2);

            if (rw1 == FileAccess.Read)
                Console.WriteLine("próba nr 1: odczyt dozwolony");
            else
                Console.WriteLine("próba nr 1: odczyt zabroniony");

            if ((rw2 & FileAccess.Read) != 0)
                Console.WriteLine("próba nr 2: odczyt dozwolony");
            else
                Console.WriteLine("próba nr 2: odczyt zabroniony");
        }

        // Przykład 54: wyliczenia a bezpieczeństwo typologiczne
        static Color MakeFakeColor()
        {
            return (Color)3;
        }

        static void AcceptColorInput(Color c)
        {
            // Sprawdzamy, czy wejściowa wartość wyliczenia jest prawidłowa
            if (!Enum.IsDefined(typeof(Color), c))
                throw new ArgumentOutOfRangeException("c");

            // Zweryfikowaliśmy dane wejściowe, możemy z nimi pracować
            Console.WriteLine("Sukces");
        }

        public static void ex54()
        {
            AcceptColorInput(Color.Niebieski);
            AcceptColorInput(MakeFakeColor());
        }

        /*** GENERYKI ***/

        // Przykład 55: przegląd
        delegate TOutput MyConverter<TInput, TOutput>(TInput input);

        class Foo55<T>
        {
            T data;

            public Foo55(T data)
            {
                this.data = data;
            }

            public U Convert<U>(MyConverter<T,U> converter)
            {
                return converter(data);
            }
        }

        public static void ex55()
        {
            Foo55<int> f = new Foo55<int>(2005);
            string s = f.Convert<string>(delegate(int i) { return i.ToString(); });
            Console.WriteLine(s);
        }

        // Przykład 56: generyki w VB.
        //     (kod oznaczony komentarzem; wymaga oddzielnego kompilatora)
        /*
            Class Foo(Of T)
                Private data As T

                Public Sub Foo(data As T)
                    Me.data = data
                End Sub

                Public Function Convert(Of U)(converter As Converter(Of T, U)) As U
                    Return converter(Me.data)
                End Function
            End Class
         */

        // Przykład 57: kolekcje bez bezpieczeństwa typologicznego
        public static void ex57()
        {
            // Trzeba rzutować podczas wyodrębniania wartości:
            System.Collections.ArrayList listOfStrings = new System.Collections.ArrayList();
            listOfStrings.Add("jakiś łańcuch");
            // ...
            string contents = (string)listOfStrings[0]; // trzeba rzutować

            // Można przypadkiem dodać obiekty niewłaściwego typu:
            listOfStrings.Add("jeden");
            listOfStrings.Add(2); // Ups!
            listOfStrings.Add("trzy");

            // A wówczas program załamie się podczas próby wyodrębnienia elementów:
            foreach (string s in listOfStrings)
            {
                // Zrobić coś z łańcuchem s...
                Console.WriteLine(s);
            }
        }

        // Przykład 58: kolekcje bezpieczne typologicznie
        public static void ex58()
        {
            List<string> listOfStrings = new List<string>();
            listOfStrings.Add("one");
            //listOfStrings.Add(2); // Tutaj kompilator zgłosi błąd
            listOfStrings.Add("three");
        }

        // Przykład 59: konstruowanie: od typów otwartych do zamkniętych
        class MyBaseType<A, B, C> {}
        class MyDerivedType1<A, B, C> : MyBaseType<A, B, C> {}
        class MyDerivedType2<A, B> : MyBaseType<A, B, int> {}
        class MyDerivedType3<B> : MyBaseType<string, B, int> {}
        class MyDerivedType4 : MyBaseType<string, object, int> {}

        // Przykład 60: zmienne statyczne i typy wewnętrzne
        class Foo60<T>
        {
            public static int staticData;
            internal enum MyEnum
            {
                One, Two, Three
            }
        }

        public static void ex60()
        {
            Foo60<int>.staticData = 2;
            Foo60<string>.staticData = 5;
            Console.WriteLine("{0}...{1}...{2}",
                Foo60<int>.staticData, Foo60<string>.staticData, Foo60<DateTime>.staticData);

            Console.WriteLine(typeof(Foo60<int>.MyEnum) == typeof(Foo60<string>.MyEnum));
        }

        // Przykład 61: ograniczenia
        class Foo61_1<T> where T : IComparable<T>
        {
            public void Bar(T x, T y)
            {
                int comparison = x.CompareTo(y);
                // ...
            }
        }

        class Foo61_2
        {
            public void Bar<T>(T x, T y) where T : IComparable<T>
            {
                int comparison = x.CompareTo(y);
                // ...
            }
        }

        class Foo61_3<T> where T : class {}

        class Foo61_4<T> where T : struct {}

        class Foo61_5
        {
            public void Bar<T>() where T : new()
            {
                T t = new T(); // Jest to możliwe tylko ze względu na T : new()
                // ...

                }
        }

    }
}
