﻿using System;
using System.Collections.Generic;
using System.Globalization;
using System.Text;
using System.Text.RegularExpressions;
using System.Reflection;

namespace Wrox.DotNetFramework2.Samples
{
    public class Chapter13
    {
        public static void _title()
        {
            Console.WriteLine("Rozdział 13 - Wyrażenia regularne");
            Console.WriteLine("=================================");
        }

        /*** PODSTAWOWA SKŁADNIA WYRAŻEŃ ***/
        /*** OBSŁUGA BCL ***/

        // Przykład 1: proste dopasowywanie kodów pocztowych
        public static void ex01()
        {
            string s = "01-432";
            string s2 = "ab-432";

            Regex r1 = new Regex("[0-9][0-9]-[0-9][0-9][0-9]");
            Console.WriteLine("{0}\r\nPasuje? {1}: {2}, {3}: {4}",
                r1.ToString(), s, r1.IsMatch(s), s2, r1.IsMatch(s2));

            Regex r2 = new Regex("[0-9]{2}-[0-9]{3}");
            Console.WriteLine("{0}\r\nPasuje? {1}: {2}, {3}: {4}",
                r2.ToString(), s, r2.IsMatch(s), s2, r2.IsMatch(s2));
        }


        // Przykład 2: dopasowywanie kwot pieniężnych
        public static void ex02()
        {
            string[] money = new string[] { "0,99 zł", "1000000,00 zł", "10,25 zł", "90 000,00 zł", "1 000 000,00 zł", "1 000000,00 zł" };
            Regex money1 = new Regex(@"^\d+,\d{2} zł");
            Regex money2 = new Regex(@"^(\d{1,3} )*\d+,\d{2} zł");
            Regex money3 = new Regex(@"^(((\d{1,3} )+\d{3})|\d+),\d{2} zł");

            foreach (string m in money)
                Console.WriteLine("{0}: 1 = {1}, 2= {2}, 3 = {3}", m, money1.IsMatch(m), money2.IsMatch(m), money3.IsMatch(m));
        }

        // Przykład 3: wyodrębnianie grup z dopasowanej kwoty
        public static void ex03()
        {
            Regex r = new Regex(@"^((\d{1,3} )*\d+),(\d{2}) zł");
            Match m = r.Match("1 035 100,99 zł");
            Console.WriteLine("Złote: {0}, grosze: {1}",
                m.Groups[1].Captures[0].Value, m.Groups[3].Captures[0].Value);
        }


        // Przykład 4: wyrażenie dopasowujące kwoty pieniężne w sposób (niemal) niezależny od kultury
        public static void ex04()
        {
            string[] money = new string[] { "$0.99", "$0,99", "$1000000.00", "$10.25", "$90,000.00", "$90.000,00", "$1,000,000.00", "$1,000000.00" };
            NumberFormatInfo info = CultureInfo.CurrentCulture.NumberFormat;
            Regex moneyint = new Regex(String.Format(@"\{0}(\d{{1,3}}\{0})*\d+\{1}\d{{2}}",
                info.CurrencyGroupSeparator, info.CurrencyDecimalSeparator));

            foreach (string m in money)
            {
                Console.WriteLine("{0}: {1}", m, moneyint.IsMatch(m));
                try
                {
                    Decimal.Parse(m, NumberStyles.Currency);
                    Console.WriteLine("!{0}: True", m);
                }
                catch (FormatException)
                {
                    Console.WriteLine("!{0}: False", m);
                }
            }
        }

        // Przykłady 5 - 33: różne wyrażenia regularne
        public static void ex05()
        {
            Regex[] rr = new Regex[] { new Regex("^\\d+"), new Regex(@"\d+$"), new Regex("^\\d+$") };
            string[] ss = new string[] { "123abc", "abc123", "123", "abc", "a123bc" };

            foreach (Regex r in rr)
                foreach (string s in ss)
                    Console.WriteLine("{0}.IsMatch({1}) = {2}", r.ToString(), s, r.IsMatch(s));
        }

        public static void ex06()
        {
            Regex r = new Regex(@"(?(^\d)^\d+$|^\D+$)");
            Regex.CompileToAssembly(new RegexCompilationInfo[] { new RegexCompilationInfo(r.ToString(), RegexOptions.None, "test", "Wrox20", true) }, new AssemblyName("TestDll"));
            Console.WriteLine(r.Match("catoo"));
            Console.WriteLine(r.Match("dog"));
            Console.WriteLine(r.Match("something3 else"));
        }

        public static void ex08()
        {
            Regex r1 = new Regex("(abc)|(xyz)*");
            Regex r2 = new Regex("((abc)|(xyz))*");
            Regex r3 = new Regex("((?:abc)|(?:xyz))*");
            Console.WriteLine(r1.IsMatch("abcxyzabcxyzxyzabcabcabc"));
            Console.WriteLine(r1.Match("abcxyzabcxyzxyzabcabcabc").Value);
            Console.WriteLine(r3.IsMatch("abcabcxyzabc"));
        }

        public static void ex09()
        {
            Regex r1 = new Regex(@"((abc)*)x(\1)");
            string s1 = "abcabcabcxabc";
            string s2 = "abcabcabcxabcabcabc";
            Console.WriteLine(r1.IsMatch(s1) + "..." + r1.Match(s1).Value);
            Console.WriteLine(r1.IsMatch(s2) + "..." + r1.Match(s2).Value);
        }

        public static void ex10()
        {
            Regex r0 = new Regex(@"<[^>]+>[^<]*</[^>]+>");
            Regex r1 = new Regex(@"<([^>]+)>[^<]*</(\1)>");
            string s0 = "<MyTag>Jakaś treść</MyTag>";
            string s1 = "<MyTag>Jakaś treść</Invalid>";
            Console.WriteLine(r0.IsMatch(s0) + "..."  + r0.IsMatch(s1));
            Console.WriteLine(r1.IsMatch(s0) + "..."  + r1.IsMatch(s1));
        }

        public static void ex11()
        {
            Regex r = new Regex(@"<([^>]+)>([^<]*)</(\1)>");
            Match m = r.Match("<MójZnacznik>Jakaś treść</MójZnacznik>");
            Console.WriteLine("Znacznik: {0}, Tekst wewnętrzny: {1}",
                m.Groups[1].Value, m.Groups[2].Value);
        }

        public static void ex12()
        {
            Regex r = new Regex(@"(^\d+$\n)+", RegexOptions.Multiline);
            Console.WriteLine(r.Match("987987\n129821").Value);
        }

        public static void ex13()
        {
            Regex r = new Regex(@"^\G\d+$\n?", RegexOptions.Multiline);
            Match m = r.Match("135\n39238\n128\n5");
            do {
                Console.WriteLine("Dopasowanie: " + m.Value);
            } while ((m = m.NextMatch()).Success);
        }

        public static void ex14()
        {
            Regex r = new Regex("(abc)*abc((abcd)|z)bc");
            Console.WriteLine(r.IsMatch("abcabcabcdbc"));
        }

        public static void ex15()
        {
            string[] inputs = new string[] {
                "<MójZnacznik><ZnacznikWewnętrzny>Tekst</ZnacznikWewnętrzny><JakiśZnacznik>Więcej tekstu</JakiśZnacznik></MójZnacznik>",
                "<MójZnacznik>Dane tekstowe</MójZnacznik>"
           };
            Regex[] rr = new Regex[] {
              new Regex(@"<(.+)>(.*|(<(.+)>.*</\4>)*)</\1>"),
              new Regex(@"<(.+)>(([^<].*)|(<(.+)>.*</\5>)*)</\1>")
            };

            foreach (string input in inputs)
            {
                Console.WriteLine(input);
                foreach (Regex r in rr)
                {
                    Console.WriteLine("  {0}", r.ToString());
                    Match m = r.Match(input);
                    Console.WriteLine("    - Dopasowanie: {0}", m.Success);
                    if (m.Success)
                        for (int i = 0; i < m.Groups.Count; i++)
                            for (int j = 0; j < m.Groups[i].Captures.Count; j++)
                                Console.WriteLine("    - Grupa {0}.{1}: {2}", i, j, m.Groups[i].Captures[j].Value);
                }
            }
        }

        public static void ex16()
        {
            Regex r = new Regex(@"\w+@(\w+\.)+\w+");
            Console.WriteLine(r.IsMatch("joe@bluebytesoftware.com"));
            Console.WriteLine(r.IsMatch("jrobinson@bwtea.com"));
            Console.WriteLine(r.IsMatch("octopus47@hotmail.com"));
        }


        public static void ex17()
        {
            string patternToEscape = "Metaznaki: .*";
            string escaped = Regex.Escape(patternToEscape);
            Console.WriteLine(escaped);
            Console.WriteLine(Regex.Unescape(escaped));
        }

        public static void ex18()
        {
            Regex numbered = new Regex(@"(\w+): (\w+)");
            Match numberedMatch = numbered.Match("Imię: Jan, Firma: Microsoft");
            Console.WriteLine("Pole '{0}' = '{1}'",
                numberedMatch.Groups[1], numberedMatch.Groups[2]);

            Regex named = new Regex(@"(?<name>\w+): (?<value>\w+)");
            Match namedMatch = named.Match("Imię: Marek, Firma: EMC");
            Console.WriteLine("Pole '{0}' = '{1}'",
                namedMatch.Groups["name"], namedMatch.Groups["value"]);
        }



        public static void ex19()
        {
            Regex named = new Regex(@"(?<name>\w+): (?<value>\w+)");
            Match namedMatch = named.Match("Name: Jerzy, Firma: EMC");
            do
            {
                if (namedMatch.Success)
                    Console.WriteLine("Pole '{0}' = '{1}'",
                        namedMatch.Groups["name"], namedMatch.Groups["value"]);
            } while ((namedMatch = namedMatch.NextMatch()).Success);
        }

        public static void ex20()
        {
            Regex named = new Regex(@"((?<name>\w+): (?<value>\w+)[,\s]*)+", RegexOptions.ExplicitCapture);
            Match namedMatch = named.Match("Imię: Jerzy, Firma: EMC");
            Group nameGroup = namedMatch.Groups["name"];
            Group valueGroup = namedMatch.Groups["value"];
            for (int i = 0; i < nameGroup.Captures.Count; i++)
                Console.WriteLine("Pole '{0}' = '{1}'",
                    nameGroup.Captures[i].Value, valueGroup.Captures[i].Value);
        }

        private static string[] genRandStrs()
        {
            Random r = new Random();
            string[] inputStrings = new string[1000];
            for (int i = 0; i < inputStrings.Length; i++)
                if (i % 2 == 0)
                    inputStrings[i] = r.Next().ToString();
                else
                    inputStrings[i] = "blahblahblahblah";
            return inputStrings;
        }

        private static void testLoad(RegexOptions opt)
        {
            string[] inputStrings = genRandStrs();

            long _bt = DateTime.Now.Ticks;
            Regex r0 = new Regex(@"(?(^\d)^\d+$|^\D+$)", opt);
            Console.WriteLine("Tworzenie zajęło {0}ms", new TimeSpan(DateTime.Now.Ticks - _bt).Milliseconds);

            _bt = DateTime.Now.Ticks;
            foreach (string str in inputStrings)
            {
                bool m = r0.IsMatch(str);
            }
            Console.WriteLine("Wykonanie zajęło {0}ms", new TimeSpan(DateTime.Now.Ticks - _bt).Milliseconds);
        }

        public static void ex21()
        {
            testLoad(RegexOptions.None);
        }

        public static void ex22()
        {
            testLoad(RegexOptions.Compiled);
        }

        public static void ex23()
        {
            Regex r0 = new Regex(@"(?<=\s+|^)\d+(?=\s+|$)");
            Console.WriteLine(r0.IsMatch("abc123xyz"));
            Console.WriteLine(r0.IsMatch("abc 123 xyz"));
            Console.WriteLine(r0.IsMatch("123"));
        }

        public static void ex24()
        {
            Regex r0 = new Regex(@"(?i:abc(?-i:\d+xyz)+)");
            Console.WriteLine(r0.IsMatch("AbC123xyz"));
            Console.WriteLine(r0.IsMatch("AbC123XyZ"));
        }

        public static void ex25()
        {
            string input = "53 koty wybrały się do miasta 13-tego lipca 2004 roku";
            Regex r = new Regex(@"\d+");
            Console.WriteLine(r.Replace(input, "<num>"));
        }

        public static void ex26()
        {
            string input = "Mamy 2004 rok, 31-szy października";
            Regex numberedCapture = new Regex(@"\d+");
            Console.WriteLine(numberedCapture.Replace(input, "<num>${1}</num>"));
            Regex namedCapture = new Regex(@"(?<rok>\d+) rok, (?<dzien>\d+)-(?<przyrostek>\w{2,3}) (?<miesiac>\w+)");
            Console.WriteLine(namedCapture.Replace(input, "${dzien}-${przyrostek} ${miesiac} ${rok} roku"));
        }

        struct Customer
        {
            public string Name;
            public string NIP;
            public string Company;
            public Customer(int i)
            {
                Name = "Janusz Kowalski";
                NIP = "xxx-xxx-xx-xx";
                Company = "Microsoft";
            }
        }

        public class CustomerEvaluator
        {
            private Dictionary<int, Customer> customers = new Dictionary<int, Customer>();
            private Customer LoadFromDb(int id)
            {
                if (customers.ContainsKey(id))
                    return customers[id];
                Customer c = new Customer(10); // wyszukujemy klienta w bazie danych
                customers.Add(id, c);
                return c;
            }
            public string Evaluate(Match match)
            {
                string id = match.Groups["custid"].Value;
                Customer c = LoadFromDb(int.Parse(id));
                string a = match.Groups["attrib"].Value;
                switch (a)
                {
                    case "name":
                        return c.Name;
                    case "nip":
                        return c.NIP;
                    case "company":
                        return c.Company;
                    default:
                        return "?nieznany?";
                }
            }
            public string Process(string input)
            {
                Regex r = new Regex(@"\$\{(?<custid>\d+):(?<attrib>\w+)}");
                return r.Replace(input, Evaluate);
            }
        }

        public static void ex27()
        {
            string input = "Klient ${1011:name} pracuje w firmie ${1011:company}.";
            CustomerEvaluator ce = new CustomerEvaluator();
            Console.WriteLine(ce.Process(input));
        }

        public static void ex28()
        {
            string input = "To jest tekst do podzielenia.";
            Regex r = new Regex(@"\s+");
            foreach (string s in r.Split(input))
                Console.WriteLine(s);
        }

        public static void ex29()
        {
            string input = "Jakiś93tekst103do38276podzielenia.";
            foreach (string s in Regex.Split(input, @"\d+"))
                Console.WriteLine(s);
            foreach (string s in Regex.Split(input, @"(\d+)"))
                Console.WriteLine(s);
        }

        public static void ex30()
        {
            string input = "Jakiś93tekst103do38276podzielenia.";
            Regex r = new Regex(@"\d+");
            foreach (string s in r.Split(input))
                Console.WriteLine(s);
        }

        public static void ex31()
        {
            Regex r = new Regex("(?>(abc)*)abc");
            Console.WriteLine(r.IsMatch("abcabcabc"));
        }

        public static void ex32()
        {
            string nl = Environment.NewLine;
            Regex r = new Regex(@"(" +
                @"(?<key>\w+):  # Dopasowuje klucz w parze klucz-wartość" + nl +
                @"\s*           # Opcjonalny odstęp" + nl +
                @"(?<value>\w+) # Dopasowuje wartość" + nl +
                @"(?# Opcjonalny przecinek i/lub odstęp)[,\s]*" +
                @")+            # Może być jedna albo wiele par ",
                RegexOptions.IgnorePatternWhitespace);
            Match m = r.Match("Name: Jan, Company: Microsoft");
            Console.WriteLine(m.Groups["key"].Value + "=" + m.Groups["value"].Value);
        }

        public static void ex33()
        {
            Regex r = new Regex("[a-z-[c-f-[de]]]");
            Console.WriteLine(r.IsMatch("d"));
            Console.WriteLine(r.IsMatch("c"));
            Console.WriteLine(r.IsMatch("m"));
        }
    }
}
