﻿using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Reflection;
using System.Runtime;
using System.Runtime.CompilerServices;
using System.Security;
using System.Text;
using System.Threading;

namespace Wrox.DotNetFramework2.Samples
{
    public class Chapter10
    {
        public static void _title()
        {
            Console.WriteLine("Rozdział 10 - Wątki, domeny i procesy");
            Console.WriteLine("=====================================");
        }

        /*** WĄTKI ***/

        // Przykład 1: przydzielanie pracy wątkom należącym do puli
        private static void MyThreadPoolWorker(object state)
        {
            ManualResetEvent mre = (ManualResetEvent)state;

            // Wykonujemy jakąś pracę; to będzie zrealizowane przez wątek z ThreadPool
            Console.WriteLine("Work occurring on the thread-pool: {0}",
                Thread.CurrentThread.ManagedThreadId);

            // Ustawiamy zdarzenie, aby wywołujący wiedział, że skończyliśmy
            mre.Set();
        }

        public static void ex01()
        {
            using (ManualResetEvent mre = new ManualResetEvent(false))
            {
                ThreadPool.QueueUserWorkItem(new WaitCallback(MyThreadPoolWorker), mre);
                
                // Kontynuujemy pracę, podczas gdy pula wątków realizuje zadanie
                Console.WriteLine("Kontynuuję pracę w głównym wątku: {0}",
                    Thread.CurrentThread.ManagedThreadId);

                // Czekamy, aż pula wątków zakończy pracę:
                mre.WaitOne();
            }
        }

        // Przykład 2: uzyskiwanie informacji o puli wątków
        public static void ex02()
        {
            // Pobieramy informacje:
            int minWorkerThreads, maxWorkerThreads, availableWorkerThreads;
            int minIoThreads, maxIoThreads, availableIoThreads;
            ThreadPool.GetMinThreads(out minWorkerThreads, out minIoThreads);
            ThreadPool.GetMaxThreads(out maxWorkerThreads, out maxIoThreads);
            ThreadPool.GetAvailableThreads(
                out availableWorkerThreads, out availableIoThreads);

            // I wypisujemy je:
            Console.WriteLine("Wątki robocze (min/maks/dostępne): {0}/{1}/{2}",
                minWorkerThreads, maxWorkerThreads, availableWorkerThreads);
            Console.WriteLine("Wątki IO (min/maks/dostępne): {0}/{1}/{2}",
                minIoThreads, maxIoThreads, availableIoThreads);
        }

        // Przykład 3: rejestracja funkcji zwrotnych do obsługi zdarzeń
        public static void ex03()
        {
            using (EventWaitHandle ewh = new ManualResetEvent(false))
            using (EventWaitHandle callbackDoneEvent = new ManualResetEvent(false))
            {
                // Funkcja zwrotna zostanie wywołana, kiedy wystąpi zdarzenie:
                ThreadPool.RegisterWaitForSingleObject(ewh,
                    delegate {
                        Console.WriteLine("Wywołano funkcję zwrotną: {0}", Thread.CurrentThread.ManagedThreadId);
                        callbackDoneEvent.Set();
                    }, null, Timeout.Infinite, true);

                // Teraz ustawiamy zdarzenie. Zauważmy, że funkcja jest wykonywana w oddzielnym wątku (z puli):
                Console.WriteLine("Ustawianie zdarzenia: {0}", Thread.CurrentThread.ManagedThreadId);
                ewh.Set();

                // I czekamy, aż funkcja zwrotna zakończy działanie (opcjonalnie):
                callbackDoneEvent.WaitOne();
            }
        }

        // Przykład 4: jawne tworzenie wątków
        private static void WorkerOperation()
        {
            Console.WriteLine("Prosty wątek roboczy");
        }
        private static void ParameterizedWorkerOperation(object o)
        {
            Console.WriteLine("Parametryzowany wątek roboczy: {0}", o);
        }

        public static void ex04()
        {
            Thread thread = new Thread(WorkerOperation);
            Thread paramThread = new Thread(ParameterizedWorkerOperation);
            thread.Start();
            paramThread.Start("Test");

            // Czekamy, aż wątki zakończą działanie:
            thread.Join();
            paramThread.Join();
        }

        // Przykład 5: kontrolowanie rozmiaru stosu wątku oraz przepełnienia stosu
        // (Ten przykład jest domyślnie wyłączony, ponieważ przepełnienie stosu
        // spowodowałoby zamknięcie procesu. Aby go właczyć, należy zmienić poniższą
        // stałą na "true")
        private const bool enableExample05 = false;

        private static void TestOverflow(object o)
        {
            int i = (int)o;
            Console.WriteLine("  TestOverflow(" + i + ")");
            TestOverflow(i + 1);
        }

        public static void ex05()
        {
            if (enableExample05)
            {
                int howManyK = 256;
                Thread t = new Thread(TestOverflow, 1024 * howManyK);
                t.Start(1);
                t.Join();
            }
        }

        // Przykład 6: używanie przerwań do łagodnego zamykania wątków
        class InterruptAwareWorker
        {
            private bool interruptRequested;
            private Thread myThread;

            public void Interrupt()
            {
                if (myThread == null)
                    throw new InvalidOperationException();
                interruptRequested = true;
                myThread.Interrupt();
                myThread.Join();
            }

            private void CheckInterrupt()
            {
                if (interruptRequested)
                    throw new ThreadInterruptedException();
            }

            public void DoWork(object obj)
            {
                myThread = Thread.CurrentThread;
                try
                {
                    while (true)
                    {
                        // Wykonaj jakąś pracę… (również operacje blokujące)
                        CheckInterrupt();
                        // Wykonaj trochę pracy…
                        CheckInterrupt();
                        // I tak dalej…
                    }
                }
                catch (ThreadInterruptedException)
                {
                    // Wątek przerwany. Posprzątaj…
                    Console.WriteLine("Przerwano wątek...");
                    return;
                }
            }
        }

        public static void ex06()
        {
            InterruptAwareWorker w = new InterruptAwareWorker();
            Thread t = new Thread(w.DoWork);
            t.Start();
            // Wykonaj pracę…
            // A teraz przerwanie:
            w.Interrupt();
            t.Join();
        }

        // Przykład 7: łączenie wątków
        public static void ex07()
        {
            int threadCount = 5;
            Thread[] threads = new Thread[threadCount];

            // Tworzymy wątki robocze: 
            for (int i = 0; i < threadCount; i++)
            {
                int idx = i;
                threads[i] = new Thread(delegate() { Console.WriteLine("Wątek roboczy {0}", idx); });
            }

            // Teraz rozpoczynamy wykonywanie wątków:
            Console.WriteLine("Rozpoczynam wykonywanie wątków...");
            Array.ForEach(threads, delegate(Thread t) { t.Start(); });

            // A wreszcie dołączamy się do nich (czekmamy na zakończenie):
            Console.WriteLine("Czekam na zakończenie...");
            Array.ForEach(threads, delegate(Thread t) { t.Join(); });
            Console.WriteLine("Wszystkie wątki zakończone");
        }

        // Przykład 8: przykład przerywania wątku
        private static object abortLock = new object();

        private static void AbortExample()
        {
            Console.WriteLine("W AbortExample()");
            using (TextReader tr = null)
            {
                Console.WriteLine("  w bloku using");
                Monitor.Enter(abortLock);
                try
                {
                    Console.WriteLine("  w bloku try");
                    //...
                    Thread.Sleep(1000);
                    Console.WriteLine("  wychodzę z bloku try");
                }
                catch (Exception e)
                {
                    Console.WriteLine("  w bloku catch: {0}", e);
                }
                finally
                {
                    Console.WriteLine("  w bloku finally");
                    Monitor.Exit(abortLock);
                }
                // CLR tutaj ponownie zgłasza wyjątek ThreadAbortException
            }
            Console.WriteLine("  poza blokiem using (tego nie zobaczymy)");
        }

        public static void ex08()
        {
            Thread t = new Thread(AbortExample);
            t.Start();
            Thread.Sleep(100);
            t.Abort();
            t.Join();
        }

        // Przykład 9: lokalny magazyn wątku
        public static void ex09()
        {
            // Sloty nienazwane:
            LocalDataStoreSlot slot = Thread.AllocateDataSlot();
            Thread.SetData(slot, 63);
            //...
            int slotValue1 = (int)Thread.GetData(slot);
            Console.WriteLine(slotValue1);

            // Sloty nazwane:
            Thread.SetData(Thread.AllocateNamedDataSlot("TestSlot"), 126);
            //...
            int slotValue2 = (int)Thread.GetData(Thread.GetNamedDataSlot("TestSlot"));
            Console.WriteLine(slotValue2);
        }

        // Przykład 10: statyczne pola wątku
        [ThreadStatic]
        private static string threadStaticData = "<unset>";

        public static void ex10()
        {
            Console.WriteLine("[Główny] przed = {0}", threadStaticData);
            threadStaticData = "Wątek główny";
            Console.WriteLine("[Główny] przed pętlą = {0}", threadStaticData);

            Thread[] threads = new Thread[3];
            for (int i = 0; i < 3; i++)
            {
                threads[i] = new Thread(delegate(object j) {
                    Console.WriteLine("[Wątek{0}] przed = {1}", j, threadStaticData);
                    threadStaticData = "Podrzędny " + j;
                    Console.WriteLine("[Wątek{0}] po = {1}", j, threadStaticData);
                });
                threads[i].Start(i);                
            }

            foreach (Thread t in threads)
                t.Join();

            Console.WriteLine("[Główny] po pętli = {0}", threadStaticData);
        }

        // Przykład 11: klasyczna sytuacja wyścigu
        private static int ex11CurrentId = 0;

        private static int Ex11NextId()
        {
            return ex11CurrentId++;
        }

        // Przykład 12: eliminowanie sytuacji wyścigu za pomocą monitorów
        private static int ex12CurrentId = 0;
        private static object ex12Mutex = new object();

        private static int Ex12NextId()
        {
            Monitor.Enter(ex12Mutex);
            try
            {
                return ex12CurrentId++;
            }
            finally
            {
                Monitor.Exit(ex12Mutex);
            }
        }

        // Przykład 13: ten sam kod co w przykładzie 12, ale z instrukcją lock języka C#
        private static int ex13CurrentId = 0;
        private static object ex13Mutex = new object();

        private static int Ex13NextId()
        {
            lock (ex13Mutex)
            {
                return ex13CurrentId++;
            }
        }

        // Przykład 14: kod podobny do przykładu 12, ale używający metod synchronizowanych
        private static int ex14CurrentId = 0;

        [MethodImpl(MethodImplOptions.Synchronized)]
        private static int Ex14NextId()
        {
            // Ten kod niejawnie blokuje się na "typeof(Chapter10); gdyby to była
            // metoda instancyjna, blokowałaby się na "this"
            return ex14CurrentId++;
        }
  
        // Przykład 15: eliminowanie sytuacji wyścigu za pomocą muteksa Windows
        private static int ex15CurrentId = 0;
        private static Mutex ex15Mutex = new Mutex();

        private static int Ex15NextId()
        {
            ex15Mutex.WaitOne();
            try
            {
                return ex15CurrentId++;
            }
            finally
            {
                ex15Mutex.ReleaseMutex();
            }
        }

        // Przykład 16: eliminowanie sytuacji wyścigu za pomocą zazębiających się operacji
        private static int ex16CurrentId = 0;

        private static int Ex16NextId()
        {
            return Interlocked.Increment(ref ex16CurrentId) - 1;
        }


        // Przykład 17: blokada typu czytelnik-pisarz
        public static void ex17()
        {
            ReaderWriterLock rwLock = new ReaderWriterLock();

            // Pozyskiwanie blokady czytelnika:
            rwLock.AcquireReaderLock(250);
            if (rwLock.IsReaderLockHeld)
            {
                Console.WriteLine("Pozyskano blokadę czytelnika");
                
                // Promocja do blokady pisarza:
                LockCookie cookie = rwLock.UpgradeToWriterLock(250);
                if (rwLock.IsWriterLockHeld)
                {
                    Console.WriteLine("Promowano do blokady pisarza");
                    rwLock.DowngradeFromWriterLock(ref cookie);
                }

                rwLock.ReleaseReaderLock();
            }

            // Pozyskiwanie blokady pisarza:
            rwLock.AcquireWriterLock(250);
            if (rwLock.IsWriterLockHeld)
            {
                Console.WriteLine("Pozyskano blokadę pisarza");
                rwLock.ReleaseWriterLock();
            }
        }

        // Przykład 18: używanie blokady czytelnik-pisarz do ochrony złożonego typu biznesowego
        class RwlAccount
        {
            // Pola
            private string company;
            private decimal balance;
            private DateTime lastUpdate;
            private ReaderWriterLock syncLock = new ReaderWriterLock();

            // Konstruktor
            public RwlAccount(string company)
            {
                this.company = company;
                this.balance = 0.0m;
                this.lastUpdate = DateTime.Now;
            }

            // Właściwości
            public string Company { get { return company; } }
            public decimal Balance { get { return balance; } }
            public ReaderWriterLock SyncLock { get { return syncLock; } }
            public DateTime LastUpdate { get { return lastUpdate; } }

            // Metody
            public decimal UpdateBalance(decimal delta)
            {
                balance += delta;
                lastUpdate = DateTime.Now;
                return balance;
            }

            public decimal AutoUpdateBalance(decimal delta)
            {
                syncLock.AcquireWriterLock(-1);
                try
                {
                    balance += delta;
                    lastUpdate = DateTime.Now;
                    return balance;
                }
                finally
                {
                    syncLock.ReleaseWriterLock();
                }
            }

            public void GetState(out string company, out decimal balance, out DateTime lastUpdate)
            {
                syncLock.AcquireReaderLock(-1);
                try
                {
                    company = Company;
                    balance = Balance;
                    lastUpdate = LastUpdate;
                }
                finally
                {
                    syncLock.ReleaseReaderLock();
                }
            }
        }

        public static void ex18()
        {
            RwlAccount account = new RwlAccount("Microsoft");

            string company;
            decimal balance;
            DateTime lastUpdate;

            // Ten wiersz jest bezpieczny dla wątków:
            account.GetState(out company, out balance, out lastUpdate);
            Console.WriteLine("{0}, balance: {1}, last updated: {2}",
                company, balance, lastUpdate);
        }

        // Przykład 19: blokada pętlowa
        class SpinLock
        {
            private int state;
            private EventWaitHandle available = new AutoResetEvent(false);

            // Sprawdzamy całkowitą liczbę dostępnych wątków sprzętowych
            // Jeśli jest tylko 1, użyjemy zoptymalizowanej ścieżki kodu
            private static bool isSingleProc = (Environment.ProcessorCount == 1);

            private const int outerTryCount = 5;
            private const int cexTryCount = 100;

            public void Enter(out bool taken)
            {
                // taken jest parametrem wyjściowym, więc raczej ustawiamy go *wewnątrz*
                // obszaru krytycznego, niż zwracamy, pozwalając na przerwanie.
                // Bez tego obiekt wywołujący funkcję mógłby pobrać blokadę,
                // ale mógłby jej nie zwolnić, bo nie wiedziałby, że musi.
                taken = false;

                while (!taken)
                {
                    if (isSingleProc)
                    {
                        // Nie czekamy aktywnie w maszynach z jednym procesorem logicznym.
                        // Próbujemy wykonać jedną zamianę; jeśli się nie uda, wracamy do
                        // EventWaitHandle.
                        Thread.BeginCriticalRegion();
                        taken = Interlocked.CompareExchange(ref state, 1, 0) == 0;
                        if (!taken)
                            Thread.EndCriticalRegion();
                    }
                    else
                    {
                        for (int i = 0; !taken && i < outerTryCount; i++)
                        {
                            // Powiadom CLR, że jesteśmy w obszarze krytycznym;
                            // Przerwanie może prowadzić do zakleszczenia.
                            Thread.BeginCriticalRegion();

                            // Próbuj "cexTryCount" razy wykonać CompareExchange
                            // zmiennej stanu
                            int tries = 0;
                            while (!(taken =
                                Interlocked.CompareExchange(ref state, 1, 0) == 0) &&
                                tries++ < cexTryCount)
                            {
                                Thread.SpinWait(1);
                            }

                            if (!taken)
                            {
                                // Nie udało się pobrać blokady pętlowej,
                                // zaznaczamy koniec obszaru krytycznego
                                // i pozwalamy innemu wątkowi wykonać się 
                                Thread.EndCriticalRegion();
                                Thread.Sleep(0);
                            }
                        }
                    }

                    // Jeśli nie otrzymaliśmy blokady, blokujemy się.
                    if (!taken) available.WaitOne();
                }

                return;
            }

            public void Enter()
            {
                // Metoda pomocnicza. Używanie jej może prowadzić do zakleszczania.
                bool b;
                Enter(out b);
            }

            public void Exit()
            {
                if (Interlocked.CompareExchange(ref state, 0, 1) == 1)
                { 
                    // Powiadamiamy wątek budzący się wewnątrz naszego obszaru krytycznego,
                    // aby przerwanie nie spowodowało utraty impulsu,
                    // co mogłoby doprowadzić do zakleszczenia
                    available.Set();
                    Thread.EndCriticalRegion();
                }
            }
        }

        public static void ex19()
        {
            SpinLock sl1 = new SpinLock();
            sl1.Enter();
            try
            {
                Console.WriteLine("Uzyskano blokadę pętlową");
            }
            finally
            {
                sl1.Exit();
                Console.WriteLine("Zwolniono blokadę pętlową");
            }
        }

        // Przykład 20: pewna blokada
        // (Ten przykład jest domyślnie wyłączony, ponieważ powoduje zakleszczenie
        // programu. Aby go włączyć, należy ustawić poniższą stałą na "true").
        private const bool enableEx20 = false;
        private static object deadlockA = new object();
        private static object deadlockB = new object();

        public static void ex20() {
            if (enableEx20)
            {
                Console.WriteLine("ex20: blokowanie a");
                lock (deadlockA) {
                    Console.WriteLine("Główny: pozyskano a");
                    ThreadPool.QueueUserWorkItem(delegate {
                        Console.WriteLine("TP  : blokowanie b");
                        lock (deadlockB) {
                            Console.WriteLine("TP  : pozyskano b");
                            Console.WriteLine("TP  : blokowanie a");
                            lock (deadlockA) {
                                Console.WriteLine("TP  : pozyskano a");
                            }
                        }
                    });
                    Thread.Sleep(100);
                    Console.WriteLine("Główny: blokowanie b");
                    lock (deadlockB) {
                        Console.WriteLine("Główny: pozyskano b");
                    }
                }
            }
        }

        // Przykład 21: producent-konsument i zdarzenia
        private static Queue<string> sharedQueue = new Queue<string>();

        private static void Producer()
        {
            // Produkujemy 20 interesujących elementów
            for (int i = 0; i < 20; i++)
            {
                string item = "Element nr " + i;
                lock (sharedQueue)
                {
                    sharedQueue.Enqueue(item);
                    Monitor.Pulse(sharedQueue);
                }
                Thread.Sleep(50);
            }
        }

        private static void Consumer()
        {
            // Konsumujemy 20 elementów
            for (int i = 0; i < 20; i++)
            {
                string item = null;
                lock (sharedQueue)
                {
                    while (sharedQueue.Count == 0)
                        Monitor.Wait(sharedQueue);
                    item = sharedQueue.Dequeue();
                }

                Console.WriteLine("Przetwarzam element: {0}", item);
            }
        }

        public static void ex21()
        {
            // Uruchamiamy producenta i konsumenta:
            Thread t1 = new Thread(Producer);
            Thread t2 = new Thread(Consumer);
            t1.Start();
            t2.Start();

            // Dołączamy się do nich:
            t1.Join();
            t2.Join();
        }

        // Przykład 22: zegary
        private static void TimerWorker(object state)
        {
            Console.WriteLine("Zdarzenie zegara! {0}", state);
        }

        public static void ex22()
        {
            Console.WriteLine("Tworzę zegar...");
            using (Timer t = new Timer(TimerWorker, "Testowanie", 0, 50))
            {
                Thread.Sleep(500);
                Console.WriteLine("Zamykam zegar...");
            }
        }

        // Przykład 23: przykład singletonu (model pamięci 2.0 ).
        class Singleton {
            private static Singleton instance;

            private string state;
            private bool initialized;

            private Singleton()
            {
                state = "…";
                initialized = true;
            }

            public static Singleton Instance
            {
                get
                {
                    if (instance == null)
                    {
                        lock (typeof(Singleton))
                        {
                            if (instance == null)
                                instance = new Singleton();
                        }
                    }
                    return instance;
                }
            }
        }

        /*** DOMENY APPDOMAIN ***/

        /*** PROCESY ***/

        private static void PrintProcess(Process p)
        {
            Console.WriteLine("  -> {0} - {1}", p.ProcessName, p.PeakWorkingSet64);
        }

        // Przykład 24: wyświetlanie listy aktywnych procesów
        public static void ex24()
        {
            Console.WriteLine("Bieżące procesy:");
            PrintProcess(Process.GetCurrentProcess());

            Console.WriteLine("Procesy IE:");
            Process[] iexplorerProcs = Process.GetProcessesByName("iexplore");
            foreach (Process p in iexplorerProcs) PrintProcess(p);

            Console.WriteLine("Wszystkie aktywne procesy:");
            Process[] allProcs = Process.GetProcesses();
            foreach (Process p in allProcs) PrintProcess(p);
        }

        // Przykład 25: wyświetlanie procesów w zdalnym komputerze
        public static void ex25()
        {
            // (Należy zmienić "developmentbox1" na nazwę prawdziwego
            // komputera w lokalnej sieci)
            Process[] allProcs = Process.GetProcesses("DevelopmentBox1");
            foreach (Process p in allProcs) PrintProcess(p);
        }

        // Przykład 26: uruchamianie procesu powłoki
        public static void ex26()
        {
            // Tworzymy informację startową naszego procesu:
            ProcessStartInfo pi = new ProcessStartInfo("dir.exe");
            pi.UseShellExecute = false;
            pi.RedirectStandardOutput = true;
            pi.Arguments = "C:\\";

            // Uruchamiamy nowy procesu:
            Process p = Process.Start(pi);
            StreamReader sr = p.StandardOutput;

            // I odczytujemy jakieś dane z jego standardowego wyjścia:
            String line;
            while ((line = sr.ReadLine()) != null)
            {
                Console.WriteLine("Czytam wiersz: {0}", line);
            }
            p.WaitForExit();
        }
    }
}
