﻿using System;
using System.Collections;
using System.Collections.Generic;
using System.IO;
using System.Reflection;
using System.Resources;
using System.Text;
using System.Threading;
using System.Globalization;

namespace Wrox.DotNetFramework2.Samples
{
    public class Chapter08
    {
        public static void _title()
        {
            Console.WriteLine("Rozdział 8 - Internacjonalizacja");
            Console.WriteLine("================================");
        }

        /*** CO TO JEST INTERNACJONALIZACJA? ***/

        /*** PRZYKŁADOWE SCENARIUSZE ***/

        // Przykład 1: psuedolokalizacja z instrukcjami if rozrzuconymi po całym programie
        public static void ex01()
        {
            bool prefersEnglish = true;
            if (prefersEnglish)
            {
                Console.WriteLine("An error has occurred.");
            }
            else
            {
                Console.WriteLine("Wystąpił błąd.");
            }
        }

        // Przykład 2: prawdziwa lokalizacja z menedżerami zasobów
        public static void ex02()
        {
            // Ścisła typizacja:
            Console.WriteLine(Chapter08Strings.ErrorOccurred);

            // Słaba typizacja:
            ResourceManager rm = new ResourceManager("Chapter08Strings",
                Assembly.GetExecutingAssembly());
            Console.WriteLine(rm.GetString("ErrorOccurred"));
        }

        // Przykład 3: formatowanie specyficzne dla kultury
        public static void ex03()
        {
            decimal money = 1250.75m;
            string[] cultures = { "en-US", "en-GB", "pl", "ja", "ar" };
            Console.OutputEncoding = Encoding.UTF8;
            foreach (string culture in cultures)
            {
                CultureInfo cultureInfo = CultureInfo.CreateSpecificCulture(culture);
                Console.WriteLine("{0} = {1}", cultureInfo.DisplayName,
                    money.ToString("C", cultureInfo));
            }
        }

        /*** KULTURA ***/

        // Przykład 4: wczytywanie specyficznych kultur
        public static void ex04()
        {
            CultureInfo enNeutral = CultureInfo.GetCultureInfo("en");
            CultureInfo enGB = CultureInfo.GetCultureInfo("en-GB");
            CultureInfo enUS = CultureInfo.GetCultureInfo("en-US");
        }

        // Przykład 5: wyliczanie kultur neutralnych
        public static void ex05()
        {
            CultureInfo[] neutralCultures =
                CultureInfo.GetCultures(CultureTypes.NeutralCultures);
            for (int i = 0; i < neutralCultures.Length; i++)
            {
                CultureInfo culture = neutralCultures[i];
                Console.WriteLine("{0}: {1} [{2}]", i, culture.DisplayName, culture.Name);
            }
        }

        // Przykład 6: uzyskiwanie bieżącej kultury formatowania i kultury UI
        public static void ex06()
        {
            CultureInfo current = CultureInfo.CurrentCulture;
            CultureInfo currentUI = CultureInfo.CurrentUICulture;

            Console.WriteLine("CurrentCulture = {0}", current);
            Console.WriteLine("CurrentUICulture = {0}", currentUI);
        }

        // Przykład 7: zmiana bieżącej kultury
        public static void ex07()
        {
            // zmiana bieżącej kultury na hiszpańską-meksykańską
            CultureInfo current = CultureInfo.CurrentCulture;
            CultureInfo currentUi = CultureInfo.CurrentUICulture;
            try
            {
                CultureInfo mexicanSpanishCi = CultureInfo.GetCultureInfo("es-MX");
                Thread.CurrentThread.CurrentCulture = mexicanSpanishCi;
                Thread.CurrentThread.CurrentUICulture = mexicanSpanishCi;
                Console.WriteLine("{0}/{1}", Thread.CurrentThread.CurrentCulture, Thread.CurrentThread.CurrentUICulture);
                Console.WriteLine("{0:C}, {1}", 1250.50m, DateTime.Now.ToLongDateString());
            }
            finally
            {
                Thread.CurrentThread.CurrentCulture = current;
                Thread.CurrentThread.CurrentUICulture = currentUi;
            }
        }

        // Przykład 8: wyświetlanie interesujących informacji o kulturze
        public static void ex08()
        {
            string[] cultures = new string[] { "en-US", "en-GB", "es-MX", "pl-PL", "ja-JP" };

            using (TextWriter sw = Console.Out)
            {
                foreach (string c in cultures)
                {
                    CultureInfo ci = new CultureInfo(c);
                    sw.WriteLine("{0}: {1}", ci.Name, ci.DisplayName);

                    sw.WriteLine("    Nazwa angielska: {0}", ci.EnglishName);
                    sw.WriteLine("    Nazwa rodzima: {0}", ci.NativeName);
                    sw.WriteLine("    LCID: {0}", ci.LCID);

                    sw.Write("    Kalendarze: ");
                    sw.Write("      {0} (podstawowy)", ci.Calendar.GetType().Name);
                    foreach (Calendar cal in ci.OptionalCalendars)
                        sw.Write(", {0}", cal.GetType().Name);
                    sw.WriteLine();

                    sw.Write("    Dni: ");
                    foreach (string d in ci.DateTimeFormat.DayNames)
                        sw.Write(", {0}", d);
                    sw.WriteLine();

                    sw.Write("    Miesiące: ");
                    foreach (string m in ci.DateTimeFormat.MonthNames)
                        sw.Write(", {0}", m);
                    sw.WriteLine();

                    DateTime dt = new DateTime(1980, 9, 12, 0, 30, 55);
                    sw.WriteLine("    Przykład krótkiej daty: {0}", dt.ToString(ci.DateTimeFormat.ShortDatePattern, ci.DateTimeFormat));
                    sw.WriteLine("    Przykład długiej daty: {0}", dt.ToString(ci.DateTimeFormat.LongDatePattern, ci.DateTimeFormat));

                    double number = -100299.55;
                    sw.WriteLine("    Waluta: {0}", number.ToString("C", ci.NumberFormat));
                    sw.WriteLine("    Format liczb: {0}", number.ToString(ci));
                    sw.WriteLine(ci.NumberFormat.NumberGroupSeparator);
                }
            }
        }

        /*** ZASOBY ***/

        /*** KODOWANIE ***/

        // Przykład 9: uzyskiwanie kodowań
        public static void ex09()
        {
            Encoding ascii = Encoding.ASCII;
            Console.WriteLine(ascii);
            Encoding utf8 = Encoding.UTF8;
            Console.WriteLine(utf8);
            Encoding westernEuropean = Encoding.GetEncoding(1252);
            Console.WriteLine(westernEuropean);
        }

        /*** PROBLEMY Z "DOMYŚLNĄ KULTUROWOŚCIĄ" ***/

        // Przykład 10: analiza składniowa liczb
        public static void ex10()
        {
            CultureInfo current = CultureInfo.CurrentCulture;
            string numberString = "33,200.500";

            try
            {
                Thread.CurrentThread.CurrentCulture = CultureInfo.GetCultureInfo("pl-PL");

                // Najpierw próbujemy przeanalizować liczbę z uwzględnieniem kultury
                try
                {
                    double number = double.Parse(numberString);
                    Console.WriteLine("Przeanalizowano! {0}", number);
                }
                catch (Exception e)
                {
                    Console.WriteLine("Przechwycono wyjątek: {0}", e.ToString());
                }

                // Teraz próbujemy analizy w kulturze niezmiennej
                try
                {
                    double number = double.Parse(numberString, CultureInfo.InvariantCulture);
                    Console.WriteLine("Przeanalizowano! {0}", number);
                }
                catch (Exception e)
                {
                    Console.WriteLine("Przechwycono wyjątek: {0}", e.ToString());
                }
            }
            finally
            {
                Thread.CurrentThread.CurrentCulture = current;
            }
        }

        // Przykład 11: ilustracja problemu tureckiego i
        public static void ex11()
        {
            CultureInfo[] cultures = new CultureInfo[] {
                CultureInfo.GetCultureInfo("en-US"),
                CultureInfo.GetCultureInfo("tr-TR")
            };

            char lower = 'i';
            char upper = 'I';

            foreach (CultureInfo culture in cultures)
            {
                Thread.CurrentThread.CurrentCulture = culture;
                Console.WriteLine("{0}", culture.DisplayName);

                // Przekształcamy małe "i" w wielkie
                char toUpper = Char.ToUpper(lower);
                Console.WriteLine("  Lower->Upper: {0} ({1:X}) -> {2} ({3:X})",
                    lower, (int)lower, toUpper, (int)toUpper);

                // Przekształcamy wielkie "i" w małe
                char toLower = Char.ToLower(upper);
                Console.WriteLine("  Upper->Lower: {0} ({1:X}) -> {2} ({3:X})",
                    upper, (int)upper, toLower, (int)toLower);
            }
        }

        // Przykład 12: naruszenie bezpieczeństwa z wykorzystaniem tureckiego i
        private static void DoSomething(string url)
        {
            if (string.Compare(url, 0, "FILE:", 0, 5, true) == 0)
                throw new Exception("Nie można tego wykonać!");
            Console.WriteLine("Zadziałało");
        }

        public static void ex12()
        {
            CultureInfo current = CultureInfo.CurrentCulture;

            try
            {
                try
                {
                    Thread.CurrentThread.CurrentCulture = CultureInfo.GetCultureInfo("en-US");
                    DoSomething("file://c:/importantfile.sys");
                }
                catch (Exception e)
                {
                    Console.WriteLine("Błąd, {0}", e.ToString());
                }
                try
                {
                    Thread.CurrentThread.CurrentCulture = CultureInfo.GetCultureInfo("tr-TR");
                    DoSomething("file://c:/importantfile.sys");
                }
                catch (Exception e)
                {
                    Console.WriteLine("Błąd, {0}", e.ToString());
                }
            }
            finally
            {
                Thread.CurrentThread.CurrentCulture = current;
            }
        }
    }
}
