using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Runtime.CompilerServices;
using System.Security.Principal;
using System.Threading;

namespace Wrox.DotNetFramework2.Samples
{
    public class Chapter03
    {
        public static void _title()
        {
            Console.WriteLine("Rozdzia 03 - Wewntrz CLR");
            Console.WriteLine("==========================");
        }

        /*** INTERMEDIATE LANGUAGE (IL) ***/

        // Przykad 1: witaj, wiecie
        public static void ex_01()
        {
            Console.WriteLine("Witaj, wiecie!");
        }

        // Przykad 2: dodawanie
        public static void ex02()
        {
            int z = 3 + 5;
            Console.WriteLine(z);
        }

        // Przykad 3: przepyw sterowania
        public static void ex03()
        {
            string f = "foo";
            if (f.Length > 5)
            {
                // Blok kodu wykonywany, jeli warunek jest speniony (true)
            }
            else
            {
                // Blok kodu wykonywany, jeli warunek nie jest speniony (false)
            }
            // Kod nastpujcy po instrukcji if
        }

        // Przykad 4: wywoania ogonowe
        void f_iterative(int n)
        {
            for (int i = n; i > 0; i--)
                Console.WriteLine(i);
        }

        void f_recursive(int n)
        {
            if (n == 0) return;
            Console.WriteLine(n);
            f_recursive(n - 1);
        }

        int fact(int n)
        {
            return fact(n, 1);
        }

        int fact(int n, int v)
        {
            if (n > 0)
                return fact(n - 1, n * v);
            else
                return v;
        }

        // Przykad 5: wywoania ograniczone
        static string GetToString<T>(T input)
        {
            return input.ToString();
        }

        public static void ex05()
        {
            Console.WriteLine(GetToString<int>(10));
        }

        // Przykad 6: niewirtualne wywoania metod wirtualnych (w C++/CLI)
        //     (kod oznaczony komentarzem; wymaga oddzielnego kompilatora)
        /*
            using namespace System;

            ref class A
            {
            public:
                virtual void f() { Console::WriteLine("A::f"); }
            };

            ref class B : public A
            {
            public:
                virtual void f() override { Console::WriteLine("B::f"); }
            };

            int main()
            {
                B b;
                b.f();
                b.A::f();
            }
         */

        // Przykad 7: is, as, rzutowanie
        public static void ex06()
        {
            object o = "test";

            string s1 = o as string;
            if (s1 != null)
            {
                // Moemy tu pracowa z "s1" jak z prawidowym acuchem
                Console.WriteLine("as: o to acuch: {0}", s1);
            }

            bool b = o is string;
            if (b)
            {
                // Moemy rzutowa "o" na "string" itd.
                Console.WriteLine("is/rzutowanie: o jest acuchem: {0}", (string)o);
            }

            string s2 = (string)o;
            // Moemy tu pracowa z "s2"; jeli "o" nie jest acuchem,
            // zostanie zgoszony wyjtek InvalidCastException
            Console.WriteLine("is: o jest acuchem: {0}", s2);
        }

        /*** WYJTKI ***/

        // Przykad 8: przegld wyjtkw
        class Customer {}
        class CustomerManager
        {
            internal Customer Load(int id) { return null; }
            internal void Close() {}
        }
        class RecordNotFoundException : Exception {}

        static void LogProgramBug() {}
        static void LogException(Exception e) {}

        static Customer LoadCustomer(int id)
        {
            Customer c = null;
            CustomerManager cm = new CustomerManager();

            try
            {
                try
                {
                    // Umieszczamy to wywoanie w bloku try, poniewa zgasza ono
                    // wyjtek, jeli klient nie istnieje.
                    c = cm.Load(id);
                }
                catch (RecordNotFoundException e)
                {
                    // Wystpi wyjtek. Niewykluczone, e moemy kontynuowa,
                    // na przykad korzystajc z domylnego klucza bazy danych.
                    // W wielu sytuacjach bdziemy chcieli zarejestrowa wyjtek.
                    LogException(e); // funkcja zdefiniowana przez uytkownika
                    c = cm.Load(-1); // -1 representuje rekord domylny
                }
                catch
                {
                    // Ten blok przechwytuje wszystkie wyjtki typu innego ni
                    // RecordNotFoundException. Jeli nasz program zakada, e z
                    // powyszego bloku try moe zosta zgoszony tylko ten typ wyjtku,
                    // powinnimyzarejestrowa to zdarzenie i pniej je zbada.
                    // By moe w programie jest usterka (albo zabrako pamici itd.).
                    LogProgramBug(); // funkcja zdefiniowana przez uytkownika
                    throw; // propagacja wyjtku
                }
            }
            finally
            {
                // Klauzula finally gwarantuje, e instancja menedera zostanie zamknita
                cm.Close();
            }

            return c;
        }

        // Przykad 9: zgaszanie
        public static void ex09()
        {
            throw new Exception("Wyjtek testowy");
        }

        // Przykad 10: przechwytywanie wedug typu
        static void DoSomething(object o)
        {
            if (o == null)
                throw new ArgumentNullException("test");
            // ...
        }

        public static void ex10()
        {
            try
            {
                // Robimy co, co moe spowodowa zgoszenie wyjtku...
                DoSomething(null);
            }
            catch (ArgumentException ae)
            {
                // (1) Robimy co z "ae"
                Console.WriteLine("Przechwycono wyjtek: {0}...", ae);
            }
            // (2) Kod nastpujcy po bloku try
            Console.WriteLine("Po bloku try");
        }

        // Przykad 11: przechwytywanie wedug filtra
        //     (oznaczony komentarzem; wymaga oddzielnego kompilatora)
        /*
            Function Foo()
                Try
                    ' Robimy co, co moe spowodowa zgoszenie wyjtku...
                Catch e As Exception When ShouldCatch(e)
                    ' Tutaj obsugujemy wyjtek
                End Try
            End Function

            Function ShouldCatch(e As Exception) As Boolean
                   ' Wykonujemy jakie obliczenia
                   ' ... i zwracamy True lub False, aby wskaza, czy wyjtek ma zosta przechwycony 
            End Function
         */

        // Przykad 12: ponowne zgaszanie wyjtku
        public static void ex12()
        {
            try
            {
                // Robimy co, co moe spowodowa zgoszenie wyjtku...
                DoSomething(null);
            }
            catch (Exception e)
            {
                // Robimy co z "e"
                Console.WriteLine("Przechwycono wyjtek: {0}... zgaszam ponownie", e);
                throw e;
            }
            Console.WriteLine("Nie dojdziemy do tego miejsca");
        }

        // Przykad 13: "bloki bdu"
        public static void ex13()
        {
            try
            {
                // Jaki kod, ktry moe zgosi wyjtek...
                DoSomething(null);
            }
            catch
            {
                // "blok bdu"
                Console.WriteLine("Co przechwyciem...");
                throw;
            }
            Console.WriteLine("Nie dojdziemy do tego miejsca");
        }

        // Przykad 14: bloki finally
        class SomeResource
        {
            public void Close() {}
        }

        static SomeResource AcquireResource() { return new SomeResource(); }

        public static void ex14()
        {
            // Pozyskujemy jaki zasb systemowy (albo wykonujemy pierwsz operacj w parze):
            SomeResource sr = AcquireResource();

            try
            {
                // Uywamy zasobu... w tym bloku moe wystpi wyjtek
                throw new Exception();
            }
            catch (Exception e)
            {
                // Logika obsugi wyjtku (opcjonalna)
                Console.WriteLine("Przechwycono wyjtek: {0}...", e);
                throw;
            }
            finally
            {
                // Bezwarunkowe zwalnianie zasobu:
                Console.WriteLine("Zamykanie zasobu...");
                sr.Close();
            }

            Console.WriteLine("Nie dojdziemy do tego miejsca");
        }

        // Przykad 15: wyjtki bezwarunkowe
        public static void ex15()
        {
            try
            {
                try
                {
                    Thread.CurrentThread.Abort();
                }
                catch (ThreadAbortException)
                {
                    // Nic nie robimy, prbujemy "pokn" wyjtek
                    Console.WriteLine("Przechwyciem wyjtek przerwania wtku, prbuj go pokn");
                }

                // Kiedy osigamy to miejsce, wyjtek jest zgaszany ponownie przez CLR
                // Kod znajdujcy si poniej nie zostanie wykonany...
                Console.WriteLine("Nie dojdziemy do tego miejsca");
            }
            catch
            {
                // Anulujemy przerwanie wtku (aby nie przerywa dziaania programu testowego)
                Thread.ResetAbort();
            }
        }

        // Przykad 16: przybieranie innej tosamoci
        public static void ex16()
        {
            // Przybieramy tosamo innego uytkownika...
            IntPtr token = WindowsIdentity.GetAnonymous().Token;
            WindowsImpersonationContext context = 
                WindowsIdentity.Impersonate(token);

            try
            {
                try
                {
                    // Wykonujemy jak chronion operacj w kontekcie zmienionej tosamoci
                }
                finally
                {
                    // Wracamy do poprzedniej tosamoci
                    if (context != null)
                        context.Undo();
                }
            }
            catch
            {
                throw;
            }
        }

        // Przykad 17: API i wyjtki analizy skadniowej
        public static void ex17()
        {
            string qty = "105zedane";
            int parsedQty;

            Stopwatch sw = new Stopwatch();
            sw.Start();
            for (int i = 0; i < 1000; i++)
            {
                // Stary sposb, tzn. wywoanie Parse i wyjtki
                try
                {
                    parsedQty = int.Parse(qty);
                }
                catch (FormatException)
                {
                    // Informujemy uytkownika, e "qty" musi by liczb
                }
            }
            sw.Stop();

            Console.WriteLine("Analiza skadniowa zaja {0} ms", sw.ElapsedMilliseconds);
            sw.Reset();

            sw.Start();
            for (int i = 0; i < 1000; i++)
            {
                // Nowy sposb, tzn. wzorzec TryParse:
                if (!int.TryParse(qty, out parsedQty))
                {
                    // Informujemy uytkownika, e "qty" musi by liczb
                }
            }
            sw.Stop();

            Console.WriteLine("Analiza skadniowa (TryParse) zaja {0} ms", sw.ElapsedMilliseconds);
        }

        /*** AUTOMATYCZNE ZARZDZANIE PAMICI ***/

        // Przykad 18: bramy pamiciowe
        public static void ex18()
        {
            using (System.Runtime.MemoryFailPoint gate = new System.Runtime.MemoryFailPoint(100))
            {
                // Jaka operacja, ktra zuywa 100 MB pamici...
                Console.WriteLine("Sukces dla punktu awarii rwnego 100 MB");
            }

            using (System.Runtime.MemoryFailPoint gate = new System.Runtime.MemoryFailPoint(1024*100))
            {
                // Jaka operacja, ktra zuywa 100 GB pamici...
                Console.WriteLine("Sukces dla punktu awarii rwnego 100 GB");
            }
        }

        /*** KOMPILACJA JIT ***/
        
        // Przykad 19: przykady wywoa metod
        class Foo
        {
            [MethodImpl(MethodImplOptions.NoInlining)]
            public int f(string s, int x, int y)
            {
                Console.WriteLine("Foo::f({0},{1},{2})", s, x, y);
                return x*y;
            }

            [MethodImpl(MethodImplOptions.NoInlining)]
            public virtual int g(string s, int x, int y)
            {
                Console.WriteLine("Foo::g({0},{1},{2})", s, x, y);
                return x+y;
            }
        }

        class Bar : Foo
        {
            [MethodImpl(MethodImplOptions.NoInlining)]
            public override int g(string s, int x, int y)
            {
                Console.WriteLine("Bar::g({0},{1},{2})", s, x, y);
                return x-y;
            }
        }

        delegate int Baz(string s, int x, int y);

        public static void ex19()
        {
            Foo f = new Foo();
            Bar b = new Bar();

            // Zwyke wywoania (call):
            int ff = f.f("Hej", 10, 10);
            int bf = b.f("Hej", 10, 10);

            // Wywoania wirtualne (callvirt):
            int fg = f.g("Hej", 10, 10);
            int bg = b.g("Hej", 10, 10);

            // Wywoania dynamiczne, wirtualne i zwyke:
            Baz bzff = f.f;
            bzff("Hi", 10, 10);
            Baz bzbg = b.g;
            bzbg("Hi", 10, 10);
        }

        // Przykad 20: ramki stosu
        class Foo20
        {
            internal static void A()
            {
                B(10, 50);
            }

            internal static void B(int x, int y)
            {
                C("Hello", x * y);
            }

            internal static void C(string label, int num)
            {
                StackTrace trace = new StackTrace();
                Console.WriteLine(trace.ToString());
            }
        }

        public static void ex20()
        {
            Foo20.A();
        }
    }

}
