﻿using System;
using System.Collections.Generic;
using System.Text;
using System.Threading;
using System.Reflection;
using System.Reflection.Emit;

namespace Wrox.DotNetFramework2.Samples
{
    public class Chapter14
    {
        public static void _title()
        {
            Console.WriteLine("Rozdział 14 - Programowanie dynamiczne");
            Console.WriteLine("======================================");
        }

        /*** API REFLEKSJI ***/

        // Przykład 1: pobieranie referencji do różnych interesujących podzespołów
        public static void ex01()
        {
            Assembly a1 = Assembly.GetExecutingAssembly();
            Console.WriteLine(a1.FullName);
            Assembly a2 = Assembly.GetEntryAssembly();
            Console.WriteLine(a2.FullName);
            Assembly a3 = Assembly.GetCallingAssembly();
            Console.WriteLine(a3.FullName);
            ThreadPool.QueueUserWorkItem(delegate {
                Console.WriteLine(Assembly.GetCallingAssembly().FullName);
            });
            Thread.Sleep(100);
        }

        // Przykład 2: uzyskiwanie informacji o platformie podzespołu
        public static void ex02()
        {
            Assembly a = Assembly.GetExecutingAssembly();
            Module m = a.ManifestModule;

            PortableExecutableKinds peKinds;
            ImageFileMachine imageFileMachine;
            m.GetPEKind(out peKinds, out imageFileMachine);

            Console.WriteLine("{0}: {1}...{2}", a.FullName, peKinds, imageFileMachine);

            if ((peKinds & PortableExecutableKinds.ILOnly) != 0)
            {
                // platform independent
            }
            else
            {
                // platform dependent
                switch (imageFileMachine)
                {
                    case ImageFileMachine.I386:
                        // ...
                        break;
                    case ImageFileMachine.IA64:
                        // ...
                        break;
                    case ImageFileMachine.AMD64:
                        // ...
                        break;
                }
            } 
        }

        // Przykład 3: dynamiczna introspekcja podzespołu
        public static void ex03()
        {
            int x = 10;
            Assembly a = Assembly.Load("mscorlib");

            Type t1 = typeof(int);
            Type t2 = x.GetType();
            Type t3 = a.GetType("System.Int32");
            Type t4 = Type.GetType("System.Int32");

            Console.WriteLine(t1 == t2);
            Console.WriteLine(t2 == t3);
            Console.WriteLine(t3 == t4);
        }
        
        // Przykład 4: uzykiwanie informacji, uchwytów i tokenów
        public static void ex04()
        {
            // Uzyskiwanie informacji o metodzie:
            Type typeInfo = typeof(object);
            MethodInfo methInfo = typeInfo.GetMethod("ToString");
            Console.WriteLine("Info: {0}", methInfo);

            // Uzyskiwanie uchwytu:
            RuntimeTypeHandle typeHandle = typeInfo.TypeHandle;
            RuntimeMethodHandle methHandle = methInfo.MethodHandle;
            Console.WriteLine("Uchwyt: {0}", methHandle);

            // Uzyskiwanie tokena metadanych:
            ModuleHandle moduleHandle = methInfo.Module.ModuleHandle;
            int typeToken = typeInfo.MetadataToken;
            int methToken = methInfo.MetadataToken;
            Console.WriteLine("Token: {0}", methToken);
        }

        // Przykład 5: uniwersalna procedura do wywoływania metod
        private static object InvokeMethod<T>(string f, T obj, object[] args)
        {
            MethodInfo method = typeof(T).GetMethod(f);
            return method.Invoke(obj, args);
        }

        // Przykład 6: badanie ciała metody
        public static void ex06()
        {
            MethodInfo m = typeof(object).GetMethod("ToString");
            MethodBody mb = m.GetMethodBody();
            byte[] ilb = mb.GetILAsByteArray();
            for (int i = 0; i < ilb.Length; i++)
                Console.Write("{0:X} ", ilb[i]);
        }

        // Przykład 7: dynamiczne wywoływanie konstruktorów
        public static void ex07()
        {
            ConstructorInfo ci = typeof(string).GetConstructor(
                new Type[] { typeof(char[]) });
            string s = (string)ci.Invoke(new object[] {
                new char[] { 'W', 'i', 'i', 'a', 'j' } });
            Console.WriteLine(s);
        }

        // Przykład 8: zastosowanie aktywatorów
        private static T CreateAndLogNewObject<T>(string s)
        {
            T t = Activator.CreateInstance<T>();
            Console.WriteLine("Utworzono nowy obiekt '{0}'; {1}", t, s);
            return t;
        }

        public static void ex08()
        {
            Customer c0 = (Customer)Activator.CreateInstance(typeof(Customer));
            Customer c1 = Activator.CreateInstance<Customer>();
            Customer c2 = CreateAndLogNewObject<Customer>("I am creating this because...");
        }

        // Przykład 9: wyświetlanie informacji o parametrach generycznych
        private static void PrintTypeParams(Type t)
        {
            Console.WriteLine(t.FullName);
            foreach (Type ty in t.GetGenericArguments())
            {
                Console.WriteLine("--> {0} {1}", ty.FullName, ty.IsGenericParameter);
                if (ty.IsGenericParameter)
                {
                    Type[] constraints = ty.GetGenericParameterConstraints();
                    foreach (Type c in constraints)
                        Console.WriteLine("    ...{0}", c.FullName);
                }
            }
        }

        public static void ex09()
        {
            PrintTypeParams(typeof(List<>));
            PrintTypeParams(typeof(List<int>));
            PrintTypeParams(typeof(Nullable<>));
        }

        // Przykład 10: otwarte i skonstruowane typy generyczne
        public static void ex10()
        {
            Type listType = typeof(List<>);
            Console.WriteLine("List<>: {0}, {1}",
                listType.IsGenericType, listType.ContainsGenericParameters);
            Type listIntType = typeof(List<int>);
            Console.WriteLine("List<int>: {0}, {1}",
                listIntType.IsGenericType, listIntType.ContainsGenericParameters);
        }

        // Przykład 11: uzyskiwanie definicji typu generycznego
        public static void ex11()
        {
            Console.WriteLine(typeof(List<>).Equals(typeof(List<int>).GetGenericTypeDefinition()));
        }

        // Przykład 12: dynamiczne konstruowanie typów
        public static void ex12()
        {
            Type listType = typeof(List<>);
            Type listOfIntType = listType.MakeGenericType(typeof(int));
            Console.WriteLine(listOfIntType.FullName);
        }

        // Przykład 13: obiekt biznesowy ogólnego przeznaczenia
        abstract class BusinessObject
        {
            public static T Create<T>(Dictionary<string, object> props)
                where T : BusinessObject
            {
                T newInstance = Activator.CreateInstance<T>();
                Type t = typeof(T);

                foreach (KeyValuePair<string, object> prop in props)
                {
                    PropertyInfo pi = t.GetProperty(prop.Key, BindingFlags.Public | BindingFlags.Instance);
                    if (pi == null)
                        throw new ArgumentException(
                            string.Format("Nie znaleziono właściwości '{0}'", prop.Key));
                    if (!pi.CanWrite)
                        throw new ArgumentException(
                            string.Format("'{0}' nie ma metody ustawiającej", prop.Key));

                    pi.SetValue(newInstance, prop.Value, null);
                }

                return newInstance;
            }

            public override string ToString()
            {
                StringBuilder sb = new StringBuilder();

                Type t = this.GetType();
                sb.Append(string.Format("#{0}<", t.FullName));

                PropertyInfo[] props = t.GetProperties();
                for (int i = 0; i < props.Length; i++)
                {
                    if (i > 0) sb.Append(",");
                    PropertyInfo prop = props[i];
                    sb.Append(string.Format("{0}={1}", prop.Name, prop.GetValue(this, null)));
                }
                sb.Append(">");

                return sb.ToString();
            }
        }

        class Customer : BusinessObject
        {
            // Pola
            private string firstName;
            private string lastName;

            // Właściwości
            public string FirstName
            {
                get { return firstName; }
                set { firstName = value; }
            }
            public string LastName
            {
                get { return lastName; }
                set { lastName = value; }
            }
        }

        public static void ex13()
        {
            Dictionary<string, object> props = new Dictionary<string, object>();
            props.Add("FirstName", "Joe");
            props.Add("LastName", "Duffy");
            Customer c = BusinessObject.Create<Customer>(props);
            Console.WriteLine(c);
        }

        // Przykład 14: buforowanie wyników wiązania z wykorzystaniem uchwytów
        class BindingCacheKey
        {
            private RuntimeTypeHandle type;
            private string method;
            private RuntimeTypeHandle[] argTypes;

            public BindingCacheKey(Type type, string method, Type[] argTypes)
            {
                this.type = type.TypeHandle;
                this.method = method;
                this.argTypes = new RuntimeTypeHandle[argTypes.Length];
                for (int i = 0; i < argTypes.Length; i++)
                    this.argTypes[i] = argTypes[i].TypeHandle;
            }

            public override bool Equals(object obj)
            {
                BindingCacheKey key = obj as BindingCacheKey;
                if (key == null)
                    return false;
                return Equals(key);
            }

            public bool Equals(BindingCacheKey obj)
            {
                if (obj == null)
                    return false;
                if (!obj.type.Equals(this.type) || obj.method != this.method ||
                    obj.argTypes.Length != this.argTypes.Length)
                    return false;
                for (int i = 0; i < obj.argTypes.Length; i++)
                    if (!obj.argTypes[i].Equals(this.argTypes[i]))
                        return false;
                return true;
            }

            public override int GetHashCode()
            {
                int hc = type.GetHashCode()%3;
                hc += method.GetHashCode()%3;
                hc += argTypes.GetHashCode()%3;
                return hc;
            }
        }

        class CachedLateBinder
        {
            private Dictionary<BindingCacheKey, RuntimeMethodHandle> cache = new Dictionary<BindingCacheKey, RuntimeMethodHandle>();

            public MethodBase GetMethod(object o, string method, object[] args)
            {
                Type type = o.GetType();
                Type[] argTypes = new Type[args.Length];
                BindingCacheKey key = new BindingCacheKey(type, method, argTypes);

                MethodBase match;
                if (cache.ContainsKey(key))
                {
                    match = MethodBase.GetMethodFromHandle(cache[key]);
                }
                else
                {
                    // Wykonujemy powolne wiązanie...
                    match = null;
                    cache[key] = match.MethodHandle;
                }

                return match;
            }
        }

        /*** ATRYBUTY NIESTANDARDOWE ***/

        // Przykład 15: prosty atrybut niestandardowy i jego zastosowanie
        class MyAttribute : Attribute { /* ... */ }

        [MyAttribute]
        class UserClass
        {
            [MyAttribute]
            private int x;

            [return: MyAttribute]
            [MyAttribute]
            public int IncrementX()
            {
                return x++;
            }

            public int IncrementXBy([MyAttribute] int y)
            {
                return x += y;
            }
        }

        [My]
        class UserClass2 {}

        // Przykład 16: atrybuty niestandardowe z właściwościami
        [AttributeUsage(AttributeTargets.Class | AttributeTargets.Struct |
                        AttributeTargets.Interface)]
        class TableAttribute : Attribute
        {
            public TableAttribute() { }

            public TableAttribute(string name)
            {
                this.name = name;
            }

            private string name;
            private string dbName;

            public string Name
            {
                get { return name; }
                set { name = value; }
            }

            public string DbName
            {
                get { return dbName; }
                set { dbName = value; }
            }
        }

        [Table("customers", DbName="Northwind")]
        class UserClass3 { /* ... */ }

        [Table(Name="customers", DbName="Northwind")]
        class UserClass4 { /* ... */ }

        // Przykład 17: dostęp do atrybutów niestandardowych
        public static void ex17()
        {
            Type t = typeof(UserClass);
            object[] attributes = t.GetCustomAttributes(typeof(MyAttribute), true);
            foreach (object attribute in attributes)
                Console.WriteLine(attribute);
        }

        /*** DELEGACJE ***/

        // Przykład 18: deklarowanie i stosowanie prostych delegacji
        delegate string MyDelegate(int x);

        class MyType
        {
            public string MyFunc(int foo)
            {
                return "Wywołano MyFunc z parametrem foo równym " + foo;
            }
        }

        public static void ex18()
        {
            MyType mt = new MyType();
            MyDelegate md = mt.MyFunc;
            Console.WriteLine(md.Invoke(5));
            Console.WriteLine(md(5));
        }

        // Przykład 19: delegacje parametryzowane
        delegate T Func<T>();
        delegate T Func<T, A>(A a);
        delegate T Func<T, A, B>(A a, B b);
        // ...
        delegate void FuncVoid();
        delegate void FuncVoid<A>(A a);
        delegate void FuncVoid<A, B>(A a, B b);
        // ...

        // Przykład 20: zdarzenia i łańcuchy delegacji
        public static void ex20()
        {
            FuncVoid<int> f1 = delegate(int x) { Console.WriteLine(x * x); };
            FuncVoid<int> f2 = delegate(int x) { Console.WriteLine(x * 2); };
            FuncVoid<int> f3 = (FuncVoid<int>)Delegate.Combine(f1, f2);
            f3(10);
        }

        // Przykład 21: delegacje ko- i kontrawariancyjne
        class A { }
        class B : A { }
        class C : B { }

        private static B Ex21Foo(B b) { Console.WriteLine("foo"); return b; }

        delegate B MyDelegate1(B b);
        delegate B MyDelegate2(C c);
        delegate A MyDelegate3(B b);
        delegate A MyDelegate4(C c);

        public static void ex21()
        {
            MyDelegate1 d1 = Ex21Foo;
            MyDelegate2 d2 = Ex21Foo;
            MyDelegate3 d3 = Ex21Foo;
            MyDelegate4 d4 = Ex21Foo;
        }

        // Przykład 22: delegacje asynchroniczne
        delegate int IntIntDelegate(int x);
        private static int Square(int x) { return x * x; }

        private static void AsyncDelegateCallback(IAsyncResult ar)
        {
            IntIntDelegate f = (IntIntDelegate)ar.AsyncState;
            Console.WriteLine(f.EndInvoke(ar));
        }

        public static void ex22()
        {
            IntIntDelegate f = Square;

            /* Wersja 1: oczekiwanie w pętli (szybka metoda delegacyjna) */
            IAsyncResult ar1 = f.BeginInvoke(10, null, null);
            while (!ar1.IsCompleted)
                // Wykonujemy jakąś kosztowną pracę w oczekiwaniu na zakończenie delegacji...
            Console.WriteLine(f.EndInvoke(ar1));

            /* Wersja 2: oczekiwanie na WaitHandle (wolniejsza metoda delegacyjna) */
            IAsyncResult ar2 = f.BeginInvoke(20, null, null);
            // Wykonujemy jakąś pracę...
            ar2.AsyncWaitHandle.WaitOne();
            Console.WriteLine(f.EndInvoke(ar2));

            /* Wersja 3: wywołanie zwrotne */
            IAsyncResult ar3 = f.BeginInvoke(30, AsyncDelegateCallback, f);
            // Wracamy z wywołania metody (a delegacja wykonuje się)...
        }

        // Przykład 23: metody anonimowe
        private static void TransformUpTo(IntIntDelegate d, int max)
        {
            for (int i = 0; i <= max; i++)
                Console.WriteLine(d(i));
        }

        public static void ex23()
        {
            TransformUpTo(delegate(int x) { return x * x; }, 10);
        }

        // Przykład 24: dodatkowe metody anonimowe
        delegate void FooBar();
        
        private static void Bar(FooBar d)
        {
            d(); d(); d();
        }

        public static void ex24()
        {
            int i = 0;
            Bar(delegate { i++; });
            Console.WriteLine(i);
        }

        /*** EMITOWANIE KODU I METADANYCH ***/

        // Przykład 25: emitowanie podzespołu "Witaj, świecie"
        public static void ex25()
        {
            // Konfigurujemy obiekty do budowania podzespolu i modułu:
            string outFilename = "foo.exe";
            AssemblyBuilder ab = AppDomain.CurrentDomain.DefineDynamicAssembly(
                new AssemblyName("foo"), AssemblyBuilderAccess.RunAndSave,
                AppDomain.CurrentDomain.Evidence);
            ModuleBuilder mb = ab.DefineDynamicModule(ab.FullName, outFilename, false);

            // Tworzymy prosty typ z jedną metodą:
            TypeBuilder tb = mb.DefineType("Program",
                TypeAttributes.Public | TypeAttributes.Sealed, typeof(object));
            MethodBuilder m = tb.DefineMethod("MyMethod",
                MethodAttributes.Public | MethodAttributes.Static);

            // Teraz emitujemy prosty kod "Witaj, świecie":
            ILGenerator ilg = m.GetILGenerator();
            ilg.Emit(OpCodes.Ldstr, "Witaj, świecie!");
            ilg.Emit(OpCodes.Call,
                typeof(Console).GetMethod("WriteLine", new Type[] { typeof(string) }));
            ilg.Emit(OpCodes.Ret);

            // Na koniec tworzymy typ, ustawiamy punkt wejścia i zapisujemy podzespół na dysku:
            tb.CreateType();
            ab.SetEntryPoint(m);
            //ab.Save(outFilename);

            // Dynamiczne wywołanie metody...
            MethodInfo mi = ab.GetType("Program").GetMethod("MyMethod");
            mi.Invoke(null, null);
        }

        // Przykład 26: ten sam program utworzony za pomocą LCG
        public static void ex26()
        {
            DynamicMethod m =  new DynamicMethod("MyMethod", typeof(void),
                new Type[0], typeof(Chapter14).Module);

            // Teraz emitujemy prosty kod "Witaj, świecie":
            ILGenerator ilg = m.GetILGenerator();
            ilg.Emit(OpCodes.Ldstr, "Witaj, świecie!... w stylu LCG");
            ilg.Emit(OpCodes.Call,
                typeof(Console).GetMethod("WriteLine", new Type[] { typeof(string) }));
            ilg.Emit(OpCodes.Ret);

            // Wywołujemy kod:
            m.Invoke(null, null);
        }
    }
}
