﻿using System;
using System.Collections.Generic;
using System.Collections.Specialized;
using System.IO;
using System.IO.Compression;
using System.Net;
using System.Net.Mail;
using System.Net.Sockets;
using System.Runtime.InteropServices;
using System.Text;
using System.Threading;
using System.Xml;

namespace Wrox.DotNetFramework2.Samples
{
    public class Chapter07
    {
        public void _title()
        {
            Console.WriteLine("Rozdział 7. Wejście-wyjście, pliki i sieć");
            Console.WriteLine("=========================================");
        }

        /*** STRUMIENIE ***/

        // Przykład 1: czytanie ze strumienia bajt po bajcie
        public static void ex01()
        {
            using (Stream s = new FileStream("ch07_01.txt", FileMode.Open))
            {
                int read;
                while ((read = s.ReadByte()) != -1)
                {
                    Console.Write("{0} ", read);
                }
            }
        }

        // Przykład 2: czytanie ze strumienia z rzutowaniem na typ char
        public static void ex02()
        {
            using (Stream s = new FileStream("ch07_01.txt", FileMode.Open))
            {
                int read;
                while ((read = s.ReadByte()) != -1)
                {
                    Console.Write("{0} ", (char)read);
                }
            }
        }

        // Przykład 3: czytanie ze strumienia z wykorzystaniem bufora
        public static void ex03()
        {
            using (Stream s = new FileStream("ch07_01.txt", FileMode.Open))
            {
                int readCount;
                byte[] buffer = new byte[4096];
                while ((readCount = s.Read(buffer, 0, buffer.Length)) != 0)
                {
                    for (int i = 0; i < readCount; i++)
                    {
                        Console.Write("{0} ", buffer[i]);
                    }
                }
            }
        }

        // Przykład 4: zapisywanie danych (kopiowanie z jednego strumienia do drugiego)
        public static void ex04()
        {
            using (Stream from = new FileStream("ch07_01.txt", FileMode.Open))
            using (Stream to = new FileStream("ch07_02.txt", FileMode.OpenOrCreate))
            {
                int readCount;
                byte[] buffer = new byte[1024];
                while ((readCount = from.Read(buffer, 0, 1024)) != 0)
                {
                    to.Write(buffer, 0, readCount);
                }
            }
        }

        // Przykład 5: pozycjonowanie
        public static void ex05()
        {
            using (Stream s = new FileStream("ch07_01.txt", FileMode.Open))
            {
                s.Seek(8, SeekOrigin.Current);
                Console.WriteLine(s.ReadByte());
                s.Seek(0, SeekOrigin.Begin);
                Console.WriteLine(s.ReadByte());
                s.Seek(-1, SeekOrigin.End);
                Console.WriteLine(s.ReadByte());
            }
        }

        // Przykład 6: asynchroniczne we-wy, bez blokowania głównego wątku
        struct StreamReadState
        {
            public Stream Stream;
            public byte[] Buffer;
            public EventWaitHandle DoneEvent;

            public StreamReadState(Stream stream, int count)
            {
                this.Stream = stream;
                this.Buffer = new byte[count];
                this.DoneEvent = new ManualResetEvent(false);
            }
        }

        private static void ReadCallback(IAsyncResult ar)
        {
            // Ta metoda zostaje wywołana po zakończeniu operacji BeginRead

            // Wywołanie EndRead zwraca odczytane bajty:
            StreamReadState state = (StreamReadState)ar.AsyncState;
            int bytesRead = state.Stream.EndRead(ar);

            // Teraz możemy przetworzyć dane. Możemy na przykład wypełnić pole tekstowe
            // w interfejsie użytkownika, zarejestrować dane itd.

            if (bytesRead == state.Buffer.Length)
            {
                // Inicjujemy kolejny asynchroniczny odczyt, aby uzyskać kolejną porcję danych:
                state.Stream.BeginRead(state.Buffer, 0, state.Buffer.Length,
                    ReadCallback, state);
            }
            else
            {
                // Napotkaliśmy koniec strumienia. Zamykamy strumień i wykonujemy kod końcowy.
                state.Stream.Close();

                // Tutaj powinien znaleźć się kod wykonywna po zakończeniu całej operacji we-wy.
                // Możemy usunąć pasek postępu, powiadomić użytkownika itd.
                state.DoneEvent.Set();
            }
        }

        public static void ex06()
        {
            // Przypuśćmy, że uzyskaliśmy strumień z okna dialogowego FileOpenDialog
            // Otwieramy go i inicjujemy odczyt...
            Stream s = new FileStream("ch07_01.txt", FileMode.Open);

            // Tworzymy nowy "pojemnik" na współdzielone informacje i rozpoczynamy operację asynchroniczną:
            StreamReadState state = new StreamReadState(s, 4096);
            s.BeginRead(state.Buffer, 0, state.Buffer.Length, ReadCallback, state);

            // Teraz możemy kontynuować pracę...
            // Jeśli wykonujemy jakąś operację na interfejsie użytkownika,
            // interfejs nie będzie zablokowany podczas asynchronicznego odczytu
            state.DoneEvent.WaitOne();
        }

        // Przykład 7: asynchroniczne we-wy z częściowym blokowaniem głównego wątku
        public static void ex07()
        {
            using (Stream s = new FileStream("ch07_01.txt", FileMode.Open))
            {
                byte[] buffer = new byte[4096];
                int bytesRead;
                do
                {
                    // Inicjujemy odczyt asynchroniczny:
                    IAsyncResult ar = s.BeginRead(buffer, 0, buffer.Length, null, null);

                    // Teraz możemy kontynuować pracę...
                    // Jeśli wykonujemy jakąś operację na interfejsie użytkownika,
                    // interfejs nie będzie zablokowany podczas asynchronicznego odczytu.

                    // Czekamy na zakończenie odczytu:
                    bytesRead = s.EndRead(ar);

                    // Teraz możemy przetworzyć dane, na przykład wypełnić pole tekstowe
                    // w interfejsie użytkownika, zarejestrować dane itd.
                }
                while (bytesRead == buffer.Length);
            }
        }

        // Przykład 8: dekodowanie strumieni za pomocą klasy StreamReader
        public static void ex08()
        {
            Stream s = new FileStream("ch07_01.txt", FileMode.Open);
            using (StreamReader sr = new StreamReader(s, Encoding.UTF8))
            {
                int readCount;
                char[] buffer = new char[1024];
                while ((readCount = sr.Read(buffer, 0, 1024)) != 0)
                {
                    for (int i = 0; i < readCount; i++)
                    {
                        Console.Write("{0} ", buffer[i]);
                    }
                }
            }
        }

        // Przykład 9: odczytywanie całych wierszy tekstu za pomocą klasy StreamReader
        public static void ex09()
        {
            Stream s = new FileStream("ch07_01.txt", FileMode.Open);
            using (StreamReader sr = new StreamReader(s, Encoding.UTF8))
            {
                string line;
                while ((line = sr.ReadLine()) != null)
                {
                    Console.WriteLine(line);
                }
            }
        }

        // Przykład 10: odczytywanie całej zawartości strumienia za pomocą klasy StreamReader
        public static void ex10()
        {
            Stream s = new FileStream("ch07_01.txt", FileMode.Open);
            using (StreamReader sr = new StreamReader(s, Encoding.UTF8))
            {
                Console.WriteLine(sr.ReadToEnd());
            }
        }

        // Przykład 11: zapisywanie danych za pomocą klasy TextWriter
        public static void ex11()
        {
            using (TextWriter tw = Console.Out)
            {
                tw.Write("Saldo: "); // wersja string
                tw.Write(30232.30m); // wersja decimal
                tw.Write(", oszczędności? "); // wersja string (ponownie)
                tw.Write(true); // wersja bool
                tw.WriteLine('.'); // wersja char
                // Itd...
            }
        }

        // Przykład 12: odczytywanie danych z łańcucha
        public static void ex12()
        {
            string contents = "Zawartość testowa";
            using (StringReader reader = new StringReader(contents))
            {
                int c;
                while ((c = reader.Read()) != -1)
                {
                    Console.Write("{0} ", c);
                }
            }
        }

        // Przykład 13: odczytywanie danych z łańcucha wiersz po wierszu
        public static void ex13()
        {
            string contents = "Zawartość testowa\r\nWszystkiego najlepszego";
            using (StringReader reader = new StringReader(contents))
            {
                int lineNo = 0;
                string line;
                while ((line = reader.ReadLine()) != null)
                {
                    Console.WriteLine("Line#{0}: {1}", ++lineNo, line);
                }
            }
        }

        // Przykład 14: zapisywanie w łańcuchu
        public static void ex14()
        {
            StringWriter writer = new StringWriter();
            writer.Write("Imię: {0}, Wiek: {1}", "Henryk", 32);
            Console.WriteLine(writer.ToString());
        }

        // Przykład 15: używanie klas BinaryReader i BinaryWriter do pseudoserializacji
        struct Employee
        {
            public string FirstName;
            public string LastName;
            public int Extension;
            public string SocialSecurityNumber;
            public bool Salaried;

            public Employee(string firstName, string lastName, int extension, string ssn, bool salaried)
            {
                this.FirstName = firstName;
                this.LastName = lastName;
                this.Extension = extension;
                this.SocialSecurityNumber = ssn;
                this.Salaried = salaried;
            }

            public override string ToString()
            {
                return string.Format("{0}, {1}; {2}; {3}; {4}", LastName, FirstName, Extension, SocialSecurityNumber, Salaried);
            }
        }

        private static List<Employee> CreateEmployees()
        {
            List<Employee> emps = new List<Employee>();
            emps.Add(new Employee("Joe", "Duffy", 32500, "000-11-2222", true));
            emps.Add(new Employee("George", "Bush", 50123, "001-01-0001", true));
            emps.Add(new Employee("Jess", "Robinson", 99332, "321-21-4321", false));
            emps.Add(new Employee("Billy", "Bob", 32332, "333-22-1111", true));
            emps.Add(new Employee("Homer", "Simpson", 93812, "999-88-7777", false));
            return emps;
        }

        private static List<Employee> DeserializeEmployees(Stream s)
        {
            List<Employee> employees = new List<Employee>();
            BinaryReader reader = new BinaryReader(s);

            try
            {
                while (true)
                {
                    Employee e = new Employee();
                    e.FirstName = reader.ReadString();
                    e.LastName = reader.ReadString();
                    e.Extension = reader.ReadInt32();
                    e.SocialSecurityNumber = reader.ReadString();
                    e.Salaried = reader.ReadBoolean();
                    employees.Add(e);
                    Console.WriteLine("Odczytano: {0}", e.ToString());
                }
            }
            catch (EndOfStreamException)
            {
                // ok, oczekiwany koniec strumienia
            }

            return employees;
        }

        private static void SerializeEmployees(Stream s, IEnumerable<Employee> employees)
        {
            BinaryWriter writer = new BinaryWriter(s);
            foreach (Employee e in employees)
            {
                writer.Write(e.FirstName);
                writer.Write(e.LastName);
                writer.Write(e.Extension);
                writer.Write(e.SocialSecurityNumber);
                writer.Write(e.Salaried);
                Console.WriteLine("Zapisano: {0}", e.ToString());
            }
        }

        public static void ex15()
        {
            Stream s = new MemoryStream();
            IEnumerable<Employee> employees = CreateEmployees();

            SerializeEmployees(s, employees);

            // Deserializacja:
            s.Seek(0, SeekOrigin.Begin);
            DeserializeEmployees(s);

            // Wyświetlamy surowe bajty:
            s.Seek(0, SeekOrigin.Begin);
            int read;
            while ((read = s.ReadByte()) != -1)
                Console.Write("{0:X} ", read);
        }

        // Przykład 16: tryby i prawa dostępu
        public static void ex16()
        {
            using (FileStream fs1 =
                File.Open("ch07_01.txt", FileMode.Open))
            {
                // fs1 jest otwierany z następującymi parametrami:
                //   FileAccess.ReadWrite (ustawienie domyślne) oraz
                //   FileShare.None (ustawienie domyślne)
                /*…*/
            }
            using (FileStream fs2 =
                File.Open("ch07_01.txt", FileMode.Append, FileAccess.Write))
            {
                // fs2 jest otwierany z następującymi parametrami:
                //   FileAccess.Write
                //   FileShare.None (ustawienie domyślne)
                // i przewijany na koniec (parametr FileMode.Append)
                /*…*/
            }
            using (FileStream fs3 =
                File.Open("ch07_01.txt", FileMode.Truncate, FileAccess.ReadWrite, FileShare.Read))
            {
                // fs3 jest otwierany z następującymi parametrami:
                //   FileAccess.ReadWrite
                //   FileShare.Read
                // a cała jego zawartość jest usuwana (parametr FileMode.Truncate)
            }
        }

        // Przykład 17: zarządzanie systemem plików
        public static void ex17()
        {
            DirectoryInfo root = new DirectoryInfo(@"C:\Program Files\");

            // Uwaga: opcja AllDirectories przetwarza rekurencyjnie wszystkie podkatalogi:
            DirectoryInfo[] dirs = root.GetDirectories("*", SearchOption.TopDirectoryOnly);
            Console.WriteLine("{0} podkatalogi:", root.FullName);
            foreach (DirectoryInfo subDir in dirs)
            {
                Console.WriteLine("    {0}", subDir.Name);
            }

            FileInfo[] files = root.GetFiles();
            Console.WriteLine("pliki:");
            foreach (FileInfo file in files)
            {
                Console.WriteLine("    {0} ({1} bajtów)", file.Name, file.Length);
            }
        }

        // Przykład 18: śledzenie zmian w systemie plików (asynchroniczne)
        private static void OnCreatedOrChanged(object sender, FileSystemEventArgs e)
        {
            switch (e.ChangeType)
            {
                case WatcherChangeTypes.Created:
                    // kod obsługujący tworzenie...
                    Console.WriteLine("utworzono '{0}'", e.FullPath);
                    break;
                case WatcherChangeTypes.Changed:
                    // kod obsługujący zmiany...
                    Console.WriteLine("zmieniono '{0}'", e.FullPath);
                    break;
            }
        }

        private static void OnDeleted(object sender, FileSystemEventArgs e)
        {
            // Uwaga: używamy oddzielnej procedury obsługi; moglibyśmy użyć OnCreatedOrChanged
            // kod obsługujący usuwanie...
            Console.WriteLine("usunięto '{0}'", e.FullPath);
        }

        private static void OnRenamed(object sender, RenamedEventArgs e)
        {
            // kod obsługujący zmianę nazwy
            Console.WriteLine("zmieniono nazwę '{0}' na '{1}'", e.OldFullPath, e.FullPath);
        }

        public static void ex18()
        {
            FileSystemWatcher watcher = new FileSystemWatcher(@"c:\", "*.txt");
            watcher.Created += OnCreatedOrChanged;
            watcher.Changed += OnCreatedOrChanged;
            watcher.Deleted += OnDeleted;
            watcher.Renamed += OnRenamed;
        }

        // Przykład 19: śledzenie zmian w systemie plików (synchroniczne)
        private static bool isEx19Enabled = false; // Zmienić na 'true', aby uruchomić ten przykład
        public static void ex19()
        {
            FileSystemWatcher watcher = new FileSystemWatcher(@"c:\", "*.*");
            while (isEx19Enabled)
            {
                // Interesują nas tylko modyfikacje pliku i zmiany nazwy:
                WaitForChangedResult result =
                    watcher.WaitForChanged(
                    WatcherChangeTypes.Changed | WatcherChangeTypes.Renamed);

                // Jeśli jesteśmy tutaj, oznacza to, że zaszła jakaś zmiana:
                switch (result.ChangeType)
                {
                    case WatcherChangeTypes.Changed:
                        Console.WriteLine("{0}: plik '{1}' zmienił się",
                            DateTime.Now, result.Name);
                        break;
                    case WatcherChangeTypes.Renamed:
                        Console.WriteLine("{0}: zmieniono nazwę pliku '{1}' na '{2}'",
                            DateTime.Now, result.OldName, result.Name);
                        break;
                }
            }
        }

        /*** URZĄDZENIA STANDARDOWE ***/

        /*** SIEĆ ***/

        // Przykład 20: wysyłanie danych przez gniazdo
        public static void ex20()
        {
            Socket s = new Socket(AddressFamily.InterNetwork,
                SocketType.Stream, ProtocolType.Tcp);
            using (s)
            {
                s.Connect("www.bluebytesoftware.com", 80);
                byte[] buffer = new byte[2048];
                int bytesSent = s.Send(buffer);
                if (bytesSent != buffer.Length)
                {
                    // Bufor transportowy prawdopodobnie był pełny...
                }
                s.Close();
            }
        }

        // Przykład 21: przyjmowanie połączeń z gniazdem (prosty serwer plików)
        private static void HandleRequest(object state)
        {
            // Konfigurujemy obiekty odpowiedzialne za komunikację:
            using (Socket client = (Socket)state)
            using (NetworkStream stream = new NetworkStream(client))
            using (StreamReader reader = new StreamReader(stream))
            using (StreamWriter writer = new StreamWriter(stream))
            {
                // Klient przesyła nazwę pliku i znak nowego wiersza;
                // serwer odczytuje plik, wysyła go i zamyka strumień
                string fileName = reader.ReadLine();
                writer.Write(File.ReadAllText(fileName));
                
            }
        }

        private static bool isEx21Enabled = false; // Zmienić na 'true', aby uruchomić ten przykład
        public static void ex21()
        {
            // Tworzymy gniazdo i wiążemy je z portem 9999, zezwalając na kolejkowanie 3 żądań
            using (Socket s = new Socket(AddressFamily.InterNetwork,
                SocketType.Stream, ProtocolType.Tcp))
            {
                s.Bind(new IPEndPoint(IPAddress.Loopback, 9999));
                s.Listen(3);

                // Nasza główna pętla przetwarzania...
                while (isEx21Enabled)
                {
                    // Akceptujemy żądanie klienta i przekazujemy je do wątku roboczego:
                    Socket client = s.Accept();
                    ThreadPool.QueueUserWorkItem(HandleRequest, client);
                }
            }
        }

        // Przykład 22: transmisje grupowe UDP
        private static bool isEx22Enabled = false; // Zmienić na 'true', aby uruchomić ten przykład
        public static void ex22()
        {
            using (UdpClient udp = new UdpClient(1024))
            {
                IPAddress groupAddress = IPAddress.Parse("0.0.0.0");
                udp.JoinMulticastGroup(groupAddress, 32);
                udp.EnableBroadcast = true;

                while (isEx22Enabled)
                {
                    IPEndPoint sentBy = null;
                    byte[] data = udp.Receive(ref sentBy);
                    // Przetwarzamy dane...
                    break;
                }

                udp.DropMulticastGroup(groupAddress);
            }
        }

        // Przykład 23: prosty serwer echa HTTP
        private static bool isEx23Enabled = false; // Zmienić na 'true', aby uruchomić ten przykład
        public static void ex23()
        {
            using (HttpListener listener = new HttpListener())
            {
                // Konfigurujemy słuchacza:
                listener.AuthenticationSchemes = AuthenticationSchemes.Negotiate;
                listener.Prefixes.Add("http://localhost:8080/echo/");
                listener.Prefixes.Add("https://localhost/echo/");
                listener.Start();

                // Przetwarzamy żądania aż do chwili zamknięcia programu (tzn. zakończenia procesu;
                // prawdopodobnie moglibyśmy zrobić to w sposób bardziej przyjazny dla użytkownika):
                while (isEx23Enabled)
                {
                    // Czekamy na następne połączenie:
                    HttpListenerContext ctx = listener.GetContext();
                    ctx.Response.StatusCode = 200; // HTTP OK
                    string name = ctx.Request.QueryString["name"];

                    // Teraz odpowiadamy za pomocą klasy StreamWriter:
                    using (StreamWriter writer = new StreamWriter(ctx.Response.OutputStream))
                    {
                        writer.WriteLine("<p>Witaj, {0}</p>", name);
                        writer.WriteLine("<ul>");
                        foreach (string header in ctx.Request.Headers.Keys)
                        {
                            writer.WriteLine("<li><b>{0}:</b> {1}</li>",
                                header, ctx.Request.Headers[header]);
                        }
                        writer.WriteLine("</ul>");
                    }

                    ctx.Response.Close();
                }

                listener.Stop();
            }
        }

        // Przykład 24: wysyłanie poczty e-mail
        private static bool isEx24Enabled = false; // Zmienić na 'true', aby uruchomić ten przykład
        public static void ex24()
        {
            if (isEx24Enabled)
            {
                SmtpClient client = new SmtpClient("smtp.localhost.com");

                // Jeśli wychodzący serwer SMTP wymaga logowania, informacje
                // o użytkowniku można wysłać za pomocą poniższego wiersza:
                client.Credentials =
                    new NetworkCredential("username", "password");

                // Konstruujemy wiadomość, ustawiamy pola "Od:", "Do:", "DW:", tematu i treści
                MailMessage message = new MailMessage(
                    new MailAddress("joe@somewhere.com", "Joe Duffy"),    // Do
                    new MailAddress("raj@kittylitter.com", "Raj Duffy")); // Do wiadomości
                message.Bcc.Add(new MailAddress("ashok@ferretspaceships.com"));
                message.Bcc.Add(new MailAddress("mika@ferretspaceships.com"));
                message.Subject = "…";
                message.Body = "…";

                // Dołączamy plik z dysku:
                Attachment att = new Attachment(@"c:\ścieżka\do\pliku.cs");
                message.Attachments.Add(att);

                // Wysyłamy wiadomość:
                client.Send(message);
            }
        }
    }
}