﻿using System;
using System.Collections;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Text;

namespace Wrox.DotNetFramework2.Samples
{
    public class Chapter06
    {
        public void _title()
        {
            Console.WriteLine("Rozdział 6. Tablice i kolekcje");
            Console.WriteLine("==============================");
        }

        /*** TABLICE ***/

        // Przykład 1: prosta tablica łańcuchów
        public static void ex01()
        {
            string[] myStringArray = new string[10]; // tablica 10-elementowa
            myStringArray[0] = "A";
            // ...
            string zerothElement = myStringArray[0];
            Console.WriteLine(zerothElement);
        }

        // Przykład 2: tablice jedno- i dwuwymiarowe
        public static void ex02()
        {
            int[] oneDimension = new int[10];
            oneDimension[5] = 0;
            int[,] twoDimension = new int[10,10];
            twoDimension[2,5] = 0;
        }
        

        // Przykład 3: tablice prostokątne i postrzępione
        public static void ex03()
        {
            // Konstruowanie tablic prostokątnych:
            int[,] rect = new int[5, 8];
            int[,] rect2 = new int[,] { { 132, 51 }, { 301, 25 } };

            // Przetwarzanie w pętli poszczególnych wierszy i kolumn:
            for (int i = 0; i < rect2.GetLength(0); i++)
                for (int j = 0; j < rect2.GetLength(1); j++)
                    Console.WriteLine("[{0},{1}] = {2}", i, j, rect2[i, j]);

            // Przetwarzanie całej zawartości:
            foreach (int x in rect2)
                Console.WriteLine(x);

            // Constructing jagged arrays:
            int[][] jagged = new int[5][];
            jagged[0] = new int[8];
            jagged[1] = new int[2];
            jagged[2] = new int[0];
            jagged[3] = new int[8];
            jagged[4] = new int[4];

            // Walk the jagged array:
            for (int i = 0; i < jagged.Length; i++)
                for (int j = 0; j < jagged[i].Length; j++)
                    Console.WriteLine("[{0}][{1}] = {2}", i, j, jagged[i][j]);

            // Wyświetlanie rzędu:
            int[][] jagged2 = new int[][] { new int[] { 132, 51 }, new int[] { 301, 25 } };
            Console.WriteLine(jagged2.Rank);
        }

        // Przykład 4: collections interoperability.
        public static void ex04()
        {
            int[] arrayOfInts = new int[] { 10, 99, 50 };
            Array a = (Array)arrayOfInts;
            IList list = (IList)arrayOfInts;
            ICollection collection = (ICollection)arrayOfInts;
            IEnumerable enumerable = (IEnumerable)arrayOfInts;
            ICollection<int> collectionGeneric = (ICollection<int>)arrayOfInts;
            IEnumerable<int> enumerableGeneric = (IEnumerable<int>)arrayOfInts;
        }

        // Przykład 5: klonowanie tablic
        class BadMonthType {
            private static readonly string[] monthConstants = new string[] {
                "styczeń", "luty", "marzec", "kwiecień", "maj", "czerwiec",
                "lipiec", "sierpień", "wrzesień", "październik", "listopad", "grudzień"
            };

            public string[] Months
            {
                get { return monthConstants; }
            }
        }

        class GoodMonthType {
            private static readonly string[] monthConstants = new string[] {
                "styczeń", "luty", "marzec", "kwiecień", "maj", "czerwiec",
                "lipiec", "sierpień", "wrzesień", "październik", "listopad", "grudzień"
            };

            public string[] Months
            {
                get { return (string[])monthConstants.Clone(); }
            }

            public IEnumerable<string> MonthsEnumerable
            {
                get { return Array.AsReadOnly<string>(monthConstants); }
            }
        }

        public static void ex05()
        {
            // Najpierw pokazujemy, czemu typ BadMonthType jest zły
            BadMonthType mt1 = new BadMonthType();
            
            // Wyświetlamy nazwy miesięcy przed modyfikacją
            Console.WriteLine("*** Przed (źle) ***");
            foreach (string m in mt1.Months) {
                Console.WriteLine(m);
            }

            // Zmieniamy miesiąc
            string[] months1 = mt1.Months;
            months1[3] = "nie-kwiecień";

            // Znów wyświetlamy miesiące. Nazwa kwietnia zmieniła się
            Console.WriteLine("*** Po (źle) ***");
            foreach (string m in mt1.Months) {
                Console.WriteLine(m);
            }

            Console.WriteLine();

            // Teraz robimy to samo z typem GoodMonthType
            GoodMonthType mt2 = new GoodMonthType();

            // Wyświetlamy nazwy miesięcy przed modyfikacją
            Console.WriteLine("*** Przed (dobrze) ***");
            foreach (string m in mt2.Months) {
                Console.WriteLine(m);
            }

            // Zmieniamy nazwę miesiąca (co nie działa, mt2.Months zwraca kopię)
            string[] months2 = mt2.Months;
            months2[3] = "Not-April";

            // Znów wyświetlamy miesiące. Nazwa kwietnia nie zmieniła się
            Console.WriteLine("*** Po (dobrze) ***");
            foreach (string m in mt2.Months) {
                Console.WriteLine(m);
            }
        }

        // Przykład 6: kopiowanie elementów między tablicami
        public static void ex06()
        {
            int[] srcArray = new int[] { 1, 2, 3 };
            int[] destArray = new int[] { 4, 5, 6, 7, 8, 9 };

            Array.Copy(srcArray, destArray, srcArray.Length);
            srcArray.CopyTo(destArray, 0);
            srcArray.CopyTo(destArray, 3);
            Array.Copy(srcArray, 0, destArray, 2, srcArray.Length);
        }

        // Przykład 7: dynamiczne tworzenie tablic
        public static void ex07()
        {
            string[] myArray = (string[])Array.CreateInstance(typeof(string), 15);
            Console.WriteLine(myArray.GetType());
            Console.WriteLine(myArray.GetLength(0));
            Console.WriteLine(myArray.GetLowerBound(0));
            myArray.SetValue("Witaj, tablico", 5);
            Console.WriteLine(myArray.GetValue(5));
        }
        
        class Foo {}

        // Przykład 8: statyczne metody pomocnicze
        public static void ex08()
        {
            // Sortowanie:
            int[] array = new int[] { 8, 2, 3, 5, 1, 3 };
            Console.Write("Nieposortowana: ");
            Array.ForEach<int>(array, delegate(int x) { Console.Write(x + " "); });
            Console.WriteLine();
            Array.Sort(array);
            Console.Write("Posortowana: ");
            Array.ForEach<int>(array, delegate(int x) { Console.Write(x + " "); });
            Console.WriteLine();

            // Wyszukiwanie binarne:
            if (Array.BinarySearch<int>(array, 9) >= 0)
                Console.WriteLine("Wyszukiwanie binarne: znaleziono liczbę 9");
            else
                Console.WriteLine("Wyszukiwanie binarne: nie znaleziono liczby 9");

            // Konwersje:
            string[] strArray = new string[] { "75,3", "25,999", "105,25" };

            Console.Write("Nieprzekształcona: ");
            Array.ForEach<string>(strArray, delegate(string x) { Console.Write(x + " "); });
            Console.WriteLine();

            double[] doubleArray = Array.ConvertAll<string, double>(
                strArray, Convert.ToDouble);
            Console.Write("Przekształcona w wartości double: ");
            Array.ForEach<double>(doubleArray, delegate(double x) { Console.Write(x + " "); });
            Console.WriteLine();

            string[] strArray2 = new string[] { "Bla", "Foo", "Kanoniczna" };
            Console.Write("Przed zamianą na wielkie litery: ");
            Array.ForEach<string>(strArray2, delegate(string x) { Console.Write(x + " "); });
            Console.WriteLine();

            string[] revArray = Array.ConvertAll<string, string>(
                strArray2, delegate(string s) { return s.ToUpper(); });
            Console.Write("Po zamianie w wielkie litery: ");
            Array.ForEach<string>(revArray, delegate(string x) { Console.Write(x + " "); });
            Console.WriteLine();
        }

        // Przykład 9: tablice stałe
        unsafe struct Ex09FixedFoo
        {
            public fixed char buffer[256];
        }

        public unsafe static void ex09()
        {
            Ex09FixedFoo foo = new Ex09FixedFoo();

            // Po prostu kopiujemy znaki łańcucha do bufora Foo:
            string s = "Hello, World";
            char[] sc = s.ToCharArray();
            for (int i = 0; i < sc.Length; i++)
                foo.buffer[i] = sc[i];

            // I pokazujemy, jak można bezpośrednio uzyskać dostęp do Foo:
            char* pchBuff = (char*)&foo;
            char c = '\0';
            for (int i = 0; i < 256 && (c = pchBuff[i]) != 0; i++)
                Console.Write(c + " ");
        }

        /*** KOLEKCJE ***/

        // Przykład 10: wady słabo typizowanych kolekcji
        public static void ex10()
        {
            IList myIntList = new ArrayList();
            myIntList.Add(55);
            myIntList.Add("Oho ho");
            myIntList.Add(10);

            foreach (int i in myIntList)
                Console.WriteLine(i);
        }

        // Przykład 11: zastosowanie kolekcji generycznych
        public static void ex11()
        {
            IList<int> myIntList = new List<int>();
            myIntList.Add(55);
            //myIntList.Add("Oho ho");
            myIntList.Add(10);

            foreach (int i in myIntList)
                Console.WriteLine(i);
        }

        // Przykład 12: przykład użycia ICollection<T>
        public static void ex12()
        {
            ICollection<int> myCollection = new Collection<int>();

            // Najpierw dodajemy kilka elementów do kolekcji:
            myCollection.Add(105);
            myCollection.Add(232);
            myCollection.Add(350);

            // ...po czym usuwamy jeden z nich:
            myCollection.Remove(232);

            // Wyszukujemy konkretne elementy:
            Console.WriteLine("Zawiera {0}? {1}", 105, myCollection.Contains(105));
            Console.WriteLine("Zawiera {0}? {1}", 232, myCollection.Contains(232));

            // Wyliczamy zawartość kolekcji:
            foreach (int i in myCollection)
                Console.WriteLine(i);

            // Kopiujemy zawartość do tablicy, aby przetworzyć elementy iteracyjnie:
            int[] myArray = new int[myCollection.Count];
            myCollection.CopyTo(myArray, 0);
            for (int i = 0; i < myArray.Length; i++)
                Console.WriteLine(myArray[i]);
        }

        // Przykład 13: zastosowanie IList<T>
        public static void ex13()
        {
            IList<double> myList = new List<double>();

            // Najpierw dodajemy, wstawiamy i usuwamy kilka elementów:
            myList.Add(10.54);
            myList.Add(209.2234);
            myList.Insert(1, 39.999);
            myList.Add(438.2);
            myList.Remove(10.54);

            // Następnie wyświetlamy elementy znajdujące się pod określonymi indeksami:
            Console.WriteLine("IndexOf {0} = {1}", 209.2234, myList.IndexOf(209.2234));
            Console.WriteLine("IndexOf {0} = {1}", 10.54, myList.IndexOf(10.54));

            // Na koniec wyliczamy listę z wykorzystaniem właściwości Count i indeksatora IList<T>:
            for (int i = 0; i < myList.Count; i++)
                Console.WriteLine(myList[i]);
        }

        // Przykład 14: zastosowanie IDictionary<TKey,TValue>
        public static void ex14()
        {
            IDictionary<string, decimal> salaryMap = new Dictionary<string, decimal>();

            // Dodajemy kilka wpisów do słownika:
            salaryMap.Add("Bartek", 62250.5M);
            salaryMap.Add("Oskar", 16000.0M);
            salaryMap.Add("Janek", 32900.99M);

            // Teraz usuwamy jeden wpis:
            salaryMap.Remove("Oskar");

            // Sprawdzamy, czy określone klucze istnieją w słowniku:
            Console.WriteLine(salaryMap.ContainsKey("Bartek")); // Wyświetla "True"
            Console.WriteLine(salaryMap.ContainsKey("Stefan")); // Wyświetla "False"

            // Pobieramy wartości ze słownika:
            Console.WriteLine("{0:C}", salaryMap["Bartek"]); // Wyświetla "62 250,50 zł"
            Console.WriteLine("{0:C}", salaryMap["Janek"]); // Wyświetla "32 900,99 zł"

            // Teraz przewarzamy słownik iteracyjnie i dodajemy wartości:
            decimal total = 0.0M;
            foreach (decimal d in salaryMap.Values)
                total += d;
            Console.WriteLine("{0:C}", total); // Wyświetla "95 151,49 zł"

            // Iteracja po parach klucz-wartość:
            // (zły sposób)
            foreach (string key in salaryMap.Keys)
                Console.WriteLine("{0} == {1}", key, salaryMap[key]);

            // (dobry sposób)
            foreach (KeyValuePair<string, decimal> kvp in salaryMap)
                Console.WriteLine("{0} == {1}", kvp.Key, kvp.Value);
        }

        // Przykład 15: zastosowanie IEnumerator<T>
        public static void ex15()
        {
            IEnumerable<string> enumerable = new string[] { "A", "B", "C" };

            // Postać skrótowa:
            foreach (string s in enumerable)
                Console.WriteLine(s);

            // Sposób ręczny:
            IEnumerator<string> enumerator = enumerable.GetEnumerator();
            while (enumerator.MoveNext())
            {
                string s = enumerator.Current;
                Console.WriteLine(s);
            }
        }

        // Przykład 16: iteratory C#, dwa sposoby
        class DoublerContainer : IEnumerable<int>
        {
            private List<int> myList = new List<int>();

            public DoublerContainer(List<int> myList)
            {
                this.myList = myList;
            }

            public IEnumerator<int> GetEnumerator()
            {
                foreach (int i in myList)
                    yield return i * 2;
            }

            IEnumerator IEnumerable.GetEnumerator()
            {
                return ((IEnumerable<int>)this).GetEnumerator();
            }
        }

        static IEnumerable<int> Fibonacci(int max)
        {
            int i1 = 1;
            int i2 = 1;

            while (i1 <= max)
            {
                yield return i1;
                int t = i1;
                i1 = i2;
                i2 = t + i2;
            }
        }

        public static void ex16()
        {
            // Najpierw przetwarzamy zawartość podwajacza:
            Console.WriteLine("Podwajacz:");
            DoublerContainer dc = new DoublerContainer(
                new List<int>(new int[] { 10, 20, 30, 40, 50 }));
            foreach (int x in dc)
                Console.WriteLine(x);

            // Następnie korzystamy z generatora ciągu Fibonacciego:
            Console.WriteLine("Fibonacci:");
            foreach (int i in Fibonacci(21))
                Console.WriteLine(i);
        }

        // Przykład 17: zastosowanie List<T>
        private static int ConvertStringToInt(string input)
        {
            int result;
            if (!int.TryParse(input, out result))
                result = -1;
            return result;
        }

        public static void ex17()
        {
            // Tworzenie listy z tablicy:
            List<int> intList = new List<int>(new int[] { 3, 5, 15, 1003, 25 });

            // Tworzenie listy ze słownika:
            IDictionary<string, DateTime> dictionary = new Dictionary<string, DateTime>();
            List<KeyValuePair<string, DateTime>> keyValueList =
                new List<KeyValuePair<string, DateTime>>(dictionary);

            // Tworzenie listy z kolejki:
            Queue <string> q = new Queue<string>(); //…
            List<string> stringList = new List<string>(q);

            // Uzyskiwanie kopii przeznaczonej tylko do odczytu:
            ReadOnlyCollection<int> roList = intList.AsReadOnly();

            // Instrukcja foreach z wykorzystaniem enumeratora:
            foreach (int x in intList)
                Console.WriteLine(x);

            // Używanie indeksatora:
            for (int i = 0; i < intList.Count; i++)
                Console.WriteLine(intList[i]);

            // Używanie delegacji Action<T>:
            intList.ForEach(delegate(int x) { Console.WriteLine(x); });

            // Przekształcanie listy:
            List<string> stringList1 = new List<string>(
                new string[] { "99", "182", "15" });
            List<int> intList1 = stringList1.ConvertAll<int>(Convert.ToInt32);

            List<string> stringList2 = new List<string>(
                new string[] { "99", "182", "błąd", "15" });
            List<int> intList2 = stringList2.ConvertAll<int>(ConvertStringToInt);
        }

        // Przykład 18: sortowane słowniki
        public static void ex18()
        {
            SortedDictionary<string, int> sd = new SortedDictionary<string, int>();

            // Dodajemy kilka par, śledząc sekwencję dodawania:
            int sequence = 0;
            sd.Add("kilka", sequence++);
            sd.Add("interesujących", sequence++);
            sd.Add("słów", sequence++);
            sd.Add("przeznaczonych", sequence++);
            sd.Add("do", sequence++);
            sd.Add("posortowania", sequence++);

            // Teraz iterujemy po liście:
            foreach (KeyValuePair<string, int> kvp in sd)
                Console.WriteLine("{0} : {1}", kvp.Key, kvp.Value);
        }

        // Przykład 19: kolekcje niestandardowe
        public class NotificationList<T> : Collection<T>
        {
            public event EventHandler<ItemInsertedArgs<T>> ItemAdded;

            protected override void InsertItem(int index, T item)
            {
                EventHandler<ItemInsertedArgs<T>> handler = ItemAdded;
                if (handler != null)
                {
                    handler(this, new ItemInsertedArgs<T>(index, item));
                }
                base.InsertItem(index, item);
            }
        }

        public class ItemInsertedArgs<T> : EventArgs
        {
            public int Index;
            public T Item;

            public ItemInsertedArgs(int index, T item)
            {
                this.Index = index;
                this.Item = item;
            }
        }

        public static void ex19()
        {
            NotificationList<int> list = new NotificationList<int>();

            // Rejestrujemy procedurę obsługi zdarzenia:
            list.ItemAdded += delegate(object o, ItemInsertedArgs<int> args) {
                Console.WriteLine("Do listy został dodany nowy element: {0} pod indeksem {1}",
                    args.Item, args.Index);
            };

            // Teraz wstawiamy kilka losowych liczb:
            Random r = new Random();
            for (int i = 0; i < 10; i++)
            {
                list.Add(r.Next());
            }
        }

        // Przykład 20: zastosowanie stosu (Stack<T>)
        public static void ex20()
        {
            Stack<int> s = new Stack<int>();

            s.Push(10); // Zawartość: 10
            s.Push(15); // Zawartość: 15, 10
            s.Push(3);  // Zawartość: 3, 15, 10

            Console.WriteLine(s.Peek()); // Wypisuje "3"

            Console.WriteLine(s.Pop()); // Wypisuje "3"; zawartość: 15, 10
            Console.WriteLine(s.Pop()); // Wypisuje "15"; zawartość: 10
            Console.WriteLine(s.Pop()); // Wypisuje "10"; zawartość: <pusty>
        }

        // Przykład 21: zastosowanie kolejki (Queue<T>)
        public static void ex21()
        {
            Queue<int> q = new Queue<int>();

            q.Enqueue(10); // Zawartość: 10
            q.Enqueue(15); // Zawartość: 15, 10
            q.Enqueue(3);  // Zawartość: 3, 15, 10

            Console.WriteLine(q.Peek()); // Wypisuje "10"

            Console.WriteLine(q.Dequeue()); // Wypisuje "10"; zawartość: 3, 15
            Console.WriteLine(q.Dequeue()); // Wypisuje "15"; zawartość: 3
            Console.WriteLine(q.Dequeue()); // Wypisuje "3";  zawartość: <pusta>
        }

        // Przykład 22: zastosowanie połączonej listy (LinkedList<T>)
        public static void ex22()
        {
            LinkedList<int> list = new LinkedList<int>();

            list.AddFirst(10);             // Zawartość: ->10
            list.AddLast(15);              // Zawartość: ->10->15
            list.AddLast(3);               // Zawartość: ->10->15->3
            list.AddLast(99);              // Zawartość: ->10->15->3->99
            list.AddBefore(list.Last, 25); // Zawartość: ->10->15->3->25->99

            LinkedListNode<int> node = list.First;
            while (node != null)
            {
                Console.WriteLine(node.Value);
                node = node.Next;
            }
        }

        // Przykład 23: typ implementujący interfejsy porównywania i równości
        class Person : IComparable<Person>, IEquatable<Person>, IComparable
        {
            public string Name;
            public int Age;
            public string Company;

            // Implementujemy IComparable<Person>.CompareTo:
            public int CompareTo(Person other)
            {
                if (other == null)
                    return -1;
                return this.Name.CompareTo(other.Name);
            }

            // Implementujemy IComparable.CompareTo:
            public int CompareTo(object obj)
            {
                Person p = obj as Person;
                return CompareTo(p);
            }

            // Implementujemy IEquatable<Person>.Equals:
            public bool Equals(Person other)
            {
                return ((IComparable<Person>)this).CompareTo(other) == 0;
            }

            // Przesłaniamy Object.Equals:
            public override bool Equals(object obj)
            {
                Person p = obj as Person;
                return Equals(p);
            }
        }

        public static void ex23()
        {
            Person[] people = new Person[3];
            people[0] = new Person();
            people[0].Name = "Szymon Gawda"; people[0].Age = 20; people[0].Company = "EMC";
            people[1] = new Person();
            people[1].Name = "Marek Machnik"; people[1].Age = 40; people[1].Company = "EMC";
            people[2] = new Person();
            people[2].Name = "Jan Dudar"; people[2].Age = 30; people[2].Company = "MSFT";

            foreach (Person p in people)
                Console.Write("'{0},{1},{2}'\t", p.Name, p.Age, p.Company);
            Console.WriteLine();

            Array.Sort(people);

            foreach (Person p in people)
                Console.Write("'{0},{1},{2}'\t", p.Name, p.Age, p.Company);
            Console.WriteLine();
        }

        // Przykład 24: niestandardowe porównywanie obiektów Person
        enum SortDirection
        {
            Ascending,
            Descending
        }

        class PersonAgeComparer : IComparer<Person>
        {
            private int modifier;

            public PersonAgeComparer(SortDirection direction)
            {
                if (direction == SortDirection.Ascending)
                    modifier = 1;
                else
                    modifier = -1;
            }

            public int Compare(Person x, Person y)
            {
                return x.Age.CompareTo(y.Age) * modifier;
            }
        }

        public static void ex24()
        {
            Person[] people = new Person[3];
            people[0] = new Person();
            people[0].Name = "Szymon Gawda"; people[0].Age = 20; people[0].Company = "EMC";
            people[1] = new Person();
            people[1].Name = "Marek Machnik"; people[1].Age = 40; people[1].Company = "EMC";
            people[2] = new Person();
            people[2].Name = "Jan Dudar"; people[2].Age = 30; people[2].Company = "MSFT";

            foreach (Person p in people)
                Console.Write("'{0},{1},{2}'\t", p.Name, p.Age, p.Company);
            Console.WriteLine();

            Array.Sort(people, new PersonAgeComparer(SortDirection.Ascending));

            foreach (Person p in people)
                Console.Write("'{0},{1},{2}'\t", p.Name, p.Age, p.Company);
            Console.WriteLine();
        }

        // Przykład 25: porównywanie oparte na delegacjach
        public static void ex25()
        {
            Person[] people = new Person[3];
            people[0] = new Person();
            people[0].Name = "Szymon Gawda"; people[0].Age = 20; people[0].Company = "EMC";
            people[1] = new Person();
            people[1].Name = "Marek Machnik"; people[1].Age = 40; people[1].Company = "EMC";
            people[2] = new Person();
            people[2].Name = "Jan Dudar"; people[2].Age = 30; people[2].Company = "MSFT";

            foreach (Person p in people)
                Console.Write("'{0},{1},{2}'\t", p.Name, p.Age, p.Company);
            Console.WriteLine();

            Array.Sort(people, delegate(Person x, Person y) {
                return x.Age.CompareTo(y.Age);
            });

            foreach (Person p in people)
                Console.Write("'{0},{1},{2}'\t", p.Name, p.Age, p.Company);
            Console.WriteLine();
        }

        // Przykład 26: kolekcja z obsługą akcji
        class ActionCollection<T> : Collection<T>
        {
            private Action<T> action;

            public ActionCollection(Action<T> action)
            {
                this.action = action;
            }

            protected override void InsertItem(int index, T item)
            {
                action(item);
                base.InsertItem(index, item);
            }
        }

        public static void ex26()
        {
            ActionCollection<string> ac = new ActionCollection<string>(
                delegate(string s) { Console.WriteLine(s); });
            ac.Add("To zostanie wypisane na konsoli");
        }

        // Przykład 27: kolekcja z obsługą przekształcania
        class ConverterCollection<T> : Collection<T>
        {
            private Converter<T,T> convert;

            public ConverterCollection(Converter<T,T> convert)
            {
                this.convert = convert;
            }

            protected override void InsertItem(int index, T item)
            {
                base.InsertItem(index, convert(item));
            }
        }

        public static void ex27()
        {
            ConverterCollection<string> c = new ConverterCollection<string>(
                delegate (string s) { return s.ToUpper(); });
            c.Add("Witaj");
            c.Add("świecie!");

            foreach (string s in c)
                Console.WriteLine(s);
        }
    }
}