﻿using System;
using System.Collections;
using System.Collections.Generic;
using System.Diagnostics;
using System.Globalization;
using System.IO;
using System.Reflection;
using System.Security;
using System.Security.Permissions;
using System.Security.Policy;
using System.Xml;

namespace Wrox.DotNetFramework2.Samples
{
    public class Chapter04
    {
        public void _title()
        {
            Console.WriteLine("Rozdział 4 - Podzespoły, wczytywanie i wdrażanie");
            Console.WriteLine("================================================");
        }

        /*** JENDOSTKI WDRAŻANIA, WYKONYWANIA I WIELOKROTNEGO UŻYTKU ***/

        // Przykład 1: przykładowy podzespół
        public static void ex01()
        {
            XmlDocument xml = new XmlDocument();
            xml.Load(Environment.GetCommandLineArgs()[0] + ".config");
            XmlTextWriter writer = new XmlTextWriter(Console.Out);
            writer.Formatting = Formatting.Indented;
            xml.WriteTo(writer);
        }

        // Przykład 2: konstruowanie nazwy podzespołu
        public static void ex02()
        {
            AssemblyName mscorlib = new AssemblyName();
            mscorlib.Name = "mscorlib";
            mscorlib.Version = new Version(2, 0, 0, 0);
            mscorlib.CultureInfo = CultureInfo.InvariantCulture;
            mscorlib.SetPublicKeyToken(new byte[] {
                0xb7, 0x7a, 0x5c, 056, 0x19, 0x34, 0xe0, 0x89 });
            mscorlib.ProcessorArchitecture = ProcessorArchitecture.X86;

            Console.WriteLine(mscorlib);
        }

        /*** WCZYTYWANIE PODZESPOŁÓW ***/

        // Przykład 3: zwalnianie podzespołu
        public static void ex03()
        {
            // Tworzymy nową domenę i wczytujemy do niej bibliotekę DLL:
            AppDomain ad = AppDomain.CreateDomain("Isolated");

            // ...

            // Teraz możemy usunąć z pamięci zawartość domeny:
            AppDomain.Unload(ad);
        }

        // Przykład 4: test zestawu roboczego
        public static void ex04()
        {
            Console.WriteLine("Przed: {0}", Environment.WorkingSet);

            // Wczytuje System.dll (jeśli nie jest jeszcze wczytana):
            Type uriType = typeof(Uri); 
            Console.WriteLine("Po wczytaniu System.dll: {0}", Environment.WorkingSet);

            // Wczytuje System.dll do 10 nowych AppDomain:
            List<AppDomain> ads = new List<AppDomain>();
            for (int i = 0; i < 10; i++)
            {
                AppDomain ad = AppDomain.CreateDomain(i.ToString());
                ad.DoCallBack(delegate { Type t = typeof(Uri); });
                ads.Add(ad);
            }
            Console.WriteLine("Po wczytaniu System.dll do 10 domen AppDomain: {0}", Environment.WorkingSet);

            // Zwalniamy domeny AppDomain i sprawdzamy zestaw roboczy:
            foreach (AppDomain ad in ads)
                AppDomain.Unload(ad);
            Console.WriteLine("Po zwolnieniu domen AppDomain: {0}", Environment.WorkingSet);
        }

        // Przykład 5: wczytywanie dynamiczne (przez Assembly.Load).
        interface IFoo
        {
            void Bar(string s);
        }

        public static void ex05()
        {
            // Tworzymy w pełni kwalifikowaną nazwę podzespołu:
            AssemblyName name = new AssemblyName();
            name.Name = "TestDll.dll";
            name.Version = new Version("1.0");
            name.KeyPair = new StrongNameKeyPair(File.OpenRead("TestDll.pubkey"));

            // Wczytujemy podzespół:
            Assembly a = Assembly.Load(name);

            // Tworzymy instancję typu z wczytanego podzespołu:
            IFoo foo = a.CreateInstance("Foo") as IFoo;
            if (foo != null)
                foo.Bar("Test");
        }

        // Przykład 6: wczytywanie tylko do refleksji
        public static void ex06()
        {
            string sysLocation = string.Format(@"{0}\microsoft.net\framework\{1}\system.dll",
                Environment.GetEnvironmentVariable("windir"), typeof(string).Assembly.ImageRuntimeVersion);
            Assembly sys = ReflectionOnlyLoadFrom(sysLocation);
            Array.ForEach<Assembly>(
                AppDomain.CurrentDomain.ReflectionOnlyGetAssemblies(),
                delegate(Assembly a) { Console.WriteLine("* " + a.FullName); });
        }

        const string currentAssemblyKey = "CurrentReflectionAssemblyBase";

        private static Assembly ReflectionOnlyLoadFrom(string assemblyPath)
        {
            AppDomain currentAd = AppDomain.CurrentDomain;
            ResolveEventHandler customResolveHandler =
                new ResolveEventHandler(CustomReflectionOnlyResolver);
            currentAd.ReflectionOnlyAssemblyResolve += customResolveHandler;

            // Zapisujemy w ALS katalog bazowy, z którego wczytujemy podzespół
            currentAd.SetData(currentAssemblyKey,
                Path.GetDirectoryName(assemblyPath));

            // Wczytujemy podzespoły i wymuszamy rozwikłanie zależności
            Assembly assembly = Assembly.ReflectionOnlyLoadFrom(assemblyPath);
            Type[] types = assembly.GetTypes();

            // Wreszcie zerujemy wpis ALS i usuwamy procedurę obsługi
            currentAd.SetData(currentAssemblyKey, null);
            currentAd.ReflectionOnlyAssemblyResolve -= customResolveHandler;

            return assembly;
        }

        private static Assembly CustomReflectionOnlyResolver(object sender, ResolveEventArgs e)
        {
            AssemblyName name = new AssemblyName(e.Name);
            string assemblyPath = Path.Combine(
                (string)AppDomain.CurrentDomain.GetData(currentAssemblyKey),
                name.Name + ".dll");

            if (File.Exists(assemblyPath))
            {
                // Podzespół zależny został znaleziony w tym samym katalogu
                return Assembly.ReflectionOnlyLoadFrom(assemblyPath);
            }
            else
            {
                // Nie znaleziono go na dysku, może znajdziemy go w GAC...
                return Assembly.ReflectionOnlyLoad(name.Name);
            }
        }

        // Przykład 7: przekazywanie typów
        //     (przykład pominięty, ponieważ potrzebne są dodatkowe podzespoły)

        /*
         * DLL a:
         
            using System.Windows.Forms;

            public class Bar
            {
                public void Baz()
                {
                    MessageBox.Show("Cześć, stary");
                }
            }

         * DLL b:
         
            using System;
            using System.Runtime.CompilerServices;

            [assembly:TypeForwardedTo(typeof(Bar))]

            public class Foo
            {
                public void Baz()
                {
                    // Jakiś przydatny kod...
                }
            }
        */
    }
}
