using System;
using System.Collections.Generic;
using System.Data;
using System.Data.SqlClient;
using System.Text;
using System.Transactions;
using System.Runtime.InteropServices;
using System.EnterpriseServices;

namespace Wrox.DotNetFramework2.Samples
{
    public class Chapter15
    {
        public static void _title()
        {
            Console.WriteLine("Rozdzia 15 - Transakcje");
            Console.WriteLine("========================");
        }

        /*** MODEL PROGRAMOWANIA TRANSAKCYJNEGO ***/

        // Przykad 1: zasigi transakcyjne
        public static void ex01()
        {
            using (TransactionScope tx = new TransactionScope())
            {
                // Pracujemy z zasobami transakcyjnymi...

                tx.Complete();
            }
        }

        // Przykad 2: jawne wycofywanie transakcji
        public static void ex02()
        {
            using (TransactionScope tx = new TransactionScope())
            {
                // Pracujemy z zasobami transakcyjnymi...

                // ...
                Transaction.Current.Rollback(new Exception("Stao si co zego"));
                // ...

                tx.Complete();
            }
        }

        // Przykad 3: zapobieganie zakleszczeniu
        public static void ex03()
        {
            using (TransactionScope tx =
                new TransactionScope(
                    TransactionScopeOption.RequiresNew,
                    new TimeSpan(0, 0, 1)))
            {
                // Operacje transakcyjne... (musz zakoczy si w cigu 1 s)

                // (Usypiamy wtek na wicej ni sekund, aby zasymulowa zakleszczenie...)
                System.Threading.Thread.Sleep(1050);

                tx.Complete();
            }

        }

        // Przykad 4: integracja z ADO.NET
        private const bool enableEx04 = false;
        public static void ex04()
        {
            if (enableEx04)
            {
                using (TransactionScope txScope = new TransactionScope())
                {
                    IDbConnection cn = /*...*/null;

                    IDbCommand cmd1 = cn.CreateCommand();
                    cmd1.CommandText = "INSERT ...";
                    cmd1.ExecuteNonQuery();

                    IDbCommand cmd2 = cn.CreateCommand();
                    cmd2.CommandText = "UPDATE ...";
                    cmd2.ExecuteNonQuery();

                    txScope.Complete();
                }
            }
        }

        // Przykad 5: zagniedone zasigi
        private static void ex05()
        {
            using (TransactionScope scope1 = new TransactionScope())
            {
                // Opcja TransactionScopeOption.Requires jest przyjmowana domylnie.
                // Transakcja T1 zostaje utworzona i jest aktywna w tym bloku.

                using (TransactionScope scope2 = new TransactionScope())
                {
                    // Opcja Requires znw jest przyjmowana domylnie, co powoduje ponowne
                    // wykorzystanie transakcji T1. Wszystkie operacje id na rachunek T1.

                    // Wywoanie Complete "gosuje" za zatwierdzeniem transakcji T1. Jeli metoda
                    // Complete nie zostanie wywoana, T1 jest skazana na zagad i zostanie wycofana.
                    // Metoda Dispose nie przetwarza fizycznie transakcji, poniewa "scope1"
                    // nie jest zasigiem gwnym transakcji T1.
                }

                B(); // B konstruuje now transakcj T2 w swoim zasigu.

                // Tutaj aktywna jest transakcja T1.

                C(); // C wstrzymuje transakcj T1 w swoim zasigu.

                // Znw jest aktywna transakcja T1

                // Jeli tutaj wywoamy Complete, "zagosujemy" za zatwierdzeniem T1.
                // Wywoanie Dispose na "scope1" powoduje fizyczne przetworzenie transakcji,
                // poniewa "scope1" jest zasigiem gwnym transakcji T1.
            }
        }

        private static void B()
        {
            using (TransactionScope scope2 =
                new TransactionScope(TransactionScopeOption.RequiresNew))
            {
                // Zawsze generuje now transakcj.

                // Nie znaleziono zagniedonych transakcji, wic zasig "scope2" jest
                // odpowiedzialny za gosowanie i fizyczne przetwarzanie nowej transakcji.
            }
        }

        private static void C()
        {
            using (TransactionScope scope3 =
                new TransactionScope(TransactionScopeOption.Suppress))
            {
                // Tutaj nie ma adnej aktywnej transakcji.
            }
        }

        // Przykad 6: transakcje zalene
        private static void ThreadPoolWorker(object state)
        {
            DependentTransaction dtx = (DependentTransaction)state;
            using (TransactionScope tx = new TransactionScope(dtx))
            {
                // Jestemy w tym samym kontekcie transakcyjnym, w ktrym
                // utworzona zostaa transakcja dtx.

                // Jakie operacje transakcyjne...

                tx.Complete();
            }
        }

        public static void ex06()
        {
            using (TransactionScope tx = new TransactionScope())
            {
                DependentTransaction dtx = Transaction.Current.DependentClone(
                    DependentCloneOption.BlockCommitUntilComplete);
                System.Threading.ThreadPool.QueueUserWorkItem(ThreadPoolWorker, dtx);

                // Jakie operacje transakcyjne...

                // Jeli dotrzemy do wywoania Complete, zanim wtek roboczy
                // z puli ThreadPool wykona swoje zadanie, wywoanie zablokuje si.
                tx.Complete();
            }
        }

        // Przykad 7: integracja z Enterprise Services
        [Transaction(TransactionOption.RequiresNew,
            Isolation = TransactionIsolationLevel.Serializable,
            Timeout = 30)]
        class MyComponent : ServicedComponent
        {
            [AutoComplete]
            public void Foo()
            {
                // Wykonujemy jakie operacje transakcyjne; jeli zakocz
                // si one powodzeniem, transakcja COM+ zostanie zatwierdzona.
            }

            public void Bar()
            {
                // Wykonujemy operacje transakcyjne. Tym razem zatwierdzamy je rcznie.
                ContextUtil.SetComplete();
            }
        }
    }

}
