﻿using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Text;

namespace Wrox.DotNetFramework2.Samples
{
    public class Chapter12
    {
        public static void _title()
        {
            Console.WriteLine("Rozdział 12 - Śledzenie i diagnostyka");
            Console.WriteLine("=====================================");
        }

        public static void __int()
        {
            DefaultTraceListener dtl = new DefaultTraceListener();
            dtl.AssertUiEnabled = true;
            Trace.Listeners.Add(dtl);
            Trace.Listeners.Add(new ConsoleTraceListener());
        }

        /*** ŚLEDZENIE ***/

        // Przykład 1: prosta struktura danych z śledzeniem
        class Customer
        {
            static TraceSource customerTraceSource = new TraceSource("Customer");
            public string Name;

            public Customer(string name)
            {
                if (name == null || name == string.Empty)
                    throw new ArgumentNullException("name");
                this.Name = name;
                Trace.WriteLine("Utworzono nowego klienta {0}", this.ToString());
            }

            public override string ToString()
            {
                Debug.Assert(this.Name != null && this.Name != string.Empty,
                    "Nazwisko klienta nie powinno tu być puste");
                if (Name == null)
                    customerTraceSource.TraceEvent(TraceEventType.Warning, 1000,
                        "Dane klienta mogą być uszkodzone (puste nazwisko)");
                return Name;
            }
        }

        public static void ex01()
        {
            Customer c = new Customer("Foo");
            c.Name = null;
            Console.WriteLine(c.ToString());
        }

        // Przykład 2: różne typy śledzenia
        static TraceSource myTraceSource = new TraceSource("myTraceSource");

        public static void ex02()
        {
            Customer customer = new Customer("Jan");
            myTraceSource.TraceEvent(TraceEventType.Error, 10000,
                "Nieprawidłowe nazwisko klienta {0}", customer.Name);
            myTraceSource.TraceData(TraceEventType.Error, 10001, customer);
            myTraceSource.TraceInformation("Poprawnie wczytano dane klienta z bazy");
            myTraceSource.Flush();
        }

        // Przykład 3: korzystanie z asercji
        private static void F(int x, int y)
        {
            Debug.Assert(x == y, "x nie jest równe y",
                "Ta metoda zakłada, że x i y są równe, a tymczasem nie są (x: " + x + ", y: " + y + "); dalsze wykonywanie może spowodować uszkodzenie danych.");
            // Wykonujemy jakąś pracę przy spełnionym niezmienniku...
        }

        public static void ex03()
        {
            F(0, 0);
            F(0, 1);
        }

        // Przykład 4: używanie przełączika źródła
        //    Można również włączyć za pomocą konfiguracji:
        //        <system.diagnostics>
        //            <sources>
        //                <source name="myTraceSource" switchValue="Error" />
        //            </sources>
        //        </system.diagnostics>
        public static void ex04()
        {
            TraceSource ts = new TraceSource("MyTraceSource");
            ts.Listeners.Add(new ConsoleTraceListener());
            ts.Switch = new SourceSwitch("mySource");
            ts.Switch.Level = SourceLevels.Error;

            ts.TraceEvent(TraceEventType.Information, 0, "Informacja");
            ts.TraceEvent(TraceEventType.Error, 0, "Błąd");
        }

        // Przykład 5: używanie przełącznika logicznego
        //     Można również włączyć za pomocą konfiguracji:
        //         <system.diagnostics>
        //             <switches>
        //                 <add name="testBooleanSwitch" value="true" />
        //             </switches>
        //         </system.diagnostics>
        public static void ex05()
        {
            BooleanSwitch testBooleanSwitch = new BooleanSwitch(
                "testBooleanSwitch", "Używany do globalnego wyłączania lub włączania śledzenia");
            Trace.WriteLineIf(testBooleanSwitch.Enabled, "To jest komunikat śledzenia");
        }

        // Przykład 6: używanie przełącznika śledzenia
        //     Można również włączyć za pomocą konfiguracji:
        //         <system.diagnostics>
        //             <switches>
        //                 <add name="testTraceSwitch" value="Verbose" />
        //             </switches>
        //         </system.diagnostics>
        public static void ex06()
        {
            TraceSwitch testTraceSwitch = new TraceSwitch(
                        "testTraceSwitch", "Wyłącza komunikaty na podstawie poziomu ważności");
            Trace.WriteLineIf(testTraceSwitch.TraceError, "Stało się coś złego");
            Trace.WriteLineIf(testTraceSwitch.TraceInfo, "Jakiś komunikat informacyjny");
        }
    }
}
