"""Tables and Figures for Chapter 4

   Learning Algorithms:
   A programmer's guide to writing better code
   Chapter 4: Heaping It On
   (C) 2021, George T. Heineman

"""
import timeit

from algs.table import DataTable, FigureNum, TableNum, process, caption

# Executes 3*N/2 add operations and 3*N/2 remove_max operations for a total of 3*N
def run_trials(clazz, N, factor):
    """Run a single trial."""
    stmt = '''
from {0} import PQ 
one_run(PQ({1}), {1}, {2})'''.format(clazz,N,factor)
    return min(timeit.repeat(stmt=stmt, setup = 'from ch04.timing import one_run',
                             repeat=5, number=10))/10

def average_performance(max_n=65536, output=True, decimals=1):
    """Generate table of average performance for different PQ implementations."""
    T = 3
    base = 256
    cutoff = 16384
    high = max_n

    heap = {}
    order_ar = {}
    order_ll = {}
    N = base
    while N <= high:
        order_ll[N] = 1000000*run_trials('ch04.ordered_list', N, T)/(T*N)
        heap[N]     = 1000000*run_trials('ch04.heap', N, T)/(T*N)
        N *= 2

    N = base
    array = {}
    linked = {}
    builtin = {}
    while N <= cutoff:
        order_ar[N]  = 1000000*run_trials('ch04.ordered', N, T)/(T*N)
        linked[N]    = 1000000*run_trials('ch04.linked', N, T)/(T*N)
        array[N]     = 1000000*run_trials('ch04.array', N, T)/(T*N)
        builtin[N]   = 1000000*run_trials('ch04.builtin', N, T)/(T*N)

        N *= 2

    N = base
    tbl = DataTable([8,8,8,8,8,8,8],
                    ['N','Kopiec','Posort. lista','Lista pow.','Posort. tab.','Wbud. listy','Tablica'],
                    output=output, decimals=decimals)
    while N <= high:
        if N <= cutoff:
            tbl.row([N, heap[N], order_ll[N], linked[N], order_ar[N], builtin[N], array[N]])
        else:
            #tbl.set_output(False)
            tbl.row([N, heap[N], order_ll[N]])
        N *= 2

    if output:
        print()
        print('Kopiec', tbl.best_model('Kopiec'))
        print('Posort. lista', tbl.best_model('Posort. lista'))
        print('Lista pow.', tbl.best_model('Lista pow.'))
        print('Posort. tab.', tbl.best_model('Posort. tab.'))
        print('Wbud. listy', tbl.best_model('Wbud. listy'))
        print('Tablica', tbl.best_model('Tablica'))
    return tbl

def output_heap(h):
    """Output a heap, roughly in ASCII with rows."""
    idx = 1
    offset = 16
    level = 0
    while idx < h.N:
        print('poziom {}\t'.format(level) + '|'.join([' {:>3} '.format(e.priority) for e in h.storage[idx:min(h.N+1,2*idx)]]))
        idx *= 2
        level += 1
        offset //= 2

def initial_heap():
    """Construct initial heap for figures."""
    from ch04.heap import PQ

    h = PQ(31)
    for i in [15, 13, 14, 9, 11, 12, 14, 8, 2, 1, 10, 8, 6, 9, 7, 4, 5]:
        h.enqueue(i, i)

    return h

def heap_enqueue_animation():
    """Show changes to storage with each swim()."""
    from ch04.entry import Entry

    heap = initial_heap()
    heap.N += 1
    heap.storage[heap.N] = Entry(12, 12)
    fig_num = 8
    print('Rys. 4-{:<2d} : '.format(fig_num),' -- |' + '|'.join([' {:>3} '.format(e.priority) for e in heap.storage[1:heap.N+1]]))
    print()
    child = heap.N
    while child > 1 and heap.less(child//2, child):
        heap.swap(child, child//2)
        child = child // 2
        fig_num += 1
        print('Rys. 4-{:<2d} : '.format(fig_num),' -- |' + '|'.join([' {:>3} '.format(e.priority) for e in heap.storage[1:heap.N+1]]))

def heap_dequeue_animation():
    """Show changes to storage with each sink()."""
    heap = initial_heap()
    heap.enqueue(12, 12)
    heap.enqueue(16, 16)
    print('Rys. 4-11 : ',' -- |' + '|'.join([' {:>3} '.format(e.priority) for e in heap.storage[1:heap.N+1]]))
    heap.storage[1] = heap.storage[heap.N]
    heap.storage[heap.N] = None
    heap.N -= 1
    fig_num = 13
    print('Rys. 4-{} : '.format(fig_num),' -- |' + '|'.join([' {:>3} '.format(e.priority) for e in heap.storage[1:heap.N+1]]))
    print()
    parent = 1
    while 2*parent <= heap.N:
        child = 2*parent
        if child < heap.N and heap.less(child, child+1):
            child += 1
        if not heap.less(parent, child):
            break
        heap.swap(child, parent)
        fig_num += 1
        print('Rys. 4-{} : '.format(fig_num),' -- |' + '|'.join([' {:>3} '.format(e.priority) for e in heap.storage[1:heap.N+1]]))

        parent = child

def generate_ch04():
    """Generate tables/figures for chapter 04."""
    chapter = 4

    with FigureNum(1) as figure_number:
        description  = 'Oczekiwanie w kolejce w klubie nocnym'
        label = caption(chapter, figure_number)
        print('Przerysowane przez grafika')
        print('{}. {}'.format(label, description))
        print()

    with FigureNum(2) as figure_number:
        description  = 'Model kolejki przed klubem nocnym z trzema węzłami'
        label = caption(chapter, figure_number)
        print('Przerysowane przez grafika')
        print('{}. {}'.format(label, description))
        print()

    with FigureNum(3) as figure_number:
        description  = 'Goście mogą wejść szybciej dzięki kupionej wejściówce'
        label = caption(chapter, figure_number)
        print('Przerysowane przez grafika')
        print('{}. {}'.format(label, description))
        print()

    # For full book output, remove "max_n=16384". Added to reduce time to generate all.
    with TableNum(1) as table_number:
        process(average_performance(max_n=16384),
                chapter, table_number,
                'Średnia wydajność operacji (czas w ns) dla instancji problemu o wielkości N',
                yaxis='Czas (w nanosekundach)')

    with FigureNum(4) as figure_number:
        description  = 'Złożoność O(log N) kopca (struktura Heap) jest lepsza niż złożoność O(N) pozostałych technik'
        label = caption(chapter, figure_number)
        print('Wygenerowane w Excelu')
        print('{}. {}'.format(label, description))
        print()

    with FigureNum(5) as figure_number:
        description  = 'Przykładowy kopiec binarny typu max'
        label = caption(chapter, figure_number)
        heap = initial_heap()
        output_heap(heap)
        print('Narysowany ręcznie')
        print('{}. {}'.format(label, description))
        print()

    with FigureNum(6) as figure_number:
        description  = 'Określanie liczby poziomów potrzebnych w kopcu binarnym z N elementami'
        label = caption(chapter, figure_number)
        print('Narysowany ręcznie')
        print('{}. {}'.format(label, description))
        print()

    with FigureNum(7) as figure_number:
        description  = 'Które z tych struktur są poprawnymi kopcami binarnymi typu max?'
        label = caption(chapter, figure_number)
        print('Narysowany ręcznie')
        print('{}. {}'.format(label, description))
        print()

    with FigureNum(8) as figure_number:
        description  = 'Pierwszy krok przy wstawianiu elementu polega na umieszczeniu go na następnej dostępnej pozycji'
        label = caption(chapter, figure_number)
        print('Narysowany ręcznie')
        print('{}. {}'.format(label, description))
        print()

    with FigureNum(9) as figure_number:
        description  = 'W drugim kroku element jest w razie potrzeby przenoszony o poziom w górę'
        label = caption(chapter, figure_number)
        print('Narysowany ręcznie')
        print('{}. {}'.format(label, description))
        print()

    with FigureNum(10) as figure_number:
        description  = 'W trzecim kroku element jest w razie potrzeby przenoszony o poziom w górę'
        label = caption(chapter, figure_number)
        heap2 = initial_heap()
        heap2.enqueue(12, 12)
        output_heap(heap2)
        print('Narysowany ręcznie')
        print('{}. {}'.format(label, description))
        print()

    with FigureNum(11) as figure_number:
        description  = 'Dodawanie elementu o priorytecie 16., który „wypływa” na samą górę'
        label = caption(chapter, figure_number)
        heap3 = initial_heap()
        heap3.enqueue(12, 12)
        heap3.enqueue(16, 16)
        output_heap(heap3)
        print('Narysowany ręcznie')
        print('{}. {}'.format(label, description))
        print()

    with FigureNum(12) as figure_number:
        description  = 'Pierwszy krok polega na usunięciu ostatniego elementu z dolnego poziomu'
        label = caption(chapter, figure_number)
        print('Narysowany ręcznie')
        print('{}. {}'.format(label, description))
        print()

    with FigureNum(13) as figure_number:
        description  = 'Nieprawidłowy kopiec powstały w wyniku wstawienia ostatniego elementu na poziomie 0.'
        label = caption(chapter, figure_number)
        print('Narysowany ręcznie')
        print('{}. {}'.format(label, description))
        print()

    with FigureNum(14) as figure_number:
        description  = 'Przestawienie górnego elementu z jego lewym dzieckiem (jest to dziecko o wyższym priorytecie)'
        label = caption(chapter, figure_number)
        print('Narysowany ręcznie')
        print('{}. {}'.format(label, description))
        print()

    with FigureNum(15) as figure_number:
        description  = '„Zatapianie” elementu o dodatkowy poziom'
        label = caption(chapter, figure_number)
        print('Narysowany ręcznie')
        print('{}. {}'.format(label, description))
        print()

    with FigureNum(16) as figure_number:
        description  = 'Wynikowy kopiec po „zatopieniu” elementu, by dotarł do prawidłowej lokalizacji'
        label = caption(chapter, figure_number)
        heap4 = initial_heap()
        heap4.enqueue(12, 12)
        heap4.enqueue(16, 16)
        heap4.dequeue()
        output_heap(heap4)
        print('Narysowany ręcznie')
        print('{}. {}'.format(label, description))
        print()

    with FigureNum(17) as figure_number:
        description  = 'Zapisywanie kopca binarnego typu max w tablicy'
        label = caption(chapter, figure_number)
        heap4 = initial_heap()
        heap4.enqueue(12, 12)
        print(' -- |' + '|'.join([' {:>3} '.format(e.priority) for e in heap4.storage[1:heap4.N+1]]))
        print('Narysowany ręcznie')
        print('{}. {}'.format(label, description))
        print()

    with FigureNum(18) as figure_number:
        description  = 'Zmiany w tablicy storage po dodaniu elementu do kopca z rysunku 4.8'
        label = caption(chapter, figure_number)
        heap_enqueue_animation()
        print('{}. {}'.format(label, description))
        print()

    with FigureNum(19) as figure_number:
        description  = 'Zmiany w tablicy storage po wywołaniu metody dequeue() dla kopca z rysunku 4.11'
        label = caption(chapter, figure_number)
        heap_dequeue_animation()
        print('{}. {}'.format(label, description))
        print()

    with FigureNum(20) as figure_number:
        description  = 'Używanie tablicy jako kolejki cyklicznej'
        label = caption(chapter, figure_number)
        print('Narysowany ręcznie')
        print('{}. {}'.format(label, description))
        print()

    with FigureNum(21) as figure_number:
        description  = 'Nowa struktura — kopiec silnia'
        label = caption(chapter, figure_number)
        print('Narysowany ręcznie')
        print('{}. {}'.format(label, description))
        print()

#######################################################################
if __name__ == '__main__':
    generate_ch04()
