#!/usr/bin/perl
# gen_dirs.pl - Wygenerowanie listy członków Ligi Historycznej
# w różnych formatach danych wyjściowych
# (ta wersja obsługuje również format XML).

use strict;
use warnings;
use DBI;

# Przyjmuje się założenie, że informacje o komputerze, użytkowniku i haśle będą pobrane z pliku opcji.

my $dsn = "DBI:mysql:sampdb;mysql_read_default_group=client";
my %conn_attrs = (RaiseError => 1, PrintError => 0, AutoCommit => 1);
my $dbh = DBI->connect ($dsn, undef, undef, \%conn_attrs);

#@ _TEXT_FORMAT_ENTRY_
sub text_format_entry
{
  printf "%s\n", format_name ($_[0]);
}
#@ _TEXT_FORMAT_ENTRY_

#@ _RTF_INIT_
sub rtf_init
{
  print "{\\rtf0\n";
  print "{\\fonttbl {\\f0 Times;}}\n";
  print "\\plain \\f0 \\fs24\n";
}
#@ _RTF_INIT_

#@ _RTF_CLEANUP_
sub rtf_cleanup
{
  print "}\n";
}
#@ _RTF_CLEANUP_

#@ _RTF_FORMAT_ENTRY_
sub rtf_format_entry
{
my $entry_ref = shift;

  printf "\\b Imię i nazwisko: %s\\b0\\par\n", format_name ($entry_ref);
  my $address = "";
  $address .= $entry_ref->{street}
                if defined ($entry_ref->{street});
  $address .= ", " . $entry_ref->{city}
                if defined ($entry_ref->{city});
  $address .= ", " . $entry_ref->{state}
                if defined ($entry_ref->{state});
  $address .= " " . $entry_ref->{zip}
                if defined ($entry_ref->{zip});
  print "Adres: $address\\par\n"
                if $address ne "";
  print "Telefon: $entry_ref->{phone}\\par\n"
                if defined ($entry_ref->{phone});
  print "E-mail: $entry_ref->{email}\\par\n"
                if defined ($entry_ref->{email});
  print "Zainteresowania: $entry_ref->{interests}\\par\n"
                if defined ($entry_ref->{interests});
  print "\\par\n";
}
#@ _RTF_FORMAT_ENTRY_

#@ _HTML_INIT_
sub html_init
{
  print "<html>\n";
  print "<head>\n";
  print "<title>Katalog członków Ligi Historycznej</title>\n";
  print "</head>\n";
  print "<body bgcolor=\"white\">\n";
  print "<h1>Katalog członków Ligi Historycznej</h1>\n";
}
#@ _HTML_INIT_

#@ _HTML_CLEANUP_
sub html_cleanup
{
  print "</body>\n";
  print "</html>\n";
}
#@ _HTML_CLEANUP_

#@ _HTML_FORMAT_ENTRY_
sub html_format_entry
{
my $entry_ref = shift;

  # Konwersja znaków &, ", > i < na odpowiadające im encje HTML,
  # (&amp;, &quot;, &gt, &lt;).
  foreach my $key (keys (%{$entry_ref}))
  {
    next unless defined ($entry_ref->{$key});
    $entry_ref->{$key} =~ s/&/&amp;/g;
    $entry_ref->{$key} =~ s/\"/&quot;/g;
    $entry_ref->{$key} =~ s/>/&gt;/g;
    $entry_ref->{$key} =~ s/</&lt;/g;
  }
  printf "<strong>Imię i nazwisko: %s</strong><br />\n", format_name ($entry_ref);
  my $address = "";
  $address .= $entry_ref->{street}
                if defined ($entry_ref->{street});
  $address .= ", " . $entry_ref->{city}
                if defined ($entry_ref->{city});
  $address .= ", " . $entry_ref->{state}
                if defined ($entry_ref->{state});
  $address .= " " . $entry_ref->{zip}
                if defined ($entry_ref->{zip});
  print "Adres: $address<br />\n"
                if $address ne "";
  print "Telefon: $entry_ref->{phone}<br />\n"
                if defined ($entry_ref->{phone});
  print "E-mail: $entry_ref->{email}<br />\n"
                if defined ($entry_ref->{email});
  print "Zainteresowania: $entry_ref->{interests}<br />\n"
                if defined ($entry_ref->{interests});
  print "<br />\n";
}
#@ _HTML_FORMAT_ENTRY_

#@ _XML_INIT_
sub xml_init
{
  print "<?xml version=\"1.0\"?>\n";
  print "<members>\n";
}
#@ _XML_INIT_

#@ _XML_CLEANUP_
sub xml_cleanup
{
  print "</members>\n";
}
#@ _XML_CLEANUP_

#@ _XML_FORMAT_ENTRY_
sub xml_format_entry
{
my $entry_ref = shift;

  print "  <member>\n";
  # Konwersja znaków &, ", > i < na odpowiadające im encje HTML,
  # (&amp;, &quot;, &gt, &lt;).
  foreach my $key (keys (%{$entry_ref}))
  {
    next unless defined ($entry_ref->{$key});
    $entry_ref->{$key} =~ s/&/&amp;/g;
    $entry_ref->{$key} =~ s/\"/&quot;/g;
    $entry_ref->{$key} =~ s/>/&gt;/g;
    $entry_ref->{$key} =~ s/</&lt;/g;
    printf "    <$key>%s</$key>\n", $entry_ref->{$key};
  }
  print "  </member>\n";
}
#@ _XML_FORMAT_ENTRY_

#@ _SWITCHBOX_1_
#@ _SWITCHBOX_WITHOUT_HTML_A_
# Tablica switchbox wymieniająca metody formatowania dla poszczególnych formatów danych wyjściowych.
my %switchbox =
(
  "text" =>                 # Metody do obsługi formatu zwykłego tekstu.
  {
    "init"    => undef,     # Inicjalizacja nie jest wymagana.
    "entry"   => \&text_format_entry,
    "cleanup" => undef      # Operacje czyszczące nie są wymagane.
  },
  "rtf" =>                  # Metody do obsługi formatu RTF.
  {
    "init"    => \&rtf_init,
    "entry"   => \&rtf_format_entry,
    "cleanup" => \&rtf_cleanup
#@ _SWITCHBOX_WITHOUT_HTML_A_
  },
  "html" =>                 # Metody do obsługi formatu HTML.
  {
    "init"    => \&html_init,
    "entry"   => \&html_format_entry,
    "cleanup" => \&html_cleanup
#@ _SWITCHBOX_WITHOUT_HTML_B_
  },
  "xml" =>                  # Metody do obsługi formatu XML.
  {
    "init"    => \&xml_init,
    "entry"   => \&xml_format_entry,
    "cleanup" => \&xml_cleanup
  }
);
#@ _SWITCHBOX_WITHOUT_HTML_B_
#@ _SWITCHBOX_1_

#@ _SWITCHBOX_2_
my $formats = join (" ", sort (keys (%switchbox)));
# Sprawdzenie, czy w wierszu poleceń został podany argument.
@ARGV == 1
  or die "Użycie: gen_dirx.pl format_type\nObsługiwane formaty: $formats\n";

# Określenie prawidłowego elementu dla argumentu wiersza poleceń.
# Jeżeli element nie zostanie znaleziony, podano nieobsługiwany format.
my $func_hashref = $switchbox{$ARGV[0]};

defined ($func_hashref)
  or die "Nieznany format: $ARGV[0]\nObsługiwane formaty: $formats\n";
#@ _SWITCHBOX_2_

#@ _MAIN_BODY_
# Wywołanie metody inicjalizującej, o ile taka istnieje.
&{$func_hashref->{init}} if defined ($func_hashref->{init});

# Pobranie wartości rekordu, o ile taki istnieje.
if (defined ($func_hashref->{entry}))
{
  my $sth = $dbh->prepare (qq{
              SELECT * FROM member ORDER BY last_name, first_name
            });
  $sth->execute ();
  while (my $entry_ref = $sth->fetchrow_hashref ("NAME_lc"))
  {
    # Przekazanie (przez odniesienie) wartości do funkcji formatującej.
    &{$func_hashref->{entry}} ($entry_ref);
  }
}

# Wywołanie metody czyszczącej, o ile taka istnieje.
&{$func_hashref->{cleanup}} if defined ($func_hashref->{cleanup});
#@ _MAIN_BODY_

$dbh->disconnect ();

#@ _FORMAT_NAME_
sub format_name
{
my $entry_ref = shift;

  my $name = $entry_ref->{first_name} . " " . $entry_ref->{last_name};
  if (defined ($entry_ref->{suffix}))     # Istnieje przyrostek dla nazwiska.
  {
    # Brak przecinka w przyrostkach typu I, II, III itd.
    $name .= "," unless $entry_ref->{suffix} =~ /^[IVX]+$/;
    $name .= " " . $entry_ref->{suffix}
  }
  return ($name);
}
#@ _FORMAT_NAME_
