#!/usr/bin/perl
# renewal_notify2.pl - Skrypt informuje członków Ligi Historycznej, którzy muszą odnowić 
# członkostwo (wymaga użycia modułu Mail::Sendmail).

# Można podać argumenty, takie jak identyfikatory członków Ligi, adresy e-mail
# lub nazwy plików. Identyfikatory są uznawane za wartości liczbowe, adresy e-mail
# za zawierające znak "@", wszystko pozostałe jest uznawane za nazwy plików. Po otwarciu
# pliku wartości w pierwszej kolumnie są interpretowane jako identyfikatory członków Ligi
# lub adresy e-mail (ale nie jako nazwy plików, co ma na celu uniknięcie rekurencji).

# Dzięki temu możesz w wierszu poleceń wyraźnie wskazać członków Ligi, którzy mają
# zostać poinformowani. Inna możliwość to umieszczenie listy członków Ligi w pliku,
# a program odczyta jego zawartość i wyśle powiadomienia. Dane wyjściowe skryptu
# need_renewal zawierają w pierwszej kolumnie identyfikatory członków Ligi.

use strict;
use warnings;
use DBI;
use Mail::Sendmail;

# Przetworzenie parametrów z wiersza poleceń, o ile zostały podane.

use Getopt::Long;
$Getopt::Long::ignorecase = 0; # Wielkość liter w opcjach ma znaczenie.
$Getopt::Long::bundling = 1;   # -uname = -u name, a nie -u -n -a -m -e

# Parametry domyślne, wszystkie początkowo niezdefiniowane.
my ($host_name, $password, $port_num, $socket_name, $user_name);

GetOptions (
  # =i oznacza, że po opcji trzeba podać wartość w postaci liczby całkowitej.
  # =s oznacza, że po opcji trzeba podać wartość w postaci ciągu tekstowego.
  "host|h=s"      => \$host_name,
  "password|p=s"  => \$password,
  "port|P=i"      => \$port_num,
  "socket|S=s"    => \$socket_name,
  "user|u=s"      => \$user_name
) or exit (1);

# Utworzenie źródła danych.
my $dsn = "DBI:mysql:sampdb";
$dsn .= ";host=$host_name" if $host_name;
$dsn .= ";port=$port_num" if $port_num;
$dsn .= ";mysql_socket=$socket_name" if $socket_name;
$dsn .= ";mysql_read_default_group=client";

# Nawiązanie połączenia z serwerem.
my %conn_attrs = (RaiseError => 1, PrintError => 0, AutoCommit => 1);
my $dbh = DBI->connect ($dsn, $user_name, $password, \%conn_attrs);

#@ _MAIN_LOOP_
if (@ARGV == 0)   # Brak argumentów, odczyt wartości z STDIN.
{
  read_file (\*STDIN);
}
else
{
  while (my $arg = shift (@ARGV))
  {
    # Interpretacja argumentu wraz z rekurencją nazwy pliku.
    interpret_argument ($arg, 1);
  }
}
#@ _MAIN_LOOP_

$dbh->disconnect ();

#@ _READ_FILE_
sub read_file
{
my $fh = shift;   # Uchwyt do już otworzonego pliku.
my $arg;

  while (defined ($arg = <$fh>))
  {
    # Usunięcie wszystkiego (także znaku nowego wiersza) poza pierwszą kolumną.
    $arg =~ s/\s.*//s;
    # Interpretacja argumentu bez  rekurencji nazwy pliku.
    interpret_argument ($arg, 0);
  }
}
#@ _READ_FILE_

#@ _INTERPRET_ARGUMENT_
sub interpret_argument
{
my ($arg, $recurse) = @_;

  if ($arg =~ /^\d+$/)    # Liczbowy identyfikator członka Ligi.
  {
    notify_member ($arg);
  }
  elsif ($arg =~ /@/)     # Adres e-mail.
  {
    # Pobranie identyfikatora członka Ligi powiązanego z adresem
     # (powinien być dokładnie jeden).
    my $stmt = qq{ SELECT member_id FROM member WHERE email = ? };
    my $ary_ref = $dbh->selectcol_arrayref ($stmt, undef, $arg);
    if (scalar (@{$ary_ref}) == 0)
    {
      warn "Dla podanego adresu e-mail $arg nie znaleziono rekordu: będzie zignorowany\n";
    }
    elsif (scalar (@{$ary_ref}) > 1)
    {
      warn "Dla podanego adresu e-mail $arg znaleziono kilka rekordów: będzie zignorowany\n";
    }
    else
    {
      notify_member ($ary_ref->[0]);
    }
  }
  else                    # Nazwa pliku.
  {
    if (!$recurse)
    {
      warn "Nazwa pliku $arg wewnątrz nazwy pliku: będzie zignorowana.\n";
    }
    else
    {
      open (IN, $arg) or die "Nie można otworzyć pliku $arg: $!\n";
      read_file (\*IN);
      close (IN);
    }
  }
}
#@ _INTERPRET_ARGUMENT_

# Poinformowanie członka Ligi o konieczności odnowienia członkostwa.

#@ _NOTIFY_MEMBER_
sub notify_member
{
my $member_id = shift;

  warn "Wysyłanie wiadomości do członka Ligi o identyfikatorze $member_id...\n";
  my $stmt = qq{ SELECT * FROM member WHERE member_id = ? };
  my $sth = $dbh->prepare ($stmt);
  $sth->execute ($member_id);
  my @col_name = @{$sth->{NAME}};
  my $entry_ref = $sth->fetchrow_hashref ();
  $sth->finish ();
  if (!$entry_ref)                      # Nie znaleziono członka Ligi!
  {
    warn "Nie znaleziono rekordu dla członka Ligi o identyfikatorze $member_id!\n";
    return;
  }
  if (!defined ($entry_ref->{email}))   # Brak podanego adresu e-mail.
  {
    warn "Osoba $member_id nie podała adresu e-mail; wiadomość nie została wysłana.\n";
    return;
  }

$entry_ref->{email} = "paul\@localhost";
  # construct the message body
  my $message =<< "EOF";
Witaj! Twoje członkostwo w Lidze Historycznej wkrótce wygaśnie.
Mamy nadzieję, że poświęcisz kilka minut na kontakt z biurem
Ligi Historycznej i odnowisz członkostwo. Poniżej przedstawiono
informacje, które podałeś o sobie do katalogu członków Ligi. Prosimy
o zwrócenie szczególnej uwagi na datę wygaśnięcia członkostwa.
Dziękujemy!

EOF
  foreach my $col_name (@col_name)
  {
    $message .= "$col_name:";
    $message .= " $entry_ref->{$col_name}"
            if defined ($entry_ref->{$col_name});
    $message .= "\n";
  }

  # Utworzenie tablicy zawierającej odpowiednie informacje.
  my $from = (getlogin () || getpwuid ($<)) . "\@localhost";
  my %mailhash = (
    From  => $from,
    To    => $entry_ref->{email},
    Subject => "Przypominamy o konieczności odnowienia członkostwa w Lidze Historycznej",
    Message => $message
  );
  sendmail (%mailhash)
    or die "Nie można wysłać wiadomości na adres $entry_ref->{email}: $!\n";
}
#@ _NOTIFY_MEMBER_
