/*
 * prepared_call.c - Program pokazuje, jak używać zapytań preinterpretowanych
 * do obsługi zapytań CALL i uzyskania dostępu do ostatecznych wartości OUT/INOUT 
 * parametrów procedury.
 *
 * W celu zachowania prostoty przyjęto założenie, że wszystkie parametry i kolumny
 * zbioru wynikowego są typu INT (a tym samym rodzaj bufora to MYSQL_TYPE_LONG).
 *
 * Do zrobienia:
 * - Być może coś innego z SERVER_PS_OUT_PARAMS.
 */

#include <my_global.h>
#include <my_sys.h>
#include <m_string.h>   /* Dla strdup(). */
#include <mysql.h>
#include <my_getopt.h>

#ifdef HAVE_OPENSSL
enum options_client
{
  OPT_SSL_SSL=256,
  OPT_SSL_KEY,
  OPT_SSL_CERT,
  OPT_SSL_CA,
  OPT_SSL_CAPATH,
  OPT_SSL_CIPHER,
  OPT_SSL_VERIFY_SERVER_CERT
};
#endif

static char *opt_host_name = NULL;    /* Komputer serwera (domyślnie = localhost). */
static char *opt_user_name = NULL;    /* Nazwa użytkownika (domyślnie = nazwa logowania). */
static char *opt_password = NULL;     /* Hasło (domyślnie = brak). */
static unsigned int opt_port_num = 0; /* Numer portu (użyj wbudowanej wartości). */
static char *opt_socket_name = NULL;  /* Nazwa gniazda (użyj wbudowanej wartości). */
static char *opt_db_name = NULL;      /* Nazwa bazy danych (domyślnie = brak). */
static unsigned int opt_flags = 0;    /* Flagi połączenia (brak). */

#include <sslopt-vars.h>

static int ask_password = 0;          /* Czy program ma pytać o hasło? */

static const char *client_groups[] = { "client", NULL };

static struct my_option my_opts[] =   /* Struktury informacji opcji. */
{
  {"help", '?', "Wyświetlenie tego komunikatu pomocy i zakończenie programu.",
  NULL, NULL, NULL,
  GET_NO_ARG, NO_ARG, 0, 0, 0, 0, 0, 0},
  {"host", 'h', "Komputer, z którym chcesz się połączyć.",
  (uchar **) &opt_host_name, NULL, NULL,
  GET_STR, REQUIRED_ARG, 0, 0, 0, 0, 0, 0},
  {"password", 'p', "Hasło.",
  (uchar **) &opt_password, NULL, NULL,
  GET_STR, OPT_ARG, 0, 0, 0, 0, 0, 0},
  {"port", 'P', "Numer portu",
  (uchar **) &opt_port_num, NULL, NULL,
  GET_UINT, REQUIRED_ARG, 0, 0, 0, 0, 0, 0},
  {"socket", 'S', "Ścieżka dostępu do gniazda.",
  (uchar **) &opt_socket_name, NULL, NULL,
  GET_STR, REQUIRED_ARG, 0, 0, 0, 0, 0, 0},
  {"user", 'u', "Nazwa użytkownika.",
  (uchar **) &opt_user_name, NULL, NULL,
  GET_STR, REQUIRED_ARG, 0, 0, 0, 0, 0, 0},

#include <sslopt-longopts.h>

  { NULL, 0, NULL, NULL, NULL, NULL, GET_NO_ARG, NO_ARG, 0, 0, 0, 0, 0, 0 }
};

/*
 * Wyświetlenie komunikatu diagnostycznego. Jeżeli conn ma wartość inną niż NULL,
 * następuje wyświetlenie zwróconych przez serwer informacji o błędzie.
 */

static void
print_error (MYSQL *conn, char *message)
{
  fprintf (stderr, "%s\n", message);
  if (conn != NULL)
  {
    fprintf (stderr, "Błąd %u (%s): %s\n",
             mysql_errno (conn), mysql_sqlstate (conn), mysql_error (conn));
  }
}

/* Funkcja podobna do print_error(), ale w celu uzyskania dostępu do informacji
 * o błędzie używa uchwytu zapytania zamiast uchwytu połączenia.
 */

static void
print_stmt_error (MYSQL_STMT *stmt, char *message)
{
  fprintf (stderr, "%s\n", message);
  if (stmt != NULL)
  {
    fprintf (stderr, "Błąd %u (%s): %s\n",
             mysql_stmt_errno (stmt),
             mysql_stmt_sqlstate (stmt),
             mysql_stmt_error (stmt));
  }
}

static my_bool
get_one_option (int optid, const struct my_option *opt, char *argument)
{
  switch (optid)
  {
  case '?':
    my_print_help (my_opts);  /* Wyświetlenie komunikatu pomocy. */
    exit (0);
  case 'p':                   /* Hasło. */
    if (!argument)            /* Brak podanej wartości; należy poprosić o nią później. */
      ask_password = 1;
    else                      /* Skopiowanie hasła i nadpisanie początkowego. */
    {
      opt_password = strdup (argument);
      if (opt_password == NULL)
      {
        print_error (NULL, "Nie udało się alokować bufora hasła.");
        exit (1);
      }
      while (*argument)
        *argument++ = 'x';
      ask_password = 0;
    }
    break;
#include <sslopt-case.h>
  }
  return (0);
}

/*
 * Przetworzenie jednego zbioru wynikowego z wykonywanego zapytania składowanego.
 * Obsługa jedynie kolumn zawierających wartości INT (MYSQL_TYPE_LONG), choć
 * wartościami może być NULL.
 *
 * Wartością zwrotną jest zero w przypadku sukcesu lub wartość niezerowa, jeśli wystąpi błąd.
 */

/* #@ _PROCESS_RESULT_SET_ */
static int process_result_set (MYSQL_STMT *stmt, int num_cols)
{
MYSQL_FIELD *fields;     /* Wskaźnik do metadanych zbioru wynikowego. */
MYSQL_BIND  *params;     /* Wskaźnik do buforów danych wyjściowych. */
int         int_data[3];  /* Wartości parametru danych wyjściowych. */
my_bool     is_null[3];  /* Czy parametr danych wyjściowych ma wartość NULL?*/
int         i;

  MYSQL_RES *metadata = mysql_stmt_result_metadata (stmt);
  if (metadata == NULL)
  {
    print_stmt_error (stmt, "Nie można pobrać metadanych zbioru wynikowego.");
    return (1);
  }

  fields = mysql_fetch_fields (metadata);

  params = (MYSQL_BIND *) malloc (sizeof (MYSQL_BIND) * num_cols);
  if (!params)
  {
    print_stmt_error (NULL, "Nie można zaalokować parametrów danych wyjściowych.");
    return (1);
  }

  /* Inicjalizacja struktur parametru i dołączenie ich do zapytania. */
  memset (params, 0, sizeof (MYSQL_BIND) * num_cols);

  for (i = 0; i < num_cols; ++i)
  {
    params[i].buffer_type = fields[i].type;
    params[i].is_null = &is_null[i];

    if (fields[i].type != MYSQL_TYPE_LONG)
    {
      fprintf (stderr, "BŁĄD: nieoczekiwany typ: %d.\n", fields[i].type);
      return (1);
    }
    params[i].buffer = (char *) &(int_data[i]);
    params[i].buffer_length = sizeof (int_data);
  }
  mysql_free_result (metadata); /* Koniec pracy z metadanymi, można je zwolnić. */

  if (mysql_stmt_bind_result (stmt, params))
  {
    print_stmt_error (stmt, "Wyniku nie można dołączyć do buforów danych wyjściowych.");
    return (1);
  }

  /* Pobranie rekordów zbioru wynikowego i wyświetlenie jego zawartości. */
  while (1)
  {
    int status = mysql_stmt_fetch (stmt);
    if (status == 1 || status == MYSQL_NO_DATA)
      break;  /* Nie ma więcej rekordów. */

    for (i = 0; i < num_cols; ++i)
    {
      printf (" val[%d] = ", i+1);
      if (params[i].buffer_type != MYSQL_TYPE_LONG)
        printf ("  nieoczekiwany typ (%d)\n", params[i].buffer_type);
      else
      {
        if (*params[i].is_null)
          printf ("NULL;");
        else
          printf ("%ld;", (long) *((int *) params[i].buffer));
      }
    }
    printf ("\n");
  }

  free (params); /* Koniec pracy z buforami danych wyjściowych, można je zwolnić. */
  return (0);
}
/* #@ _PROCESS_RESULT_SET_ */

/*
 * Przetworzenie wyników wygenerowanych przez wykonanie zapytania CALL.
 */

/* #@ _PROCESS_CALL_RESULT_ */
static void process_call_result (MYSQL *conn, MYSQL_STMT *stmt)
{
int status;
int num_cols;

  /*
   * Trzeba sprawdzić liczbę kolumn w każdym wyniku. Jeżeli wynik nie zawiera żadnej,
   * to jest ostatecznym pakietem stanu i oznacza brak kolejnych danych do przetworzenia.
   * Wartość niezerowa oznacza konieczność pobrania zbioru wynikowego.
   */
  do {
    if ((num_cols = mysql_stmt_field_count (stmt)) > 0)
    {
      /* Poinformowanie, czy zbiór wynikowy zawiera parametry lub dane. */
      if (conn->server_status & SERVER_PS_OUT_PARAMS)
        printf ("Wartości parametrów OUT/INOUT:\n");
      else
        printf ("Wartości zbioru wynikowego zapytania:\n");

      if (process_result_set (stmt, num_cols))
        break; /* Wystąpił błąd. */
    }

    /* Wartość stanu: –1 = koniec pracy, 0 = więcej wyników, >0 = błąd. */
    status = mysql_stmt_next_result (stmt);
    if (status > 0)
      print_stmt_error (stmt, "d w trakcie sprawdzania istnienia kolejnego zbioru wyników.");
  } while (status == 0);
}
/* #@ _PROCESS_CALL_RESULT_ */

/*
 * Wykonanie preinterpretowanego zapytania CALL.
 *
 * Wartością zwrotną jest zero w przypadku sukcesu lub wartość niezerowa, jeśli wystąpi błąd.
 */

/* #@ _EXEC_PREPARED_CALL_ */
static int exec_prepared_call (MYSQL_STMT *stmt)
{
MYSQL_BIND params[3];   /* Bufory parametru. */
int        int_data[3]; /* Wartości parametru. */
int        i;

  /* Przygotowanie zapytania CALL. */
  if (mysql_stmt_prepare (stmt, "CALL grade_event_stats(?, ?, ?)", 31))
  {
    print_stmt_error (stmt, "Nie można przygotować zapytania.");
    return (1);
  }

  /* Inicjalizacja struktur parametru i dołączenie ich do zapytania. */
  memset (params, 0, sizeof (params));

  for (i = 0; i < 3; ++i)
  {
    params[i].buffer_type = MYSQL_TYPE_LONG;
    params[i].buffer = (char *) &int_data[i];
    params[i].length = 0;
    params[i].is_null = 0;
  }

  if (mysql_stmt_bind_param (stmt, params))
  {
    print_stmt_error (stmt, "Nie można dołączyć parametrów.");
    return (1);
  }

  /* Przypisanie wartości parametrów i wykonanie zapytania. */
  int_data[0]= 4;  /* p_event_id */
  int_data[1]= 0;  /* p_min (Parametr OUT; wartość początkowa ignorowana przez procedurę). */
  int_data[2]= 0;  /* p_min (Parametr OUT; wartość początkowa ignorowana przez procedurę). */

  if (mysql_stmt_execute (stmt))
  {
    print_stmt_error (stmt, "Nie można wykonać zapytania.");
    return (1);
  }
  return (0);
}
/* #@ _EXEC_PREPARED_CALL_ */

int
main (int argc, char *argv[])
{
int        opt_err;
MYSQL      *conn;   /* Wskaźnik do uchwytu połączenia. */
MYSQL_STMT *stmt;   /* Wskaźnik do uchwytu zapytania. */

  MY_INIT (argv[0]);
  load_defaults ("my", client_groups, &argc, &argv);

  if ((opt_err = handle_options (&argc, &argv, my_opts, get_one_option)))
    exit (opt_err);

  /* Wyświetlenie pytania o hasło, o ile zachodzi potrzeba. */
  if (ask_password)
    opt_password = get_tty_password (NULL);

  /* Pobranie nazwy bazy danych, jeśli została podana w wierszu poleceń. */
  if (argc > 0)
  {
    opt_db_name = argv[0];
    --argc; ++argv;
  }

  /* Inicjalizacja biblioteki klienta. */
  if (mysql_library_init (0, NULL, NULL))
  {
    print_error (NULL, "Wywołanie mysql_library_init() zakończyło się niepowodzeniem.");
    exit (1);
  }

  /* Inicjalizacja uchwytu połączenia. */
  conn = mysql_init (NULL);
  if (conn == NULL)
  {
    print_error (NULL, "Wywołanie mysql_init() zakończyło się niepowodzeniem (prawdopodobnie z powodu braku pamięci).");
    exit (1);
  }

#ifdef HAVE_OPENSSL
  /* Przekazanie bibliotece klienta informacji SSL. */
  if (opt_use_ssl)
    mysql_ssl_set (conn, opt_ssl_key, opt_ssl_cert, opt_ssl_ca,
                   opt_ssl_capath, opt_ssl_cipher);
  mysql_options (conn,MYSQL_OPT_SSL_VERIFY_SERVER_CERT,
                 (char*)&opt_ssl_verify_server_cert);
#endif

  /* Nawiązanie połączenia z serwerem. */
  if (mysql_real_connect (conn, opt_host_name, opt_user_name, opt_password,
      opt_db_name, opt_port_num, opt_socket_name, opt_flags) == NULL)
  {
    print_error (conn, "Wywołanie mysql_real_connect() zakończyło się niepowodzeniem.");
    mysql_close (conn);
    exit (1);
  }

/* #@ _VERIFY_SERVER_VERSION_ */
  if (mysql_get_server_version (conn) < 50503)
  {
    print_error (NULL, "Wywołanie CALL wymaga serwera MySQL 5.5.3 lub nowszego");
    mysql_close (conn);
    exit (1);
  }
/* #@ _VERIFY_SERVER_VERSION_ */

  /* Inicjalizacja uchwytu zapytania, wykonanie zapytania CALL i zamknięcie uchwytu. */
/* #@ _CALL_PROCEDURE_ */
  stmt = mysql_stmt_init (conn);
  if (!stmt)
    print_error (NULL, "Nie można zainicjalizować uchwytu zapytania.");
  else
  {
    if (exec_prepared_call (stmt) == 0)
      process_call_result (conn, stmt);
    mysql_stmt_close (stmt);
  }
/* #@ _CALL_PROCEDURE_ */

  /* Zamknięcie połączenia z serwerem i zamknięcie biblioteki klienta. */
  mysql_close (conn);
  mysql_library_end ();
  exit (0);
}
