/*
 * exec_stmt.c -  Prosty program do interaktywnego wykonywania zapytań.
 */

#include <my_global.h>
#include <my_sys.h>
#include <m_string.h>   /* Dla strdup(). */
#include <mysql.h>
#include <my_getopt.h>

static char *opt_host_name = NULL;    /* Komputer serwera (domyślnie = localhost). */
static char *opt_user_name = NULL;    /* Nazwa użytkownika (domyślnie = nazwa logowania). */
static char *opt_password = NULL;     /* Hasło (domyślnie = brak). */
static unsigned int opt_port_num = 0; /* Numer portu (użyj wbudowanej wartości). */
static char *opt_socket_name = NULL;  /* Nazwa gniazda (użyj wbudowanej wartości). */
static char *opt_db_name = NULL;      /* Nazwa bazy danych (domyślnie = brak). */
static unsigned int opt_flags = 0;    /* Flagi połączenia (brak). */

static int ask_password = 0;          /* Czy program ma pytać o hasło? */

static MYSQL *conn;                   /* Wskaźnik do uchwytu połączenia. */

static const char *client_groups[] = { "client", NULL };

static struct my_option my_opts[] =   /* Struktury informacji opcji. */
{
  {"help", '?', "Wyświetlenie tego komunikatu pomocy i zakończenie programu.",
  NULL, NULL, NULL,
  GET_NO_ARG, NO_ARG, 0, 0, 0, 0, 0, 0},
  {"host", 'h', "Komputer, z którym chcesz się połączyć.",
  (uchar **) &opt_host_name, NULL, NULL,
  GET_STR, REQUIRED_ARG, 0, 0, 0, 0, 0, 0},
  {"password", 'p', "Hasło.",
  (uchar **) &opt_password, NULL, NULL,
  GET_STR, OPT_ARG, 0, 0, 0, 0, 0, 0},
  {"port", 'P', "Numer portu",
  (uchar **) &opt_port_num, NULL, NULL,
  GET_UINT, REQUIRED_ARG, 0, 0, 0, 0, 0, 0},
  {"socket", 'S', "Ścieżka dostępu do gniazda.",
  (uchar **) &opt_socket_name, NULL, NULL,
  GET_STR, REQUIRED_ARG, 0, 0, 0, 0, 0, 0},
  {"user", 'u', "Nazwa użytkownika.",
  (uchar **) &opt_user_name, NULL, NULL,
  GET_STR, REQUIRED_ARG, 0, 0, 0, 0, 0, 0},
  { NULL, 0, NULL, NULL, NULL, NULL, GET_NO_ARG, NO_ARG, 0, 0, 0, 0, 0, 0 }
};

/*
 * Wyświetlenie komunikatu diagnostycznego. Jeżeli conn ma wartość inną niż NULL,
 * następuje wyświetlenie zwróconych przez serwer informacji o błędzie.
 */

static void
print_error (MYSQL *conn, char *message)
{
  fprintf (stderr, "%s\n", message);
  if (conn != NULL)
  {
    fprintf (stderr, "Błąd %u (%s): %s\n",
             mysql_errno (conn), mysql_sqlstate (conn), mysql_error (conn));
  }
}

static my_bool
get_one_option (int optid, const struct my_option *opt, char *argument)
{
  switch (optid)
  {
  case '?':
    my_print_help (my_opts);  /* Wyświetlenie komunikatu pomocy. */
    exit (0);
  case 'p':                   /* Hasło. */
    if (!argument)            /* Brak podanej wartości; należy poprosić o nią później. */
      ask_password = 1;
    else                      /* Skopiowanie hasła i nadpisanie początkowego. */
    {
      opt_password = strdup (argument);
      if (opt_password == NULL)
      {
        print_error (NULL, "Nie udało się alokować bufora hasła.");
        exit (1);
      }
      while (*argument)
        *argument++ = 'x';
      ask_password = 0;
    }
    break;
  }
  return (0);
}

#include "process_result_set.c"

#include "process_statement.c"

int
main (int argc, char *argv[])
{
int opt_err;

  MY_INIT (argv[0]);
  load_defaults ("my", client_groups, &argc, &argv);

  if ((opt_err = handle_options (&argc, &argv, my_opts, get_one_option)))
    exit (opt_err);

  /* Wyświetlenie pytania o hasło, o ile zachodzi potrzeba. */
  if (ask_password)
    opt_password = get_tty_password (NULL);

  /* Pobranie nazwy bazy danych, jeśli została podana w wierszu poleceń. */
  if (argc > 0)
  {
    opt_db_name = argv[0];
    --argc; ++argv;
  }

  /* Inicjalizacja biblioteki klienta. */
  if (mysql_library_init (0, NULL, NULL))
  {
    print_error (NULL, "Wywołanie mysql_library_init() zakończyło się niepowodzeniem.");
    exit (1);
  }

  /* Inicjalizacja uchwytu połączenia. */
  conn = mysql_init (NULL);
  if (conn == NULL)
  {
    print_error (NULL, "Wywołanie mysql_init() zakończyło się niepowodzeniem (prawdopodobnie z powodu braku pamięci).");
    exit (1);
  }

  /* Nawiązanie połączenia z serwerem. */
  if (mysql_real_connect (conn, opt_host_name, opt_user_name, opt_password,
      opt_db_name, opt_port_num, opt_socket_name, opt_flags) == NULL)
  {
    print_error (conn, "Wywołanie mysql_real_connect() zakończyło się niepowodzeniem.");
    mysql_close (conn);
    exit (1);
  }

/* #@ _MAIN_LOOP_ */
  while (1)
  {
    char  buf[10000];

    fprintf (stderr, "query> ");                  /* Wyświetlenie znaku zachęty. */
    if (fgets (buf, sizeof (buf), stdin) == NULL) /* Odczyt polecenia. */
      break;
    if (strcmp (buf, "quit\n") == 0 || strcmp (buf, "\\q\n") == 0)
      break;
    process_statement (conn, buf);                /* Wykonanie zapytania. */
  }
/* #@ _MAIN_LOOP_ */

  /* Zamknięcie połączenia z serwerem i zamknięcie biblioteki klienta. */
  mysql_close (conn);
  mysql_library_end ();
  exit (0);
}
