/*
** Globalna tabela adresw uytkownikw i adresw uywanych na fakturach
*/
DROP TABLE IF EXISTS address;
CREATE TABLE address (
        ID INT NOT NULL AUTO_INCREMENT,
        Name_Prefix CHAR(16),
        Name_First CHAR(255),
        Name_Middle CHAR(255),
        Name_Last CHAR(255) NOT NULL,
        Name_Suffix CHAR(16),
        Company CHAR(255),
        Street1 CHAR(255),
        Street2 CHAR(255),
        Street3 CHAR(255),
        City  CHAR(255) NOT NULL,
        StateProv CHAR(255) NOT NULL,
        PostalCode CHAR(255) NOT NULL,
        CountryCode CHAR(2), /* ISO 3166 */
        Phone1 CHAR(32),
        Phone2 CHAR(32),
        Fax CHAR(32),
        Email CHAR(255) NOT NULL,
        

        PRIMARY KEY(ID)
);


/*
** Atrybuty towaru, SKU (kolor, rozmiar)
*/
DROP TABLE IF EXISTS attribute;
CREATE TABLE attribute (
  ID INT NOT NULL AUTO_INCREMENT,
  Name CHAR(64) NOT NULL,
  Graphic CHAR(255),
  DisplayPrecedence INT NOT NULL,
  
  PRIMARY KEY(ID),
  INDEX (DisplayPrecedence, Name)
);


/*
** Rodzaje patnoci za faktur (Visa, MasterCard)
*/
DROP TABLE IF EXISTS billing;
CREATE TABLE billing (
  ID INT NOT NULL AUTO_INCREMENT,
  ECML CHAR(4),
  Name CHAR(64) NOT NULL,
  DisplayPrecedence INT NOT NULL,

  PRIMARY KEY(ID),
  INDEX (DisplayPrecedence)
);


/*
** Rabaty na fakturze
*/
DROP TABLE IF EXISTS coupon;
CREATE TABLE coupon (
  ID INT DEFAULT '0' NOT NULL AUTO_INCREMENT,
  Name CHAR(32) NOT NULL,
  DollarOff DECIMAL(5,2),
  PercentageOff DECIMAL(5,2),
  MinAmountPurchased DECIMAL(5,2),
  StartDate TIMESTAMP(14),
  EndDate TIMESTAMP(14),
  NeverExpires ENUM('N', 'Y') NOT NULL DEFAULT 'N',
  Combineable ENUM('N', 'Y') NOT NULL DEFAULT 'N',
  DisplayPrecedence INT,
  
  PRIMARY KEY (ID),
  INDEX (Name),
  INDEX (DisplayPrecedence)
);

/*
** Sprawdza, czy uytkownik skorzysta z kuponu rabatowego
** (Denormalizacja: invoice_coupon)
*/
DROP TABLE IF EXISTS coupon_user;
CREATE TABLE coupon_user (
  Coupon INT NOT NULL,
  User INT NOT NULL,
  
  PRIMARY KEY (Coupon, User),
  
  FOREIGN KEY (Coupon) REFERENCES coupon (ID),
  FOREIGN KEY (User) REFERENCES user (ID)
);


/*
** Przyznane kredyty
*/
DROP TABLE IF EXISTS credit;
CREATE TABLE credit (
  ID INT NOT NULL AUTO_INCREMENT,
  ExternalID CHAR(32) NOT NULL,
  Purchaser INT NOT NULL,
  Purchased DATETIME,
  Expires DATETIME,
  InitialValue DECIMAL(11,2) NOT NULL,
  RemainingValue DECIMAL(11,2) NOT NULL,
        
  PRIMARY KEY(ID),
  FOREIGN KEY (Purchaser) REFERENCES User (ID)
);


/*
** Hierarchia kategorii produktw
*/
DROP TABLE IF EXISTS department;
CREATE TABLE department (
  ID INT NOT NULL AUTO_INCREMENT,
  Name CHAR(64) NOT NULL,
  Graphic CHAR(255),
  Parent INT NOT NULL,
  Description blob,
  DisplayPrecedence INT NOT NULL,

  PRIMARY KEY(ID), 
  INDEX (Parent),
  INDEX (DisplayPrecedence, Name),
  
  FOREIGN KEY (Parent) REFERENCES department (ID)
);


/*
** Dobiera pozycje towarowe w rne grupy (kategorie)
*/
DROP TABLE IF EXISTS department_item;
CREATE TABLE department_item (
  Department INT NOT NULL,
  Item INT NOT NULL,

  PRIMARY KEY(Department, Item),

  FOREIGN KEY (Department) REFERENCES department (ID),
  FOREIGN KEY (Item) REFERENCES item (ID)
);


/*
** Rodzaje opat umieszczanych na fakturze
*/
DROP TABLE IF EXISTS fee;
CREATE TABLE fee (
  ID INT NOT NULL AUTO_INCREMENT,
  Name CHAR(64) NOT NULL,
  
  PRIMARY KEY(ID),
  INDEX (Name)
);

/*
** Kontenery na zamwienia umieszczane w systemie
*/
DROP TABLE IF EXISTS invoice;
CREATE TABLE invoice (
  ID INT NOT NULL AUTO_INCREMENT,
  
  User INT NOT NULL,
  Active ENUM('N', 'Y') NOT NULL DEFAULT 'N',
  Created DATETIME NOT NULL,

  PRIMARY KEY (ID),
  INDEX (User),

  FOREIGN KEY (User) REFERENCES user (ID)
);


/*
** Rozliczenie faktury
*/
DROP TABLE IF EXISTS invoice_billing;
CREATE TABLE invoice_billing (
        ID INT NOT NULL AUTO_INCREMENT,
        
        Invoice INT NOT NULL,
        Billing INT NOT NULL,
        Address INT NOT NULL,
        
        # dane z karty
        CreditCardOwner CHAR(64) NOT NULL, 
        CreditCardNumber CHAR(32) NOT NULL,
        CreditCardExpiration DATETIME NOT NULL,
        
        # dodatkowe informacje o posiadaczu karty, niezakodowane
        # w pasku magnetycznym
        CreditCardVerification CHAR(4),

	      PRIMARY KEY(ID),
        INDEX (Invoice),
        
        FOREIGN KEY (Invoice) REFERENCES invoice (ID),
        FOREIGN KEY (Address) REFERENCES address (ID),
        FOREIGN KEY (Billing) REFERENCES billing (ID)
);


/*
** Kupony rabatowe do faktury
** (Denormalizacja: coupon_user)
*/
DROP TABLE IF EXISTS invoice_coupon;
CREATE TABLE invoice_coupon (
  Invoice INT NOT NULL,
  Coupon INT NOT NULL,
  
  PRIMARY KEY (Invoice, Coupon),
  
  FOREIGN KEY (Invoice) REFERENCES invoice (ID),
  FOREIGN KEY (Coupon) REFERENCES coupon (ID)
);


/*
** Opaty na fakturze
*/
DROP TABLE IF EXISTS invoice_fee;
CREATE TABLE invoice_fee (
        Invoice INT NOT NULL,
        Fee INT NOT NULL,
        Value DECIMAL(11,2) NOT NULL,
        
        INDEX (Invoice),
        INDEX (Fee),
        
        FOREIGN KEY (Invoice) REFERENCES invoice (ID),
        FOREIGN KEY (Fee) REFERENCES fee (ID)
);


/*
** Stan faktury (stworzona, wysana)
*/
DROP TABLE IF EXISTS invoice_status;
CREATE TABLE invoice_status (
  ID INT NOT NULL AUTO_INCREMENT,

  Invoice INT NOT NULL,
  Status INT NOT NULL,
  Created DATETIME NOT NULL,
  Description CHAR(255),

  PRIMARY KEY(ID),
  INDEX (Invoice),
  
  FOREIGN KEY (Invoice) REFERENCES invoice (ID),
  FOREIGN KEY (Status) REFERENCES status (ID)
);


/*
** Miejsca wysyki faktury
*/
DROP TABLE IF EXISTS invoice_shipping;
CREATE TABLE invoice_shipping (
  ID INT NOT NULL AUTO_INCREMENT,

  Invoice INT NOT NULL,
  Address INT NOT NULL,
  Shipping INT NOT NULL,
  Message CHAR(255),

  PRIMARY KEY(ID),
  INDEX (Invoice),
  
  FOREIGN KEY (Invoice) REFERENCES invoice (ID),
  FOREIGN KEY (Address) REFERENCES address (ID),
  FOREIGN KEY (Shipping) REFERENCES shipping (ID)
);


/*
** Pozycje towarw (towary na fakturze)
*/
DROP TABLE IF EXISTS invoice_sku;
CREATE TABLE invoice_sku (
  ID INT NOT NULL AUTO_INCREMENT,
  Invoice INT NOT NULL,
  SKU INT NOT NULL,
  Quantity INT NOT NULL,
  ExternalSKU CHAR(64) NOT NULL,
  Name CHAR(64) NOT NULL,
  ListPrice DECIMAL(11,2) NOT NULL,
  SalePrice DECIMAL(11,2) NOT NULL,
  Freight DECIMAL(11,2) NOT NULL,
  Shipping INT NOT NULL,
  GiftWarp char(1) NOT NULL DEFAULT 'N',
  
  PRIMARY KEY(ID),
  INDEX (Invoice),
  INDEX (SKU),
  INDEX (ExternalSKU),
  
  FOREIGN KEY (Invoice) REFERENCES invoice (ID),
  FOREIGN KEY (SKU) REFERENCES sku (ID)
);


/*
** Odmiany dotyczce towarw z poszczeglnych pozycji
*/
DROP TABLE IF EXISTS invoice_sku_variation;
CREATE TABLE invoice_sku_variation (
  InvoiceSKU INT NOT NULL,
  Variation INT NOT NULL,
  Qualifier CHAR(255),
  

  PRIMARY KEY(InvoiceSKU, Variation)
);


/*
** Pozycje w katalogu, przypisane do jednego lub wicej towarw
*/
DROP TABLE IF EXISTS item;
CREATE TABLE item (
  ID INT NOT NULL AUTO_INCREMENT,
  Name CHAR(64) NOT NULL,
  Description BLOB,
  Keywords VARCHAR(255),
  Thumbnail VARCHAR(255),
  Graphic VARCHAR(255),
  LargeGraphic VARCHAR(255),
  DisplayPrecedence INT,
  Active ENUM('N', 'Y') NOT NULL DEFAULT 'N',

  PRIMARY KEY(ID),
  INDEX (Name),
  INDEX (Active)
);


/*
** Dowolne zwizki midzy pozycjami indeksu towarowego
*/
DROP TABLE IF EXISTS item_item;
CREATE TABLE item_item (
  Item INT NOT NULL,
  Related_Item INT NOT NULL,
  Relationship INT NOT NULL,
  
  PRIMARY KEY (Item, Related_Item, Relationship),
  FOREIGN KEY (Item) REFERENCES item (ID),
  FOREIGN KEY (Related_Item) REFERENCES item (ID),
  FOREIGN KEY (Relationship) REFERENCES relationship (ID)
);

/*
** Uprawnienia w systemie
*/
DROP TABLE IF EXISTS permission;
CREATE TABLE permission
(
        ID INT NOT NULL AUTO_INCREMENT,
        Name CHAR(255) NOT NULL,
        
        PRIMARY KEY(ID),
        INDEX (Name)
);


/*
** Rodzaje zwizkw pozycji indeksu towarowego
*/
DROP TABLE IF EXISTS relationship;
CREATE TABLE relationship (
  ID INT NOT NULL AUTO_INCREMENT,
  Name CHAR(255) NOT NULL,
  DisplayPrecedence INT NOT NULL,
  
  PRIMARY KEY(ID),
  INDEX (DisplayPrecedence, Name)
);


/*
** Warto inicjujca generator liczb losowych
*/
DROP TABLE IF EXISTS seed;
CREATE TABLE seed (
        seed INT NOT NULL DEFAULT 314
);


/*
** Sesje uytkownika
*/
DROP TABLE IF EXISTS session;
CREATE TABLE session
(
        ID CHAR(16) NOT NULL,
        User INT,
        LastAction DATETIME,
        Invoice INT,
        
        
        PRIMARY KEY(ID),
	      INDEX (User),
	      INDEX (Invoice),
	      FOREIGN KEY (User) REFERENCES user (ID),
        FOREIGN KEY (Invoice) REFERENCES invoice (ID),
);


/*
** Koszyki (towary dla sesji)
*/
DROP TABLE IF EXISTS session_sku;
CREATE TABLE session_sku (
  ID INT NOT NULL AUTO_INCREMENT,
  Session CHAR(24) NOT NULL,
  SKU INT NOT NULL,
  Quantity INT NOT NULL,
  Notes blob,
  
  PRIMARY KEY(ID),
  INDEX (Session),
  INDEX (SKU),
  
  FOREIGN KEY (Session) REFERENCES session (ID),
  FOREIGN KEY (SKU) REFERENCES sku (ID)
);


/*
** Odmiany towarw z koszyka
*/
DROP TABLE IF EXISTS session_sku_variation;
CREATE TABLE session_sku_variation (
  SessionSKU INT NOT NULL,
  Variation INT NOT NULL,
  Qualifier CHAR(255),

  PRIMARY KEY(SessionSKU, Variation)
);


/*
** Sposb wysyki (UPS, FedEx)
*/
DROP TABLE IF EXISTS shipping;
CREATE TABLE shipping (
  ID INT NOT NULL AUTO_INCREMENT,
  Name CHAR(255) NOT NULL,
  DisplayPrecedence INT NOT NULL,
  
  PRIMARY KEY(ID),
  INDEX (DisplayPrecedence, Name)
);


/*
** Co mona kupi, czyli towary, pogrupowane wg 'pozycji'
*/
DROP TABLE IF EXISTS sku;
CREATE TABLE sku (
  ID INT NOT NULL AUTO_INCREMENT,
  Item INT NOT NULL,
  ExternalSKU CHAR(64) NOT NULL,
  Name CHAR(64) NOT NULL,
  ListPrice DECIMAL(11,2) NOT NULL,
  SalePrice DECIMAL(11,2) NOT NULL,
  AdditionalShipping DECIMAL(11,2) NOT NULL,
  
  DisplayPrecedence INT NOT NULL,
  Active ENUM('N', 'Y') NOT NULL DEFAULT 'N',
  InventoryAvailable INT NOT NULL,
  
  CanBackorder ENUM('N', 'Y') NOT NULL DEFAULT 'N',

  PRIMARY KEY(ID),
  INDEX (Item),
  INDEX (ExternalSKU),
  INDEX (DisplayPrecedence, Name),

  FOREIGN KEY (Item) REFERENCES item (ID)
);


/*
** Odmiany towarw
*/
DROP TABLE IF EXISTS sku_variation;
CREATE TABLE sku_variation (
  SKU INT NOT NULL,
  Variation INT NOT NULL,
  
  PRIMARY KEY (SKU, Variation),
  
  FOREIGN KEY (SKU) REFERENCES sku (ID),
  FOREIGN KEY (Variation) REFERENCES variation (ID)
);


/*
** Rodzaje komunikatw stanu dla faktur
*/
DROP TABLE IF EXISTS status;
CREATE TABLE status (
  ID INT NOT NULL AUTO_INCREMENT,

  Name CHAR(64) NOT NULL,

  PRIMARY KEY(ID),
  INDEX (Name)
);

/*
** Podatki od sprzeday w USA
*/
DROP TABLE IF EXISTS tax;
CREATE TABLE tax (
        State CHAR(2) NOT NULL,
        Rate DECIMAL(4,5) NOT NULL,
        TaxShipping ENUM('N', 'Y') NOT NULL DEFAULT 'N',
        
        PRIMARY KEY(State)
);


/*
** Klienci i administratorzy
*/
DROP TABLE IF EXISTS user;
CREATE TABLE user (
  ID INT NOT NULL AUTO_INCREMENT,
  Login CHAR(32) NOT NULL,
  Password CHAR(32) NOT NULL,
  Address INT NOT NULL,
  
  PRIMARY KEY(ID),
  INDEX (Login),
  INDEX (Address),
  
  FOREIGN KEY (Address) REFERENCES address (ID)
);


/*
** Uprawnienia uytkownikw (login, admin)
*/
DROP TABLE IF EXISTS user_permission;
CREATE TABLE user_permission
(
  User INT NOT NULL,
  Permission INT NOT NULL,
        
  PRIMARY KEY (User, Permission),

  FOREIGN KEY (User) REFERENCES user (ID),
  FOREIGN KEY (Permission) REFERENCES permission (ID)
);


/*
** Lista ycze (towar dla uytkownika)
*/
DROP TABLE IF EXISTS user_sku;
CREATE TABLE user_sku (
  ID INT NOT NULL AUTO_INCREMENT,
  User INT NOT NULL,
  SKU INT NOT NULL,
  Quantity INT NOT NULL,

  PRIMARY KEY(ID),
  INDEX (User),
  INDEX (SKU),
  
  FOREIGN KEY (User) REFERENCES user (ID),
  FOREIGN KEY (SKU) REFERENCES sku (ID)
);


/*
** Odmiany dla pozycji z koszyka
*/
DROP TABLE IF EXISTS user_sku_variation;
CREATE TABLE user_sku_variation (
  UserSKU INT NOT NULL,
  Variation INT NOT NULL,
  Qualifier CHAR(255),

  PRIMARY KEY(UserSKU, Variation)
);


/*
** Moliwe wartoci atrybutw (dla rozmiaru S,M,L)
*/
DROP TABLE IF EXISTS variation;
CREATE TABLE variation (
  ID INT NOT NULL AUTO_INCREMENT,
  Name CHAR(64) NOT NULL,
  Attribute INT NOT NULL,
  Description CHAR(64),
  Graphic CHAR(255),
  DisplayPrecedence INT NOT NULL,
  
  
  PRIMARY KEY(ID),
  INDEX (Attribute),
  INDEX (DisplayPrecedence, Name),
  
  FOREIGN KEY (Attribute) REFERENCES attribute (ID)
);
