package Cookbook::CacheContent;

use Apache;
use Apache::Constants qw(OK SERVER_ERROR DECLINED);
use Apache::File ();

@Cookbook::CacheContent::ISA = qw(Apache);

use strict;

sub disk_cache ($$) {

  my ($self, $r) = @_;

  my $log = $r->server->log;

  my $file = $r->filename;

  # Zamieniamy minuty na doby dla testu -M.
  my $timeout = $self->ttl($r) / (24*60);

  # Sprawdzamy wiek pliku.
  if (-f $r->finfo && -M _ < $timeout) {
    $log->info("Uywamy pliku '$file' z bufora.");
    return DECLINED;
  }

  # Nie ma starego pliku, musimy wygenerowa nowy.
  $log->info("Tworzymy plik '$file'.");

  # Najpierw tworzymy obiekt dania dla naszej klasy Capture poniej.
  my $fake_r = Cookbook::CacheContent::Capture->new($r);

  # Wywoujemy metod handler() podklasy, ale z "faszywym" obiektem
  # dania, abymy mogli przej wygenerowan zawarto.
  $self->handler($fake_r);

  # Tworzymy plik buforujcy.
  my $fh = Apache::File->new(">$file");

  unless ($fh) {
    $log->error("Nie mona otworzy pliku '$file': $!");
    return SERVER_ERROR;
  }

  # Zapisujemy wygenerowan zawarto do pliku.
  print $fh $fake_r->data();

  # Musimy jawnie zamkn plik, gdy inaczej nagwek Content-Length
  # bdzie mia nieprawidow warto.
  $fh->close;

  # Wreszcie ustawiamy nazw pliku na plik buforujcy i pozwalamy
  # domylnemu programowi obsugi serwera Apache wysa jego zawarto.
  $r->filename($file);

  return OK;
}

sub ttl {
  # Podaj czas buforowania w minutach.
  # Domylna warto: 1 godzina.

  return shift->dir_config('CacheTTL') || 60;
}

sub handler {

  my ($self, $r) = @_;

  $r->send_http_header('text/html'); # ignorowane...

  $r->print(" --- brak podklasy! --- ");
}  

package Cookbook::CacheContent::Capture;
# Przechwytujemy wynik dziaania programu obsugi i zapamitujemy go.

@Cookbook::CacheContent::Capture::ISA = qw(Apache);

sub new {

  my ($class, $r) = @_;

  $r ||= Apache->request;

  tie *STDOUT, $class, $r;

  return tied *STDOUT;
}

sub print {
  # Przechwytujemy metod print, aby zapamitywa dane.

  shift->{_data} .= join('', @_);
}

sub data {
  # Zwracamy zapamitane dane.

  return shift->{_data};
}

sub send_http_header {
  # Nic nie robimy - nie wysyamy nagwkw z programu obsugi PerlFixupHandler.
};

sub TIEHANDLE {

  my ($class, $r) = @_;

  return bless { _r    => $r,
                 _data => undef
  }, $class;
}

sub PRINT {
  shift->print(@_);
}
1;
