package Cookbook::CookieAuthentication;

use Apache::Constants qw (OK REDIRECT SERVER_ERROR DECLINED FORBIDDEN);
use Apache::Cookie;
use Apache::Log;
use Apache::Request;

use MIME::Base64 qw(encode_base64 decode_base64);

use strict;

@Cookbook::CookieAuthentication::ISA = qw(Apache::Request);

sub new {
  
  my ($class, $r) = @_;
  
  $r = Apache::Request->new($r);
  
  return bless {r => $r}, $class;
}

sub get_cookie_auth_pw {
  
  my $r = shift;
  
  my $log = $r->server->log;
  
  my $auth_type = $r->auth_type;
  my $auth_name = $r->auth_name;
  
  # Sprawd, czy podano nazw wasnego formularza logowania.
  my $login = $r->dir_config('Login');
  
  unless ($login) {
    $log->error("Nie podano nazwy formularza logowania");
    return SERVER_ERROR;
  }
  
  $r->custom_response(FORBIDDEN, $login);
  
  # Sprawd, czy powinnimy zaj si tym daniem.
  unless (lc($auth_type) eq 'cookie') {
    $log->info("AuthType $auth_type nie jest obsugiwany przez ", ref($r));
    return DECLINED;
  }
  
  # Sprawd, czy ustawiono AuthName.
  unless ($auth_name) {
    $log->error("Nie ustawiono AuthName");
    return SERVER_ERROR;
  }
  
  # Sprbuj pobra autoryzacyjne cookie.
  my %cookiejar = Apache::Cookie->new($r)->parse;
  
  unless ($cookiejar{$auth_name}) {
    $r->note_cookie_auth_failure;
    return FORBIDDEN;
  }
  
  # Pobierz nazw uytkownika i haso z cookie.
  my %auth_cookie = $cookiejar{$auth_name}->value;
  
  my ($user, $password) = split /:/, decode_base64($auth_cookie{Basic});
  
  unless ($user && $password) {
    # Ojej, cookie wrcio bez danych uytkownika.
    
    # Zobacz, czy mamy dane z formularza logowania.
    $user = $r->param('user');
    $password = $r->param('password');
    
    # Nie nadpisuj URI w starym cookie, po prostu powrt.
    return FORBIDDEN unless ($user && $password);
    
    # Mamy jakie dane, wic ustaw cookie autoryzacyjne.
    my @values = (uri => $auth_cookie{uri},
                  Basic => encode_base64(join ":", ($user, $password)),
                 );

    $cookiejar{$auth_name}->value(\@values);
    $cookiejar{$auth_name}->path('/');
    
    $cookiejar{$auth_name}->bake;
    
    # Teraz przekieruj z powrotem do miejsca, gdzie zmierza
    # uytkownik i rozpocznij nowy cykl autoryzacji.
    $r->headers_out->set(Location => $auth_cookie{uri});
    return REDIRECT;
  }
  
  # Otrzymalimy poprawne cookie, wic zwr odpowiednie informacje.
  $r->user($user);
  $r->connection->auth_type($auth_type);
  
  return (OK, $password);
}

sub note_cookie_auth_failure {
  
  my $r = shift;
  
  my $auth_cookie = Apache::Cookie->new($r,
                                        -name => $r->auth_name,
                                        -value => { uri => $r->uri },
                                        -path => '/'
                                       );
  $auth_cookie->bake;
}
1;
