//
//
// Opis: Demonstruje uzycie pomocy online
//
//
unit SimpleHelpViewer;

interface

{ Ten modul nie ma interfejsu! }

implementation

uses
  SysUtils, classes, HelpIntfs,
  { Wymagane dla poprawki GetHelpFileName }
  QForms,
  frmHelpTopic, frmTOC;

const
  STR_VIEWER_NAME = 'Simple Help Viewer';

type
  TSimpleHelpViewer = class (TInterfacedObject, ICustomHelpViewer)
  private
    FHelpManager: IHelpManager;
    FViewerID: Integer;
    FHelpStrings: TStringList;
    FlstHelpFile: TStringList;
    FLastHelpFileName: String;
    procedure InternalShutdown;
    procedure LoadHelpFile;
  public
    constructor Create;
    destructor Destroy; override;
    { ICustomHelpViewer methods }
    function  GetViewerName : String;
    function  UnderstandsKeyword(const HelpString: String): Integer;
    function  GetHelpStrings(const HelpString: String): TStringList;
    function  CanShowTableOfContents : Boolean;
    procedure ShowTableOfContents;
    procedure ShowHelp(const HelpString: String);
    procedure NotifyID(const ViewerID: Integer);
    procedure SoftShutDown;
    procedure ShutDown;
  end;


var
  { HelpViewer jest globalny dla modulu i jest inicjalizowany
    podczas uruchamiania programu }
  HelpViewer: TSimpleHelpViewer;

{
  GetHelpFileName stanowi zastepstwo dla THelpManager.GetHelpFile, gdyz
  Application.GetCurrentHelpFile jest uszkodzony.
}
function GetHelpFileName: String;
var
  ActiveForm: TCustomForm;
begin
  ActiveForm := Screen.ActiveCustomForm;
  if Assigned(ActiveForm) and (ActiveForm.HelpFile <> '') then
    Result := ActiveForm.HelpFile
  else
    Result := Application.HelpFile;
end;


{ TSimpleHelpViewer }
constructor TSimpleHelpViewer.Create;
begin
  inherited Create;
  FHelpStrings := TStringList.Create;
end;

destructor TSimpleHelpViewer.Destroy;
begin
  FHelpStrings.Free;
  inherited Destroy;
end;

{
  Kod finalizujacy modulu wywoluje InternalShutdown w celu zamkniecia
  przegladarki i powiadaomienia menedzera pomocy ze przegladarka
  zostala zamknieta.
}
procedure TSimpleHelpViewer.InternalShutdown;
begin
  if Assigned (FHelpManager) then
    FHelpManager.Release (FViewerID);
  Shutdown;
end;

{
  Menedzer pomocy wywoluje GetViewerName w celu zapewnienia unikalnego
  lancuchowego identyfikatora, ktorego uzytkownicy moga uzywac do
  wybrania z pomiedzy roznych systemow pomocy.
}
function  TSimpleHelpViewer.GetViewerName : String;
begin
  Result := STR_VIEWER_NAME;
end;

{ LoadHelpFile laduje biezacy plik pomocy do TStringList }
procedure TSimpleHelpViewer.LoadHelpFile;
var
  HelpFile: String;
begin
  if FlstHelpFile = nil then
    FlstHelpFile := TStringList.Create;
  { Uzylibysmy FHelpManager.GetHelpFile, ale ta funkcja jest uszkodzona } 
  HelpFile := GetHelpFileName;
  { zabezpieczamy sie przed ponownym zaladowaniem tego samego pliku }
  if HelpFile <> FLastHelpFileName then
  begin
    FlstHelpFile.LoadFromFile (HelpFile);
    FLastHelpFileName := HelpFile;
  end;
end;

{
  Menedzer pomocy wywoluje UnderstandsKeywords w celu zapytania czy
  przegladarka obsluguje dane slowo kluczowe, Przegladarka zwraca
  ilosc tematow zwiazanych z tym slowem kluczowym. Zwrocona wartosc zero
  wskazuje ze przegladarka nie obsluguje tego slowa kluczowego. 
}
function  TSimpleHelpViewer.UnderstandsKeyword(const HelpString: String): Integer;
var
  lstLine: TStringList;
  i : Integer;
begin
  LoadHelpFile;
  lstLine := TStringList.Create;
  try
    FHelpStrings.Clear;
    for i := 0 to FlstHelpFile.Count-1 do
    begin
      lstLine.CommaText := FlstHelpFile[i];
      if lstLine.Count >= 3 then
        if SameText (lstLine[0], HelpString) then
          FHelpStrings.Add (lstLine[1]);
    end;
  finally
    lstLine.Free;
  end;
  Result := FHelpStrings.Count;
end;

{
  Gdy przegladarka oferuje wiecej niz jeden temat dla danego slowa
  kluczowego i menedzer pomocy musi wyswietlic element interfejsu uzytkownika
  pozwalajacy na wybranie tematu, wywoluje GetHelpStrings w celu uzyskania
  listy, budowanej przez przegladarke w funkcji UnderstandsKeywords.

  Menedzer pomocy gwarantuje ze wywola te funkcje tylko po wczesniejszym
  wywolaniu funkcji UnderstandsKeywords, wiec mozemy w niej zbudowac liste,
  ktora tu zwrocimy.
}
function  TSimpleHelpViewer.GetHelpStrings(const HelpString: String): TStringList;
begin
  Result := FHelpStrings;
end;

{
  Menedzer pomocy wywoluje te funkcje w celu zapytania czy przegladarka
  moze wyswietlic spis tresci. Jesli tak, powinna zwrocici wartosc True.
}
function  TSimpleHelpViewer.CanShowTableOfContents : Boolean;
begin
  Result := True;
end;

{
  Menedzer pomocy wywoluje te procedure gdy chce by przegladarka
  wyswietlila spis tresci. Menedzer pomocy wywola te procedure tylko
  wtedy gdy funkcja CanShowTableOfContents zwroci wartosc True.
}
procedure TSimpleHelpViewer.ShowTableOfContents;
begin
  LoadHelpFile;
  ShowTOC (GetHelpFileName, FlstHelpFile);
end;

{
  Menedzer pomocy wywoluje te procedure gdy chce by przegladarka
  wyswietlila pomoc dla konkretnego tematu.
}
procedure TSimpleHelpViewer.ShowHelp(const HelpString: String);
var
  lstLine: TStringList;
  i: Integer;
begin
  LoadHelpFile;
  lstLine := TStringList.Create;
  try
	{ Lokalizujemy pierwsze wystapienie HelpString w pliku }
    for i := 0 to FlstHelpFile.Count-1 do
    begin
      lstLine.CommaText := FlstHelpFile[i];
      if SameText (lstLine[0], HelpString) then
      begin
        frmHelpTopic.ShowTopic (lstLine[1], lstLine[2]);
        exit;
      end;
    end;
  finally
    lstLine.Free;
  end;
end;

{
  Menedzer pomocy wywoluje NotifyID po poprawnej rejestracji przegladarki
  w celu dostarczenia jej unikalnej wartosci identyfikujacej przegladarke
  podczas komunikacji z menedzerem pomocy.
}
procedure TSimpleHelpViewer.NotifyID(const ViewerID: Integer);
begin
  FViewerID := ViewerID;
end;

{
  Menedzer pomocy wywoluje SoftShutDown do zamkniecia wszystkich
  widocznych czesci przegladarki bez zamykania samej przegladarki.
}
procedure TSimpleHelpViewer.SoftShutDown;
begin
  frmTOC.ShutDown;
  frmHelpTopic.ShutDown;
end;

{
  Menedzer pomocy wywoluje ShutDown do zamkniecia przegladarki.
  Poniewaz to menedzer pomocy nakazal zamkniecie przegladarki, nie
  musimy go o tym powiadamiac.
}
procedure TSimpleHelpViewer.ShutDown;
begin
  SoftShutDown;
  if Assigned(FHelpManager) then
    FHelpManager := nil;
end;


initialization
  {
    Tworzymy i inicjalizujemy przegladarke
    po czym rejesrtujemy ja w menederze pomocy.
  }
  if not Assigned(HelpViewer) then
  begin
    HelpViewer := TSimpleHelpViewer.Create;
    HelpIntfs.RegisterViewer(HelpViewer, HelpViewer.FHelpManager);
  end;

finalization
  { Zamykamy i zwalniamy przegladarke. }
  if Assigned(HelpViewer) then
  begin
    HelpViewer.InternalShutDown;
  end;

end.

